// Licensed under the Apache License, Version 2.0
// <LICENSE-APACHE or http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your option.
// All files in the project carrying such notice may not be copied, modified, or distributed
// except according to those terms.
//! This module defines the 32-Bit Windows Device I/O control codes.
use shared::basetsd::DWORD64;
use shared::devpropdef::DEVPROPKEY;
use shared::guiddef::GUID;
use shared::minwindef::{BYTE, DWORD, WORD};
use um::winnt::{
    ANYSIZE_ARRAY, BOOLEAN, FILE_READ_DATA, FILE_WRITE_DATA, HANDLE, LARGE_INTEGER, WCHAR,
};
DEFINE_GUID!{GUID_DEVINTERFACE_DISK,
    0x53f56307, 0xb6bf, 0x11d0, 0x94, 0xf2, 0x00, 0xa0, 0xc9, 0x1e, 0xfb, 0x8b}
DEFINE_GUID!{GUID_DEVINTERFACE_CDROM,
    0x53f56308, 0xb6bf, 0x11d0, 0x94, 0xf2, 0x00, 0xa0, 0xc9, 0x1e, 0xfb, 0x8b}
DEFINE_GUID!{GUID_DEVINTERFACE_PARTITION,
    0x53f5630a, 0xb6bf, 0x11d0, 0x94, 0xf2, 0x00, 0xa0, 0xc9, 0x1e, 0xfb, 0x8b}
DEFINE_GUID!{GUID_DEVINTERFACE_TAPE,
    0x53f5630b, 0xb6bf, 0x11d0, 0x94, 0xf2, 0x00, 0xa0, 0xc9, 0x1e, 0xfb, 0x8b}
DEFINE_GUID!{GUID_DEVINTERFACE_WRITEONCEDISK,
    0x53f5630c, 0xb6bf, 0x11d0, 0x94, 0xf2, 0x00, 0xa0, 0xc9, 0x1e, 0xfb, 0x8b}
DEFINE_GUID!{GUID_DEVINTERFACE_VOLUME,
    0x53f5630d, 0xb6bf, 0x11d0, 0x94, 0xf2, 0x00, 0xa0, 0xc9, 0x1e, 0xfb, 0x8b}
DEFINE_GUID!{GUID_DEVINTERFACE_MEDIUMCHANGER,
    0x53f56310, 0xb6bf, 0x11d0, 0x94, 0xf2, 0x00, 0xa0, 0xc9, 0x1e, 0xfb, 0x8b}
DEFINE_GUID!{GUID_DEVINTERFACE_FLOPPY,
    0x53f56311, 0xb6bf, 0x11d0, 0x94, 0xf2, 0x00, 0xa0, 0xc9, 0x1e, 0xfb, 0x8b}
DEFINE_GUID!{GUID_DEVINTERFACE_CDCHANGER,
    0x53f56312, 0xb6bf, 0x11d0, 0x94, 0xf2, 0x00, 0xa0, 0xc9, 0x1e, 0xfb, 0x8b}
DEFINE_GUID!{GUID_DEVINTERFACE_STORAGEPORT,
    0x2accfe60, 0xc130, 0x11d2, 0xb0, 0x82, 0x00, 0xa0, 0xc9, 0x1e, 0xfb, 0x8b}
DEFINE_GUID!{GUID_DEVINTERFACE_VMLUN,
    0x6f416619, 0x9f29, 0x42a5, 0xb2, 0x0b, 0x37, 0xe2, 0x19, 0xca, 0x02, 0xb0}
DEFINE_GUID!{GUID_DEVINTERFACE_SES,
    0x1790c9ec, 0x47d5, 0x4df3, 0xb5, 0xaf, 0x9a, 0xdf, 0x3c, 0xf2, 0x3e, 0x48}
DEFINE_GUID!{WDI_STORAGE_PREDICT_FAILURE_DPS_GUID,
    0xe9f2d03a, 0x747c, 0x41c2, 0xbb, 0x9a, 0x02, 0xc6, 0x2b, 0x6d, 0x5f, 0xcb}
DEFINE_GUID!{GUID_DEVINTERFACE_SERVICE_VOLUME,
    0x6ead3d82, 0x25ec, 0x46bc, 0xb7, 0xfd, 0xc1, 0xf0, 0xdf, 0x8f, 0x50, 0x37}
DEFINE_GUID!{GUID_DEVINTERFACE_HIDDEN_VOLUME,
    0x7f108a28, 0x9833, 0x4b3b, 0xb7, 0x80, 0x2c, 0x6b, 0x5f, 0xa5, 0xc0, 0x62}
DEFINE_GUID!{GUID_DEVINTERFACE_UNIFIED_ACCESS_RPMB,
    0x27447c21, 0xbcc3, 0x4d07, 0xa0, 0x5b, 0xa3, 0x39, 0x5b, 0xb4, 0xee, 0xe7}
DEFINE_GUID!{GUID_DEVINTERFACE_SCM_PHYSICAL_DEVICE,
    0x4283609d, 0x4dc2, 0x43be, 0xbb, 0xb4, 0x4f, 0x15, 0xdf, 0xce, 0x2c, 0x61}
DEFINE_GUID!{GUID_SCM_PD_HEALTH_NOTIFICATION,
    0x9da2d386, 0x72f5, 0x4ee3, 0x81, 0x55, 0xec, 0xa0, 0x67, 0x8e, 0x3b, 0x06}
DEFINE_GUID!{GUID_SCM_PD_PASSTHROUGH_INVDIMM,
    0x4309AC30, 0x0D11, 0x11E4, 0x91, 0x91, 0x08, 0x00, 0x20, 0x0C, 0x9A, 0x66}
DEFINE_GUID!{GUID_DEVINTERFACE_COMPORT,
    0x86E0D1E0, 0x8089, 0x11D0, 0x9C, 0xE4, 0x08, 0x00, 0x3E, 0x30, 0x1F, 0x73}
DEFINE_GUID!{GUID_DEVINTERFACE_SERENUM_BUS_ENUMERATOR,
    0x4D36E978, 0xE325, 0x11CE, 0xBF, 0xC1, 0x08, 0x00, 0x2B, 0xE1, 0x03, 0x18}
//146
DEFINE_DEVPROPKEY!{DEVPKEY_Storage_Portable,
    0x4d1ebee8, 0x803, 0x4774, 0x98, 0x42, 0xb7, 0x7d, 0xb5, 0x2, 0x65, 0xe9, 2}
DEFINE_DEVPROPKEY!{DEVPKEY_Storage_Removable_Media,
    0x4d1ebee8, 0x803, 0x4774, 0x98, 0x42, 0xb7, 0x7d, 0xb5, 0x2, 0x65, 0xe9, 3}
DEFINE_DEVPROPKEY!{DEVPKEY_Storage_System_Critical,
    0x4d1ebee8, 0x803, 0x4774, 0x98, 0x42, 0xb7, 0x7d, 0xb5, 0x2, 0x65, 0xe9, 4}
//177
pub type DEVICE_TYPE = DWORD;
pub const FILE_DEVICE_BEEP: DEVICE_TYPE = 0x00000001;
pub const FILE_DEVICE_CD_ROM: DEVICE_TYPE = 0x00000002;
pub const FILE_DEVICE_CD_ROM_FILE_SYSTEM: DEVICE_TYPE = 0x00000003;
pub const FILE_DEVICE_CONTROLLER: DEVICE_TYPE = 0x00000004;
pub const FILE_DEVICE_DATALINK: DEVICE_TYPE = 0x00000005;
pub const FILE_DEVICE_DFS: DEVICE_TYPE = 0x00000006;
pub const FILE_DEVICE_DISK: DEVICE_TYPE = 0x00000007;
pub const FILE_DEVICE_DISK_FILE_SYSTEM: DEVICE_TYPE = 0x00000008;
pub const FILE_DEVICE_FILE_SYSTEM: DEVICE_TYPE = 0x00000009;
pub const FILE_DEVICE_INPORT_PORT: DEVICE_TYPE = 0x0000000a;
pub const FILE_DEVICE_KEYBOARD: DEVICE_TYPE = 0x0000000b;
pub const FILE_DEVICE_MAILSLOT: DEVICE_TYPE = 0x0000000c;
pub const FILE_DEVICE_MIDI_IN: DEVICE_TYPE = 0x0000000d;
pub const FILE_DEVICE_MIDI_OUT: DEVICE_TYPE = 0x0000000e;
pub const FILE_DEVICE_MOUSE: DEVICE_TYPE = 0x0000000f;
pub const FILE_DEVICE_MULTI_UNC_PROVIDER: DEVICE_TYPE = 0x00000010;
pub const FILE_DEVICE_NAMED_PIPE: DEVICE_TYPE = 0x00000011;
pub const FILE_DEVICE_NETWORK: DEVICE_TYPE = 0x00000012;
pub const FILE_DEVICE_NETWORK_BROWSER: DEVICE_TYPE = 0x00000013;
pub const FILE_DEVICE_NETWORK_FILE_SYSTEM: DEVICE_TYPE = 0x00000014;
pub const FILE_DEVICE_NULL: DEVICE_TYPE = 0x00000015;
pub const FILE_DEVICE_PARALLEL_PORT: DEVICE_TYPE = 0x00000016;
pub const FILE_DEVICE_PHYSICAL_NETCARD: DEVICE_TYPE = 0x00000017;
pub const FILE_DEVICE_PRINTER: DEVICE_TYPE = 0x00000018;
pub const FILE_DEVICE_SCANNER: DEVICE_TYPE = 0x00000019;
pub const FILE_DEVICE_SERIAL_MOUSE_PORT: DEVICE_TYPE = 0x0000001a;
pub const FILE_DEVICE_SERIAL_PORT: DEVICE_TYPE = 0x0000001b;
pub const FILE_DEVICE_SCREEN: DEVICE_TYPE = 0x0000001c;
pub const FILE_DEVICE_SOUND: DEVICE_TYPE = 0x0000001d;
pub const FILE_DEVICE_STREAMS: DEVICE_TYPE = 0x0000001e;
pub const FILE_DEVICE_TAPE: DEVICE_TYPE = 0x0000001f;
pub const FILE_DEVICE_TAPE_FILE_SYSTEM: DEVICE_TYPE = 0x00000020;
pub const FILE_DEVICE_TRANSPORT: DEVICE_TYPE = 0x00000021;
pub const FILE_DEVICE_UNKNOWN: DEVICE_TYPE = 0x00000022;
pub const FILE_DEVICE_VIDEO: DEVICE_TYPE = 0x00000023;
pub const FILE_DEVICE_VIRTUAL_DISK: DEVICE_TYPE = 0x00000024;
pub const FILE_DEVICE_WAVE_IN: DEVICE_TYPE = 0x00000025;
pub const FILE_DEVICE_WAVE_OUT: DEVICE_TYPE = 0x00000026;
pub const FILE_DEVICE_8042_PORT: DEVICE_TYPE = 0x00000027;
pub const FILE_DEVICE_NETWORK_REDIRECTOR: DEVICE_TYPE = 0x00000028;
pub const FILE_DEVICE_BATTERY: DEVICE_TYPE = 0x00000029;
pub const FILE_DEVICE_BUS_EXTENDER: DEVICE_TYPE = 0x0000002a;
pub const FILE_DEVICE_MODEM: DEVICE_TYPE = 0x0000002b;
pub const FILE_DEVICE_VDM: DEVICE_TYPE = 0x0000002c;
pub const FILE_DEVICE_MASS_STORAGE: DEVICE_TYPE = 0x0000002d;
pub const FILE_DEVICE_SMB: DEVICE_TYPE = 0x0000002e;
pub const FILE_DEVICE_KS: DEVICE_TYPE = 0x0000002f;
pub const FILE_DEVICE_CHANGER: DEVICE_TYPE = 0x00000030;
pub const FILE_DEVICE_SMARTCARD: DEVICE_TYPE = 0x00000031;
pub const FILE_DEVICE_ACPI: DEVICE_TYPE = 0x00000032;
pub const FILE_DEVICE_DVD: DEVICE_TYPE = 0x00000033;
pub const FILE_DEVICE_FULLSCREEN_VIDEO: DEVICE_TYPE = 0x00000034;
pub const FILE_DEVICE_DFS_FILE_SYSTEM: DEVICE_TYPE = 0x00000035;
pub const FILE_DEVICE_DFS_VOLUME: DEVICE_TYPE = 0x00000036;
pub const FILE_DEVICE_SERENUM: DEVICE_TYPE = 0x00000037;
pub const FILE_DEVICE_TERMSRV: DEVICE_TYPE = 0x00000038;
pub const FILE_DEVICE_KSEC: DEVICE_TYPE = 0x00000039;
pub const FILE_DEVICE_FIPS: DEVICE_TYPE = 0x0000003A;
pub const FILE_DEVICE_INFINIBAND: DEVICE_TYPE = 0x0000003B;
pub const FILE_DEVICE_VMBUS: DEVICE_TYPE = 0x0000003E;
pub const FILE_DEVICE_CRYPT_PROVIDER: DEVICE_TYPE = 0x0000003F;
pub const FILE_DEVICE_WPD: DEVICE_TYPE = 0x00000040;
pub const FILE_DEVICE_BLUETOOTH: DEVICE_TYPE = 0x00000041;
pub const FILE_DEVICE_MT_COMPOSITE: DEVICE_TYPE = 0x00000042;
pub const FILE_DEVICE_MT_TRANSPORT: DEVICE_TYPE = 0x00000043;
pub const FILE_DEVICE_BIOMETRIC: DEVICE_TYPE = 0x00000044;
pub const FILE_DEVICE_PMI: DEVICE_TYPE = 0x00000045;
pub const FILE_DEVICE_EHSTOR: DEVICE_TYPE = 0x00000046;
pub const FILE_DEVICE_DEVAPI: DEVICE_TYPE = 0x00000047;
pub const FILE_DEVICE_GPIO: DEVICE_TYPE = 0x00000048;
pub const FILE_DEVICE_USBEX: DEVICE_TYPE = 0x00000049;
pub const FILE_DEVICE_CONSOLE: DEVICE_TYPE = 0x00000050;
pub const FILE_DEVICE_NFP: DEVICE_TYPE = 0x00000051;
pub const FILE_DEVICE_SYSENV: DEVICE_TYPE = 0x00000052;
pub const FILE_DEVICE_VIRTUAL_BLOCK: DEVICE_TYPE = 0x00000053;
pub const FILE_DEVICE_POINT_OF_SERVICE: DEVICE_TYPE = 0x00000054;
pub const FILE_DEVICE_STORAGE_REPLICATION: DEVICE_TYPE = 0x00000055;
pub const FILE_DEVICE_TRUST_ENV: DEVICE_TYPE = 0x00000056;
pub const FILE_DEVICE_UCM: DEVICE_TYPE = 0x00000057;
pub const FILE_DEVICE_UCMTCPCI: DEVICE_TYPE = 0x00000058;
#[inline]
pub fn CTL_CODE(
    DeviceType: DWORD,
    Function: DWORD,
    Method: DWORD,
    Access: DWORD,
) -> DWORD {
    (DeviceType << 16) | (Access << 14) | (Function << 2) | Method
}
//288
pub const METHOD_BUFFERED: DWORD = 0;
pub const METHOD_IN_DIRECT: DWORD = 1;
pub const METHOD_OUT_DIRECT: DWORD = 2;
pub const METHOD_NEITHER: DWORD = 3;
//317
pub const FILE_ANY_ACCESS: DWORD = 0;
pub const FILE_SPECIAL_ACCESS: DWORD = FILE_ANY_ACCESS;
pub const FILE_READ_ACCESS: DWORD = 0x0001;
pub const FILE_WRITE_ACCESS: DWORD = 0x0002;
//347
pub const IOCTL_STORAGE_BASE: DWORD = FILE_DEVICE_MASS_STORAGE;
pub const IOCTL_STORAGE_CHECK_VERIFY: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0200,
    METHOD_BUFFERED, FILE_READ_ACCESS);
pub const IOCTL_STORAGE_CHECK_VERIFY2: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0200,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const IOCTL_STORAGE_MEDIA_REMOVAL: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0201,
    METHOD_BUFFERED, FILE_READ_ACCESS);
pub const IOCTL_STORAGE_EJECT_MEDIA: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0202,
    METHOD_BUFFERED, FILE_READ_ACCESS);
pub const IOCTL_STORAGE_LOAD_MEDIA: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0203,
    METHOD_BUFFERED, FILE_READ_ACCESS);
pub const IOCTL_STORAGE_LOAD_MEDIA2: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0203,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const IOCTL_STORAGE_RESERVE: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0204, METHOD_BUFFERED,
    FILE_READ_ACCESS);
pub const IOCTL_STORAGE_RELEASE: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0205, METHOD_BUFFERED,
    FILE_READ_ACCESS);
pub const IOCTL_STORAGE_FIND_NEW_DEVICES: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0206,
    METHOD_BUFFERED, FILE_READ_ACCESS);
pub const IOCTL_STORAGE_EJECTION_CONTROL: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0250,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const IOCTL_STORAGE_MCN_CONTROL: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0251,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const IOCTL_STORAGE_GET_MEDIA_TYPES: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0300,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const IOCTL_STORAGE_GET_MEDIA_TYPES_EX: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0301,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const IOCTL_STORAGE_GET_MEDIA_SERIAL_NUMBER: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0304,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const IOCTL_STORAGE_GET_HOTPLUG_INFO: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0305,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const IOCTL_STORAGE_SET_HOTPLUG_INFO: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0306,
    METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_STORAGE_RESET_BUS: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0400, METHOD_BUFFERED,
    FILE_READ_ACCESS);
pub const IOCTL_STORAGE_RESET_DEVICE: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0401,
    METHOD_BUFFERED, FILE_READ_ACCESS);
pub const IOCTL_STORAGE_BREAK_RESERVATION: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0405,
    METHOD_BUFFERED, FILE_READ_ACCESS);
pub const IOCTL_STORAGE_PERSISTENT_RESERVE_IN: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0406,
    METHOD_BUFFERED, FILE_READ_ACCESS);
pub const IOCTL_STORAGE_PERSISTENT_RESERVE_OUT: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0407,
    METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_STORAGE_GET_DEVICE_NUMBER: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0420,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const IOCTL_STORAGE_PREDICT_FAILURE: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0440,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const IOCTL_STORAGE_FAILURE_PREDICTION_CONFIG: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0441,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const IOCTL_STORAGE_READ_CAPACITY: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0450,
    METHOD_BUFFERED, FILE_READ_ACCESS);
pub const IOCTL_STORAGE_GET_DEVICE_TELEMETRY: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0470,
    METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_STORAGE_DEVICE_TELEMETRY_NOTIFY: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0471,
    METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_STORAGE_DEVICE_TELEMETRY_QUERY_CAPS: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE,
    0x0472, METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_STORAGE_GET_DEVICE_TELEMETRY_RAW: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0473,
    METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_STORAGE_QUERY_PROPERTY: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0500,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const IOCTL_STORAGE_MANAGE_DATA_SET_ATTRIBUTES: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0501,
    METHOD_BUFFERED, FILE_WRITE_ACCESS);
pub const IOCTL_STORAGE_GET_LB_PROVISIONING_MAP_RESOURCES: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE,
    0x0502, METHOD_BUFFERED, FILE_READ_ACCESS);
pub const IOCTL_STORAGE_GET_BC_PROPERTIES: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0600,
    METHOD_BUFFERED, FILE_READ_ACCESS);
pub const IOCTL_STORAGE_ALLOCATE_BC_STREAM: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0601,
    METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_STORAGE_FREE_BC_STREAM: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0602,
    METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_STORAGE_CHECK_PRIORITY_HINT_SUPPORT: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE,
    0x0620, METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const IOCTL_STORAGE_START_DATA_INTEGRITY_CHECK: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0621,
    METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_STORAGE_STOP_DATA_INTEGRITY_CHECK: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0622,
    METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const OBSOLETE_IOCTL_STORAGE_RESET_BUS: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0400,
    METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const OBSOLETE_IOCTL_STORAGE_RESET_DEVICE: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0401,
    METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_STORAGE_ENABLE_IDLE_POWER: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0720,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const IOCTL_STORAGE_GET_IDLE_POWERUP_REASON: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0721,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const IOCTL_STORAGE_POWER_ACTIVE: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0722,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const IOCTL_STORAGE_POWER_IDLE: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0723,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const IOCTL_STORAGE_EVENT_NOTIFICATION: DWORD = CTL_CODE!(IOCTL_STORAGE_BASE, 0x0724,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
//565
STRUCT!{struct STORAGE_DEVICE_NUMBER {
    DeviceType: DEVICE_TYPE,
    DeviceNumber: DWORD,
    PartitionNumber: DWORD,
}}
pub type PSTORAGE_DEVICE_NUMBER = *mut STORAGE_DEVICE_NUMBER;
STRUCT!{struct STORAGE_DEVICE_NUMBERS {
    NumberOfDevices: DWORD,
    Devices: [STORAGE_DEVICE_NUMBER; ANYSIZE_ARRAY],
}}
pub type PSTORAGE_DEVICE_NUMBERS = *mut STORAGE_DEVICE_NUMBERS;
//1040
ENUM!{enum STORAGE_QUERY_TYPE {
    PropertyStandardQuery = 0,
    PropertyExistsQuery,
    PropertyMaskQuery,
    PropertyQueryMaxDefined,
}}
pub type PSTORAGE_QUERY_TYPE = *mut STORAGE_QUERY_TYPE;
ENUM!{enum STORAGE_PROPERTY_ID {
    StorageDeviceProperty = 0,
    StorageAdapterProperty,
    StorageDeviceIdProperty,
    StorageDeviceUniqueIdProperty,
    StorageDeviceWriteCacheProperty,
    StorageMiniportProperty,
    StorageAccessAlignmentProperty,
    StorageDeviceSeekPenaltyProperty,
    StorageDeviceTrimProperty,
    StorageDeviceWriteAggregationProperty,
    StorageDeviceDeviceTelemetryProperty,
    StorageDeviceLBProvisioningProperty,
    StorageDevicePowerProperty,
    StorageDeviceCopyOffloadProperty,
    StorageDeviceResiliencyProperty,
    StorageDeviceMediumProductType,
    StorageAdapterCryptoProperty,
    StorageDeviceIoCapabilityProperty = 48,
    StorageAdapterProtocolSpecificProperty,
    StorageDeviceProtocolSpecificProperty,
    StorageAdapterTemperatureProperty,
    StorageDeviceTemperatureProperty,
    StorageAdapterPhysicalTopologyProperty,
    StorageDevicePhysicalTopologyProperty,
    StorageDeviceAttributesProperty,
    StorageDeviceManagementStatus,
    StorageAdapterSerialNumberProperty,
    StorageDeviceLocationProperty,
    StorageDeviceNumaProperty,
    StorageDeviceZonedDeviceProperty,
    StorageDeviceUnsafeShutdownCount,
}}
pub type PSTORAGE_PROPERTY_ID = *mut STORAGE_PROPERTY_ID;
STRUCT!{struct STORAGE_PROPERTY_QUERY {
    PropertyId: STORAGE_PROPERTY_ID,
    QueryType: STORAGE_QUERY_TYPE,
    AdditionalParameters: [BYTE; 1],
}}
pub type PSTORAGE_PROPERTY_QUERY = *mut STORAGE_PROPERTY_QUERY;
//1574
STRUCT!{struct DEVICE_TRIM_DESCRIPTOR {
    Version: DWORD,
    Size: DWORD,
    TrimEnabled: BOOLEAN,
}}
pub type PDEVICE_TRIM_DESCRIPTOR = *mut DEVICE_TRIM_DESCRIPTOR;
//7540
pub const IOCTL_DISK_BASE: DWORD = FILE_DEVICE_DISK;
pub const IOCTL_DISK_GET_DRIVE_GEOMETRY: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0000,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const IOCTL_DISK_GET_PARTITION_INFO: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0001,
    METHOD_BUFFERED, FILE_READ_ACCESS);
pub const IOCTL_DISK_SET_PARTITION_INFO: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0002,
    METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_DISK_GET_DRIVE_LAYOUT: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0003,
    METHOD_BUFFERED, FILE_READ_ACCESS);
pub const IOCTL_DISK_SET_DRIVE_LAYOUT: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0004,
    METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_DISK_VERIFY: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0005, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const IOCTL_DISK_FORMAT_TRACKS: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0006, METHOD_BUFFERED,
    FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_DISK_REASSIGN_BLOCKS: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0007, METHOD_BUFFERED,
    FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_DISK_PERFORMANCE: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0008, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const IOCTL_DISK_IS_WRITABLE: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0009, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const IOCTL_DISK_LOGGING: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x000a, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const IOCTL_DISK_FORMAT_TRACKS_EX: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x000b,
    METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_DISK_HISTOGRAM_STRUCTURE: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x000c,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const IOCTL_DISK_HISTOGRAM_DATA: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x000d, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const IOCTL_DISK_HISTOGRAM_RESET: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x000e, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const IOCTL_DISK_REQUEST_STRUCTURE: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x000f,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const IOCTL_DISK_REQUEST_DATA: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0010, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const IOCTL_DISK_PERFORMANCE_OFF: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0018, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const IOCTL_DISK_CONTROLLER_NUMBER: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0011,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const SMART_GET_VERSION: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0020, METHOD_BUFFERED,
    FILE_READ_ACCESS);
pub const SMART_SEND_DRIVE_COMMAND: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0021, METHOD_BUFFERED,
    FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const SMART_RCV_DRIVE_DATA: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0022, METHOD_BUFFERED,
    FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_DISK_GET_PARTITION_INFO_EX: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0012,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const IOCTL_DISK_SET_PARTITION_INFO_EX: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0013,
    METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_DISK_GET_DRIVE_LAYOUT_EX: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0014,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const IOCTL_DISK_SET_DRIVE_LAYOUT_EX: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0015,
    METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_DISK_CREATE_DISK: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0016, METHOD_BUFFERED,
    FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_DISK_GET_LENGTH_INFO: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0017, METHOD_BUFFERED,
    FILE_READ_ACCESS);
pub const IOCTL_DISK_GET_DRIVE_GEOMETRY_EX: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0028,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const IOCTL_DISK_REASSIGN_BLOCKS_EX: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0029,
    METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_DISK_UPDATE_DRIVE_SIZE: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0032,
    METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_DISK_GROW_PARTITION: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0034, METHOD_BUFFERED,
    FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_DISK_GET_CACHE_INFORMATION: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0035,
    METHOD_BUFFERED, FILE_READ_ACCESS);
pub const IOCTL_DISK_SET_CACHE_INFORMATION: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0036,
    METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_DISK_GET_WRITE_CACHE_STATE: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0037,
    METHOD_BUFFERED, FILE_READ_ACCESS);
pub const OBSOLETE_DISK_GET_WRITE_CACHE_STATE: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0037,
    METHOD_BUFFERED, FILE_READ_ACCESS);
pub const IOCTL_DISK_DELETE_DRIVE_LAYOUT: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0040,
    METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_DISK_UPDATE_PROPERTIES: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0050,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const IOCTL_DISK_FORMAT_DRIVE: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x00f3, METHOD_BUFFERED,
    FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_DISK_SENSE_DEVICE: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x00f8, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const IOCTL_DISK_CHECK_VERIFY: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0200, METHOD_BUFFERED,
    FILE_READ_ACCESS);
pub const IOCTL_DISK_MEDIA_REMOVAL: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0201, METHOD_BUFFERED,
    FILE_READ_ACCESS);
pub const IOCTL_DISK_EJECT_MEDIA: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0202, METHOD_BUFFERED,
    FILE_READ_ACCESS);
pub const IOCTL_DISK_LOAD_MEDIA: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0203, METHOD_BUFFERED,
    FILE_READ_ACCESS);
pub const IOCTL_DISK_RESERVE: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0204, METHOD_BUFFERED,
    FILE_READ_ACCESS);
pub const IOCTL_DISK_RELEASE: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0205, METHOD_BUFFERED,
    FILE_READ_ACCESS);
pub const IOCTL_DISK_FIND_NEW_DEVICES: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0206,
    METHOD_BUFFERED, FILE_READ_ACCESS);
pub const IOCTL_DISK_GET_MEDIA_TYPES: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0300, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const IOCTL_DISK_GET_DISK_ATTRIBUTES: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x003c,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const IOCTL_DISK_SET_DISK_ATTRIBUTES: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x003d,
    METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_DISK_RESET_SNAPSHOT_INFO: DWORD = CTL_CODE!(IOCTL_DISK_BASE, 0x0084,
    METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS);
//7810
ENUM!{enum MEDIA_TYPE {
    Unknown,
    F5_1Pt2_512,
    F3_1Pt44_512,
    F3_2Pt88_512,
    F3_20Pt8_512,
    F3_720_512,
    F5_360_512,
    F5_320_512,
    F5_320_1024,
    F5_180_512,
    F5_160_512,
    RemovableMedia,
    FixedMedia,
    F3_120M_512,
    F3_640_512,
    F5_640_512,
    F5_720_512,
    F3_1Pt2_512,
    F3_1Pt23_1024,
    F5_1Pt23_1024,
    F3_128Mb_512,
    F3_230Mb_512,
    F8_256_128,
    F3_200Mb_512,
    F3_240M_512,
    F3_32M_512,
}}
pub type PMEDIA_TYPE = *mut MEDIA_TYPE;
//7884
STRUCT!{struct DISK_GEOMETRY {
    Cylinders: LARGE_INTEGER,
    MediaType: MEDIA_TYPE,
    TracksPerCylinder: DWORD,
    SectorsPerTrack: DWORD,
    BytesPerSector: DWORD,
}}
pub type PDISK_GEOMETRY = *mut DISK_GEOMETRY;
DEFINE_GUID!{WMI_DISK_GEOMETRY_GUID,
    0x25007f51, 0x57c2, 0x11d1, 0xa5, 0x28, 0x00, 0xa0, 0xc9, 0x06, 0x29, 0x10}
STRUCT!{struct PARTITION_INFORMATION {
    StartingOffset: LARGE_INTEGER,
    PartitionLength: LARGE_INTEGER,
    HiddenSectors: DWORD,
    PartitionNumber: DWORD,
    PartitionType: BYTE,
    BootIndicator: BOOLEAN,
    RecognizedPartition: BOOLEAN,
    RewritePartition: BOOLEAN,
}}
pub type PPARTITION_INFORMATION = *mut PARTITION_INFORMATION;
STRUCT!{struct SET_PARTITION_INFORMATION {
    PartitionType: BYTE,
}}
pub type PSET_PARTITION_INFORMATION = *mut SET_PARTITION_INFORMATION;
STRUCT!{struct DRIVE_LAYOUT_INFORMATION {
    PartitionCount: DWORD,
    Signature: DWORD,
    PartitionEntry: [PARTITION_INFORMATION; 1],
}}
pub type PDRIVE_LAYOUT_INFORMATION = *mut DRIVE_LAYOUT_INFORMATION;
STRUCT!{struct VERIFY_INFORMATION {
    StartingOffset: LARGE_INTEGER,
    Length: DWORD,
}}
pub type PVERIFY_INFORMATION = *mut VERIFY_INFORMATION;
STRUCT!{struct REASSIGN_BLOCKS {
    Reserved: WORD,
    Count: WORD,
    BlockNumber: [DWORD; 1],
}}
pub type PREASSIGN_BLOCKS = *mut REASSIGN_BLOCKS;
STRUCT!{#[repr(packed)] struct REASSIGN_BLOCKS_EX {
    Reserved: WORD,
    Count: WORD,
    BlockNumber: [LARGE_INTEGER; 1],
}}
pub type PREASSIGN_BLOCKS_EX = *mut REASSIGN_BLOCKS_EX;
ENUM!{enum PARTITION_STYLE {
    PARTITION_STYLE_MBR,
    PARTITION_STYLE_GPT,
    PARTITION_STYLE_RAW,
}}
STRUCT!{struct PARTITION_INFORMATION_GPT {
    PartitionType: GUID,
    PartitionId: GUID,
    Attributes: DWORD64,
    Name: [WCHAR; 36],
}}
pub type PPARTITION_INFORMATION_GPT = *mut PARTITION_INFORMATION_GPT;
//8059
STRUCT!{struct PARTITION_INFORMATION_MBR {
    PartitionType: BYTE,
    BootIndicator: BOOLEAN,
    RecognizedPartition: BOOLEAN,
    HiddenSectors: DWORD,
    PartitionId: GUID,
}}
pub type PPARTITION_INFORMATION_MBR = *mut PARTITION_INFORMATION_MBR;
pub type SET_PARTITION_INFORMATION_MBR = SET_PARTITION_INFORMATION;
pub type SET_PARTITION_INFORMATION_GPT = PARTITION_INFORMATION_GPT;
STRUCT!{struct SET_PARTITION_INFORMATION_EX {
    PartitionStyle: PARTITION_STYLE,
    u: SET_PARTITION_INFORMATION_EX_u,
}}
UNION!{union SET_PARTITION_INFORMATION_EX_u {
    [u64; 14],
    Mbr Mbr_mut: SET_PARTITION_INFORMATION_MBR,
    Gpt Gpt_mut: SET_PARTITION_INFORMATION_GPT,
}}
STRUCT!{struct CREATE_DISK_GPT {
    DiskId: GUID,
    MaxPartitionCount: DWORD,
}}
pub type PCREATE_DISK_GPT = *mut CREATE_DISK_GPT;
STRUCT!{struct CREATE_DISK_MBR {
    Signature: DWORD,
}}
pub type PCREATE_DISK_MBR = *mut CREATE_DISK_MBR;
STRUCT!{struct CREATE_DISK {
    PartitionStyle: PARTITION_STYLE,
    u: CREATE_DISK_u,
}}
pub type PCREATE_DISK = *mut CREATE_DISK;
UNION!{union CREATE_DISK_u {
    [u32; 5],
    Mbr Mbr_mut: CREATE_DISK_MBR,
    Gpt Gpt_mut: CREATE_DISK_GPT,
}}
STRUCT!{struct GET_LENGTH_INFORMATION {
    Length: LARGE_INTEGER,
}}
pub type PGET_LENGTH_INFORMATION = *mut GET_LENGTH_INFORMATION;
STRUCT!{struct PARTITION_INFORMATION_EX {
    PartitionStyle: PARTITION_STYLE,
    StartingOffset: LARGE_INTEGER,
    PartitionLength: LARGE_INTEGER,
    PartitionNumber: DWORD,
    RewritePartition: BOOLEAN,
    u: PARTITION_INFORMATION_EX_u,
}}
pub type PPARTITION_INFORMATION_EX = *mut PARTITION_INFORMATION_EX;
UNION!{union PARTITION_INFORMATION_EX_u {
    [u64; 14],
    Mbr Mbr_mut: PARTITION_INFORMATION_MBR,
    Gpt Gpt_mut: PARTITION_INFORMATION_GPT,
}}
STRUCT!{struct DRIVE_LAYOUT_INFORMATION_GPT {
    DiskId: GUID,
    StartingUsableOffset: LARGE_INTEGER,
    UsableLength: LARGE_INTEGER,
    MaxPartitionCount: DWORD,
}}
pub type PDRIVE_LAYOUT_INFORMATION_GPT = *mut DRIVE_LAYOUT_INFORMATION_GPT;
STRUCT!{struct DRIVE_LAYOUT_INFORMATION_MBR {
    Signature: DWORD,
    CheckSum: DWORD,
}}
pub type PDRIVE_LAYOUT_INFORMATION_MBR = *mut DRIVE_LAYOUT_INFORMATION_MBR;
STRUCT!{struct DRIVE_LAYOUT_INFORMATION_EX {
    PartitionStyle: DWORD,
    PartitionCount: DWORD,
    u: DRIVE_LAYOUT_INFORMATION_EX_u,
    PartitionEntry: [PARTITION_INFORMATION_EX; 1],
}}
pub type PDRIVE_LAYOUT_INFORMATION_EX = *mut DRIVE_LAYOUT_INFORMATION_EX;
UNION! {union DRIVE_LAYOUT_INFORMATION_EX_u {
    [u64; 5],
    Mbr Mbr_mut: DRIVE_LAYOUT_INFORMATION_MBR,
    Gpt Gpt_mut: DRIVE_LAYOUT_INFORMATION_GPT,
}}
//8350
STRUCT!{struct DISK_GEOMETRY_EX {
    Geometry: DISK_GEOMETRY,
    DiskSize: LARGE_INTEGER,
    Data: [BYTE; 1],
}}
//8933
pub const IOCTL_CHANGER_BASE: DWORD = FILE_DEVICE_CHANGER;
pub const IOCTL_CHANGER_GET_PARAMETERS: DWORD = CTL_CODE!(IOCTL_CHANGER_BASE, 0x0000,
    METHOD_BUFFERED, FILE_READ_ACCESS);
pub const IOCTL_CHANGER_GET_STATUS: DWORD = CTL_CODE!(IOCTL_CHANGER_BASE, 0x0001,
    METHOD_BUFFERED, FILE_READ_ACCESS);
pub const IOCTL_CHANGER_GET_PRODUCT_DATA: DWORD = CTL_CODE!(IOCTL_CHANGER_BASE, 0x0002,
    METHOD_BUFFERED, FILE_READ_ACCESS);
pub const IOCTL_CHANGER_SET_ACCESS: DWORD = CTL_CODE!(IOCTL_CHANGER_BASE, 0x0004,
    METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_CHANGER_GET_ELEMENT_STATUS: DWORD = CTL_CODE!(IOCTL_CHANGER_BASE, 0x0005,
    METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_CHANGER_INITIALIZE_ELEMENT_STATUS: DWORD = CTL_CODE!(IOCTL_CHANGER_BASE, 0x0006,
    METHOD_BUFFERED, FILE_READ_ACCESS);
pub const IOCTL_CHANGER_SET_POSITION: DWORD = CTL_CODE!(IOCTL_CHANGER_BASE, 0x0007,
    METHOD_BUFFERED, FILE_READ_ACCESS);
pub const IOCTL_CHANGER_EXCHANGE_MEDIUM: DWORD = CTL_CODE!(IOCTL_CHANGER_BASE, 0x0008,
    METHOD_BUFFERED, FILE_READ_ACCESS);
pub const IOCTL_CHANGER_MOVE_MEDIUM: DWORD = CTL_CODE!(IOCTL_CHANGER_BASE, 0x0009,
    METHOD_BUFFERED, FILE_READ_ACCESS);
pub const IOCTL_CHANGER_REINITIALIZE_TRANSPORT: DWORD = CTL_CODE!(IOCTL_CHANGER_BASE, 0x000A,
    METHOD_BUFFERED, FILE_READ_ACCESS);
pub const IOCTL_CHANGER_QUERY_VOLUME_TAGS: DWORD = CTL_CODE!(IOCTL_CHANGER_BASE, 0x000B,
    METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_SERIAL_LSRMST_INSERT: DWORD = CTL_CODE!(FILE_DEVICE_SERIAL_PORT, 31,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const IOCTL_SERENUM_EXPOSE_HARDWARE: DWORD = CTL_CODE!(FILE_DEVICE_SERENUM, 128,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const IOCTL_SERENUM_REMOVE_HARDWARE: DWORD = CTL_CODE!(FILE_DEVICE_SERENUM, 129,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const IOCTL_SERENUM_PORT_DESC: DWORD = CTL_CODE!(FILE_DEVICE_SERENUM, 130,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const IOCTL_SERENUM_GET_PORT_NAME: DWORD = CTL_CODE!(FILE_DEVICE_SERENUM, 131,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
//9717
pub const FSCTL_REQUEST_OPLOCK_LEVEL_1: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 0,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_REQUEST_OPLOCK_LEVEL_2: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 1,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_REQUEST_BATCH_OPLOCK: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 2,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_OPLOCK_BREAK_ACKNOWLEDGE: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 3,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_OPBATCH_ACK_CLOSE_PENDING: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 4,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_OPLOCK_BREAK_NOTIFY: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 5,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_LOCK_VOLUME: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 6, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const FSCTL_UNLOCK_VOLUME: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 7, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const FSCTL_DISMOUNT_VOLUME: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 8, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const FSCTL_IS_VOLUME_MOUNTED: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 10,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_IS_PATHNAME_VALID: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 11,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_MARK_VOLUME_DIRTY: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 12,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_QUERY_RETRIEVAL_POINTERS: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 14,
    METHOD_NEITHER, FILE_ANY_ACCESS);
pub const FSCTL_GET_COMPRESSION: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 15, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const FSCTL_SET_COMPRESSION: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 16, METHOD_BUFFERED,
    FILE_READ_DATA | FILE_WRITE_DATA);
pub const FSCTL_SET_BOOTLOADER_ACCESSED: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 19,
    METHOD_NEITHER, FILE_ANY_ACCESS);
pub const FSCTL_MARK_AS_SYSTEM_HIVE: DWORD = FSCTL_SET_BOOTLOADER_ACCESSED;
pub const FSCTL_OPLOCK_BREAK_ACK_NO_2: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 20,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_INVALIDATE_VOLUMES: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 21,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_QUERY_FAT_BPB: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 22, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const FSCTL_REQUEST_FILTER_OPLOCK: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 23,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_FILESYSTEM_GET_STATISTICS: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 24,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_GET_NTFS_VOLUME_DATA: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 25,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_GET_NTFS_FILE_RECORD: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 26,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_GET_VOLUME_BITMAP: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 27, METHOD_NEITHER,
    FILE_ANY_ACCESS);
pub const FSCTL_GET_RETRIEVAL_POINTERS: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 28,
    METHOD_NEITHER, FILE_ANY_ACCESS);
pub const FSCTL_MOVE_FILE: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 29, METHOD_BUFFERED,
    FILE_SPECIAL_ACCESS);
pub const FSCTL_IS_VOLUME_DIRTY: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 30, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const FSCTL_ALLOW_EXTENDED_DASD_IO: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 32,
    METHOD_NEITHER, FILE_ANY_ACCESS);
pub const FSCTL_FIND_FILES_BY_SID: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 35, METHOD_NEITHER,
    FILE_ANY_ACCESS);
pub const FSCTL_SET_OBJECT_ID: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 38, METHOD_BUFFERED,
    FILE_SPECIAL_ACCESS);
pub const FSCTL_GET_OBJECT_ID: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 39, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const FSCTL_DELETE_OBJECT_ID: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 40, METHOD_BUFFERED,
    FILE_SPECIAL_ACCESS);
pub const FSCTL_SET_REPARSE_POINT: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 41,
    METHOD_BUFFERED, FILE_SPECIAL_ACCESS);
pub const FSCTL_GET_REPARSE_POINT: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 42,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_DELETE_REPARSE_POINT: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 43,
    METHOD_BUFFERED, FILE_SPECIAL_ACCESS);
pub const FSCTL_ENUM_USN_DATA: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 44,
    METHOD_NEITHER, FILE_ANY_ACCESS);
pub const FSCTL_SECURITY_ID_CHECK: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 45, METHOD_NEITHER,
    FILE_READ_DATA);
pub const FSCTL_READ_USN_JOURNAL: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 46, METHOD_NEITHER,
    FILE_ANY_ACCESS);
pub const FSCTL_SET_OBJECT_ID_EXTENDED: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 47,
    METHOD_BUFFERED, FILE_SPECIAL_ACCESS);
pub const FSCTL_CREATE_OR_GET_OBJECT_ID: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 48,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_SET_SPARSE: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 49, METHOD_BUFFERED,
    FILE_SPECIAL_ACCESS);
pub const FSCTL_SET_ZERO_DATA: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 50, METHOD_BUFFERED,
    FILE_WRITE_DATA);
pub const FSCTL_QUERY_ALLOCATED_RANGES: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 51,
    METHOD_NEITHER, FILE_READ_DATA);
pub const FSCTL_ENABLE_UPGRADE: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 52, METHOD_BUFFERED,
    FILE_WRITE_DATA);
pub const FSCTL_SET_ENCRYPTION: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 53, METHOD_NEITHER,
    FILE_ANY_ACCESS);
pub const FSCTL_ENCRYPTION_FSCTL_IO: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 54,
    METHOD_NEITHER, FILE_ANY_ACCESS);
pub const FSCTL_WRITE_RAW_ENCRYPTED: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 55,
    METHOD_NEITHER, FILE_SPECIAL_ACCESS);
pub const FSCTL_READ_RAW_ENCRYPTED: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 56,
    METHOD_NEITHER, FILE_SPECIAL_ACCESS);
pub const FSCTL_CREATE_USN_JOURNAL: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 57,
    METHOD_NEITHER, FILE_ANY_ACCESS);
pub const FSCTL_READ_FILE_USN_DATA: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 58,
    METHOD_NEITHER, FILE_ANY_ACCESS);
pub const FSCTL_WRITE_USN_CLOSE_RECORD: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 59,
    METHOD_NEITHER, FILE_ANY_ACCESS);
pub const FSCTL_EXTEND_VOLUME: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 60, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const FSCTL_QUERY_USN_JOURNAL: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 61,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_DELETE_USN_JOURNAL: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 62,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_MARK_HANDLE: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 63, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const FSCTL_SIS_COPYFILE: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 64, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const FSCTL_SIS_LINK_FILES: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 65, METHOD_BUFFERED,
    FILE_READ_DATA | FILE_WRITE_DATA);
pub const FSCTL_RECALL_FILE: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 69, METHOD_NEITHER,
    FILE_ANY_ACCESS);
pub const FSCTL_READ_FROM_PLEX: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 71, METHOD_OUT_DIRECT,
    FILE_READ_DATA);
pub const FSCTL_FILE_PREFETCH: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 72, METHOD_BUFFERED,
    FILE_SPECIAL_ACCESS);
pub const FSCTL_MAKE_MEDIA_COMPATIBLE: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 76,
    METHOD_BUFFERED, FILE_WRITE_DATA);
pub const FSCTL_SET_DEFECT_MANAGEMENT: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 77,
    METHOD_BUFFERED, FILE_WRITE_DATA);
pub const FSCTL_QUERY_SPARING_INFO: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 78,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_QUERY_ON_DISK_VOLUME_INFO: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 79,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_SET_VOLUME_COMPRESSION_STATE: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 80,
    METHOD_BUFFERED, FILE_SPECIAL_ACCESS);
pub const FSCTL_TXFS_MODIFY_RM: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 81, METHOD_BUFFERED,
    FILE_WRITE_DATA);
pub const FSCTL_TXFS_QUERY_RM_INFORMATION: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 82,
    METHOD_BUFFERED, FILE_READ_DATA);
pub const FSCTL_TXFS_ROLLFORWARD_REDO: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 84,
    METHOD_BUFFERED, FILE_WRITE_DATA);
pub const FSCTL_TXFS_ROLLFORWARD_UNDO: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 85,
    METHOD_BUFFERED, FILE_WRITE_DATA);
pub const FSCTL_TXFS_START_RM: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 86, METHOD_BUFFERED,
    FILE_WRITE_DATA);
pub const FSCTL_TXFS_SHUTDOWN_RM: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 87, METHOD_BUFFERED,
    FILE_WRITE_DATA);
pub const FSCTL_TXFS_READ_BACKUP_INFORMATION: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 88,
    METHOD_BUFFERED, FILE_READ_DATA);
pub const FSCTL_TXFS_WRITE_BACKUP_INFORMATION: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 89,
    METHOD_BUFFERED, FILE_WRITE_DATA);
pub const FSCTL_TXFS_CREATE_SECONDARY_RM: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 90,
    METHOD_BUFFERED, FILE_WRITE_DATA);
pub const FSCTL_TXFS_GET_METADATA_INFO: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 91,
    METHOD_BUFFERED, FILE_READ_DATA);
pub const FSCTL_TXFS_GET_TRANSACTED_VERSION: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 92,
    METHOD_BUFFERED, FILE_READ_DATA);
pub const FSCTL_TXFS_SAVEPOINT_INFORMATION: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 94,
    METHOD_BUFFERED, FILE_WRITE_DATA);
pub const FSCTL_TXFS_CREATE_MINIVERSION: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 95,
    METHOD_BUFFERED, FILE_WRITE_DATA);
pub const FSCTL_TXFS_TRANSACTION_ACTIVE: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 99,
    METHOD_BUFFERED, FILE_READ_DATA);
pub const FSCTL_SET_ZERO_ON_DEALLOCATION: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 101,
    METHOD_BUFFERED, FILE_SPECIAL_ACCESS);
pub const FSCTL_SET_REPAIR: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 102, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const FSCTL_GET_REPAIR: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 103, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const FSCTL_WAIT_FOR_REPAIR: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 104, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const FSCTL_INITIATE_REPAIR: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 106, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const FSCTL_CSC_INTERNAL: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 107, METHOD_NEITHER,
    FILE_ANY_ACCESS);
pub const FSCTL_SHRINK_VOLUME: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 108, METHOD_BUFFERED,
    FILE_SPECIAL_ACCESS);
pub const FSCTL_SET_SHORT_NAME_BEHAVIOR: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 109,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_DFSR_SET_GHOST_HANDLE_STATE: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 110,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_TXFS_LIST_TRANSACTION_LOCKED_FILES: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM,
    120, METHOD_BUFFERED, FILE_READ_DATA);
pub const FSCTL_TXFS_LIST_TRANSACTIONS: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 121,
    METHOD_BUFFERED, FILE_READ_DATA);
pub const FSCTL_QUERY_PAGEFILE_ENCRYPTION: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 122,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_RESET_VOLUME_ALLOCATION_HINTS: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 123,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_QUERY_DEPENDENT_VOLUME: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 124,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_SD_GLOBAL_CHANGE: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 125,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_TXFS_READ_BACKUP_INFORMATION2: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 126,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_LOOKUP_STREAM_FROM_CLUSTER: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 127,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_TXFS_WRITE_BACKUP_INFORMATION2: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 128,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_FILE_TYPE_NOTIFICATION: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 129,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_FILE_LEVEL_TRIM: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 130, METHOD_BUFFERED,
    FILE_WRITE_DATA);
pub const FSCTL_GET_BOOT_AREA_INFO: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 140,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_GET_RETRIEVAL_POINTER_BASE: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 141,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_SET_PERSISTENT_VOLUME_STATE: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 142,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_QUERY_PERSISTENT_VOLUME_STATE: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 143,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_REQUEST_OPLOCK: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 144, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const FSCTL_CSV_TUNNEL_REQUEST: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 145,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_IS_CSV_FILE: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 146, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const FSCTL_QUERY_FILE_SYSTEM_RECOGNITION: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 147,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_CSV_GET_VOLUME_PATH_NAME: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 148,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_CSV_GET_VOLUME_NAME_FOR_VOLUME_MOUNT_POINT: DWORD = CTL_CODE!(
    FILE_DEVICE_FILE_SYSTEM, 149, METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_CSV_GET_VOLUME_PATH_NAMES_FOR_VOLUME_NAME: DWORD = CTL_CODE!(
    FILE_DEVICE_FILE_SYSTEM, 150, METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_IS_FILE_ON_CSV_VOLUME: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 151,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_CORRUPTION_HANDLING: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 152,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_OFFLOAD_READ: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 153, METHOD_BUFFERED,
    FILE_READ_ACCESS);
pub const FSCTL_OFFLOAD_WRITE: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 154, METHOD_BUFFERED,
    FILE_WRITE_ACCESS);
pub const FSCTL_CSV_INTERNAL: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 155, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const FSCTL_SET_PURGE_FAILURE_MODE: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 156,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_QUERY_FILE_LAYOUT: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 157,
    METHOD_NEITHER, FILE_ANY_ACCESS);
pub const FSCTL_IS_VOLUME_OWNED_BYCSVFS: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 158,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_GET_INTEGRITY_INFORMATION: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 159,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_SET_INTEGRITY_INFORMATION: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 160,
    METHOD_BUFFERED, FILE_READ_DATA | FILE_WRITE_DATA);
pub const FSCTL_QUERY_FILE_REGIONS: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 161,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_DEDUP_FILE: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 165, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const FSCTL_DEDUP_QUERY_FILE_HASHES: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 166,
    METHOD_NEITHER, FILE_READ_DATA);
pub const FSCTL_DEDUP_QUERY_RANGE_STATE: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 167,
    METHOD_NEITHER, FILE_READ_DATA);
pub const FSCTL_DEDUP_QUERY_REPARSE_INFO: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 168,
    METHOD_NEITHER, FILE_ANY_ACCESS);
pub const FSCTL_RKF_INTERNAL: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 171, METHOD_NEITHER,
    FILE_ANY_ACCESS);
pub const FSCTL_SCRUB_DATA: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 172, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const FSCTL_REPAIR_COPIES: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 173, METHOD_BUFFERED,
    FILE_READ_DATA | FILE_WRITE_DATA);
pub const FSCTL_DISABLE_LOCAL_BUFFERING: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 174,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_CSV_MGMT_LOCK: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 175, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const FSCTL_CSV_QUERY_DOWN_LEVEL_FILE_SYSTEM_CHARACTERISTICS: DWORD = CTL_CODE!(
    FILE_DEVICE_FILE_SYSTEM, 176, METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_ADVANCE_FILE_ID: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 177, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const FSCTL_CSV_SYNC_TUNNEL_REQUEST: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 178,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_CSV_QUERY_VETO_FILE_DIRECT_IO: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 179,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_WRITE_USN_REASON: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 180,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_CSV_CONTROL: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 181, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const FSCTL_GET_REFS_VOLUME_DATA: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 182,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_CSV_H_BREAKING_SYNC_TUNNEL_REQUEST: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM,
    185, METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_QUERY_STORAGE_CLASSES: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 187,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_QUERY_REGION_INFO: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 188,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_USN_TRACK_MODIFIED_RANGES: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 189,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_QUERY_SHARED_VIRTUAL_DISK_SUPPORT: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM,
    192, METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_SVHDX_SYNC_TUNNEL_REQUEST: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 193,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_SVHDX_SET_INITIATOR_INFORMATION: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 194,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_SET_EXTERNAL_BACKING: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 195,
    METHOD_BUFFERED, FILE_SPECIAL_ACCESS);
pub const FSCTL_GET_EXTERNAL_BACKING: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 196,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_DELETE_EXTERNAL_BACKING: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 197,
    METHOD_BUFFERED, FILE_SPECIAL_ACCESS);
pub const FSCTL_ENUM_EXTERNAL_BACKING: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 198,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
pub const FSCTL_ENUM_OVERLAY: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 199, METHOD_NEITHER,
    FILE_ANY_ACCESS);
pub const FSCTL_ADD_OVERLAY: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 204, METHOD_BUFFERED,
    FILE_WRITE_DATA);
pub const FSCTL_REMOVE_OVERLAY: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 205, METHOD_BUFFERED,
    FILE_WRITE_DATA);
pub const FSCTL_UPDATE_OVERLAY: DWORD = CTL_CODE!(FILE_DEVICE_FILE_SYSTEM, 206, METHOD_BUFFERED,
    FILE_WRITE_DATA);
// FILE_DEVICE_AVIO is defined nowhere
//pub const IOCTL_AVIO_ALLOCATE_STREAM: DWORD = CTL_CODE!(FILE_DEVICE_AVIO, 1, METHOD_BUFFERED,
//    FILE_SPECIAL_ACCESS);
//pub const IOCTL_AVIO_FREE_STREAM: DWORD = CTL_CODE!(FILE_DEVICE_AVIO, 2, METHOD_BUFFERED,
//    FILE_SPECIAL_ACCESS);
//pub const IOCTL_AVIO_MODIFY_STREAM: DWORD = CTL_CODE!(FILE_DEVICE_AVIO, 3, METHOD_BUFFERED,
//    FILE_SPECIAL_ACCESS);
STRUCT!{struct PATHNAME_BUFFER {
    PathNameLength: DWORD,
    Name: [WCHAR; 1],
}}
pub type PPATHNAME_BUFFER = *mut PATHNAME_BUFFER;
STRUCT!{struct FSCTL_QUERY_FAT_BPB_BUFFER {
    First0x24BytesOfBootSector: [BYTE; 0x24],
}}
pub type PFSCTL_QUERY_FAT_BPB_BUFFER = *mut FSCTL_QUERY_FAT_BPB_BUFFER;
STRUCT!{struct NTFS_VOLUME_DATA_BUFFER {
    VolumeSerialNumber: LARGE_INTEGER,
    NumberSectors: LARGE_INTEGER,
    TotalClusters: LARGE_INTEGER,
    FreeClusters: LARGE_INTEGER,
    TotalReserved: LARGE_INTEGER,
    BytesPerSector: DWORD,
    BytesPerCluster: DWORD,
    BytesPerFileRecordSegment: DWORD,
    ClustersPerFileRecordSegment: DWORD,
    MftValidDataLength: LARGE_INTEGER,
    MftStartLcn: LARGE_INTEGER,
    Mft2StartLcn: LARGE_INTEGER,
    MftZoneStart: LARGE_INTEGER,
    MftZoneEnd: LARGE_INTEGER,
}}
pub type PNTFS_VOLUME_DATA_BUFFER = *mut NTFS_VOLUME_DATA_BUFFER;
STRUCT!{struct NTFS_EXTENDED_VOLUME_DATA {
    ByteCount: DWORD,
    MajorVersion: WORD,
    MinorVersion: WORD,
    BytesPerPhysicalSector: DWORD,
    LfsMajorVersion: WORD,
    LfsMinorVersion: WORD,
    MaxDeviceTrimExtentCount: DWORD,
    MaxDeviceTrimByteCount: DWORD,
    MaxVolumeTrimExtentCount: DWORD,
    MaxVolumeTrimByteCount: DWORD,
}}
pub type PNTFS_EXTENDED_VOLUME_DATA = *mut NTFS_EXTENDED_VOLUME_DATA;
STRUCT!{struct REFS_VOLUME_DATA_BUFFER {
    ByteCount: DWORD,
    MajorVersion: DWORD,
    MinorVersion: DWORD,
    BytesPerPhysicalSector: DWORD,
    VolumeSerialNumber: LARGE_INTEGER,
    NumberSectors: LARGE_INTEGER,
    TotalClusters: LARGE_INTEGER,
    FreeClusters: LARGE_INTEGER,
    TotalReserved: LARGE_INTEGER,
    BytesPerSector: DWORD,
    BytesPerCluster: DWORD,
    MaximumSizeOfResidentFile: LARGE_INTEGER,
    Reserved: [LARGE_INTEGER; 10],
}}
pub type PREFS_VOLUME_DATA_BUFFER = *mut REFS_VOLUME_DATA_BUFFER;
STRUCT!{struct STARTING_LCN_INPUT_BUFFER {
    StartingLcn: LARGE_INTEGER,
}}
pub type PSTARTING_LCN_INPUT_BUFFER = *mut STARTING_LCN_INPUT_BUFFER;
STRUCT!{struct VOLUME_BITMAP_BUFFER {
    StartingLcn: LARGE_INTEGER,
    BitmapSize: LARGE_INTEGER,
    Buffer: [BYTE; 1],
}}
pub type PVOLUME_BITMAP_BUFFER = *mut VOLUME_BITMAP_BUFFER;
STRUCT!{struct STARTING_VCN_INPUT_BUFFER {
    StartingVcn: LARGE_INTEGER,
}}
pub type PSTARTING_VCN_INPUT_BUFFER = *mut STARTING_VCN_INPUT_BUFFER;
STRUCT!{struct RETRIEVAL_POINTERS_BUFFER_INTERNAL {
    NextVcn: LARGE_INTEGER,
    Lcn: LARGE_INTEGER,
}}
STRUCT!{struct RETRIEVAL_POINTERS_BUFFER {
    ExtentCount: DWORD,
    StartingVcn: LARGE_INTEGER,
    Extents: [RETRIEVAL_POINTERS_BUFFER_INTERNAL; 1],
}}
pub type PRETRIEVAL_POINTERS_BUFFER = *mut RETRIEVAL_POINTERS_BUFFER;
STRUCT!{struct NTFS_FILE_RECORD_INPUT_BUFFER {
    FileReferenceNumber: LARGE_INTEGER,
}}
pub type PNTFS_FILE_RECORD_INPUT_BUFFER = *mut NTFS_FILE_RECORD_INPUT_BUFFER;
STRUCT!{struct NTFS_FILE_RECORD_OUTPUT_BUFFER {
    FileReferenceNumber: LARGE_INTEGER,
    FileRecordLength: DWORD,
    FileRecordBuffer: [BYTE; 1],
}}
pub type PNTFS_FILE_RECORD_OUTPUT_BUFFER = *mut NTFS_FILE_RECORD_OUTPUT_BUFFER;
STRUCT!{struct MOVE_FILE_DATA {
    FileHandle: HANDLE,
    StartingVcn: LARGE_INTEGER,
    StartingLcn: LARGE_INTEGER,
    ClusterCount: DWORD,
}}
pub type PMOVE_FILE_DATA = *mut MOVE_FILE_DATA;
STRUCT!{struct MOVE_FILE_RECORD_DATA {
    FileHandle: HANDLE,
    SourceFileRecord: LARGE_INTEGER,
    TargetFileRecord: LARGE_INTEGER,
}}
pub type PMOVE_FILE_RECORD_DATA = *mut MOVE_FILE_RECORD_DATA;
//15468
pub const IOCTL_VOLUME_BASE: DWORD = 0x00000056;
pub const IOCTL_VOLUME_GET_VOLUME_DISK_EXTENTS: DWORD = CTL_CODE!(IOCTL_VOLUME_BASE, 0,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
STRUCT!{struct DISK_EXTENT {
    DiskNumber: DWORD,
    StartingOffset: LARGE_INTEGER,
    ExtentLength: LARGE_INTEGER,
}}
pub type PDISK_EXTENT = *mut DISK_EXTENT;
STRUCT!{struct VOLUME_DISK_EXTENTS {
    NumberOfDiskExtents: DWORD,
    Extents: [DISK_EXTENT; ANYSIZE_ARRAY],
}}
pub type PVOLUME_DISK_EXTENTS = *mut VOLUME_DISK_EXTENTS;
pub const IOCTL_VOLUME_ONLINE: DWORD = CTL_CODE!(IOCTL_VOLUME_BASE, 2, METHOD_BUFFERED,
    FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_VOLUME_OFFLINE: DWORD = CTL_CODE!(IOCTL_VOLUME_BASE, 3, METHOD_BUFFERED,
    FILE_READ_ACCESS | FILE_WRITE_ACCESS);
pub const IOCTL_VOLUME_IS_CLUSTERED: DWORD = CTL_CODE!(IOCTL_VOLUME_BASE, 12, METHOD_BUFFERED,
    FILE_ANY_ACCESS);
pub const IOCTL_VOLUME_GET_GPT_ATTRIBUTES: DWORD = CTL_CODE!(IOCTL_VOLUME_BASE, 14,
    METHOD_BUFFERED, FILE_ANY_ACCESS);
STRUCT!{struct DISK_PERFORMANCE {
    BytesRead: LARGE_INTEGER,
    BytesWritten: LARGE_INTEGER,
    ReadTime: LARGE_INTEGER,
    WriteTime: LARGE_INTEGER,
    IdleTime: LARGE_INTEGER,
    ReadCount: DWORD,
    WriteCount: DWORD,
    QueueDepth: DWORD,
    SplitCount: DWORD,
    QueryTime: LARGE_INTEGER,
    StorageDeviceNumber: DWORD,
    StorageManagerName: [WCHAR; 8],
}}
pub type PDISK_PERFORMANCE = *mut DISK_PERFORMANCE;
