#include <glib.h>
#include <glib-object.h>
#include <blockdev/utils.h>


#ifndef BD_LVM_API
#define BD_LVM_API

GQuark  bd_lvm_error_quark (void);


#define BD_LVM_ERROR bd_lvm_error_quark ()
typedef enum {
    BD_LVM_ERROR_TECH_UNAVAIL,
    BD_LVM_ERROR_FAIL,
    BD_LVM_ERROR_PARSE,
    BD_LVM_ERROR_NOEXIST,
    BD_LVM_ERROR_DM_ERROR,
    BD_LVM_ERROR_NOT_ROOT,
    BD_LVM_ERROR_CACHE_INVAL,
    BD_LVM_ERROR_CACHE_NOCACHE,
    BD_LVM_ERROR_NOT_SUPPORTED,
    BD_LVM_ERROR_VDO_POLICY_INVAL,
    BD_LVM_ERROR_DEVICES_DISABLED,
} BDLVMError;

typedef enum {
    BD_LVM_CACHE_POOL_STRIPED =  1 << 0,
    BD_LVM_CACHE_POOL_RAID1 =    1 << 1,
    BD_LVM_CACHE_POOL_RAID5 =    1 << 2,
    BD_LVM_CACHE_POOL_RAID6 =    1 << 3,
    BD_LVM_CACHE_POOL_RAID10 =   1 << 4,

    BD_LVM_CACHE_POOL_META_STRIPED =  1 << 10,
    BD_LVM_CACHE_POOL_META_RAID1 =    1 << 11,
    BD_LVM_CACHE_POOL_META_RAID5 =    1 << 12,
    BD_LVM_CACHE_POOL_META_RAID6 =    1 << 13,
    BD_LVM_CACHE_POOL_META_RAID10 =   1 << 14,
} BDLVMCachePoolFlags;

typedef enum {
    BD_LVM_CACHE_MODE_UNKNOWN,
    BD_LVM_CACHE_MODE_WRITETHROUGH,
    BD_LVM_CACHE_MODE_WRITEBACK,
} BDLVMCacheMode;

typedef enum {
    BD_LVM_VDO_MODE_UNKNOWN,
    BD_LVM_VDO_MODE_RECOVERING,
    BD_LVM_VDO_MODE_READ_ONLY,
    BD_LVM_VDO_MODE_NORMAL,
} BDLVMVDOOperatingMode;

typedef enum {
    BD_LVM_VDO_COMPRESSION_UNKNOWN,
    BD_LVM_VDO_COMPRESSION_ONLINE,
    BD_LVM_VDO_COMPRESSION_OFFLINE,
} BDLVMVDOCompressionState;

typedef enum {
    BD_LVM_VDO_INDEX_UNKNOWN,
    BD_LVM_VDO_INDEX_ERROR,
    BD_LVM_VDO_INDEX_CLOSED,
    BD_LVM_VDO_INDEX_OPENING,
    BD_LVM_VDO_INDEX_CLOSING,
    BD_LVM_VDO_INDEX_OFFLINE,
    BD_LVM_VDO_INDEX_ONLINE,
} BDLVMVDOIndexState;

typedef enum {
    BD_LVM_VDO_WRITE_POLICY_UNKNOWN,
    BD_LVM_VDO_WRITE_POLICY_AUTO,
    BD_LVM_VDO_WRITE_POLICY_SYNC,
    BD_LVM_VDO_WRITE_POLICY_ASYNC,
} BDLVMVDOWritePolicy;


#define BD_LVM_TYPE_PVDATA (bd_lvm_pvdata_get_type ())
GType  bd_lvm_pvdata_get_type ();


typedef struct BDLVMPVdata {
    gchar *pv_name;
    gchar *pv_uuid;
    guint64 pv_free;
    guint64 pv_size;
    guint64 pe_start;
    gchar *vg_name;
    gchar *vg_uuid;
    guint64 vg_size;
    guint64 vg_free;
    guint64 vg_extent_size;
    guint64 vg_extent_count;
    guint64 vg_free_count;
    guint64 vg_pv_count;
    gchar **pv_tags;
    gboolean missing;
} BDLVMPVdata;

/**
 * BDLVMPVdata:
 * @pv_name: name of the PV
 * @pv_uuid: UUID of the PV
 * @pv_free: size of the free space in the PV
 * @pv_size: size of the PV
 * @pe_start: start of the physical extents area (i.e. offset of the first PE)
 * @vg_name: name of the VG the PV belongs to
 * @vg_uuid: UUID of the VG the PV belongs to
 * @vg_size: size of the VG the PV belongs to
 * @vg_free: size of the free space in the PV's VG
 * @vg_extent_size: extent size used by the PV's VG
 * @vg_extent_count: number of extents in the PV's VG
 * @vg_free_count: number of free extents in the PV's VG
 * @vg_pv_count: number of PVs that belong to this PV's VG
 * @pv_tags: (array zero-terminated=1): list of LVM tags for this PV
 * @missing: whether this PV is missing from the system or not
 */
/**
 * bd_lvm_pvdata_copy: (skip)
 * @data: (nullable): %BDLVMPVdata to copy
 *
 * Creates a new copy of @data.
 */
BDLVMPVdata* bd_lvm_pvdata_copy (BDLVMPVdata *data);


/**
 * bd_lvm_pvdata_free: (skip)
 * @data: (nullable): %BDLVMPVdata to free
 *
 * Frees @data.
 */
void  bd_lvm_pvdata_free (BDLVMPVdata *data);



#define BD_LVM_TYPE_VGDATA (bd_lvm_vgdata_get_type ())
GType  bd_lvm_vgdata_get_type ();


typedef struct BDLVMVGdata {
    gchar *name;
    gchar *uuid;
    guint64 size;
    guint64 free;
    guint64 extent_size;
    guint64 extent_count;
    guint64 free_count;
    guint64 pv_count;
    gboolean exported;
    gchar **vg_tags;
} BDLVMVGdata;

/**
 * BDLVMVGdata:
 * @name: name of the VG
 * @uuid: UUID of the VG
 * @size: size of the VG
 * @free: size of the free space in the VG
 * @extent_size: extent size used by the VG
 * @extent_count: number of extents in the VG
 * @free_count: number of free extents in the VG
 * @pv_count: number of PVs that belong to the VG
 * @exported: whether the VG is exported or not
 * @vg_tags: (array zero-terminated=1): list of LVM tags for this VG
 */
/**
 * bd_lvm_vgdata_copy: (skip)
 * @data: (nullable): %BDLVMVGdata to copy
 *
 * Creates a new copy of @data.
 */
BDLVMVGdata* bd_lvm_vgdata_copy (BDLVMVGdata *data);


/**
 * bd_lvm_vgdata_free: (skip)
 * @data: (nullable): %BDLVMVGdata to free
 *
 * Frees @data.
 */
void  bd_lvm_vgdata_free (BDLVMVGdata *data);



#define BD_LVM_TYPE_SEGDATA (bd_lvm_segdata_get_type ())
GType  bd_lvm_segdata_get_type ();


typedef struct BDLVMSEGdata {
    guint64 size_pe;
    guint64 pv_start_pe;
    gchar *pvdev;
} BDLVMSEGdata;

/**
 * BDLVMSEGdata:
 * @size_pe: Size of this segment in extents
 * @pv_start_pe: Where it starts in the physical volume in extents
 * @pvdev: Device name of the physical volume or NULL
 */
/**
 * bd_lvm_segdata_copy: (skip)
 * @data: (nullable): %BDLVMSEGdata to copy
 *
 * Creates a new copy of @data.
 */
BDLVMSEGdata* bd_lvm_segdata_copy (BDLVMSEGdata *data);


/**
 * bd_lvm_segdata_free: (skip)
 * @data: (nullable): %BDLVMSEGdata to free
 *
 * Frees @data.
 */
void  bd_lvm_segdata_free (BDLVMSEGdata *data);



static BDLVMSEGdata **copy_segs (BDLVMSEGdata **segs) {
    int i, len;
    BDLVMSEGdata **new_segs;

    if (segs == NULL)
       return NULL;

    for (len = 0; segs[len]; len++)
        ;

    new_segs = g_new0 (BDLVMSEGdata *, len+1);
    for (i = 0; i < len; i++)
        new_segs[i] = bd_lvm_segdata_copy (segs[i]);

    return new_segs;
}

static void free_segs (BDLVMSEGdata **segs) {
    int i;

    if (segs == NULL)
       return;

    for (i = 0; segs[i]; i++)
        bd_lvm_segdata_free (segs[i]);
    (g_free) (segs);
}

#define BD_LVM_TYPE_LVDATA (bd_lvm_lvdata_get_type ())
GType  bd_lvm_lvdata_get_type ();


typedef struct BDLVMLVdata {
    gchar *lv_name;
    gchar *vg_name;
    gchar *uuid;
    guint64 size;
    gchar *attr;
    gchar *segtype;
    gchar *origin;
    gchar *pool_lv;
    gchar *data_lv;
    gchar *metadata_lv;
    gchar *roles;
    gchar *move_pv;
    guint64 data_percent;
    guint64 metadata_percent;
    guint64 copy_percent;
    gchar **lv_tags;
    gchar **data_lvs;
    gchar **metadata_lvs;
    BDLVMSEGdata **segs;
} BDLVMLVdata;

/**
 * BDLVMLVdata:
 * @lv_name: name of the LV
 * @vg_name: name of the VG the LV belongs to
 * @uuid: UUID of the LV
 * @size: size of the LV
 * @attr: attributes of the LV
 * @segtype: segment type of the LV
 * @origin: origin of the LV (for snapshots, etc.)
 * @pool_lv: pool LV of the LV (for thin and cached LVs)
 * @data_lv: data LV of the LV (for thin and cache pools)
 * @metadata_lv: metadata LV of the LV (for thin and cache pools)
 * @roles: comma separated list of this LV's roles
 * @move_pv: source physical volume of a temporary logical volume created with the pvmove command
 * @data_percent: available data space in a thin pool
 * @metadata_percent: available metadata space in a thin pool
 * @copy_percent: synchronization percentage of a mirrored logical volume
 * @lv_tags: (array zero-terminated=1): list of LVM tags for this LV
 * @data_lvs: (nullable) (array zero-terminated=1): list of data sub-LVs this LV (for raids, etc).
 * @metadata_lvs: (nullable) (array zero-terminated=1): list of metadata sub-LVS for this LV (for raids, etc).
 * @segs: (nullable) (array zero-terminated=1): The segments of this LV.
 *
 * The @segs, @data_lvs, and @metadata_lvs fields are only filled by a
 * call to bd_lvm_lvinfo_tree or bd_lvm_lvs_tree.  They are all NULL
 * normally.  If they are filled, they follow these rules:
 *
 * A LV is either stored directly on physical volumes, or is made up
 * of sub-LVs.  If it is stored on PVs, the @segs list is non-empty
 * and the @data_lvs and @metadata_lvs fields are both NULL.  If it is
 * made up of sub-LVs, then @segs is NULL, @data_lvs is non-empty and
 * @metadata_lvs might or might not be empty but is non-NULL.  The
 * sub-LVs contained in @data_lv and @metadata_lv are always included
 * in @data_lvs and @metadata_lvs, respectively.
 *
 * For a partial LV, the @segs list might not be complete; i.e., the
 * sum of the sizes of the listed segments might not be equal to the
 * size reported for the LV itself.
 *
 * Also, the order of entries in @segs must be assumed to be random;
 * it does not correspond to the order of segments in the logical
 * volume itself.
 */
/**
 * bd_lvm_lvdata_copy: (skip)
 * @data: (nullable): %BDLVMLVdata to copy
 *
 * Creates a new copy of @data.
 */
BDLVMLVdata* bd_lvm_lvdata_copy (BDLVMLVdata *data);


/**
 * bd_lvm_lvdata_free: (skip)
 * @data: (nullable): %BDLVMLVdata to free
 *
 * Frees @data.
 */
void  bd_lvm_lvdata_free (BDLVMLVdata *data);



#define BD_LVM_TYPE_VDOPOOLDATA (bd_lvm_vdopooldata_get_type ())
GType  bd_lvm_vdopooldata_get_type ();



typedef struct BDLVMVDOPooldata {
    BDLVMVDOOperatingMode operating_mode;
    BDLVMVDOCompressionState compression_state;
    BDLVMVDOIndexState index_state;
    BDLVMVDOWritePolicy write_policy;
    guint64 used_size;
    gint32 saving_percent;
    guint64 index_memory_size;
    gboolean deduplication;
    gboolean compression;
} BDLVMVDOPooldata;

/**
 * BDLVMVDOPooldata:
 * @operating_mode: operating mode of the VDO pool (e.g. %BD_LVM_VDO_MODE_NORMAL)
 * @compression_state: state of the compression
 * @index_state: state of the VDO index
 * @write_policy: write policy of the VDO LV
 * @used_size: currently used space
 * @saving_percent: percentage of physical blocks saved
 * @index_memory_size: index memory size of the VDO volume
 * @deduplication: whether deduplication is enabled
 * @compression: whether compression is enabled
 */
/**
 * bd_lvm_vdopooldata_copy: (skip)
 * @data: (nullable): %BDLVMVDOPooldata to copy
 *
 * Creates a new copy of @data.
 */
BDLVMVDOPooldata* bd_lvm_vdopooldata_copy (BDLVMVDOPooldata *data);


/**
 * bd_lvm_vdopooldata_free: (skip)
 * @data: (nullable): %BDLVMVDOPooldata to free
 *
 * Frees @data.
 */
void  bd_lvm_vdopooldata_free (BDLVMVDOPooldata *data);



#define BD_LVM_TYPE_VDO_STATS (bd_lvm_vdo_stats_get_type ())
GType  bd_lvm_vdo_stats_get_type ();


typedef struct BDLVMVDOStats {
    gint64 block_size;
    gint64 logical_block_size;
    gint64 physical_blocks;
    gint64 data_blocks_used;
    gint64 overhead_blocks_used;
    gint64 logical_blocks_used;
    gint64 used_percent;
    gint64 saving_percent;
    gdouble write_amplification_ratio;
} BDLVMVDOStats;

/**
 * BDLVMVDOStats:
 * @block_size: The block size of a VDO volume, in bytes.
 * @logical_block_size: The logical block size, in bytes.
 * @physical_blocks: The total number of physical blocks allocated for a VDO volume.
 * @data_blocks_used: The number of physical blocks currently in use by a VDO volume
 *                    to store data.
 * @overhead_blocks_used: The number of physical blocks currently in use by a VDO volume
 *                        to store VDO metadata.
 * @logical_blocks_used: The number of logical blocks currently mapped.
 * @used_percent: The percentage of physical blocks used on a VDO volume
 *                (= used blocks / allocated blocks * 100).
 * @saving_percent: The percentage of physical blocks saved on a VDO volume
 *                  (= [logical blocks used - physical blocks used] / logical blocks used).
 * @write_amplification_ratio: The average number of block writes to the underlying storage
 *                             per block written to the VDO device.
 */
/**
 * bd_lvm_vdo_stats_copy: (skip)
 * @stats: (nullable): %BDLVMVDOStats to copy
 *
 * Creates a new copy of @stats.
 */
BDLVMVDOStats* bd_lvm_vdo_stats_copy (BDLVMVDOStats *stats);


/**
 * bd_lvm_vdo_stats_free: (skip)
 * @stats: (nullable): %BDLVMVDOStats to free
 *
 * Frees @stats.
 */
void  bd_lvm_vdo_stats_free (BDLVMVDOStats *stats);



#define BD_LVM_TYPE_CACHE_STATS (bd_lvm_cache_stats_get_type ())
GType  bd_lvm_cache_stats_get_type ();


typedef struct BDLVMCacheStats {
    guint64 block_size;
    guint64 cache_size;
    guint64 cache_used;
    guint64 md_block_size;
    guint64 md_size;
    guint64 md_used;
    guint64 read_hits;
    guint64 read_misses;
    guint64 write_hits;
    guint64 write_misses;
    BDLVMCacheMode mode;
} BDLVMCacheStats;

/**
 * BDLVMCacheStats:
 * @block_size: block size used by the cache
 * @cache_size: size of the cache
 * @cache_used: size of the used space in the cache
 * @md_block_size: block size used for cache metadata
 * @md_size: size of the metadata space of the cache
 * @md_used: size of the used metadata space in the cache
 * @read_hits: number of read hits
 * @read_misses: number of read misses
 * @write_hits: number of write hits
 * @write_misses: number of write misses
 * @mode: mode the cache is operating in
 */
/**
 * bd_lvm_cache_stats_copy: (skip)
 * @data: (nullable): %BDLVMCacheStats to copy
 *
 * Creates a new copy of @data.
 */
BDLVMCacheStats* bd_lvm_cache_stats_copy (BDLVMCacheStats *data);


/**
 * bd_lvm_cache_stats_free: (skip)
 * @data: (nullable): %BDLVMCacheStats to free
 *
 * Frees @data.
 */
void  bd_lvm_cache_stats_free (BDLVMLVdata *data);



typedef enum {
    BD_LVM_TECH_BASIC = 0,
    BD_LVM_TECH_BASIC_SNAP,
    BD_LVM_TECH_THIN,
    BD_LVM_TECH_CACHE,
    BD_LVM_TECH_CALCS,
    BD_LVM_TECH_THIN_CALCS,
    BD_LVM_TECH_CACHE_CALCS,
    BD_LVM_TECH_GLOB_CONF,
    BD_LVM_TECH_VDO,
    BD_LVM_TECH_WRITECACHE,
    BD_LVM_TECH_DEVICES,
    BD_LVM_TECH_SHARED,
} BDLVMTech;

typedef enum {
    BD_LVM_TECH_MODE_CREATE = 1 << 0,
    BD_LVM_TECH_MODE_REMOVE = 1 << 2,
    BD_LVM_TECH_MODE_MODIFY = 1 << 3,
    BD_LVM_TECH_MODE_QUERY  = 1 << 4,
} BDLVMTechMode;


/**
 * bd_lvm_is_tech_avail:
 * @tech: the queried tech
 * @mode: a bit mask of queried modes of operation (#BDLVMTechMode) for @tech
 * @error: (out) (optional): place to store error (details about why the @tech-@mode combination is not available)
 *
 * Returns: whether the @tech-@mode combination is available -- supported by the
 *          plugin implementation and having all the runtime dependencies available
 */
gboolean  bd_lvm_is_tech_avail (BDLVMTech tech, guint64 mode, GError **error);


/**
 * bd_lvm_is_supported_pe_size:
 * @size: size (in bytes) to test
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the given size is supported physical extent size or not
 *
 * Tech category: %BD_LVM_TECH_CALCS no mode (it is ignored)
 */
gboolean  bd_lvm_is_supported_pe_size (guint64 size, GError **error);


/**
 * bd_lvm_get_supported_pe_sizes:
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: (transfer full) (array fixed-size=25): list of supported PE sizes
 *
 * Tech category: %BD_LVM_TECH_CALCS no mode (it is ignored)
 */
guint64 * bd_lvm_get_supported_pe_sizes (GError **error);


/**
 * bd_lvm_get_max_lv_size:
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: maximum LV size in bytes
 *
 * Tech category: %BD_LVM_TECH_CALCS no mode (it is ignored)
 */
guint64  bd_lvm_get_max_lv_size (GError **error);


/**
 * bd_lvm_round_size_to_pe:
 * @size: size to be rounded
 * @pe_size: physical extent (PE) size or 0 to use the default
 * @roundup: whether to round up or down (ceil or floor)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: @size rounded to @pe_size according to the @roundup
 *
 * Rounds given @size up/down to a multiple of @pe_size according to the value
 * of the @roundup parameter. If the rounded value is too big to fit in the
 * return type, the result is rounded down (floored) regardless of the @roundup
 * parameter.
 *
 * Tech category: %BD_LVM_TECH_CALCS no mode (it is ignored)
 */
guint64  bd_lvm_round_size_to_pe (guint64 size, guint64 pe_size, gboolean roundup, GError **error);


/**
 * bd_lvm_get_lv_physical_size:
 * @lv_size: LV size
 * @pe_size: PE size
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: space taken on disk(s) by the LV with given @size
 *
 * Gives number of bytes needed for an LV with the size @lv_size on an LVM stack
 * using given @pe_size.
 *
 * Tech category: %BD_LVM_TECH_CALCS no mode (it is ignored)
 */
guint64  bd_lvm_get_lv_physical_size (guint64 lv_size, guint64 pe_size, GError **error);


/**
 * bd_lvm_get_thpool_padding:
 * @size: size of the thin pool
 * @pe_size: PE size or 0 if the default value should be used
 * @included: if padding is already included in the size
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: size of the padding needed for a thin pool with the given @size
 *         according to the @pe_size and @included
 *
 * Tech category: %BD_LVM_TECH_THIN_CALCS no mode (it is ignored)
 */
guint64  bd_lvm_get_thpool_padding (guint64 size, guint64 pe_size, gboolean included, GError **error);


/**
 * bd_lvm_get_thpool_meta_size:
 * @size: size of the thin pool
 * @chunk_size: chunk size of the thin pool or 0 to use the default
 * @n_snapshots: ignored
 * @error: (out) (optional): place to store error (if any)
 *
 * Note: This function will be changed in 3.0: the @n_snapshots parameter
 *       is currently not used and will be removed.
 *
 * Returns: recommended size of the metadata space for the specified pool
 *
 * Tech category: %BD_LVM_TECH_THIN_CALCS no mode (it is ignored)
 */
guint64  bd_lvm_get_thpool_meta_size (guint64 size, guint64 chunk_size, guint64 n_snapshots, GError **error);


/**
 * bd_lvm_is_valid_thpool_md_size:
 * @size: the size to be tested
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the given size is a valid thin pool metadata size or not
 *
 * Tech category: %BD_LVM_TECH_THIN_CALCS no mode (it is ignored)
 */
gboolean  bd_lvm_is_valid_thpool_md_size (guint64 size, GError **error);


/**
 * bd_lvm_is_valid_thpool_chunk_size:
 * @size: the size to be tested
 * @discard: whether discard/TRIM is required to be supported or not
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the given size is a valid thin pool chunk size or not
 *
 * Tech category: %BD_LVM_TECH_THIN_CALCS no mode (it is ignored)
 */
gboolean  bd_lvm_is_valid_thpool_chunk_size (guint64 size, gboolean discard, GError **error);


/**
 * bd_lvm_pvcreate:
 * @device: the device to make PV from
 * @data_alignment: data (first PE) alignment or 0 to use the default
 * @metadata_size: size of the area reserved for metadata or 0 to use the default
 * @extra: (nullable) (array zero-terminated=1): extra options for the PV creation
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the PV was successfully created or not
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_CREATE
 */
gboolean  bd_lvm_pvcreate (const gchar *device, guint64 data_alignment, guint64 metadata_size, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_pvresize:
 * @device: the device to resize
 * @size: the new requested size of the PV or 0 if it should be adjusted to device's size
 * @extra: (nullable) (array zero-terminated=1): extra options for the PV resize
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the PV's size was successfully changed or not
 *
 * If given @size different from 0, sets the PV's size to the given value (see
 * pvresize(8)). If given @size 0, adjusts the PV's size to the underlying
 * block device's size.
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_MODIFY
 */
gboolean  bd_lvm_pvresize (const gchar *device, guint64 size, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_pvremove:
 * @device: the PV device to be removed/destroyed
 * @extra: (nullable) (array zero-terminated=1): extra options for the PV removal
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the PV was successfully removed/destroyed or not
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_REMOVE
 */
gboolean  bd_lvm_pvremove (const gchar *device, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_pvmove:
 * @src: the PV device to move extents off of
 * @dest: (nullable): the PV device to move extents onto or %NULL
 * @extra: (nullable) (array zero-terminated=1): extra options for the PV move
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the extents from the @src PV where successfully moved or not
 *
 * If @dest is %NULL, VG allocation rules are used for the extents from the @src
 * PV (see pvmove(8)).
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_MODIFY
 */
gboolean  bd_lvm_pvmove (const gchar *src, const gchar *dest, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_pvscan:
 * @device: (nullable): the device to scan for PVs or %NULL
 * @update_cache: whether to update the lvmetad cache or not
 * @extra: (nullable) (array zero-terminated=1): extra options for the PV scan
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the system or @device was successfully scanned for PVs or not
 *
 * The @device argument is used only if @update_cache is %TRUE. Otherwise the
 * whole system is scanned for PVs.
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_QUERY
 */
gboolean  bd_lvm_pvscan (const gchar *device, gboolean update_cache, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_add_pv_tags:
 * @device: the device to set PV tags for
 * @tags: (array zero-terminated=1): list of tags to add
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the tags were successfully added to @device or not
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_QUERY
 */
gboolean  bd_lvm_add_pv_tags (const gchar *device, const gchar **tags, GError **error);


/**
 * bd_lvm_delete_pv_tags:
 * @device: the device to set PV tags for
 * @tags: (array zero-terminated=1): list of tags to remove
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the tags were successfully removed from @device or not
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_QUERY
 */
gboolean  bd_lvm_delete_pv_tags (const gchar *device, const gchar **tags, GError **error);


/**
 * bd_lvm_pvinfo:
 * @device: a PV to get information about or %NULL
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: (transfer full): information about the PV on the given @device or
 * %NULL in case of error (the @error) gets populated in those cases)
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_QUERY
 */
BDLVMPVdata* bd_lvm_pvinfo (const gchar *device, GError **error);


/**
 * bd_lvm_pvs:
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: (array zero-terminated=1): information about PVs found in the system
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_QUERY
 */
BDLVMPVdata** bd_lvm_pvs (GError **error);


/**
 * bd_lvm_vgcreate:
 * @name: name of the newly created VG
 * @pv_list: (array zero-terminated=1): list of PVs the newly created VG should use
 * @pe_size: PE size or 0 if the default value should be used
 * @extra: (nullable) (array zero-terminated=1): extra options for the VG creation
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the VG @name was successfully created or not
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_CREATE
 */
gboolean  bd_lvm_vgcreate (const gchar *name, const gchar **pv_list, guint64 pe_size, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_vgremove:
 * @vg_name: name of the to be removed VG
 * @extra: (nullable) (array zero-terminated=1): extra options for the VG removal
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the VG was successfully removed or not
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_REMOVE
 */
gboolean  bd_lvm_vgremove (const gchar *vg_name, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_vgrename:
 * @old_vg_name: old name of the VG to rename
 * @new_vg_name: new name for the @old_vg_name VG
 * @extra: (nullable) (array zero-terminated=1): extra options for the VG rename
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the VG was successfully renamed or not
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_MODIFY
 */
gboolean  bd_lvm_vgrename (const gchar *old_vg_name, const gchar *new_vg_name, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_vgactivate:
 * @vg_name: name of the to be activated VG
 * @extra: (nullable) (array zero-terminated=1): extra options for the VG activation
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the VG was successfully activated or not
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_MODIFY
 */
gboolean  bd_lvm_vgactivate (const gchar *vg_name, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_vgdeactivate:
 * @vg_name: name of the to be deactivated VG
 * @extra: (nullable) (array zero-terminated=1): extra options for the VG deactivation
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the VG was successfully deactivated or not
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_MODIFY
 */
gboolean  bd_lvm_vgdeactivate (const gchar *vg_name, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_vgextend:
 * @vg_name: name of the to be extended VG
 * @device: PV device to extend the @vg_name VG with
 * @extra: (nullable) (array zero-terminated=1): extra options for the VG extension
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the VG @vg_name was successfully extended with the given @device or not.
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_MODIFY
 */
gboolean  bd_lvm_vgextend (const gchar *vg_name, const gchar *device, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_vgreduce:
 * @vg_name: name of the to be reduced VG
 * @device: (nullable): PV device the @vg_name VG should be reduced of or %NULL
 *                        if the VG should be reduced of the missing PVs
 * @extra: (nullable) (array zero-terminated=1): extra options for the VG reduction
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the VG @vg_name was successfully reduced of the given @device or not
 *
 * Note: This function does not move extents off of the PV before removing
 *       it from the VG. You must do that first by calling #bd_lvm_pvmove.
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_MODIFY
 */
gboolean  bd_lvm_vgreduce (const gchar *vg_name, const gchar *device, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_add_vg_tags:
 * @vg_name: the VG to set tags on
 * @tags: (array zero-terminated=1): list of tags to add
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the tags were successfully added to @vg_name or not
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_QUERY
 */
gboolean  bd_lvm_add_vg_tags (const gchar *vg_name, const gchar **tags, GError **error);


/**
 * bd_lvm_delete_vg_tags:
 * @vg_name: the VG to set tags on
 * @tags: (array zero-terminated=1): list of tags to remove
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the tags were successfully removed from @vg_name or not
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_QUERY
 */
gboolean  bd_lvm_delete_vg_tags (const gchar *vg_name, const gchar **tags, GError **error);


/**
 * bd_lvm_vglock_start:
 * @vg_name: a shared VG to start the lockspace in lvmlockd
 * @extra: (nullable) (array zero-terminated=1): extra options for the vgchange command
 *                                               (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the lock was successfully started for @vg_name or not
 *
 * Tech category: %BD_LVM_TECH_SHARED-%BD_LVM_TECH_MODE_MODIFY
 */
gboolean  bd_lvm_vglock_start (const gchar *vg_name, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_vglock_stop:
 * @vg_name: a shared VG to stop the lockspace in lvmlockd
 * @extra: (nullable) (array zero-terminated=1): extra options for the vgchange command
 *                                               (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the lock was successfully stopped for @vg_name or not
 *
 * Tech category: %BD_LVM_TECH_SHARED-%BD_LVM_TECH_MODE_MODIFY
 */
gboolean  bd_lvm_vglock_stop (const gchar *vg_name, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_vginfo:
 * @vg_name: a VG to get information about
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: (transfer full): information about the @vg_name VG or %NULL in case
 * of error (the @error) gets populated in those cases)
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_QUERY
 */
BDLVMVGdata* bd_lvm_vginfo (const gchar *vg_name, GError **error);


/**
 * bd_lvm_vgs:
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: (array zero-terminated=1): information about VGs found in the system
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_QUERY
 */
BDLVMVGdata** bd_lvm_vgs (GError **error);


/**
 * bd_lvm_lvorigin:
 * @vg_name: name of the VG containing the queried LV
 * @lv_name: name of the queried LV
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: (transfer full): the origin volume for the @vg_name/@lv_name LV or
 * %NULL if failed to determine (@error) is set in those cases)
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_QUERY
 */
gchar* bd_lvm_lvorigin (const gchar *vg_name, const gchar *lv_name, GError **error);


/**
 * bd_lvm_lvcreate:
 * @vg_name: name of the VG to create a new LV in
 * @lv_name: name of the to-be-created LV
 * @size: requested size of the new LV
 * @type: (nullable): type of the new LV ("striped", "raid1",..., see lvcreate (8))
 * @pv_list: (nullable) (array zero-terminated=1): list of PVs the newly created LV should use or %NULL
 * if not specified
 * @extra: (nullable) (array zero-terminated=1): extra options for the LV creation
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the given @vg_name/@lv_name LV was successfully created or not
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_CREATE
 */
gboolean  bd_lvm_lvcreate (const gchar *vg_name, const gchar *lv_name, guint64 size, const gchar *type, const gchar **pv_list, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_lvremove:
 * @vg_name: name of the VG containing the to-be-removed LV
 * @lv_name: name of the to-be-removed LV
 * @force: whether to force removal or not
 * @extra: (nullable) (array zero-terminated=1): extra options for the LV removal
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the @vg_name/@lv_name LV was successfully removed or not
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_REMOVE
 */
gboolean  bd_lvm_lvremove (const gchar *vg_name, const gchar *lv_name, gboolean force, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_lvrename:
 * @vg_name: name of the VG containing the to-be-renamed LV
 * @lv_name: name of the to-be-renamed LV
 * @new_name: new name for the @vg_name/@lv_name LV
 * @extra: (nullable) (array zero-terminated=1): extra options for the LV rename
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the @vg_name/@lv_name LV was successfully renamed to
 * @vg_name/@new_name or not
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_MODIFY
 */
gboolean  bd_lvm_lvrename (const gchar *vg_name, const gchar *lv_name, const gchar *new_name, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_lvresize:
 * @vg_name: name of the VG containing the to-be-resized LV
 * @lv_name: name of the to-be-resized LV
 * @size: the requested new size of the LV
 * @extra: (nullable) (array zero-terminated=1): extra options for the LV resize
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the @vg_name/@lv_name LV was successfully resized or not
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_MODIFY
 */
gboolean  bd_lvm_lvresize (const gchar *vg_name, const gchar *lv_name, guint64 size, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_lvrepair:
 * @vg_name: name of the VG containing the to-be-repaired LV
 * @lv_name: name of the to-be-repaired LV
 * @pv_list: (array zero-terminated=1): list of PVs to be used for the repair
 * @extra: (nullable) (array zero-terminated=1): extra options for the LV repair
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the @vg_name/@lv_name LV was successfully repaired or not
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_MODIFY
 */
gboolean  bd_lvm_lvrepair (const gchar *vg_name, const gchar *lv_name, const gchar **pv_list, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_lvactivate:
 * @vg_name: name of the VG containing the to-be-activated LV
 * @lv_name: name of the to-be-activated LV
 * @ignore_skip: whether to ignore the skip flag or not
 * @shared: whether to activate the LV in shared mode (used for shared LVM setups with lvmlockd,
 *          use %FALSE if not sure)
 * @extra: (nullable) (array zero-terminated=1): extra options for the LV activation
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the @vg_name/@lv_name LV was successfully activated or not
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_MODIFY
 */
gboolean  bd_lvm_lvactivate (const gchar *vg_name, const gchar *lv_name, gboolean ignore_skip, gboolean shared, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_lvdeactivate:
 * @vg_name: name of the VG containing the to-be-deactivated LV
 * @lv_name: name of the to-be-deactivated LV
 * @extra: (nullable) (array zero-terminated=1): extra options for the LV deactivation
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the @vg_name/@lv_name LV was successfully deactivated or not
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_MODIFY
 */
gboolean  bd_lvm_lvdeactivate (const gchar *vg_name, const gchar *lv_name, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_lvsnapshotcreate:
 * @vg_name: name of the VG containing the LV a new snapshot should be created of
 * @origin_name: name of the LV a new snapshot should be created of
 * @snapshot_name: name of the to-be-created snapshot
 * @size: requested size for the snapshot
 * @extra: (nullable) (array zero-terminated=1): extra options for the LV snapshot creation
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the @snapshot_name snapshot of the @vg_name/@origin_name LV
 * was successfully created or not.
 *
 * Tech category: %BD_LVM_TECH_BASIC_SNAP-%BD_LVM_TECH_MODE_CREATE
 */
gboolean  bd_lvm_lvsnapshotcreate (const gchar *vg_name, const gchar *origin_name, const gchar *snapshot_name, guint64 size, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_lvsnapshotmerge:
 * @vg_name: name of the VG containing the to-be-merged LV snapshot
 * @snapshot_name: name of the to-be-merged LV snapshot
 * @extra: (nullable) (array zero-terminated=1): extra options for the LV snapshot merge
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the @vg_name/@snapshot_name LV snapshot was successfully merged or not
 *
 * Tech category: %BD_LVM_TECH_BASIC_SNAP-%BD_LVM_TECH_MODE_MODIFY
 */
gboolean  bd_lvm_lvsnapshotmerge (const gchar *vg_name, const gchar *snapshot_name, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_add_lv_tags:
 * @vg_name: name of the VG that contains the LV to set tags on
 * @lv_name: name of the LV to set tags on
 * @tags: (array zero-terminated=1): list of tags to add
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the tags were successfully added to @device or not
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_QUERY
 */
gboolean  bd_lvm_add_lv_tags (const gchar *vg_name, const gchar *lv_name, const gchar **tags, GError **error);


/**
 * bd_lvm_delete_lv_tags:
 * @vg_name: name of the VG that contains the LV to set tags on
 * @lv_name: name of the LV to set tags on
 * @tags: (array zero-terminated=1): list of tags to remove
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the tags were successfully removed from @device or not
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_QUERY
 */
gboolean  bd_lvm_delete_lv_tags (const gchar *vg_name, const gchar *lv_name, const gchar **tags, GError **error);


/**
 * bd_lvm_lvinfo:
 * @vg_name: name of the VG that contains the LV to get information about
 * @lv_name: name of the LV to get information about
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: (transfer full): information about the @vg_name/@lv_name LV or %NULL in case
 * of error (the @error) gets populated in those cases)
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_QUERY
 */
BDLVMLVdata* bd_lvm_lvinfo (const gchar *vg_name, const gchar *lv_name, GError **error);


/**
 * bd_lvm_lvinfo_tree:
 * @vg_name: name of the VG that contains the LV to get information about
 * @lv_name: name of the LV to get information about
 * @error: (out) (optional): place to store error (if any)
 *
 * This function will fill out the data_lvs, metadata_lvs, and segs fields as well.
 *
 * Returns: (transfer full): information about the @vg_name/@lv_name LV or %NULL in case
 * of error (the @error) gets populated in those cases)
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_QUERY
 */
BDLVMLVdata* bd_lvm_lvinfo_tree (const gchar *vg_name, const gchar *lv_name, GError **error);


/**
 * bd_lvm_lvs:
 * @vg_name: (nullable): name of the VG to get information about LVs from
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: (array zero-terminated=1): information about LVs found in the given
 * @vg_name VG or in system if @vg_name is %NULL
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_QUERY
 */
BDLVMLVdata** bd_lvm_lvs (const gchar *vg_name, GError **error);


/**
 * bd_lvm_lvs_tree:
 * @vg_name: (nullable): name of the VG to get information about LVs from
 * @error: (out) (optional): place to store error (if any)
 *
 * This function will fill out the data_lvs, metadata_lvs, and segs fields as well.
 *
 * Returns: (array zero-terminated=1): information about LVs found in the given
 * @vg_name VG or in system if @vg_name is %NULL.
 *
 * Tech category: %BD_LVM_TECH_BASIC-%BD_LVM_TECH_MODE_QUERY
 */
BDLVMLVdata** bd_lvm_lvs_tree (const gchar *vg_name, GError **error);


/**
 * bd_lvm_thpoolcreate:
 * @vg_name: name of the VG to create a thin pool in
 * @lv_name: name of the to-be-created pool LV
 * @size: requested size of the to-be-created pool
 * @md_size: requested metadata size or 0 to use the default
 * @chunk_size: requested chunk size or 0 to use the default
 * @profile: (nullable): profile to use (see lvm(8) for more information) or %NULL to use
 *                         the default
 * @extra: (nullable) (array zero-terminated=1): extra options for the thin pool creation
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the @vg_name/@lv_name thin pool was successfully created or not
 *
 * Tech category: %BD_LVM_TECH_THIN-%BD_LVM_TECH_MODE_CREATE
 */
gboolean  bd_lvm_thpoolcreate (const gchar *vg_name, const gchar *lv_name, guint64 size, guint64 md_size, guint64 chunk_size, const gchar *profile, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_thlvcreate:
 * @vg_name: name of the VG containing the thin pool providing extents for the to-be-created thin LV
 * @pool_name: name of the pool LV providing extents for the to-be-created thin LV
 * @lv_name: name of the to-be-created thin LV
 * @size: requested virtual size of the to-be-created thin LV
 * @extra: (nullable) (array zero-terminated=1): extra options for the thin LV creation
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the @vg_name/@lv_name thin LV was successfully created or not
 *
 * Tech category: %BD_LVM_TECH_THIN-%BD_LVM_TECH_MODE_CREATE
 */
gboolean  bd_lvm_thlvcreate (const gchar *vg_name, const gchar *pool_name, const gchar *lv_name, guint64 size, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_thlvpoolname:
 * @vg_name: name of the VG containing the queried thin LV
 * @lv_name: name of the queried thin LV
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: (transfer full): the name of the pool volume for the @vg_name/@lv_name
 * thin LV or %NULL if failed to determine (@error) is set in those cases)
 *
 * Tech category: %BD_LVM_TECH_THIN-%BD_LVM_TECH_MODE_QUERY
 */
gchar* bd_lvm_thlvpoolname (const gchar *vg_name, const gchar *lv_name, GError **error);


/**
 * bd_lvm_thsnapshotcreate:
 * @vg_name: name of the VG containing the thin LV a new snapshot should be created of
 * @origin_name: name of the thin LV a new snapshot should be created of
 * @snapshot_name: name of the to-be-created snapshot
 * @pool_name: (nullable): name of the thin pool to create the snapshot in or %NULL if not specified
 * @extra: (nullable) (array zero-terminated=1): extra options for the thin LV snapshot creation
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the @snapshot_name snapshot of the @vg_name/@origin_name
 * thin LV was successfully created or not.
 *
 * Tech category: %BD_LVM_TECH_THIN-%BD_LVM_TECH_MODE_CREATE
 */
gboolean  bd_lvm_thsnapshotcreate (const gchar *vg_name, const gchar *origin_name, const gchar *snapshot_name, const gchar *pool_name, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_set_global_config:
 * @new_config: (nullable): string representation of the new global LVM
 *                            configuration to set or %NULL to reset to default
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the new requested global config @new_config was successfully
 *          set or not
 *
 * Tech category: %BD_LVM_TECH_GLOB_CONF no mode (it is ignored)
 */
gboolean  bd_lvm_set_global_config (const gchar *new_config, GError **error);


/**
 * bd_lvm_get_global_config:
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: a copy of a string representation of the currently set LVM global
 *          configuration
 *
 * Tech category: %BD_LVM_TECH_GLOB_CONF no mode (it is ignored)
 */
gchar* bd_lvm_get_global_config (GError **error);


/**
 * bd_lvm_set_devices_filter:
 * @devices: (nullable) (array zero-terminated=1): list of devices for lvm commands to work on
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the devices filter was successfully set or not
 *
 * Tech category: %BD_LVM_TECH_DEVICES no mode (it is ignored)
 */
gboolean  bd_lvm_set_devices_filter (const gchar **devices, GError **error);


/**
 * bd_lvm_get_devices_filter:
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: (transfer full) (array zero-terminated=1): a copy of a string representation of
 *                                                     the currently set LVM devices filter
 *
 * Tech category: %BD_LVM_TECH_DEVICES no mode (it is ignored)
 */
gchar** bd_lvm_get_devices_filter (GError **error);


/**
 * bd_lvm_cache_get_default_md_size:
 * @cache_size: size of the cache to determine MD size for
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: recommended default size of the cache metadata LV or 0 in case of error
 *
 * Tech category: %BD_LVM_TECH_CACHE_CALCS no mode (it is ignored)
 */
guint64  bd_lvm_cache_get_default_md_size (guint64 cache_size, GError **error);


/**
 * bd_lvm_cache_get_mode_str:
 * @mode: mode to get the string representation for
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: string representation of @mode or %NULL in case of error
 *
 * Tech category: always provided/supported
 */
const gchar* bd_lvm_cache_get_mode_str (BDLVMCacheMode mode, GError **error);


/**
 * bd_lvm_cache_get_mode_from_str:
 * @mode_str: string representation of a cache mode
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: cache mode for the @mode_str or %BD_LVM_CACHE_MODE_UNKNOWN if
 *          failed to determine
 *
 * Tech category: always provided/supported
 */
BDLVMCacheMode  bd_lvm_cache_get_mode_from_str (const gchar *mode_str, GError **error);


/**
 * bd_lvm_cache_create_pool:
 * @vg_name: name of the VG to create @pool_name in
 * @pool_name: name of the cache pool LV to create
 * @pool_size: desired size of the cache pool @pool_name
 * @md_size: desired size of the @pool_name cache pool's metadata LV or 0 to
 *           use the default
 * @mode: cache mode of the @pool_name cache pool
 * @flags: a combination of (ORed) #BDLVMCachePoolFlags
 * @fast_pvs: (array zero-terminated=1): list of (fast) PVs to create the @pool_name
 *                                       cache pool (and the metadata LV)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the cache pool @vg_name/@pool_name was successfully created or not
 *
 * Tech category: %BD_LVM_TECH_CACHE-%BD_LVM_TECH_MODE_CREATE
 */
gboolean  bd_lvm_cache_create_pool (const gchar *vg_name, const gchar *pool_name, guint64 pool_size, guint64 md_size, BDLVMCacheMode mode, BDLVMCachePoolFlags flags, const gchar **fast_pvs, GError **error);


/**
 * bd_lvm_cache_attach:
 * @vg_name: name of the VG containing the @data_lv and the @cache_pool_lv LVs
 * @data_lv: data LV to attach the @cache_pool_lv to
 * @cache_pool_lv: cache pool LV to attach to the @data_lv
 * @extra: (nullable) (array zero-terminated=1): extra options for the cache attachment
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the @cache_pool_lv was successfully attached to the @data_lv or not
 *
 * Note: Both @data_lv and @cache_lv will be deactivated before the operation.
 *
 * Tech category: %BD_LVM_TECH_CACHE-%BD_LVM_TECH_MODE_MODIFY
 */
gboolean  bd_lvm_cache_attach (const gchar *vg_name, const gchar *data_lv, const gchar *cache_pool_lv, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_cache_detach:
 * @vg_name: name of the VG containing the @cached_lv
 * @cached_lv: name of the cached LV to detach its cache from
 * @destroy: whether to destroy the cache after detach or not
 * @extra: (nullable) (array zero-terminated=1): extra options for the cache detachment
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the cache was successfully detached from the @cached_lv or not
 *
 * Note: synces the cache first
 *
 * Tech category: %BD_LVM_TECH_CACHE-%BD_LVM_TECH_MODE_MODIFY
 */
gboolean  bd_lvm_cache_detach (const gchar *vg_name, const gchar *cached_lv, gboolean destroy, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_cache_create_cached_lv:
 * @vg_name: name of the VG to create a cached LV in
 * @lv_name: name of the cached LV to create
 * @data_size: size of the data LV
 * @cache_size: size of the cache (or cached LV more precisely)
 * @md_size: size of the cache metadata LV or 0 to use the default
 * @mode: cache mode for the cached LV
 * @flags: a combination of (ORed) #BDLVMCachePoolFlags
 * @slow_pvs: (array zero-terminated=1): list of slow PVs (used for the data LV)
 * @fast_pvs: (array zero-terminated=1): list of fast PVs (used for the cache LV)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the cached LV @lv_name was successfully created or not
 *
 * Tech category: %BD_LVM_TECH_CACHE-%BD_LVM_TECH_MODE_CREATE
 */
gboolean  bd_lvm_cache_create_cached_lv (const gchar *vg_name, const gchar *lv_name, guint64 data_size, guint64 cache_size, guint64 md_size, BDLVMCacheMode mode, BDLVMCachePoolFlags flags, const gchar **slow_pvs, const gchar **fast_pvs, GError **error);


/**
 * bd_lvm_cache_pool_name:
 * @vg_name: name of the VG containing the @cached_lv
 * @cached_lv: cached LV to get the name of the its pool LV for
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: name of the cache pool LV used by the @cached_lv or %NULL in case of error
 *
 * Tech category: %BD_LVM_TECH_CACHE-%BD_LVM_TECH_MODE_QUERY
 */
gchar* bd_lvm_cache_pool_name (const gchar *vg_name, const gchar *cached_lv, GError **error);


/**
 * bd_lvm_cache_stats:
 * @vg_name: name of the VG containing the @cached_lv
 * @cached_lv: cached LV to get stats for
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: stats for the @cached_lv or %NULL in case of error
 *
 * Tech category: %BD_LVM_TECH_CACHE-%BD_LVM_TECH_MODE_QUERY
 */
BDLVMCacheStats* bd_lvm_cache_stats (const gchar *vg_name, const gchar *cached_lv, GError **error);


/**
 * bd_lvm_writecache_attach:
 * @vg_name: name of the VG containing the @data_lv and the @cache_pool_lv LVs
 * @data_lv: data LV to attach the @cache_lv to
 * @cache_lv: cache (fast) LV to attach to the @data_lv
 * @extra: (nullable) (array zero-terminated=1): extra options for the cache attachment
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the @cache_lv was successfully attached to the @data_lv or not
 *
 * Tech category: %BD_LVM_TECH_WRITECACHE-%BD_LVM_TECH_MODE_MODIFY
 */
gboolean  bd_lvm_writecache_attach (const gchar *vg_name, const gchar *data_lv, const gchar *cache_lv, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_writecache_detach:
 * @vg_name: name of the VG containing the @cached_lv
 * @cached_lv: name of the cached LV to detach its cache from
 * @destroy: whether to destroy the cache after detach or not
 * @extra: (nullable) (array zero-terminated=1): extra options for the cache detachment
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the cache was successfully detached from the @cached_lv or not
 *
 * Note: synces the cache first
 *
 * Tech category: %BD_LVM_TECH_WRITECACHE-%BD_LVM_TECH_MODE_MODIFY
 */
gboolean  bd_lvm_writecache_detach (const gchar *vg_name, const gchar *cached_lv, gboolean destroy, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_writecache_create_cached_lv:
 * @vg_name: name of the VG to create a cached LV in
 * @lv_name: name of the cached LV to create
 * @data_size: size of the data LV
 * @cache_size: size of the cache (or cached LV more precisely)
 * @slow_pvs: (array zero-terminated=1): list of slow PVs (used for the data LV)
 * @fast_pvs: (array zero-terminated=1): list of fast PVs (used for the cache LV)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the cached LV @lv_name was successfully created or not
 *
 * Tech category: %BD_LVM_TECH_WRITECACHE-%BD_LVM_TECH_MODE_CREATE
 */
gboolean  bd_lvm_writecache_create_cached_lv (const gchar *vg_name, const gchar *lv_name, guint64 data_size, guint64 cache_size, const gchar **slow_pvs, const gchar **fast_pvs, GError **error);


/**
 * bd_lvm_thpool_convert:
 * @vg_name: name of the VG to create the new thin pool in
 * @data_lv: name of the LV that should become the data part of the new pool
 * @metadata_lv: name of the LV that should become the metadata part of the new pool
 * @name: (nullable): name for the thin pool (if %NULL, the name @data_lv is inherited)
 * @extra: (nullable) (array zero-terminated=1): extra options for the thin pool creation
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Converts the @data_lv and @metadata_lv into a new thin pool in the @vg_name
 * VG.
 *
 * Returns: whether the new thin pool was successfully created from @data_lv and
 *          @metadata_lv or not
 *
 * Tech category: %BD_LVM_TECH_THIN-%BD_LVM_TECH_MODE_CREATE
 */
gboolean  bd_lvm_thpool_convert (const gchar *vg_name, const gchar *data_lv, const gchar *metadata_lv, const gchar *name, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_cache_pool_convert:
 * @vg_name: name of the VG to create the new thin pool in
 * @data_lv: name of the LV that should become the data part of the new pool
 * @metadata_lv: name of the LV that should become the metadata part of the new pool
 * @name: (nullable): name for the thin pool (if %NULL, the name @data_lv is inherited)
 * @extra: (nullable) (array zero-terminated=1): extra options for the thin pool creation
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Converts the @data_lv and @metadata_lv into a new cache pool in the @vg_name
 * VG.
 *
 * Returns: whether the new cache pool was successfully created from @data_lv and
 *          @metadata_lv or not
 *
 * Tech category: %BD_LVM_TECH_CACHE-%BD_LVM_TECH_MODE_CREATE
 */
gboolean  bd_lvm_cache_pool_convert (const gchar *vg_name, const gchar *data_lv, const gchar *metadata_lv, const gchar *name, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_vdo_pool_create:
 * @vg_name: name of the VG to create a new LV in
 * @lv_name: name of the to-be-created VDO LV
 * @pool_name: (nullable): name of the to-be-created VDO pool LV or %NULL for default name
 * @data_size: requested size of the data VDO LV (physical size of the @pool_name VDO pool LV)
 * @virtual_size: requested virtual_size of the @lv_name VDO LV
 * @index_memory: amount of index memory (in bytes) or 0 for default
 * @compression: whether to enable compression or not
 * @deduplication: whether to enable deduplication or not
 * @write_policy: write policy for the volume
 * @extra: (nullable) (array zero-terminated=1): extra options for the VDO LV creation
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the given @vg_name/@lv_name VDO LV was successfully created or not
 *
 * Tech category: %BD_LVM_TECH_VDO-%BD_LVM_TECH_MODE_CREATE
 */
gboolean  bd_lvm_vdo_pool_create (const gchar *vg_name, const gchar *lv_name, const gchar *pool_name, guint64 data_size, guint64 virtual_size, guint64 index_memory, gboolean compression, gboolean deduplication, BDLVMVDOWritePolicy write_policy, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_vdo_enable_compression:
 * @vg_name: name of the VG containing the to-be-changed VDO pool LV
 * @pool_name: name of the VDO pool LV to enable compression on
 * @extra: (nullable) (array zero-terminated=1): extra options for the VDO change
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether compression was successfully enabled on @vg_name/@pool_name LV or not
 *
 * Tech category: %BD_LVM_TECH_VDO-%BD_LVM_TECH_MODE_MODIFY
 */
gboolean  bd_lvm_vdo_enable_compression (const gchar *vg_name, const gchar *pool_name, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_vdo_disable_compression:
 * @vg_name: name of the VG containing the to-be-changed VDO pool LV
 * @pool_name: name of the VDO pool LV to disable compression on
 * @extra: (nullable) (array zero-terminated=1): extra options for the VDO change
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether compression was successfully disabled on @vg_name/@pool_name LV or not
 *
 * Tech category: %BD_LVM_TECH_VDO-%BD_LVM_TECH_MODE_MODIFY
 */
gboolean  bd_lvm_vdo_disable_compression (const gchar *vg_name, const gchar *pool_name, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_vdo_enable_deduplication:
 * @vg_name: name of the VG containing the to-be-changed VDO pool LV
 * @pool_name: name of the VDO pool LV to enable deduplication on
 * @extra: (nullable) (array zero-terminated=1): extra options for the VDO change
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether deduplication was successfully enabled on @vg_name/@pool_name LV or not
 *
 * Tech category: %BD_LVM_TECH_VDO-%BD_LVM_TECH_MODE_MODIFY
 */
gboolean  bd_lvm_vdo_enable_deduplication (const gchar *vg_name, const gchar *pool_name, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_vdo_disable_deduplication:
 * @vg_name: name of the VG containing the to-be-changed VDO pool LV
 * @pool_name: name of the VDO pool LV to disable deduplication on
 * @extra: (nullable) (array zero-terminated=1): extra options for the VDO change
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether deduplication was successfully disabled on @vg_name/@pool_name LV or not
 *
 * Tech category: %BD_LVM_TECH_VDO-%BD_LVM_TECH_MODE_MODIFY
 */
gboolean  bd_lvm_vdo_disable_deduplication (const gchar *vg_name, const gchar *pool_name, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_vdo_info:
 * @vg_name: name of the VG that contains the LV to get information about
 * @lv_name: name of the LV to get information about
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: (transfer full): information about the @vg_name/@lv_name LV or %NULL in case
 * of error (the @error) gets populated in those cases)
 *
 * Tech category: %BD_LVM_TECH_VDO-%BD_LVM_TECH_MODE_QUERY
 */
BDLVMVDOPooldata* bd_lvm_vdo_info (const gchar *vg_name, const gchar *lv_name, GError **error);


/**
 * bd_lvm_vdo_resize:
 * @vg_name: name of the VG containing the to-be-resized VDO LV
 * @lv_name: name of the to-be-resized VDO LV
 * @size: the requested new size of the VDO LV
 * @extra: (nullable) (array zero-terminated=1): extra options for the VDO LV resize
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the @vg_name/@lv_name VDO LV was successfully resized or not
 *
 * Note: Reduction needs to process TRIM for reduced disk area to unmap used data blocks
 *       from the VDO pool LV and it may take a long time.
 *
 * Tech category: %BD_LVM_TECH_VDO-%BD_LVM_TECH_MODE_MODIFY
 */
gboolean  bd_lvm_vdo_resize (const gchar *vg_name, const gchar *lv_name, guint64 size, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_vdo_pool_resize:
 * @vg_name: name of the VG containing the to-be-resized VDO pool LV
 * @pool_name: name of the to-be-resized VDO pool LV
 * @size: the requested new size of the VDO pool LV
 * @extra: (nullable) (array zero-terminated=1): extra options for the VDO pool LV resize
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the @vg_name/@pool_name VDO pool LV was successfully resized or not
 *
 * Note: Size of the VDO pool LV can be only extended, not reduced.
 *
 * Tech category: %BD_LVM_TECH_VDO-%BD_LVM_TECH_MODE_MODIFY
 */
gboolean  bd_lvm_vdo_pool_resize (const gchar *vg_name, const gchar *pool_name, guint64 size, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_vdo_pool_convert:
 * @vg_name: name of the VG that contains @pool_lv
 * @pool_lv: name of the LV that should become the new VDO pool LV
 * @name: (nullable): name for the VDO LV or %NULL for default name
 * @virtual_size: virtual size for the new VDO LV
 * @index_memory: amount of index memory (in bytes) or 0 for default
 * @compression: whether to enable compression or not
 * @deduplication: whether to enable deduplication or not
 * @write_policy: write policy for the volume
 * @extra: (nullable) (array zero-terminated=1): extra options for the VDO pool creation
 *                                                 (just passed to LVM as is)
 * @error: (out) (optional): place to store error (if any)
 *
 * Converts the @pool_lv into a new VDO pool LV in the @vg_name VG and creates a new
 * @name VDO LV with size @virtual_size.
 *
 * Note: All data on @pool_lv will be irreversibly destroyed.
 *
 * Returns: whether the new VDO pool LV was successfully created from @pool_lv and or not
 *
 * Tech category: %BD_LVM_TECH_VDO-%BD_LVM_TECH_MODE_CREATE&%BD_LVM_TECH_MODE_MODIFY
 */
gboolean  bd_lvm_vdo_pool_convert (const gchar *vg_name, const gchar *pool_lv, const gchar *name, guint64 virtual_size, guint64 index_memory, gboolean compression, gboolean deduplication, BDLVMVDOWritePolicy write_policy, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_vdolvpoolname:
 * @vg_name: name of the VG containing the queried VDO LV
 * @lv_name: name of the queried VDO LV
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: (transfer full): the name of the pool volume for the @vg_name/@lv_name
 * VDO LV or %NULL if failed to determine (@error) is set in those cases)
 *
 * Tech category: %BD_LVM_TECH_VDO-%BD_LVM_TECH_MODE_QUERY
 */
gchar* bd_lvm_vdolvpoolname (const gchar *vg_name, const gchar *lv_name, GError **error);


/**
 * bd_lvm_get_vdo_operating_mode_str:
 * @mode: mode to get the string representation for
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: string representation of @mode or %NULL in case of error
 *
 * Tech category: always provided/supported
 */
const gchar* bd_lvm_get_vdo_operating_mode_str (BDLVMVDOOperatingMode mode, GError **error);


/**
 * bd_lvm_get_vdo_compression_state_str:
 * @state: state to get the string representation for
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: string representation of @state or %NULL in case of error
 *
 * Tech category: always provided/supported
 */
const gchar* bd_lvm_get_vdo_compression_state_str (BDLVMVDOCompressionState state, GError **error);


/**
 * bd_lvm_get_vdo_index_state_str:
 * @state: state to get the string representation for
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: string representation of @state or %NULL in case of error
 *
 * Tech category: always provided/supported
 */
const gchar* bd_lvm_get_vdo_index_state_str (BDLVMVDOIndexState state, GError **error);


/**
 * bd_lvm_get_vdo_write_policy_str:
 * @policy: policy to get the string representation for
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: string representation of @policy or %NULL in case of error
 *
 * Tech category: always provided/supported
 */
const gchar* bd_lvm_get_vdo_write_policy_str (BDLVMVDOWritePolicy policy, GError **error);


/**
 * bd_lvm_get_vdo_write_policy_from_str:
 * @policy_str: string representation of a policy
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: write policy for the @policy_str or %BD_LVM_VDO_WRITE_POLICY_UNKNOWN if
 *          failed to determine
 *
 * Tech category: always provided/supported
 */
BDLVMVDOWritePolicy  bd_lvm_get_vdo_write_policy_from_str (const gchar *policy_str, GError **error);


/**
 * bd_lvm_vdo_get_stats_full:
 * @vg_name: name of the VG that contains @pool_name VDO pool
 * @pool_name: name of the VDO pool to get statistics for
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: (transfer full) (element-type utf8 utf8): hashtable of type string - string of available
 *                                                    statistics or %NULL in case of error
 *                                                    (@error gets populated in those cases)
 *
 * Statistics are collected from the values exposed by the kernel `kvdo` module
 * at the `/sys/kvdo/<VDO_NAME>/statistics/` path.
 * Some of the keys are computed to mimic the information produced by the vdo tools.
 * Please note the contents of the hashtable may vary depending on the actual kvdo module version.
 *
 * Tech category: %BD_LVM_TECH_VDO-%BD_LVM_TECH_MODE_QUERY
 */
GHashTable* bd_lvm_vdo_get_stats_full (const gchar *vg_name, const gchar *pool_name, GError **error);


/**
 * bd_lvm_vdo_get_stats:
 * @vg_name: name of the VG that contains @pool_name VDO pool
 * @pool_name: name of the VDO pool to get statistics for
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: (transfer full): a structure containing selected statistics or %NULL in case of error
 *                           (@error gets populated in those cases)
 *
 * In contrast to @bd_lvm_vdo_get_stats_full this function will only return selected statistics
 * in a fixed structure. In case a value is not available, -1 would be returned.
 *
 * Tech category: %BD_LVM_TECH_VDO-%BD_LVM_TECH_MODE_QUERY
 */
BDLVMVDOStats* bd_lvm_vdo_get_stats (const gchar *vg_name, const gchar *pool_name, GError **error);


/**
 * bd_lvm_devices_add:
 * @device: device (PV) to add to the devices file
 * @devices_file: (nullable): LVM devices file or %NULL for default
 * @extra: (nullable) (array zero-terminated=1): extra options for the lvmdevices command
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the @device was successfully added to @devices_file or not
 *
 * Tech category: %BD_LVM_TECH_DEVICES no mode (it is ignored)
 */
gboolean  bd_lvm_devices_add (const gchar *device, const gchar *devices_file, const BDExtraArg **extra, GError **error);


/**
 * bd_lvm_devices_delete:
 * @device: device (PV) to delete from the devices file
 * @devices_file: (nullable): LVM devices file or %NULL for default
 * @extra: (nullable) (array zero-terminated=1): extra options for the lvmdevices command
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: whether the @device was successfully removed from @devices_file or not
 *
 * Tech category: %BD_LVM_TECH_DEVICES no mode (it is ignored)
 */
gboolean  bd_lvm_devices_delete (const gchar *device, const gchar *devices_file, const BDExtraArg **extra, GError **error);


#endif  /* BD_LVM_API */
