/*
 * Copyright (C) 2022 - 2025, Stephan Mueller <smueller@chronox.de>
 *
 * License, 0x see LICENSE file in root directory
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ALL OF
 * WHICH ARE HEREBY DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF NOT ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

#include "chacha20_c.h"
#include "chacha20_neon.h"
#include "chacha20_riscv64_v_zbb.h"
#include "chacha20_avx2.h"
#include "chacha20_avx512.h"

#include "alignment.h"
#include "conv_be_le.h"
#include "compare.h"
#include "lc_chacha20.h"
#include "math_helper.h"
#include "test_helper_common.h"
#include "ret_checkers.h"
#include "visibility.h"

#define LC_EXEC_ONE_TEST(chacha20_impl)                                        \
	if (chacha20_impl) {                                                   \
		ret += chacha20_enc_selftest(chacha20_impl, #chacha20_impl);   \
		ret += chacha20_enc_large(chacha20_impl, #chacha20_impl);      \
		ret += chacha20_stream_test(chacha20_impl, #chacha20_impl);    \
	}

static int chacha20_enc_large(const struct lc_sym *chacha20_sym,
			      const char *name)
{
	/* Test vector according to RFC 7539 section 2.4.2 */
	static const uint8_t key[] = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
				       0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d,
				       0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14,
				       0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b,
				       0x1c, 0x1d, 0x1e, 0x1f };
	static const uint8_t iv[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				      0x00, 0x4a, 0x00, 0x00, 0x00, 0x00 };
	static const uint8_t string[] = {
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x00, 0x01, 0x02, 0x03, 0x04,
	};
	static const uint8_t exp[] = {
		0x22, 0x4e, 0x53, 0xf0, 0x44, 0x1e, 0xdf, 0xe6, 0x2f, 0xdf,
		0x25, 0x6c, 0xbc, 0x66, 0x1b, 0xea, 0x8c, 0x12, 0x1d, 0x81,
		0x39, 0x29, 0x00, 0xe5, 0x7e, 0x4e, 0xc8, 0xef, 0x9a, 0xf6,
		0xc9, 0x7f, 0x8a, 0x3a, 0x08, 0xa0, 0x76, 0x65, 0x0c, 0x95,
		0xb5, 0x78, 0x76, 0xce, 0xe9, 0x2e, 0x95, 0x33, 0x79, 0x4d,
		0xb8, 0x43, 0xc2, 0x3b, 0x32, 0xca, 0xea, 0x20, 0x2e, 0x4f,
		0xf4, 0x78, 0x47, 0xb0, 0x69, 0xa7, 0x76, 0x9c, 0x3b, 0x66,
		0x09, 0x46, 0x22, 0xcb, 0xfc, 0x2b, 0xe8, 0x48, 0xc2, 0x79,
		0x26, 0xd5, 0x36, 0x6e, 0x74, 0xbc, 0x8a, 0x74, 0xf3, 0xe8,
		0xc7, 0x39, 0xc0, 0x09, 0x5f, 0x42, 0x39, 0x8a, 0x6c, 0xd9,
		0x1e, 0x86, 0x2a, 0x8e, 0xc1, 0x66, 0xe8, 0xce, 0x94, 0x18,
		0x78, 0x2c, 0xf3, 0x62, 0x76, 0x00, 0x77, 0x25, 0x1c, 0xa6,
		0x88, 0xfa, 0xbe, 0xeb, 0x3d, 0x94, 0xc2, 0xea, 0xc8, 0xec,
		0x04, 0x4f, 0x6a, 0x96, 0x9b, 0x47, 0x31, 0x74, 0xec, 0xe2,
		0x43, 0x55, 0xf5, 0xe2, 0xa9, 0xaf, 0x89, 0x45, 0xc5, 0x48,
		0xab, 0x4d, 0x96, 0x29, 0xc8, 0xac, 0x02, 0x4f, 0x27, 0xd7,
		0x0a, 0x73, 0x51, 0xae, 0xfc, 0xca, 0x88, 0xb7, 0x98, 0x6d,
		0x3c, 0x6b, 0x76, 0xc3, 0xd8, 0x62, 0x4d, 0x81, 0x74, 0xc8,
		0x9e, 0xa4, 0x9a, 0x10, 0x41, 0x3c, 0x84, 0xbf, 0x0a, 0xd0,
		0x6d, 0x14, 0xae, 0xd2, 0xf7, 0xe4, 0xfe, 0x88, 0x0f, 0x5b,
		0x30, 0x64, 0x45, 0x2b, 0x31, 0x61, 0xca, 0x94, 0x61, 0xfe,
		0x21, 0x46, 0x29, 0xb2, 0xc0, 0x71, 0x65, 0x74, 0x87, 0xa3,
		0xe1, 0x74, 0x7d, 0x06, 0xc9, 0x8c, 0x3f, 0x4d, 0x87, 0xc2,
		0xa0, 0xdc, 0xac, 0xf4, 0x26, 0xe3, 0xc0, 0x97, 0x47, 0xa8,
		0x87, 0x0a, 0x11, 0x70, 0x11, 0xa0, 0x23, 0x7e, 0xe0, 0xd0,
		0x47, 0x7e, 0x7d, 0xa4, 0xcb, 0x21, 0xcb, 0x50, 0xb7, 0xa1,
		0x19, 0xb8, 0x3e, 0x0f, 0xec, 0x76, 0x53, 0xe8, 0x9d, 0x35,
		0xd5, 0x5f, 0x5f, 0xc0, 0x38, 0xc4, 0xb3, 0x45, 0xb5, 0x0a,
		0x08, 0x07, 0x58, 0x0f, 0xd6, 0x44, 0x27, 0xaa, 0x31, 0x7e,
		0xb5, 0x41, 0xa4, 0xe4, 0x5f, 0x7a, 0x9d, 0xc8, 0x17, 0x6c,
		0x04, 0xed, 0x53, 0xa2, 0xc9, 0xd5, 0xda, 0xbf, 0xc7, 0xb1,
		0x26, 0x30, 0x22, 0x2a, 0xb9, 0xaf, 0xc9, 0x97, 0x7f, 0x3a,
		0x89, 0xfc, 0xd5, 0x33, 0x6f, 0x8d, 0xc6, 0x6a, 0x7b, 0xea,
		0x16, 0x07, 0x04, 0x99, 0x56, 0xbc, 0x67, 0xe5, 0x3b, 0x8f,
		0x30, 0x91, 0x74, 0x2b, 0xe7, 0x6f, 0x72, 0xcc, 0x9c, 0x2e,
		0x03, 0x64, 0x3a, 0x1d, 0x14, 0x89, 0xe2, 0x50, 0x24, 0x24,
		0xa9, 0xa4, 0xe5, 0xba, 0x14, 0x13, 0x99, 0xd4, 0x1b, 0x61,
		0x5a, 0x1b, 0x47, 0xe0, 0xec, 0x88, 0xb9, 0xa7, 0x3d, 0x3c,
		0x68, 0x28, 0x12, 0x18, 0x1a, 0x9a, 0xa0, 0xd7, 0xbf, 0x6b,
		0x72, 0x89, 0x7c, 0x21, 0x8c, 0x4e, 0x6c, 0x24, 0x24, 0x93,
		0x8e, 0x3e, 0x6f, 0x56, 0x3c, 0xad, 0xe1, 0xc2, 0x4d, 0xd6,
		0x19, 0x43, 0x4a, 0xf0, 0x41, 0xe7, 0xb7, 0xda, 0xe7, 0xa7,
		0x13, 0x0c, 0x10, 0x8f, 0x18, 0xa8, 0x4f, 0x05, 0x1c, 0x77,
		0x86, 0x3a, 0x0e, 0x0b, 0x82, 0x31, 0xc3, 0x15, 0xdb, 0xc1,
		0x31, 0xfb, 0x97, 0x4a, 0xdb, 0xa8, 0x47, 0x31, 0xb8, 0x8d,
		0x98, 0x97, 0x25, 0xe6, 0xa1, 0xbb, 0xe1, 0x6a, 0x7d, 0x8f,
		0x7f, 0x0f, 0x2b, 0x7b, 0x21, 0x7b, 0x1b, 0x81, 0x73, 0xd7,
		0x18, 0x99, 0xf4, 0x97, 0x60, 0xc7, 0x14, 0x7f, 0xd9, 0xdf,
		0x13, 0x83, 0xd5, 0xb0, 0x1a, 0x9a, 0x71, 0x54, 0x68, 0xf1,
		0x39, 0x15, 0xaa, 0x3f, 0x9b, 0x84, 0x45, 0xfc, 0xba, 0x50,
		0x92, 0xb7, 0xba, 0xd6, 0xb6, 0x0d, 0x06, 0xdd, 0x2e, 0x26,
		0x70, 0xa7, 0xe1, 0x3f, 0x3e, 0x9c, 0xe2, 0x0e, 0xb7, 0x16,
		0x4a, 0x0a, 0x1f, 0x64, 0x4c, 0x15, 0x59, 0x00, 0xf0, 0xab,
		0x4d, 0xe6, 0x73, 0xa4, 0xe8, 0xf8, 0x72, 0xda, 0x4f, 0x67,
		0xb2, 0x93, 0xd9, 0x6f, 0xbc, 0x93, 0xf4, 0x81, 0x4c, 0xf2,
		0xb2, 0x36, 0x3d, 0x61, 0xbc, 0xb6, 0x6e, 0x34, 0xf0, 0xf7,
		0x87, 0xa7, 0xb0, 0xf0, 0x66, 0x8e, 0x34, 0xb9, 0x5a, 0x92,
		0xa6, 0x1c, 0x0f, 0xf4, 0x03, 0xfe, 0x83, 0x93, 0xa6, 0xd9,
		0x53, 0xd3, 0x66, 0x93, 0x31, 0x45, 0x6e, 0x18, 0x9c, 0x58,
		0x36, 0xce, 0x21, 0xd5, 0xce, 0x99, 0x11, 0xa6, 0x26, 0xa0,
		0x46, 0xa7, 0x1c, 0x29, 0xc1, 0x6a, 0xcb, 0x76, 0x80, 0x76,
		0xa6, 0x15, 0x79, 0x37, 0x49, 0x13, 0x2b, 0x25, 0x25, 0xc1,
		0xeb, 0x0e, 0xd8, 0xf6, 0x6e, 0x97, 0x8f, 0x16, 0xd4, 0xf4,
		0x0e, 0x23, 0xd2, 0x5a, 0xaf, 0x55, 0xde, 0x59, 0x96, 0xd3,
		0xba, 0x7d, 0x5c, 0xdd, 0xfd, 0x84, 0xa4, 0x6d, 0x2d, 0xdd,
		0x03, 0x10, 0x86, 0x70, 0x0d, 0x0a, 0xe6, 0x2c, 0x95, 0x4f,
		0x1f, 0xe2, 0x5e, 0x4f, 0x59, 0x8f, 0x1e, 0xae, 0x40, 0xc2,
		0x48, 0xcd, 0xfa, 0xcd, 0xc6, 0xd8, 0xa6, 0x89, 0x59, 0xd6,
		0xa4, 0x79, 0x2c, 0x4c, 0x15, 0x30, 0x7d, 0xd6, 0x5b, 0x60,
		0xb7, 0xbb, 0x1a, 0xc0, 0xd2, 0xe0, 0xef, 0xb7, 0x99, 0xcd,
		0xb9, 0xf4, 0xc8, 0xcb, 0xb7, 0xac, 0x2d, 0x9a, 0x6b, 0x2e,
		0x35, 0x13, 0xdc, 0x56, 0x8f, 0x80, 0x3c, 0x83, 0xbe, 0x84,
		0x40, 0xe3, 0xa3, 0xc2, 0x83, 0xe3, 0x9b, 0xd9, 0xaa, 0xcf,
		0x06, 0xaf, 0x71, 0x39, 0x61, 0x2b, 0x46, 0x2a, 0x26, 0x92,
		0xe1, 0x11, 0xf7, 0xc5, 0xee, 0x50, 0xd4, 0x74, 0x84, 0xde,
		0x73, 0xd1, 0x91, 0xad, 0x53, 0x9a, 0x0b, 0x02, 0x4f, 0x22,
		0xa3, 0x87, 0x66, 0x4a, 0xc1, 0x49, 0xed, 0xae, 0xa6, 0x2a,
		0x29, 0x03, 0x1a, 0x00, 0xcd, 0xb5, 0x44, 0x76, 0x24, 0xc1,
		0x33, 0x28, 0x3c, 0xcf, 0xfd, 0xf6, 0x6e, 0x30, 0x68, 0x79,
		0xa6, 0x42, 0x63, 0x5e, 0x1e, 0xf0, 0x23, 0xd3, 0xb1, 0xa0,
		0x6d, 0x6f, 0xc3, 0xd1, 0xf1, 0x13, 0xd9, 0x0f, 0x2e, 0x7c,
		0x8f, 0xa4, 0x6c, 0x15, 0xd5, 0x80, 0x35, 0x5b, 0xbd, 0xe7,
		0x7e, 0xa6, 0x22, 0xff, 0x7d, 0x81, 0xd8, 0x96, 0x30, 0x63,
		0xc7, 0xd8, 0xb4, 0x32, 0xac, 0xd7, 0x88, 0xdc, 0x31, 0xfe,
		0xd2, 0x37, 0xb4, 0x92, 0xd1, 0x82, 0xf4, 0x7b, 0x08, 0x69,
		0xca, 0x50, 0x17, 0x1b, 0x50, 0xa5, 0xe4, 0x86, 0x8e, 0xbe,
		0x07, 0xe7, 0x76, 0xce, 0x7b, 0x6b, 0x42, 0x11, 0x9a, 0xa7,
		0x58, 0xae, 0x0c, 0x3b, 0xfd, 0x84, 0x28, 0x1e, 0x82, 0xf5,
		0xbc, 0x89, 0x87, 0x2e, 0x5b, 0xd9, 0xa3, 0x20, 0xf0, 0x7d,
		0x2a, 0x12, 0x6f, 0x77, 0x9c, 0xc9, 0x2f, 0xe0, 0xfa, 0x0d,
		0x6b, 0x36, 0x3f, 0xdb, 0x15, 0xab, 0x2e, 0x3a, 0x62, 0xe4,
		0x0f, 0x64, 0x36, 0x86, 0xc2, 0x9e, 0x96, 0x62, 0xa3, 0x3e,
		0x8e, 0x31, 0x9d, 0x30, 0xc0, 0x10, 0xef, 0x28, 0x51, 0xc5,
		0x20, 0x5d, 0x61, 0xeb, 0x06, 0x18, 0x12, 0xe9, 0xd0, 0x6d,
		0x4c, 0x8d, 0xf7, 0x5a, 0x13, 0x3d, 0x85, 0xd0, 0x93, 0xa3,
		0x25, 0x65, 0xe6, 0x06, 0x3c, 0xbe, 0x27, 0xac, 0x5d, 0x92,
		0x73, 0x92, 0x54, 0xd3, 0x54, 0xff, 0xcf, 0x64, 0x71, 0x34,
		0xb8, 0xc8, 0x20, 0x7d, 0x85, 0x4e, 0xaa, 0xb7, 0x02, 0xf9,
		0x11, 0xc0, 0xaa, 0x35, 0x6c, 0xac, 0xff, 0x9c, 0x01, 0xaa,
		0xfc, 0x89, 0xc0, 0xff, 0xc8, 0x34, 0xc4, 0xd1, 0x97, 0x3a,
		0x03, 0xd6, 0x01, 0x2b, 0xc6, 0xf6, 0x03, 0x5d, 0x41, 0x21,
		0x7d, 0x73, 0x0a, 0xc5, 0x40, 0xf3, 0xf7, 0x97, 0x60, 0x3a,
		0xa6, 0x9b, 0xd5, 0xfd, 0x9d, 0x87, 0x0f, 0xbb, 0x63, 0x1e,
		0xa4, 0x9d, 0xdd, 0x6b, 0x26, 0xc4, 0x9d, 0x0a, 0x5f, 0xc2,
		0xe7, 0x9b, 0x41, 0x55, 0x1d, 0x3d, 0x57, 0x07, 0x02, 0x3e,
		0x6a, 0x05, 0x48, 0xde, 0xd6, 0xf3, 0x42, 0xbf, 0x78, 0xd5,
		0x5c, 0xaf, 0xe1, 0xd0, 0x00, 0x71, 0xa1, 0xce, 0xae, 0x06,
		0x85, 0x9a, 0xe1, 0x66, 0xae, 0x06, 0x2d, 0xf0, 0x19, 0x82,
		0x2b, 0xc1, 0x9b, 0x26, 0x35, 0xec, 0x97, 0x13, 0x5e, 0x38,
		0x57, 0x89, 0xde, 0xaf, 0x80, 0xc2, 0xb0, 0x3f, 0x4c, 0xde,
		0x2c, 0xd8, 0x26, 0xa8, 0xb4, 0xe3, 0xb5, 0xb7, 0x92, 0xea,
		0x90, 0x3c, 0x8b, 0x00, 0x15, 0xf1, 0x2d, 0x62, 0x5d, 0x68,
		0xeb, 0x19, 0x4c, 0xe2, 0x4e, 0x47, 0x79, 0x83, 0xef, 0x4f,
		0x90, 0xcf, 0xae, 0x1d, 0x02, 0x47, 0x22, 0x93, 0xa0, 0x71,
		0x50, 0xc4, 0x8e, 0xc5, 0x7f, 0x7b, 0x1c, 0x4f, 0x55, 0xb3,
		0xb5, 0x05, 0x9f, 0xc5, 0x4c, 0xc7, 0x93, 0x26, 0xd6, 0x1a,
		0xb5, 0x40, 0x6a, 0x0a, 0x94, 0x1c, 0x2c, 0x7b, 0x61, 0xe6,
		0x51, 0xb7, 0x0f, 0xfc, 0x37, 0xa7, 0x9c, 0x6a, 0x51, 0x6d,
		0x4c, 0x87, 0x06, 0x5e, 0xf9, 0xad, 0x4e, 0x50, 0xaf, 0xf1,
		0x89, 0x48, 0xb4, 0x61, 0x79, 0xd9, 0xf4, 0xf7, 0xd8, 0xc8,
		0xa3, 0xd6, 0x24, 0x85, 0xab, 0xbf, 0x3e, 0x99, 0x7e, 0x54,
		0xf9, 0x06, 0x12, 0x37, 0x56, 0x66, 0xde, 0x5e, 0xa3, 0x65,
		0x58, 0x2b, 0x7f, 0x8b, 0x2a, 0x09, 0x95, 0xb3, 0xf1, 0x71,
		0x84, 0xbd, 0x0f, 0xfa, 0xfc, 0x7e, 0xe8, 0x14, 0x81, 0xda,
		0x5d, 0x25, 0xeb, 0x4e, 0x84, 0x06, 0x83, 0x04, 0x97, 0x51,
		0x1b, 0xb0, 0x50, 0x8a, 0x87, 0x07, 0xc5, 0x30, 0x3d, 0x44,
		0x29, 0x0f, 0x4f, 0xa8, 0x20, 0x33, 0x0b, 0x2d, 0x18, 0x6f,
		0x54, 0x0a, 0xf7, 0x5a, 0x5a, 0x65, 0xd6, 0x8d, 0xb2, 0x0b,
		0x23, 0xf3, 0xb1, 0x87, 0xbe,
	};
	uint8_t res[sizeof(exp)];
	int ret;
	char str[30];
	LC_SYM_CTX_ON_STACK(chacha20, chacha20_sym);

	/* Encrypt */
	CKINT(lc_sym_init(chacha20));
	CKINT(lc_sym_setkey(chacha20, (uint8_t *)key, sizeof(key)));
	CKINT(lc_sym_setiv(chacha20, (uint8_t *)iv, sizeof(iv)));
	lc_sym_encrypt(chacha20, (uint8_t *)string, res, sizeof(string));
	snprintf(str, sizeof(str), "ChaCha20 large enc - %s", name);
	ret = lc_compare(res, exp, sizeof(exp), str);
	if (ret) {
		ret = -EINVAL;
		goto out;
	}
	lc_sym_zero(chacha20);

	/* Decrypt */
	CKINT(lc_sym_init(chacha20));
	CKINT(lc_sym_setkey(chacha20, (uint8_t *)key, sizeof(key)));
	CKINT(lc_sym_setiv(chacha20, (uint8_t *)iv, sizeof(iv)));
	lc_sym_decrypt(chacha20, res, res, sizeof(res));
	snprintf(str, sizeof(str), "ChaCha20 large enc - %s", name);
	ret = lc_compare(res, string, sizeof(string), str);
	if (ret) {
		ret = -EINVAL;
		goto out;
	}

out:
	lc_sym_zero(chacha20);
	return !!ret;
}

/* Test vector according to RFC 7539 section 2.4.2 */
static const uint8_t rfc_key[] = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
				   0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d,
				   0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14,
				   0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b,
				   0x1c, 0x1d, 0x1e, 0x1f };
static const uint8_t rfc_iv[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				  0x00, 0x4a, 0x00, 0x00, 0x00, 0x00 };
static const char *rfc_string =
	"Ladies and Gentlemen of the class of '99: If I could offer you only one tip for the future, sunscreen would be it.";
static const uint8_t rfc_exp[] = {
	0x6e, 0x2e, 0x35, 0x9a, 0x25, 0x68, 0xf9, 0x80, 0x41, 0xba, 0x07, 0x28,
	0xdd, 0x0d, 0x69, 0x81, 0xe9, 0x7e, 0x7a, 0xec, 0x1d, 0x43, 0x60, 0xc2,
	0x0a, 0x27, 0xaf, 0xcc, 0xfd, 0x9f, 0xae, 0x0b, 0xf9, 0x1b, 0x65, 0xc5,
	0x52, 0x47, 0x33, 0xab, 0x8f, 0x59, 0x3d, 0xab, 0xcd, 0x62, 0xb3, 0x57,
	0x16, 0x39, 0xd6, 0x24, 0xe6, 0x51, 0x52, 0xab, 0x8f, 0x53, 0x0c, 0x35,
	0x9f, 0x08, 0x61, 0xd8, 0x07, 0xca, 0x0d, 0xbf, 0x50, 0x0d, 0x6a, 0x61,
	0x56, 0xa3, 0x8e, 0x08, 0x8a, 0x22, 0xb6, 0x5e, 0x52, 0xbc, 0x51, 0x4d,
	0x16, 0xcc, 0xf8, 0x06, 0x81, 0x8c, 0xe9, 0x1a, 0xb7, 0x79, 0x37, 0x36,
	0x5a, 0xf9, 0x0b, 0xbf, 0x74, 0xa3, 0x5b, 0xe6, 0xb4, 0x0b, 0x8e, 0xed,
	0xf2, 0x78, 0x5e, 0x42, 0x87, 0x4d
};

static int chacha20_enc_selftest(const struct lc_sym *chacha20_sym,
				 const char *name)
{
	uint8_t res[sizeof(rfc_exp)];
	int ret;
	char str[30];
	LC_SYM_CTX_ON_STACK(chacha20, chacha20_sym);

	printf("ChaCha20 %s ctx size: %u\n", name,
	       (unsigned int)LC_SYM_CTX_SIZE(chacha20_sym));

	/* Encrypt */
	CKINT(lc_sym_init(chacha20));
	CKINT(lc_sym_setkey(chacha20, (uint8_t *)rfc_key, sizeof(rfc_key)));
	CKINT(lc_sym_setiv(chacha20, (uint8_t *)rfc_iv, sizeof(rfc_iv)));
	lc_sym_encrypt(chacha20, (uint8_t *)rfc_string, res, sizeof(res));
	snprintf(str, sizeof(str), "ChaCha20 enc - %s", name);
	ret = lc_compare(res, rfc_exp, sizeof(rfc_exp), str);
	if (ret) {
		ret = -EINVAL;
		goto out;
	}
	lc_sym_zero(chacha20);

	/* Decrypt */
	CKINT(lc_sym_init(chacha20));
	CKINT(lc_sym_setkey(chacha20, (uint8_t *)rfc_key, sizeof(rfc_key)));
	CKINT(lc_sym_setiv(chacha20, (uint8_t *)rfc_iv, sizeof(rfc_iv)));
	lc_sym_decrypt(chacha20, res, res, sizeof(res));
	snprintf(str, sizeof(str), "ChaCha20 dec - %s", name);
	ret = lc_compare(res, (uint8_t *)rfc_string, sizeof(res), str);
	if (ret) {
		ret = -EINVAL;
		goto out;
	}

out:
	lc_sym_zero(chacha20);
	return !!ret;
}

static int chacha20_stream_test(const struct lc_sym *chacha20_sym,
				const char *name)
{
	uint8_t res[sizeof(rfc_exp)];
	char str[30];
	size_t len;
	const uint8_t *in_p;
	uint8_t *out_p;
	unsigned int i;
	int ret;
	LC_SYM_CTX_ON_STACK(chacha20, chacha20_sym);

	/* Encrypt */
	CKINT(lc_sym_init(chacha20));
	CKINT(lc_sym_setkey(chacha20, (uint8_t *)rfc_key, sizeof(rfc_key)));
	CKINT(lc_sym_setiv(chacha20, (uint8_t *)rfc_iv, sizeof(rfc_iv)));

	len = sizeof(res);
	i = 1;
	in_p = (uint8_t *)rfc_string;
	out_p = res;
	while (len) {
		size_t todo = min_size(len, i);

		lc_sym_encrypt(chacha20, in_p, out_p, todo);

		len -= todo;
		in_p += todo;
		out_p += todo;
		i++;
	}

	snprintf(str, sizeof(str), "ChaCha20 stream enc - %s", name);
	ret = lc_compare(res, rfc_exp, sizeof(rfc_exp), str);
	if (ret) {
		ret = -EINVAL;
		goto out;
	}
	lc_sym_zero(chacha20);

	/* Decrypt */
	CKINT(lc_sym_init(chacha20));
	CKINT(lc_sym_setkey(chacha20, (uint8_t *)rfc_key, sizeof(rfc_key)));
	CKINT(lc_sym_setiv(chacha20, (uint8_t *)rfc_iv, sizeof(rfc_iv)));

	len = sizeof(res);
	i = 1;
	in_p = res;
	out_p = res;
	while (len) {
		size_t todo = min_size(len, i);

		lc_sym_decrypt(chacha20, in_p, out_p, todo);

		len -= todo;
		in_p += todo;
		out_p += todo;
		i++;
	}

	snprintf(str, sizeof(str), "ChaCha20 stream dec - %s", name);
	ret = lc_compare(res, (uint8_t *)rfc_string, sizeof(res), str);
	if (ret) {
		ret = -EINVAL;
		goto out;
	}

out:
	lc_sym_zero(chacha20);
	return !!ret;
}

LC_TEST_FUNC(int, main, int argc, char *argv[])
{
	int ret = 0;

	(void)argc;
	(void)argv;

	LC_EXEC_ONE_TEST(lc_chacha20)
	LC_EXEC_ONE_TEST(lc_chacha20_c)
	LC_EXEC_ONE_TEST(lc_chacha20_neon)
	LC_EXEC_ONE_TEST(lc_chacha20_riscv64_v_zbb)
	LC_EXEC_ONE_TEST(lc_chacha20_avx2)
	LC_EXEC_ONE_TEST(lc_chacha20_avx512)

	ret = test_validate_status(ret, LC_ALG_STATUS_CHACHA20, 0);
	ret += test_print_status();

	return ret;
}
