/*
 * Copyright (C) 2020 - 2025, Stephan Mueller <smueller@chronox.de>
 *
 * License: see LICENSE file in root directory
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ALL OF
 * WHICH ARE HEREBY DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF NOT ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

#include "compare.h"
#include "lc_cshake.h"
#include "lc_sha3.h"
#include "test_helper_common.h"
#include "visibility.h"

#include "sha3_c.h"
#include "sha3_arm_asm.h"
#include "sha3_arm_ce.h"
#include "sha3_arm_neon.h"
#include "sha3_avx2.h"
#include "sha3_avx512.h"
#include "sha3_riscv_asm.h"

#define LC_EXEC_ONE_TEST(sha3_impl)                                            \
	if (sha3_impl)                                                         \
	ret += _shake_512_tester(sha3_impl, #sha3_impl)

static int _shake_512_tester(const struct lc_hash *shake_512, const char *name)
{
	static const uint8_t msg1[] = { 0x6C, 0x9E, 0xC8, 0x5C, 0xBA, 0xBA,
					0x62, 0xF5, 0xBC, 0xFE, 0xA1, 0x9E,
					0xB9, 0xC9, 0x20, 0x52, 0xD8, 0xFF,
					0x18, 0x81, 0x52, 0xE9, 0x61, 0xC1,
					0xEC, 0x5C, 0x75, 0xBF, 0xC3, 0xC9,
					0x1C, 0x8D };
	static const uint8_t exp1[] = { 0xc7, 0x9e, 0xcf, 0x01, 0xff, 0x17,
					0xbd, 0x81, 0x22, 0x99, 0xae, 0x46,
					0xb1, 0x09, 0x7b, 0x93, 0x32, 0x73,
					0x40, 0x60, 0x81, 0x30, 0xf3, 0x0b,
					0xb5, 0xc6, 0x5d, 0x95, 0x41, 0xb5,
					0x33, 0x10, 0x97 };
	uint8_t act1[sizeof(exp1)];
	static const uint8_t msg2[] = { 0x69, 0xF9, 0x02, 0x27, 0x27, 0xA7,
					0x5B, 0xB9, 0xDA, 0xCC, 0xBA, 0x6C,
					0x66, 0x9E, 0x52, 0x88, 0x96, 0xBF,
					0xEF, 0x8D, 0xFD, 0xD1, 0xBD, 0x7E,
					0x42, 0xAA, 0x3B, 0x22, 0x71, 0xE4,
					0x79, 0xC5 };
	static const uint8_t exp2[] = {
		0xd3, 0x6b, 0x3c, 0x7f, 0x40, 0x3b, 0xe7, 0x71,
		0x67, 0xd7, 0x4f, 0xf0, 0x1a, 0x18, 0x24, 0xed,
		0xf3, 0x24, 0xcb, 0x0e, 0x36, 0xe9, 0x33, 0xbf,
		0x3a, 0xfb, 0x3b, 0xbc, 0xf9, 0xf7, 0xad, 0x65,
		0x9b, 0xfa, 0x05, 0x30, 0xdb, 0x55, 0x55, 0xe4,
		0xad, 0x73, 0xa0, 0x8e, 0x8a, 0xfa, 0xff, 0x07,
		0x53, 0x8d, 0x4f, 0x7d, 0xce, 0x23, 0xaa, 0x24,
		0x4e, 0x4c, 0x83, 0xf4, 0xfb, 0xde, 0x26, 0xd4,
		0xa2, 0xc8, 0x08, 0xe5, 0x50, 0xb6, 0xf5, 0x94,
		0x32, 0xe2, 0xb8, 0x41, 0xb0, 0xba, 0xc9, 0x3c,
		0xa0, 0xe3, 0xf6, 0x68, 0x51, 0x87, 0xbd, 0x2d,
		0xfb, 0x74, 0x18, 0x86, 0x28, 0xde, 0x3d, 0x82,
		0xbf, 0xca, 0xfe, 0x2a, 0xa7, 0x4c, 0xa9, 0x64,
		0xa4, 0x10, 0xba, 0x96, 0xd0, 0x6c, 0xbb, 0x52,
		0x98, 0x78, 0x2b, 0x16, 0x7b, 0xce, 0x6b, 0xd5,
		0x3d, 0x36, 0x3e, 0x63, 0x06, 0x42, 0xab, 0xc0,
		0xff, 0x73, 0x0d, 0xaa, 0x0c, 0x03, 0xbc, 0x99,
		0x71, 0x7c, 0x0e, 0x20, 0x2d, 0xb8, 0x5c, 0x1d,
		0x63, 0xdd, 0x18, 0x32, 0x62, 0x37, 0x22, 0xee,
		0x52, 0x85, 0xd7, 0x4b, 0x4f, 0xd3, 0xe9, 0x7f,
		0xd1, 0x2f, 0x28, 0x6a, 0x06, 0x7c, 0xc1, 0x05,
		0x20, 0x5b, 0x86, 0x49, 0x01, 0x7c, 0x48, 0x14,
		0x39, 0x6a, 0x5b, 0xe0, 0xd8, 0x9d, 0xf3, 0x09,
		0x30, 0xbf, 0xb4, 0x40, 0x91, 0x73, 0x4e, 0x9b,
		0xf7, 0x36, 0xa3, 0x89, 0x09, 0xd0, 0x0b, 0x8f,
		0x57, 0x5c, 0x01, 0x7c, 0xd8, 0x8c, 0x47, 0xd7,
		0x44, 0x67, 0x1c, 0xba, 0x80, 0x43, 0x51, 0xcf,
		0x06, 0x2c, 0x15, 0x50, 0xf6, 0x7a, 0x21, 0xde,
		0xb0, 0xc2, 0x89, 0x39, 0xf7, 0xbb, 0xda, 0x8d,
		0xa5, 0x90, 0x03, 0x5b, 0x45, 0x73, 0xcd, 0xd5,
		0x7d, 0x84, 0x71, 0x07, 0x76, 0x81, 0xe5, 0x73,
		0x74, 0x73, 0x2e, 0x7d, 0x5f, 0xa9, 0x00, 0xf2,
		0x7b, 0x11, 0xdd, 0x86, 0xc5, 0x96, 0x1e, 0x31,
		0xb3, 0x14, 0xfc, 0x88, 0x2d, 0xef, 0x8a, 0xd5,
		0xfd, 0xc7, 0x13, 0x31, 0xf8, 0x5b, 0x9b, 0xf9,
		0x78, 0xfa, 0x1a, 0xd1, 0xf8, 0xfe, 0xfb, 0x8a,
		0x9a, 0x3e, 0x1a, 0x61, 0x18, 0xe4, 0x2d, 0xe2,
		0xc0, 0x8f, 0x4d, 0xcd, 0xcf, 0xa3, 0x84, 0x4f,
		0xf7, 0x51, 0x02, 0x09, 0x1c, 0xd7, 0xfc, 0x23,
		0x2d, 0xa5, 0x77, 0xe4, 0xf3, 0xf3, 0xd5, 0x9b,
		0x64, 0x0e, 0x1e, 0x1e, 0x3b, 0xfe, 0xf7, 0x87,
		0x9e, 0xca
	};
	uint8_t act2[sizeof(exp2)];
	int ret;
	LC_HASH_CTX_ON_STACK(ctx, shake_512);
	LC_SHAKE_512_CTX_ON_STACK(shake512_stack);
	LC_HASH_CTX_ON_STACK(cctx, shake_512);

	printf("hash ctx %s (%s implementation) len %u\n", name,
	       shake_512 == lc_shake512_c ? "C" : "accelerated",
	       (unsigned int)LC_HASH_CTX_SIZE(shake_512));

	if (lc_hash_init(ctx))
		return 1;
	lc_hash_update(ctx, msg1, sizeof(msg1));
	lc_hash_set_digestsize(ctx, sizeof(act1));
	lc_hash_final(ctx, act1);
	ret = lc_compare(act1, exp1, sizeof(act1), "SHAKE512 1");
	lc_memset_secure(act1, 0, sizeof(act1));
	if (lc_sponge_extract_bytes(shake_512, ctx->hash_state, act1, 0,
				    sizeof(act1)))
		return 1;
	ret += lc_compare(act1, exp1, sizeof(act1), "SHAKE512 1 extact data");
	lc_hash_zero(ctx);

	if (ret)
		return ret;

	if (lc_hash_init(shake512_stack))
		return 1;
	lc_hash_update(shake512_stack, msg1, sizeof(msg1));
	lc_hash_set_digestsize(shake512_stack, sizeof(act1));
	lc_hash_final(shake512_stack, act1);
	ret = lc_compare(act1, exp1, sizeof(act1), "SHAKE512 1 - separate ctx");
	lc_hash_zero(shake512_stack);
	if (ret)
		return ret;

	if (lc_hash_init(ctx))
		return 1;
	lc_hash_update(ctx, msg2, sizeof(msg2));
	lc_hash_set_digestsize(ctx, sizeof(act2));
	lc_hash_final(ctx, act2);
	ret = lc_compare(act2, exp2, sizeof(act2), "SHAKE512 2");
	lc_hash_zero(ctx);

	//TODO reenable, once cSHAKE is updated for 512 bits
#if 0
	if (ret)
		return ret;

	/* Check if cSHAKE falls back to SHAKE */
	if (lc_cshake_init(cctx, NULL, 0, NULL, 0))
		return 1;
	lc_hash_update(cctx, msg2, sizeof(msg2));
	lc_hash_set_digestsize(cctx, sizeof(act2));
	lc_hash_final(cctx, act2);
	ret = lc_compare(act2, exp2, sizeof(act2), "SHAKE512 3");
	lc_hash_zero(cctx);
#endif

	return ret;
}

static int shake512_tester(void)
{
	int ret = 0;

	LC_EXEC_ONE_TEST(lc_shake512);
	LC_EXEC_ONE_TEST(lc_shake512_c);
	LC_EXEC_ONE_TEST(lc_shake512_arm_asm);
	LC_EXEC_ONE_TEST(lc_shake512_arm_ce);
	LC_EXEC_ONE_TEST(lc_shake512_arm_neon);
	LC_EXEC_ONE_TEST(lc_shake512_avx2);
	LC_EXEC_ONE_TEST(lc_shake512_avx512);
	LC_EXEC_ONE_TEST(lc_shake512_riscv_asm);
	LC_EXEC_ONE_TEST(lc_shake512_riscv_asm_zbb);

	return ret;
}

LC_TEST_FUNC(int, main, int argc, char *argv[])
{
	int ret;

	(void)argc;
	(void)argv;

	ret = shake512_tester();

	ret = test_validate_status(ret, LC_ALG_STATUS_SHAKE512, 0);
	ret += test_print_status();

	return ret;
}
