"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Match = void 0;
const assert = require("assert");
const edge_1 = require("../edge");
const utils_1 = require("../utils");
const base_1 = require("./base");
/**
 * This node matches characters/sequences and forwards the execution according
 * to matched character with optional attached value (See `.select()`).
 */
class Match extends base_1.Node {
    /**
     * Set character transformation function.
     *
     * @param transform  Transformation to apply. Can be created with
     *                   `builder.transform.*()` methods.
     */
    transform(transformFn) {
        this.transformFn = transformFn;
        return this;
    }
    /**
     * Match sequence/character and forward execution to `next` on success,
     * consuming matched bytes of the input.
     *
     * No value is attached on such execution forwarding, and the target node
     * **must not** be an `Invoke` node with a callback expecting the value.
     *
     * @param value  Sequence/character to be matched
     * @param next   Target node to be executed on success.
     */
    match(value, next) {
        if (Array.isArray(value)) {
            for (const subvalue of value) {
                this.match(subvalue, next);
            }
            return this;
        }
        const buffer = utils_1.toBuffer(value);
        const edge = new edge_1.Edge(next, false, buffer, undefined);
        this.addEdge(edge);
        return this;
    }
    /**
     * Match character and forward execution to `next` on success
     * without consuming one byte of the input.
     *
     * No value is attached on such execution forwarding, and the target node
     * **must not** be an `Invoke` with a callback expecting the value.
     *
     * @param value  Character to be matched
     * @param next   Target node to be executed on success.
     */
    peek(value, next) {
        if (Array.isArray(value)) {
            for (const subvalue of value) {
                this.peek(subvalue, next);
            }
            return this;
        }
        const buffer = utils_1.toBuffer(value);
        assert.strictEqual(buffer.length, 1, '`.peek()` accepts only single character keys');
        const edge = new edge_1.Edge(next, true, buffer, undefined);
        this.addEdge(edge);
        return this;
    }
    /**
     * Match character/sequence and forward execution to `next` on success
     * consumed matched bytes of the input.
     *
     * Value is attached on such execution forwarding, and the target node
     * **must** be an `Invoke` with a callback expecting the value.
     *
     * Possible signatures:
     *
     * * `.select(key, value [, next ])`
     * * `.select({ key: value } [, next])`
     *
     * @param keyOrMap     Either a sequence to match, or a map from sequences to
     *                     values
     * @param valueOrNext  Either an integer value to be forwarded to the target
     *                     node, or an otherwise node
     * @param next         Convenience param. Same as calling `.otherwise(...)`
     */
    select(keyOrMap, valueOrNext, next) {
        // .select({ key: value, ... }, next)
        if (typeof keyOrMap === 'object') {
            assert(valueOrNext instanceof base_1.Node, 'Invalid `next` argument of `.select()`');
            assert.strictEqual(next, undefined, 'Invalid argument count of `.select()`');
            const map = keyOrMap;
            next = valueOrNext;
            Object.keys(map).forEach((mapKey) => {
                const numKey = mapKey;
                this.select(numKey, map[numKey], next);
            });
            return this;
        }
        // .select(key, value, next)
        assert.strictEqual(typeof valueOrNext, 'number', 'Invalid `value` argument of `.select()`');
        assert.notStrictEqual(next, undefined, 'Invalid `next` argument of `.select()`');
        const key = utils_1.toBuffer(keyOrMap);
        const value = valueOrNext;
        const edge = new edge_1.Edge(next, false, key, value);
        this.addEdge(edge);
        return this;
    }
    // Limited public use
    /**
     * Get tranformation function
     */
    getTransform() {
        return this.transformFn;
    }
}
exports.Match = Match;
//# sourceMappingURL=match.js.map