"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Edge = void 0;
const assert = require("assert");
const buffer_1 = require("buffer");
const node_1 = require("./node");
/**
 * This class represents an edge in the parser graph.
 */
class Edge {
    /**
     * @param node       Edge target
     * @param noAdvance  If `true` - the parent should not consume bytes before
     *                   moving to the target `node`
     * @param key        `Buffer` for `node.Match`, `number` for `node.Invoke`,
     *                   `undefined` for edges created with `.otherwise()`
     * @param value      `.select()` value associated with the edge
     */
    constructor(node, noAdvance, key, value) {
        this.node = node;
        this.noAdvance = noAdvance;
        this.key = key;
        this.value = value;
        if (node instanceof node_1.Invoke) {
            if (value === undefined) {
                assert.strictEqual(node.code.signature, 'match', 'Invalid Invoke\'s code signature');
            }
            else {
                assert.strictEqual(node.code.signature, 'value', 'Invalid Invoke\'s code signature');
            }
        }
        else {
            assert.strictEqual(value, undefined, 'Attempting to pass value to non-Invoke node');
        }
        if (buffer_1.Buffer.isBuffer(key)) {
            assert(key.length > 0, 'Invalid edge buffer length');
            if (noAdvance) {
                assert.strictEqual(key.length, 1, 'Only 1-char keys are allowed in `noAdvance` edges');
            }
        }
    }
    /**
     * Comparator for `.sort()` function.
     */
    static compare(a, b) {
        if (typeof a.key === 'number') {
            return a.key - b.key;
        }
        return a.key.compare(b.key);
    }
}
exports.Edge = Edge;
//# sourceMappingURL=edge.js.map