/*
 * Copyright (C) 2024 - 2025, Stephan Mueller <smueller@chronox.de>
 *
 * License: see LICENSE file in root directory
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ALL OF
 * WHICH ARE HEREBY DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF NOT ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

#ifndef SPHINCS_TESTER_VECTORS_SHAKE_128S_H
#define SPHINCS_TESTER_VECTORS_SHAKE_128S_H

#ifdef __cplusplus
extern "C" {
#endif

struct lc_sphincs_test {
	uint8_t seed[LC_SPX_SEEDBYTES];
	uint8_t msg[33];
	uint8_t pk[sizeof(struct lc_sphincs_pk)];
	uint8_t sk[sizeof(struct lc_sphincs_sk)];
	uint8_t sig[sizeof(struct lc_sphincs_sig)];
};

static const struct lc_sphincs_test tests[] =
{
	{
		.seed = {
			0x7c, 0x99, 0x35, 0xa0, 0xb0, 0x76, 0x94, 0xaa,
			0x0c, 0x6d, 0x10, 0xe4, 0xdb, 0x6b, 0x1a, 0xdd,
			0x2f, 0xd8, 0x1a, 0x25, 0xcc, 0xb1, 0x48, 0x03,
			0x2d, 0xcd, 0x73, 0x99, 0x36, 0x73, 0x7f, 0x2d,
			0xb5, 0x05, 0xd7, 0xcf, 0xad, 0x1b, 0x49, 0x74,
			0x99, 0x32, 0x3c, 0x86, 0x86, 0x32, 0x5e, 0x47,
		},
		.msg = {
			0xD8, 0x1C, 0x4D, 0x8D, 0x73, 0x4F, 0xCB, 0xFB,
			0xEA, 0xDE, 0x3D, 0x3F, 0x8A, 0x03, 0x9F, 0xAA,
			0x2A, 0x2C, 0x99, 0x57, 0xE8, 0x35, 0xAD, 0x55,
			0xB2, 0x2E, 0x75, 0xBF, 0x57, 0xBB, 0x55, 0x6A,
			0xC8
		},
		.pk = {
			0xB5, 0x05, 0xD7, 0xCF, 0xAD, 0x1B, 0x49, 0x74,
			0x99, 0x32, 0x3C, 0x86, 0x86, 0x32, 0x5E, 0x47,
			0xAC, 0x52, 0x49, 0x02, 0xFC, 0x81, 0xF5, 0x03,
			0x2B, 0xC2, 0x7B, 0x17, 0xD9, 0x26, 0x1E, 0xBD
		},
		.sk = {
			0x7C, 0x99, 0x35, 0xA0, 0xB0, 0x76, 0x94, 0xAA,
			0x0C, 0x6D, 0x10, 0xE4, 0xDB, 0x6B, 0x1A, 0xDD,
			0x2F, 0xD8, 0x1A, 0x25, 0xCC, 0xB1, 0x48, 0x03,
			0x2D, 0xCD, 0x73, 0x99, 0x36, 0x73, 0x7F, 0x2D,
			0xB5, 0x05, 0xD7, 0xCF, 0xAD, 0x1B, 0x49, 0x74,
			0x99, 0x32, 0x3C, 0x86, 0x86, 0x32, 0x5E, 0x47,
			0xAC, 0x52, 0x49, 0x02, 0xFC, 0x81, 0xF5, 0x03,
			0x2B, 0xC2, 0x7B, 0x17, 0xD9, 0x26, 0x1E, 0xBD
		},
		.sig = {
			0x43, 0xD2, 0x19, 0x8D, 0xB5, 0x09, 0x42, 0x3D,
			0xA3, 0x0B, 0xFF, 0x2D, 0x05, 0x44, 0x3F, 0x11,
			0x1B, 0xA6, 0x6D, 0x6F, 0x78, 0x2B, 0xDD, 0x24,
			0x85, 0x58, 0x9E, 0xA1, 0x5D, 0x2B, 0x8F, 0xF0,
			0xA0, 0x0C, 0x06, 0xEA, 0xED, 0xC8, 0xC2, 0x2D,
			0xCB, 0x86, 0xF3, 0xDF, 0x8B, 0x52, 0xA3, 0xBD,
			0x14, 0x4D, 0x4E, 0xD6, 0xF1, 0x16, 0x74, 0x31,
			0xA9, 0x5D, 0xC6, 0x01, 0x88, 0x79, 0xB6, 0xB0,
			0xF9, 0x81, 0x37, 0x97, 0x20, 0x4E, 0xC2, 0xB0,
			0x55, 0x8B, 0xAD, 0x17, 0xB3, 0x2E, 0x6D, 0xD9,
			0x88, 0x08, 0x6A, 0x03, 0x2C, 0x0A, 0xCB, 0xCF,
			0x2C, 0x13, 0x49, 0xFF, 0xC1, 0x6C, 0x4A, 0xF7,
			0x59, 0x36, 0x5F, 0xF7, 0x4A, 0xFE, 0x4B, 0x8D,
			0xC3, 0xFA, 0xC5, 0xB2, 0xCD, 0xA7, 0xBA, 0x65,
			0x6C, 0x36, 0xC0, 0x86, 0xE5, 0x84, 0x68, 0xC0,
			0x1E, 0xDD, 0xFC, 0x95, 0x9F, 0xBD, 0xC8, 0x53,
			0x2D, 0x75, 0xE7, 0x9C, 0xF3, 0x37, 0x47, 0x56,
			0xCC, 0x04, 0x91, 0xCF, 0xEF, 0x55, 0x59, 0x21,
			0xEC, 0x85, 0x67, 0xBF, 0xF0, 0xB6, 0xF2, 0x16,
			0xAC, 0x4A, 0x4F, 0x20, 0x0D, 0xA6, 0x3B, 0x5C,
			0x6D, 0x3A, 0x5C, 0x32, 0x73, 0xAA, 0x7A, 0x42,
			0x66, 0xAD, 0xF0, 0x83, 0xD3, 0x12, 0x61, 0x03,
			0xC7, 0x3F, 0xE6, 0x3A, 0x6E, 0x05, 0xE4, 0x7E,
			0x8B, 0x9A, 0x52, 0x0F, 0x00, 0xF3, 0x2A, 0x69,
			0x7D, 0x0F, 0xF3, 0xA5, 0xEE, 0x84, 0x09, 0x31,
			0x37, 0x73, 0x18, 0x8F, 0x30, 0x0B, 0x39, 0xE5,
			0xB4, 0x96, 0x7F, 0xEB, 0xF7, 0x7C, 0x0F, 0x23,
			0x22, 0x67, 0x85, 0xEE, 0x9D, 0xCE, 0x33, 0x5C,
			0xEF, 0xB1, 0xCE, 0x84, 0xB0, 0x67, 0x30, 0x58,
			0x1B, 0xCE, 0xF4, 0xD4, 0x5F, 0x24, 0xAE, 0xE4,
			0x96, 0xA6, 0x0B, 0xD4, 0x75, 0x9B, 0x7B, 0x20,
			0x69, 0x22, 0x41, 0x85, 0x0E, 0xAE, 0x1D, 0xE7,
			0x0C, 0x7C, 0x42, 0x87, 0xB9, 0xF3, 0xB9, 0x62,
			0xA6, 0x6E, 0x0F, 0x23, 0xD1, 0x30, 0x1B, 0x84,
			0x48, 0xBB, 0x3D, 0xC5, 0x45, 0xBE, 0x0E, 0xF8,
			0xD0, 0xEC, 0x0B, 0xE0, 0x45, 0xD3, 0x3A, 0xE4,
			0xB2, 0xDC, 0x0C, 0x5D, 0x00, 0x2C, 0x26, 0x99,
			0xE8, 0xF4, 0x9B, 0xF3, 0xBC, 0xB1, 0x36, 0x76,
			0xBE, 0xEF, 0xE1, 0x11, 0x86, 0xA2, 0x0A, 0x95,
			0x70, 0x27, 0xAC, 0x48, 0xEE, 0x6D, 0xD3, 0x3B,
			0xC0, 0x89, 0x5D, 0xF9, 0x84, 0x7F, 0xD1, 0xC6,
			0x7A, 0x75, 0x37, 0x77, 0xD2, 0x1A, 0xC4, 0x64,
			0x27, 0x51, 0x13, 0x90, 0x61, 0xCC, 0xA8, 0x36,
			0x99, 0x82, 0x2C, 0x13, 0x56, 0x78, 0x33, 0xB9,
			0x41, 0xFE, 0x59, 0x54, 0xBF, 0xF0, 0x96, 0x9A,
			0x4B, 0x20, 0xE0, 0x82, 0x9D, 0x77, 0xE2, 0x4E,
			0xD0, 0xE0, 0x2A, 0x00, 0xA2, 0xCE, 0x9F, 0x7E,
			0x64, 0x92, 0x3F, 0xA6, 0x1F, 0x0D, 0xA1, 0xAF,
			0xDC, 0x59, 0x78, 0xDC, 0x06, 0x3A, 0xFE, 0xAC,
			0xB7, 0x10, 0x8E, 0xE0, 0x8A, 0xAA, 0x55, 0xB7,
			0x11, 0xDF, 0x00, 0xBB, 0xF1, 0xC7, 0x1D, 0x69,
			0x8B, 0x38, 0x9E, 0x6A, 0xD0, 0xEE, 0x2A, 0xF4,
			0xFA, 0xD1, 0xD8, 0xF8, 0xC8, 0x7D, 0x53, 0xEE,
			0xAC, 0x1F, 0x82, 0xA1, 0x62, 0xA9, 0x5C, 0xD5,
			0xCC, 0xE6, 0xDF, 0xC9, 0x90, 0x8A, 0x1D, 0xE3,
			0x3A, 0x2B, 0x26, 0xB4, 0x1B, 0xBC, 0x4F, 0xFD,
			0xA8, 0xE1, 0x36, 0x87, 0x9F, 0x10, 0x34, 0x1A,
			0x71, 0x3D, 0x62, 0xC1, 0x07, 0xF3, 0x23, 0x8C,
			0x38, 0x69, 0x3A, 0xFF, 0x2E, 0x1F, 0xE1, 0x5A,
			0xDD, 0x83, 0x80, 0x67, 0x1B, 0x2F, 0xAC, 0x8F,
			0xDB, 0x3A, 0x4F, 0xFA, 0xCD, 0x14, 0x3F, 0x5A,
			0x00, 0xE2, 0x1C, 0xAC, 0xCB, 0xE7, 0xD9, 0x5D,
			0x4C, 0x31, 0xC4, 0xDA, 0xF7, 0x11, 0x05, 0x29,
			0xDE, 0x59, 0x9F, 0xB6, 0xE8, 0xAA, 0x4F, 0x71,
			0x8C, 0x61, 0x72, 0xF4, 0xF1, 0x0C, 0x4C, 0x1D,
			0xD7, 0x31, 0x0F, 0x8E, 0x44, 0xD1, 0x8F, 0xB9,
			0xBB, 0x6B, 0x90, 0x6A, 0xE7, 0xCE, 0x97, 0x3D,
			0xEA, 0xDF, 0xC8, 0x2D, 0x67, 0x04, 0x76, 0x2E,
			0xD6, 0x11, 0x65, 0xF6, 0xCA, 0x11, 0x83, 0x13,
			0x4B, 0x6C, 0x83, 0x4B, 0xF6, 0xB4, 0xE4, 0xCE,
			0x19, 0x70, 0x0B, 0xB5, 0x4F, 0xB2, 0xD0, 0xF2,
			0x82, 0xB1, 0x1E, 0xE5, 0xB7, 0xF6, 0x8C, 0x72,
			0xCF, 0x32, 0xFF, 0x9E, 0x7D, 0x13, 0x56, 0xBD,
			0x53, 0xFD, 0xCC, 0xE0, 0xD0, 0x3C, 0x43, 0xD0,
			0xEB, 0xF6, 0xF7, 0xD8, 0x84, 0x1A, 0x16, 0xDD,
			0xB4, 0x99, 0x44, 0xD0, 0x13, 0x74, 0xEC, 0xB9,
			0x45, 0xE6, 0xC5, 0xF0, 0x65, 0x9D, 0x5F, 0x51,
			0xDE, 0x0B, 0x27, 0xA8, 0x34, 0xE2, 0xE7, 0xBE,
			0xA7, 0x8A, 0x60, 0x9D, 0xA7, 0x5D, 0x7F, 0x2C,
			0x6A, 0x40, 0xBA, 0x91, 0x10, 0xA2, 0xC3, 0x31,
			0x9D, 0xB7, 0x77, 0x5B, 0xF6, 0x22, 0x6B, 0x9A,
			0x8E, 0x32, 0x4D, 0xC4, 0x41, 0x18, 0x24, 0xA8,
			0x8D, 0xB9, 0x5C, 0xC2, 0xFD, 0x96, 0xE4, 0xBC,
			0x24, 0xF1, 0xEC, 0xB6, 0xCE, 0x2B, 0x92, 0x93,
			0x02, 0x0C, 0x28, 0xDE, 0xAC, 0xEC, 0x1E, 0xB3,
			0x31, 0x3D, 0x4E, 0x3D, 0xFD, 0x24, 0xB4, 0x03,
			0x68, 0x6F, 0x16, 0x27, 0x2C, 0xAC, 0x3A, 0xCA,
			0x95, 0x08, 0x02, 0x57, 0x07, 0x1A, 0x54, 0xF7,
			0x45, 0xFF, 0xB4, 0x70, 0x8F, 0xF2, 0xD0, 0x2D,
			0xE9, 0x4D, 0x7E, 0x9B, 0xF8, 0xD4, 0x5F, 0x64,
			0x59, 0x17, 0xC7, 0x13, 0x5D, 0x6B, 0xC0, 0xA1,
			0xCA, 0x0A, 0x99, 0xBC, 0x4E, 0x33, 0xA6, 0x89,
			0x07, 0xAA, 0x65, 0xA5, 0x8B, 0x58, 0x6C, 0x56,
			0xE1, 0xD8, 0x1A, 0xF6, 0xCB, 0x57, 0xFA, 0x5D,
			0x56, 0xB3, 0x56, 0x76, 0x87, 0xEC, 0xEF, 0x53,
			0x2B, 0xB5, 0xAA, 0xAF, 0x20, 0x41, 0xB5, 0x10,
			0x15, 0x38, 0x29, 0x42, 0x96, 0xAE, 0x4C, 0x11,
			0x89, 0xA5, 0x10, 0x0E, 0xB1, 0x9B, 0x55, 0x31,
			0x20, 0x16, 0xC5, 0x75, 0xCB, 0xBB, 0x68, 0x8F,
			0x20, 0xBA, 0x18, 0x6D, 0xD4, 0x8E, 0x41, 0x61,
			0x64, 0xC2, 0x9B, 0x2E, 0xB7, 0xB5, 0x99, 0x79,
			0x81, 0x4B, 0x5A, 0x8E, 0x76, 0x55, 0x39, 0x97,
			0x99, 0xBF, 0x79, 0xEE, 0xAB, 0x3E, 0xE7, 0x6D,
			0x4C, 0x97, 0xDF, 0x28, 0x22, 0x65, 0x56, 0x4F,
			0x2F, 0xA8, 0x97, 0x1A, 0x1E, 0xCC, 0xA0, 0xC4,
			0x6B, 0x59, 0xCC, 0x6B, 0xA2, 0x53, 0x53, 0x1C,
			0x17, 0xAB, 0x71, 0x25, 0xCF, 0x2A, 0xAD, 0x60,
			0xA1, 0x20, 0xC7, 0xD4, 0xB6, 0x31, 0xB1, 0xBA,
			0xF1, 0x87, 0x18, 0x2C, 0x7D, 0x75, 0x82, 0xDA,
			0x32, 0x32, 0x25, 0x12, 0x15, 0xFF, 0xD6, 0xA2,
			0xA5, 0x5C, 0x62, 0x7B, 0xA8, 0xD5, 0xCA, 0xFE,
			0x76, 0x15, 0x04, 0xD8, 0x34, 0x1F, 0x29, 0x3E,
			0x71, 0x39, 0x87, 0xD6, 0xE0, 0xCA, 0x2E, 0xAB,
			0x37, 0x3C, 0x51, 0x31, 0xC2, 0xD3, 0x80, 0x51,
			0xC3, 0x5B, 0x17, 0x91, 0x89, 0x37, 0xB9, 0xFE,
			0xE9, 0x83, 0x82, 0xC2, 0x77, 0x64, 0x0D, 0xE0,
			0xCC, 0xEC, 0x45, 0xBA, 0x22, 0xD9, 0xD1, 0x89,
			0xEE, 0xA5, 0x05, 0xA2, 0x1C, 0x85, 0x94, 0xDD,
			0x9B, 0x12, 0xE6, 0x9A, 0x7F, 0xAF, 0x58, 0xED,
			0x26, 0x9B, 0x71, 0x8A, 0xBE, 0xEA, 0x46, 0x21,
			0x39, 0x1D, 0x7F, 0xB4, 0xC6, 0xE0, 0x03, 0x7B,
			0xDA, 0xF4, 0xA9, 0xAC, 0x73, 0x19, 0x16, 0x74,
			0x9E, 0x2A, 0x17, 0xD7, 0x04, 0xCF, 0x56, 0x16,
			0x8D, 0x97, 0xC1, 0x7B, 0x25, 0x7E, 0x24, 0x83,
			0x16, 0xAA, 0x9D, 0xA1, 0x5D, 0x82, 0x2E, 0xE3,
			0xC3, 0x25, 0xBC, 0x05, 0x19, 0x17, 0x36, 0x41,
			0x70, 0x07, 0xEA, 0x82, 0x08, 0x86, 0x18, 0xD7,
			0x53, 0x1F, 0xFC, 0xEF, 0x25, 0x5B, 0x2D, 0xE2,
			0xBF, 0x9F, 0xCA, 0xEB, 0x29, 0xD8, 0x3E, 0x56,
			0x7A, 0x08, 0xDD, 0x3D, 0x3C, 0x22, 0x92, 0x09,
			0xAF, 0x96, 0xBA, 0x71, 0xD8, 0x27, 0x4F, 0xDA,
			0xE3, 0x24, 0x70, 0x28, 0x78, 0xD9, 0x9A, 0xC0,
			0x5E, 0x99, 0x0E, 0x0D, 0x6F, 0x34, 0xC8, 0x79,
			0xD1, 0x92, 0x79, 0xF5, 0x75, 0x41, 0xF2, 0x94,
			0x96, 0x64, 0x5C, 0xAD, 0x4A, 0x63, 0x67, 0x93,
			0x38, 0x5B, 0x0A, 0x5D, 0xC2, 0x1D, 0x56, 0x59,
			0x37, 0xFC, 0x36, 0x38, 0x4D, 0xEA, 0x4B, 0xEB,
			0xB5, 0x74, 0x6C, 0x10, 0x74, 0x8E, 0xFC, 0xBD,
			0x6B, 0x19, 0x25, 0xA7, 0x4E, 0x3A, 0xC4, 0x67,
			0xD7, 0xAF, 0x45, 0x6E, 0x0B, 0xA1, 0xE4, 0x7A,
			0x2F, 0xAB, 0x24, 0xE8, 0x31, 0x1C, 0x14, 0xD8,
			0x40, 0xB4, 0x99, 0xC9, 0x14, 0x0E, 0x99, 0xA4,
			0x99, 0x33, 0x79, 0xB9, 0xB7, 0x62, 0xB3, 0xAD,
			0xC9, 0x49, 0x9D, 0x5C, 0x86, 0xD0, 0x7B, 0xC6,
			0xA1, 0x59, 0x87, 0x6A, 0x99, 0x62, 0xD8, 0xA4,
			0x43, 0x51, 0x4E, 0x81, 0x2F, 0x75, 0xC6, 0x0D,
			0xC5, 0x00, 0x28, 0x38, 0x8C, 0x62, 0x73, 0x29,
			0x6E, 0x72, 0x08, 0xA3, 0xFA, 0x61, 0x82, 0x56,
			0x2D, 0x10, 0xD7, 0x14, 0x2A, 0x99, 0xDA, 0x06,
			0x86, 0xEF, 0x8F, 0x05, 0xE5, 0x64, 0x44, 0x6C,
			0x6B, 0xB3, 0x2F, 0xFD, 0xEE, 0x9E, 0xDD, 0x13,
			0xAE, 0xE5, 0x80, 0x27, 0xD2, 0x9E, 0x71, 0x95,
			0x48, 0xB6, 0x7D, 0x75, 0xEF, 0xE9, 0x58, 0x1A,
			0x33, 0x74, 0xC6, 0x6F, 0x65, 0xA1, 0xCD, 0x9E,
			0xF9, 0xE9, 0x8E, 0x6B, 0x57, 0xC4, 0x03, 0x21,
			0x27, 0x39, 0xDF, 0x6F, 0xD2, 0xDE, 0x6C, 0x8A,
			0x39, 0xDE, 0xCC, 0x7C, 0xD3, 0x3E, 0x37, 0xDB,
			0x3A, 0x0F, 0x43, 0x29, 0x6C, 0xB9, 0x87, 0xE8,
			0x75, 0x6D, 0x4B, 0x29, 0xDC, 0x22, 0x77, 0x33,
			0xBD, 0xEE, 0x1D, 0x2F, 0x01, 0x67, 0x9D, 0xAB,
			0x92, 0xCE, 0x50, 0x6E, 0x2F, 0xC7, 0x7A, 0x70,
			0x79, 0x87, 0x87, 0xB2, 0xE9, 0x5B, 0xE8, 0xE9,
			0xBF, 0x80, 0xD0, 0xB6, 0x4A, 0xF8, 0xEA, 0xA6,
			0xCE, 0xDA, 0x80, 0xFE, 0x85, 0xA0, 0xCE, 0xAF,
			0x81, 0xF3, 0x35, 0xB9, 0x9A, 0x18, 0x99, 0xA3,
			0xD9, 0xD6, 0x09, 0xE7, 0xBA, 0x60, 0x6E, 0xB2,
			0xAB, 0xAB, 0xF2, 0xBB, 0xCE, 0x1B, 0xC8, 0xF9,
			0x9E, 0xAF, 0x60, 0x74, 0xCF, 0x1C, 0x7E, 0x07,
			0x98, 0x96, 0xEB, 0x09, 0x82, 0x7C, 0x16, 0xD0,
			0xCD, 0x48, 0x33, 0x37, 0x7C, 0x46, 0xA3, 0x37,
			0xA7, 0x95, 0x0B, 0x31, 0xB6, 0x56, 0x66, 0x24,
			0x02, 0xE8, 0xBA, 0x83, 0x86, 0x68, 0xA3, 0x15,
			0xAC, 0x53, 0x13, 0x15, 0xA9, 0xA5, 0x6A, 0xF1,
			0x87, 0x29, 0xEE, 0x25, 0xF5, 0x37, 0x11, 0xC0,
			0x9D, 0x25, 0xC1, 0x73, 0xAA, 0x0E, 0x4D, 0x2B,
			0xEC, 0x72, 0xDB, 0x4B, 0x9C, 0xB4, 0x21, 0x0D,
			0x52, 0xA8, 0xFB, 0x2F, 0x8B, 0x26, 0x71, 0xB1,
			0xEC, 0x71, 0x1A, 0x4D, 0xA8, 0xA3, 0x57, 0xDF,
			0xBB, 0x0D, 0x2E, 0xC9, 0x73, 0x4A, 0x50, 0xE1,
			0xDB, 0x92, 0x35, 0x2A, 0xCE, 0x0F, 0x26, 0xF5,
			0x0C, 0xFB, 0x76, 0xFD, 0x17, 0xA0, 0x8D, 0xEC,
			0xBB, 0x19, 0xC2, 0x41, 0x7A, 0x9D, 0xC7, 0x19,
			0xF2, 0xEC, 0xAC, 0x4A, 0x8E, 0x7C, 0x48, 0x27,
			0x55, 0x33, 0xDE, 0xF5, 0xC0, 0x87, 0x88, 0xDC,
			0x4B, 0x47, 0xEC, 0x81, 0x96, 0x0B, 0x25, 0xA5,
			0x7D, 0xBA, 0x27, 0x62, 0xF5, 0xA0, 0x70, 0x03,
			0x7C, 0x50, 0xA4, 0x88, 0x3F, 0xE9, 0x02, 0xEB,
			0xCB, 0x15, 0x74, 0x99, 0x8D, 0xD5, 0xE8, 0xB1,
			0xE3, 0x4E, 0xA5, 0xAE, 0xA2, 0x0B, 0xBB, 0xEF,
			0xFD, 0xB5, 0xD6, 0x16, 0x36, 0x88, 0xE4, 0xE1,
			0xBD, 0xC9, 0x61, 0x9F, 0x12, 0xB7, 0x8D, 0x20,
			0xE8, 0xC0, 0x73, 0xF8, 0x1D, 0xA8, 0xBB, 0xE4,
			0x8B, 0xDE, 0x89, 0x34, 0xBC, 0x71, 0x86, 0xDA,
			0x9D, 0x29, 0xD1, 0xF6, 0x70, 0xA3, 0x22, 0xBF,
			0x9F, 0xEB, 0xCA, 0x92, 0x91, 0x5E, 0x39, 0x3C,
			0x18, 0x78, 0x89, 0x5C, 0x04, 0xB8, 0xC3, 0x65,
			0xE4, 0xD3, 0x99, 0xAC, 0x55, 0x1A, 0x55, 0xC5,
			0x42, 0x64, 0xE3, 0xFC, 0x61, 0x76, 0xCB, 0xCD,
			0x10, 0x17, 0x90, 0x86, 0x3C, 0xDA, 0xB3, 0x95,
			0x74, 0xA4, 0xDD, 0x5C, 0x9E, 0xDD, 0x69, 0xA0,
			0x1D, 0xF2, 0x0A, 0x10, 0xE5, 0xAB, 0xFF, 0x31,
			0xB4, 0xE2, 0x04, 0xF5, 0xCF, 0x7E, 0x1D, 0xC9,
			0xA2, 0x76, 0x26, 0xEC, 0x3B, 0xF0, 0x6D, 0x28,
			0xFA, 0xD0, 0x8C, 0x10, 0x67, 0x48, 0x30, 0xD7,
			0xAB, 0xC5, 0x47, 0x72, 0xD9, 0x5A, 0xCE, 0x66,
			0x76, 0x57, 0x57, 0x34, 0x00, 0x07, 0xA3, 0x53,
			0x63, 0xD2, 0x70, 0xF4, 0x10, 0xA6, 0xBC, 0xF2,
			0x0F, 0x2C, 0xA5, 0x4D, 0xFD, 0xB0, 0x0D, 0x9B,
			0xE8, 0xFA, 0x7E, 0xA5, 0xB7, 0x9B, 0xF8, 0x18,
			0x2F, 0x16, 0xB9, 0x5F, 0x98, 0x50, 0xCE, 0x4C,
			0xAC, 0xFF, 0x1E, 0x66, 0xBE, 0xC2, 0x02, 0xB5,
			0x7B, 0x85, 0xB3, 0x7C, 0xDD, 0x2C, 0x39, 0x00,
			0x1D, 0x29, 0x50, 0x66, 0x63, 0x68, 0xAF, 0xA2,
			0x1D, 0xE5, 0xCE, 0x68, 0xF5, 0x48, 0x33, 0xA8,
			0x8D, 0xA1, 0x7B, 0x49, 0xC4, 0xE6, 0x62, 0x43,
			0x56, 0x0E, 0xC6, 0x1A, 0x6E, 0xFD, 0x5D, 0x3A,
			0x29, 0x66, 0xA7, 0x6D, 0xF2, 0xDC, 0x08, 0xC4,
			0xE5, 0xF0, 0x2F, 0x8B, 0x8C, 0xD7, 0x1B, 0x90,
			0x4D, 0xDD, 0x4B, 0xFD, 0x73, 0xA5, 0xC8, 0x48,
			0x9B, 0x7E, 0xB8, 0x13, 0xCA, 0x3D, 0xA6, 0xB8,
			0xDB, 0xEA, 0x53, 0x63, 0x54, 0xE0, 0x14, 0x28,
			0xDD, 0x6D, 0xC4, 0x2D, 0xB2, 0x32, 0x57, 0xA2,
			0x0E, 0x32, 0x2F, 0x68, 0x5B, 0xB8, 0x2B, 0x20,
			0xF0, 0xED, 0xC4, 0x83, 0x51, 0xC2, 0x2B, 0x75,
			0xE0, 0xAA, 0x8A, 0xDC, 0x56, 0x7F, 0x17, 0x2E,
			0x65, 0x43, 0x60, 0xE0, 0x94, 0xC1, 0x97, 0x54,
			0x2F, 0x39, 0x96, 0x5B, 0xD9, 0x00, 0x46, 0x21,
			0xC9, 0xEE, 0x32, 0x97, 0x87, 0x0E, 0xD8, 0x18,
			0xF9, 0x80, 0xA7, 0x1E, 0xC4, 0xA8, 0xF8, 0x18,
			0x1E, 0x9B, 0xE5, 0xBE, 0x1E, 0xF6, 0xA6, 0x60,
			0xCB, 0xF6, 0x86, 0x37, 0xE5, 0x4B, 0x5A, 0xFA,
			0xBB, 0xC5, 0xF9, 0xDC, 0x61, 0x93, 0x30, 0x14,
			0xCB, 0x52, 0xB4, 0xD2, 0x62, 0x4A, 0x24, 0xAC,
			0xA3, 0xC6, 0xF5, 0xCA, 0x80, 0xDD, 0x5A, 0xEE,
			0x93, 0xD0, 0x15, 0x5A, 0xF7, 0x03, 0xC0, 0xAC,
			0xA4, 0xA9, 0x26, 0x6C, 0xD9, 0xB5, 0x6F, 0x3F,
			0x15, 0x2F, 0xC4, 0xFC, 0xA8, 0xE7, 0xDC, 0xE3,
			0x21, 0xA1, 0x88, 0x68, 0x2F, 0xB3, 0x6E, 0x6F,
			0x7A, 0x73, 0x6F, 0xD4, 0xE9, 0x97, 0x2A, 0x9F,
			0x71, 0xF1, 0x1D, 0x50, 0xC3, 0x51, 0x55, 0x1E,
			0x3C, 0x45, 0x5F, 0x1B, 0x05, 0x1B, 0xEF, 0xCB,
			0xC1, 0xFD, 0x83, 0x23, 0x9B, 0x74, 0x89, 0x51,
			0xF7, 0xE1, 0x8C, 0x20, 0x27, 0x62, 0x73, 0x39,
			0x71, 0x2D, 0xF2, 0x77, 0x2D, 0xCD, 0x57, 0xDE,
			0x9A, 0x15, 0xF2, 0x18, 0xE2, 0x5A, 0x44, 0x93,
			0xCE, 0x20, 0xD0, 0x39, 0xE2, 0x88, 0x08, 0x81,
			0x69, 0x44, 0x5F, 0x24, 0x4F, 0x14, 0xD5, 0x6E,
			0x6E, 0xFE, 0x9E, 0xD0, 0x05, 0x09, 0x43, 0x33,
			0x1A, 0x4E, 0xF2, 0x97, 0x11, 0x9C, 0xF5, 0xC0,
			0xE2, 0x1E, 0x2B, 0xBC, 0x53, 0x5D, 0xAE, 0xBF,
			0x3F, 0xCE, 0x3C, 0xAF, 0x9D, 0x86, 0xB6, 0x2C,
			0x37, 0xA4, 0xC9, 0xBD, 0x89, 0x91, 0xB8, 0xFF,
			0x01, 0xE9, 0x92, 0xF2, 0x6A, 0x77, 0xE9, 0x87,
			0xCA, 0x8D, 0xDF, 0x6C, 0xF4, 0x7D, 0x47, 0xD5,
			0x43, 0x9E, 0xB6, 0x62, 0x2B, 0x24, 0x11, 0x72,
			0xA8, 0xD5, 0xA2, 0x51, 0xDC, 0xB5, 0xD4, 0xD2,
			0x26, 0xA6, 0x8B, 0xEF, 0x9D, 0x2E, 0x77, 0xDF,
			0xE4, 0xDB, 0x3E, 0xBD, 0x43, 0x42, 0xF4, 0x9B,
			0xEE, 0x82, 0xB2, 0x8F, 0xC3, 0x50, 0x63, 0xE9,
			0x36, 0x58, 0x9F, 0x86, 0xF8, 0xFF, 0x2D, 0xB0,
			0xF2, 0xA7, 0xFC, 0xBF, 0x0D, 0x46, 0x14, 0x84,
			0x18, 0x4F, 0x64, 0xBF, 0x18, 0xE5, 0xBF, 0xF6,
			0x84, 0x54, 0x5E, 0x61, 0x12, 0xF8, 0x76, 0x62,
			0x60, 0x98, 0x7B, 0xCF, 0xE7, 0x6B, 0xED, 0x5A,
			0x17, 0xDF, 0xB8, 0x8A, 0x9B, 0x7D, 0x7C, 0xAC,
			0xCB, 0x42, 0x83, 0xAF, 0xB4, 0xEE, 0x21, 0xEF,
			0xB4, 0x3D, 0x69, 0x8C, 0x41, 0x3D, 0xE8, 0x13,
			0x48, 0x30, 0x9B, 0xC1, 0xEC, 0x10, 0xCD, 0xB3,
			0x3A, 0x7E, 0x2E, 0x4A, 0xAE, 0xD4, 0x1C, 0xFE,
			0xBF, 0x80, 0x8B, 0x08, 0xE7, 0xF6, 0x4F, 0x8F,
			0x6F, 0x25, 0x09, 0x60, 0x37, 0x5C, 0x3A, 0x3E,
			0xD0, 0x61, 0x70, 0x00, 0xAC, 0x6E, 0x54, 0xA6,
			0x12, 0x72, 0x78, 0x61, 0xDA, 0xF4, 0xD8, 0x93,
			0x7A, 0xE1, 0x33, 0xA5, 0xE9, 0x9C, 0x60, 0x7E,
			0x09, 0xE8, 0x09, 0x7F, 0x87, 0x6E, 0xF8, 0xCD,
			0x75, 0xE2, 0x44, 0xB7, 0x8E, 0xAA, 0xBF, 0x83,
			0x1D, 0xB9, 0xEF, 0xD0, 0xBA, 0x40, 0x5B, 0x52,
			0x71, 0x58, 0x25, 0x97, 0x45, 0x79, 0xA6, 0x27,
			0x9F, 0x77, 0x75, 0xAB, 0x87, 0xDE, 0x6E, 0x26,
			0x99, 0x79, 0x53, 0x0E, 0x3F, 0xFF, 0x6D, 0x8F,
			0xF6, 0x42, 0x1A, 0xB3, 0xBA, 0x1E, 0xC6, 0x1B,
			0x9E, 0xBC, 0x1A, 0x2A, 0x7A, 0xA5, 0x90, 0x02,
			0xAC, 0x91, 0x6C, 0x26, 0xF5, 0x5B, 0xC3, 0x69,
			0xB2, 0xE1, 0x10, 0x30, 0xF3, 0x34, 0x65, 0x48,
			0x28, 0x28, 0x59, 0x30, 0x22, 0x8A, 0xD0, 0x81,
			0x25, 0x00, 0xC8, 0x22, 0xBD, 0x41, 0xEA, 0xD4,
			0x80, 0xB5, 0x30, 0x33, 0x1F, 0x86, 0x42, 0xF2,
			0x6D, 0x54, 0x54, 0xFE, 0x75, 0xCC, 0x38, 0x70,
			0xD8, 0x07, 0xEF, 0x92, 0x49, 0x6B, 0x27, 0xE0,
			0x45, 0xB3, 0x31, 0x7F, 0x10, 0xE9, 0x85, 0x33,
			0x59, 0x87, 0x5E, 0xC0, 0x41, 0x11, 0x7F, 0x3B,
			0xB3, 0x7D, 0x88, 0xC5, 0x26, 0xEF, 0x1A, 0x34,
			0x2B, 0x6E, 0xA2, 0x89, 0xFA, 0x69, 0xBC, 0x91,
			0x4D, 0x8F, 0xEF, 0x84, 0xA2, 0x73, 0x29, 0xF3,
			0x0A, 0x73, 0x26, 0xC8, 0x47, 0x10, 0xF9, 0x72,
			0x54, 0x32, 0xA5, 0x25, 0xF3, 0xBF, 0x9A, 0xF3,
			0xD9, 0x3F, 0x9F, 0xAD, 0xED, 0x57, 0x66, 0xF9,
			0x06, 0x7A, 0x5B, 0x1B, 0x7A, 0x0D, 0xAC, 0x92,
			0x75, 0x20, 0x7B, 0x67, 0x76, 0xC4, 0x04, 0xB1,
			0x78, 0x01, 0xA7, 0x37, 0x26, 0x66, 0xF1, 0x53,
			0x78, 0xCD, 0xF9, 0x1B, 0xB4, 0xC2, 0x9D, 0x6A,
			0xCF, 0x79, 0xEE, 0xD1, 0x69, 0x18, 0x94, 0x7C,
			0x76, 0x92, 0x83, 0xE8, 0x29, 0xEC, 0x1E, 0x97,
			0xCB, 0x90, 0x63, 0x04, 0x73, 0x22, 0x4D, 0x88,
			0x95, 0xF2, 0xA0, 0x21, 0x9D, 0x30, 0x95, 0x07,
			0x17, 0x3F, 0x42, 0x59, 0x43, 0x72, 0x69, 0x6F,
			0x6E, 0xF8, 0x46, 0x8B, 0x84, 0x3D, 0x4A, 0xD5,
			0x81, 0xAD, 0x78, 0xC2, 0x21, 0xBB, 0xB8, 0x77,
			0x0C, 0xA2, 0x32, 0x38, 0x58, 0x01, 0x6D, 0xC6,
			0xF9, 0xC3, 0x11, 0xBD, 0x45, 0x1A, 0x5B, 0x68,
			0xCE, 0x23, 0xC6, 0xFE, 0xB8, 0xC1, 0xF5, 0x43,
			0x82, 0xA8, 0x51, 0x2D, 0x28, 0x6E, 0x6B, 0xD5,
			0x62, 0xAD, 0xA1, 0xC6, 0xC8, 0xC7, 0xC4, 0x6D,
			0x7A, 0x97, 0x22, 0xD7, 0xA9, 0x09, 0xB7, 0xCC,
			0xFC, 0xE3, 0x25, 0x8B, 0xB3, 0x7B, 0x78, 0xC0,
			0xD0, 0x76, 0xE4, 0xBB, 0x58, 0x7B, 0xFE, 0x05,
			0x95, 0x25, 0x7C, 0x98, 0x85, 0x43, 0xED, 0xEB,
			0xD2, 0xF2, 0x4F, 0x9E, 0x12, 0x4D, 0xD0, 0xE3,
			0x35, 0xEA, 0x2A, 0xDD, 0x17, 0x20, 0x1D, 0xF7,
			0xF2, 0xE6, 0x8F, 0xBC, 0xC0, 0x2D, 0xA7, 0xD4,
			0x3B, 0x7A, 0x9A, 0x8F, 0x83, 0xDE, 0x73, 0x75,
			0xBE, 0x2C, 0x61, 0xB4, 0xC2, 0xB8, 0x72, 0xBB,
			0xDE, 0x25, 0xEA, 0x65, 0x9A, 0x59, 0xB1, 0xA6,
			0x3C, 0xBC, 0x9C, 0x5E, 0xFB, 0x6C, 0x44, 0x9D,
			0x98, 0x18, 0x24, 0x52, 0x91, 0xC6, 0xC2, 0x32,
			0x17, 0xAE, 0x6C, 0xB0, 0x18, 0xCD, 0xF7, 0xA9,
			0xA4, 0x92, 0x40, 0xF3, 0x7A, 0x48, 0x43, 0x61,
			0xB4, 0x50, 0xBA, 0x8F, 0xCC, 0xED, 0xD4, 0xF4,
			0x55, 0x6C, 0xA8, 0x42, 0x3F, 0xD1, 0xE9, 0x07,
			0x6A, 0x87, 0x63, 0x06, 0x95, 0x8E, 0xE2, 0x64,
			0x46, 0x46, 0x63, 0x3C, 0x27, 0x77, 0x28, 0x0A,
			0xC7, 0xA8, 0x2E, 0x44, 0x1D, 0x79, 0xB5, 0x56,
			0xC6, 0x29, 0xD7, 0xC9, 0x7B, 0x4C, 0x78, 0x95,
			0x4B, 0xAE, 0x0E, 0x76, 0xCB, 0x4A, 0xB1, 0xB2,
			0x0B, 0x51, 0x12, 0x6A, 0xC8, 0xF1, 0x25, 0xE2,
			0xF0, 0x1C, 0x26, 0x6D, 0xF3, 0x1B, 0x2A, 0xE6,
			0xD5, 0x0E, 0xB0, 0x2F, 0x96, 0xB3, 0x90, 0x44,
			0x81, 0xA3, 0x22, 0x54, 0x79, 0x9B, 0xC2, 0x33,
			0x88, 0xF7, 0xD8, 0x6B, 0x6B, 0x60, 0x87, 0x6D,
			0x20, 0xCF, 0x9E, 0x8A, 0x44, 0x68, 0xFB, 0x3E,
			0xBE, 0x48, 0x83, 0xFB, 0x90, 0x76, 0x5A, 0x50,
			0x5D, 0x6A, 0xE9, 0x98, 0x27, 0xA0, 0xFF, 0x96,
			0xD5, 0xEB, 0x28, 0x4A, 0xC7, 0xDF, 0x81, 0x5C,
			0x0F, 0xD5, 0xAA, 0x2B, 0xDF, 0xFA, 0x56, 0x0B,
			0xDC, 0x37, 0xBE, 0xB9, 0xA7, 0xA6, 0xA4, 0xE3,
			0xFC, 0x07, 0x4A, 0x9F, 0x81, 0x21, 0x32, 0xA8,
			0x6B, 0xE3, 0xA1, 0xF7, 0x34, 0x33, 0xA1, 0x98,
			0x0A, 0x16, 0x8B, 0xBE, 0x54, 0x91, 0x0F, 0xF5,
			0x95, 0xA4, 0x7B, 0x67, 0x47, 0xF4, 0x3A, 0x67,
			0x8F, 0xE5, 0xA7, 0xC9, 0x6E, 0x63, 0x6B, 0x4B,
			0x87, 0x4F, 0x34, 0x8D, 0x24, 0xB7, 0x93, 0x37,
			0xDB, 0x43, 0x15, 0xCB, 0x10, 0xFD, 0x0E, 0x56,
			0x24, 0x31, 0x51, 0x1C, 0x32, 0x33, 0x53, 0xCF,
			0x1E, 0x59, 0xFD, 0x5A, 0x55, 0x35, 0x7E, 0x5F,
			0x6B, 0x7C, 0xCE, 0x60, 0xF1, 0xF8, 0x21, 0x1F,
			0xD1, 0xF5, 0xBE, 0x68, 0xF7, 0xC8, 0xBD, 0x70,
			0xC2, 0x9F, 0x03, 0xC0, 0xA6, 0x61, 0x3C, 0x64,
			0xDD, 0x10, 0xA6, 0x5D, 0xB5, 0xE0, 0xC5, 0x46,
			0xF5, 0x38, 0x24, 0x03, 0xFF, 0x8B, 0xA3, 0x6B,
			0xAD, 0x49, 0x87, 0x92, 0x31, 0x91, 0x2A, 0x4B,
			0x21, 0x9A, 0x08, 0xA1, 0x98, 0x58, 0xB1, 0x2C,
			0x27, 0x44, 0xFD, 0x65, 0x60, 0x37, 0x75, 0xB5,
			0x6B, 0xF4, 0x45, 0x95, 0x12, 0xE7, 0x91, 0x88,
			0x92, 0xDA, 0x55, 0xF8, 0x7D, 0x7C, 0xC0, 0x2C,
			0x68, 0x85, 0xC0, 0xEC, 0x02, 0x55, 0x0B, 0x60,
			0x9E, 0x3F, 0xA7, 0xD9, 0xFB, 0x0D, 0x13, 0xAB,
			0xE1, 0x93, 0x3D, 0xE1, 0x0E, 0x3F, 0xFA, 0xCE,
			0x5F, 0xB8, 0xF8, 0x70, 0x7C, 0x35, 0xAC, 0x13,
			0x74, 0xDC, 0x14, 0xEE, 0x85, 0x18, 0x48, 0x1C,
			0x7E, 0x63, 0xD9, 0x36, 0xEC, 0xC6, 0x2F, 0x50,
			0xC7, 0xF9, 0x51, 0xB8, 0x7B, 0xC7, 0x16, 0xDC,
			0x45, 0xE9, 0xBC, 0xFE, 0xC6, 0xF6, 0xD9, 0x7E,
			0x7F, 0xAF, 0xDA, 0xCB, 0x6D, 0xB0, 0x5E, 0xD3,
			0x77, 0x8F, 0x21, 0x85, 0x1F, 0x32, 0x5E, 0x25,
			0x47, 0x0D, 0xA8, 0xDD, 0x81, 0xC4, 0x12, 0x23,
			0x6D, 0x66, 0xCB, 0xEE, 0x9F, 0xFA, 0x9C, 0x50,
			0xB8, 0x6A, 0xA4, 0x0B, 0xAF, 0x21, 0x34, 0x94,
			0xDF, 0xAC, 0xCA, 0x22, 0xAA, 0x0F, 0xB4, 0x79,
			0x53, 0x92, 0x87, 0x35, 0xCA, 0x42, 0x12, 0xCF,
			0x53, 0xA0, 0x9A, 0xB0, 0x33, 0x5D, 0x20, 0xA8,
			0xE6, 0x2E, 0xDE, 0x79, 0x7C, 0x8E, 0x74, 0x93,
			0x54, 0xD6, 0x36, 0xF1, 0x5F, 0x31, 0x50, 0xC5,
			0x52, 0x79, 0x7B, 0x76, 0xC0, 0x91, 0xA4, 0x1F,
			0x94, 0x9F, 0x7F, 0xB5, 0x7B, 0x42, 0xF7, 0x44,
			0x1C, 0xCA, 0x41, 0x02, 0x64, 0xD6, 0x42, 0x1F,
			0x4A, 0xA2, 0xC7, 0xE2, 0xFF, 0x48, 0x34, 0xA8,
			0xDB, 0x0E, 0x6E, 0x77, 0x50, 0xB2, 0xE1, 0x1F,
			0xF1, 0xC8, 0x9A, 0x42, 0xD1, 0xFB, 0xC2, 0x71,
			0x83, 0x58, 0xE3, 0x83, 0x25, 0x88, 0x6A, 0xD1,
			0x23, 0x46, 0xCD, 0x69, 0x4F, 0x9E, 0xAB, 0x73,
			0x46, 0xC9, 0xA2, 0x3B, 0x5E, 0xBE, 0x76, 0x37,
			0xBF, 0xD8, 0x34, 0xA4, 0x12, 0x31, 0x8B, 0x01,
			0x18, 0x8B, 0x0F, 0x29, 0xE3, 0xBD, 0x97, 0x9F,
			0x8A, 0xE7, 0x34, 0xAC, 0xF1, 0x56, 0x3A, 0xF3,
			0x03, 0xD3, 0xC0, 0x95, 0xE9, 0xEA, 0xEB, 0xA3,
			0x52, 0x07, 0xB9, 0xDF, 0x3A, 0xCF, 0x9E, 0xE4,
			0x7D, 0xA5, 0xC1, 0xE2, 0x65, 0x2F, 0x3B, 0x86,
			0x41, 0x69, 0x8F, 0x3D, 0x22, 0x60, 0x59, 0x1B,
			0x07, 0xD0, 0x05, 0x65, 0xE5, 0xD6, 0xBE, 0x18,
			0x36, 0x03, 0x3D, 0x2B, 0x7E, 0xF2, 0xC3, 0x3B,
			0xDC, 0x5C, 0xF3, 0xBB, 0xA9, 0x5B, 0x42, 0xDF,
			0x6F, 0x73, 0x34, 0x5B, 0x83, 0x53, 0x41, 0xB2,
			0x50, 0xE2, 0x86, 0x2C, 0x9F, 0x2F, 0x9C, 0xEF,
			0x77, 0xCF, 0xA7, 0x4C, 0xFB, 0x04, 0xC5, 0x60,
			0xD8, 0xA0, 0x03, 0x8C, 0x4E, 0x96, 0xCB, 0x0D,
			0xA2, 0xB3, 0xE9, 0xB2, 0xCD, 0x3C, 0xEC, 0xF5,
			0x22, 0xFD, 0xA0, 0xD6, 0x7E, 0x5F, 0x62, 0xB2,
			0xEE, 0x23, 0xBD, 0x42, 0xA6, 0x1C, 0x7D, 0xA4,
			0x8F, 0x0E, 0xA3, 0x0B, 0x81, 0xAF, 0x7C, 0xCB,
			0x6B, 0xB0, 0x2C, 0xDE, 0x27, 0x2D, 0x25, 0x74,
			0x13, 0x25, 0xE9, 0xD9, 0x15, 0x35, 0x61, 0x5C,
			0x01, 0x84, 0xF2, 0xD7, 0xF2, 0x26, 0x14, 0x1D,
			0x79, 0xB4, 0x24, 0x12, 0x72, 0x1F, 0xD3, 0x45,
			0x61, 0xD9, 0x36, 0x63, 0x65, 0x0B, 0x3C, 0x1B,
			0x69, 0x01, 0x87, 0x2B, 0xC4, 0xC0, 0xBB, 0x15,
			0xBC, 0xD9, 0x03, 0x89, 0x50, 0xB7, 0x71, 0x7F,
			0x7F, 0x44, 0x8B, 0x61, 0x26, 0x59, 0x20, 0x76,
			0xA2, 0xBA, 0xD2, 0xD6, 0x3C, 0x55, 0x39, 0x9C,
			0x24, 0x3F, 0xDB, 0xDB, 0x0C, 0x8D, 0x67, 0x6B,
			0x2A, 0xE4, 0x55, 0xE7, 0xF0, 0xA9, 0xB1, 0x8D,
			0x3F, 0xC8, 0x89, 0xC4, 0x33, 0x87, 0xF2, 0xCB,
			0xC4, 0xDC, 0x73, 0xD7, 0xC8, 0x5B, 0xFA, 0xB6,
			0xB4, 0xB0, 0x44, 0x63, 0xA3, 0xDD, 0x35, 0x9C,
			0x3A, 0x8F, 0x61, 0xBF, 0xA6, 0xC4, 0xB0, 0x42,
			0x4A, 0xEB, 0xA4, 0xDD, 0x8A, 0x95, 0xEC, 0x12,
			0x43, 0xB2, 0xE3, 0x6C, 0x29, 0xF8, 0x2E, 0x1D,
			0x71, 0x12, 0x81, 0x59, 0x9B, 0x3E, 0x05, 0xE7,
			0x54, 0x92, 0xAE, 0x34, 0x25, 0xEA, 0xA7, 0xF1,
			0x4F, 0xF8, 0xC6, 0xA9, 0x63, 0x0B, 0xBA, 0x6E,
			0xBD, 0x23, 0x6F, 0x19, 0x52, 0x69, 0xA4, 0x81,
			0xE8, 0x7E, 0xB3, 0xD4, 0x44, 0x82, 0x5B, 0xA4,
			0x42, 0x4E, 0xE5, 0xB2, 0xD9, 0xEF, 0xB5, 0x95,
			0xD5, 0xA3, 0x38, 0xF4, 0xC2, 0x53, 0xF7, 0x9D,
			0xE9, 0xD0, 0x45, 0x35, 0x20, 0x6C, 0xA6, 0xDB,
			0xC2, 0xD4, 0xC9, 0xA1, 0xEC, 0x20, 0x84, 0x9B,
			0x0D, 0xB3, 0xFB, 0xE1, 0x0C, 0x14, 0x46, 0xD5,
			0x61, 0x2D, 0x5B, 0xAC, 0x91, 0x5A, 0x39, 0x96,
			0x2C, 0xDB, 0xCA, 0xCE, 0xE0, 0x96, 0x9D, 0xCF,
			0x8E, 0xCF, 0xB8, 0x30, 0xCE, 0xA2, 0x20, 0x6C,
			0x37, 0x74, 0x9C, 0x65, 0xB8, 0xF6, 0x73, 0xDB,
			0x09, 0x0B, 0x1E, 0x2A, 0xDE, 0x6C, 0x2A, 0x2F,
			0x34, 0x9B, 0x59, 0x15, 0x10, 0x3A, 0x3A, 0xC7,
			0x84, 0x82, 0xC3, 0x9E, 0x99, 0xFF, 0xC4, 0x62,
			0x6F, 0xB4, 0xCF, 0x4A, 0x11, 0x68, 0x04, 0xAB,
			0x9D, 0x93, 0xD7, 0x10, 0x46, 0x60, 0xFE, 0xFA,
			0x07, 0x53, 0xCF, 0x87, 0x5C, 0xB2, 0x2F, 0xD6,
			0x0E, 0x55, 0xDC, 0x2F, 0x30, 0x3D, 0xE0, 0x36,
			0x47, 0x71, 0x2B, 0x12, 0x06, 0x7A, 0x55, 0xF7,
			0xA4, 0x67, 0x89, 0x7B, 0xBE, 0xD0, 0xD3, 0xA0,
			0x9D, 0x50, 0xE9, 0xDE, 0xAA, 0xDF, 0xF7, 0x8D,
			0xE9, 0xAC, 0x65, 0xC1, 0xFD, 0x05, 0xD0, 0x76,
			0x10, 0xA7, 0x9C, 0x8C, 0x46, 0x51, 0x41, 0xAD,
			0x65, 0xE6, 0x03, 0x40, 0x53, 0x1F, 0xAB, 0x08,
			0xF1, 0xF4, 0x33, 0xEF, 0x82, 0x32, 0x83, 0xFE,
			0xA1, 0xB0, 0x0E, 0x0B, 0xA0, 0x57, 0x84, 0x80,
			0x25, 0x08, 0x54, 0x7A, 0x83, 0x12, 0x91, 0xA4,
			0x1B, 0x6A, 0x57, 0x86, 0xD3, 0x60, 0x32, 0x35,
			0xB2, 0x5B, 0x97, 0x6A, 0x3E, 0x51, 0x82, 0xB3,
			0xFA, 0x49, 0x06, 0xA3, 0x7A, 0xDE, 0x49, 0x2B,
			0x94, 0x2C, 0x2A, 0x1A, 0x63, 0x32, 0xEC, 0x0B,
			0x4E, 0x34, 0x6C, 0xC2, 0xD4, 0xF3, 0xB4, 0xBA,
			0x9E, 0x96, 0x1B, 0x55, 0x1A, 0x59, 0xDC, 0xD2,
			0x20, 0x2C, 0xC3, 0xF2, 0xC3, 0xDC, 0x46, 0x2E,
			0xA4, 0x9D, 0x70, 0x1A, 0x84, 0x57, 0xCC, 0x5C,
			0x0B, 0x4C, 0x83, 0x65, 0x4E, 0x19, 0x02, 0x8F,
			0xB7, 0xF5, 0x0C, 0xBB, 0xD7, 0x55, 0xB3, 0xFE,
			0xF4, 0x91, 0x79, 0xE7, 0xDA, 0xFE, 0x06, 0x18,
			0x90, 0x95, 0xEA, 0x18, 0x94, 0xC6, 0x68, 0x99,
			0x8C, 0x83, 0x60, 0xD0, 0x09, 0x62, 0x43, 0xC1,
			0xD2, 0x48, 0xC5, 0xF2, 0x67, 0xA3, 0x17, 0xD6,
			0x95, 0x6D, 0x9A, 0xF5, 0xC4, 0x7C, 0xD0, 0x26,
			0xE3, 0x7D, 0x3F, 0x67, 0x8C, 0x65, 0x36, 0x18,
			0xD0, 0xE6, 0x74, 0x4F, 0x57, 0x1B, 0xFE, 0x50,
			0xC8, 0x99, 0xE3, 0x64, 0x6B, 0xDD, 0xA7, 0xDD,
			0x33, 0x8E, 0x50, 0xDB, 0x0E, 0xA8, 0x56, 0x96,
			0x18, 0x95, 0xC6, 0x8F, 0x5F, 0xFB, 0xE9, 0x88,
			0x2A, 0x8D, 0x6B, 0xBF, 0x01, 0x41, 0x9B, 0xDB,
			0x95, 0xC4, 0x7A, 0x85, 0x53, 0xEB, 0x16, 0xCE,
			0x75, 0xA8, 0x22, 0x73, 0x1B, 0x9E, 0x45, 0x4D,
			0x1A, 0xB4, 0x83, 0x54, 0x34, 0x9D, 0x9E, 0xD7,
			0x1C, 0xE7, 0x9D, 0x53, 0x31, 0x8E, 0x2E, 0xCB,
			0xC5, 0xFF, 0x2B, 0xC1, 0xAF, 0xD7, 0x4E, 0x20,
			0x01, 0x26, 0x1C, 0xA3, 0xCC, 0xCA, 0xF8, 0x30,
			0x7A, 0xD5, 0x19, 0x24, 0x65, 0xC1, 0x45, 0x59,
			0x76, 0x48, 0x87, 0x53, 0x7E, 0x6B, 0x69, 0x84,
			0x74, 0x45, 0xD0, 0x67, 0x13, 0x42, 0x7E, 0x19,
			0x1C, 0x86, 0xBE, 0xA0, 0x54, 0x03, 0xDC, 0xBB,
			0xA6, 0x23, 0xA6, 0x61, 0x24, 0x70, 0x3C, 0x51,
			0x5F, 0x48, 0x72, 0x08, 0x9E, 0x33, 0xE8, 0xBB,
			0xCE, 0xD7, 0x8E, 0x0F, 0xED, 0xF2, 0x29, 0x3F,
			0xE4, 0x16, 0xB7, 0x00, 0x97, 0x5E, 0x46, 0xF1,
			0xED, 0x93, 0x7B, 0xDF, 0x90, 0x51, 0x8A, 0x1E,
			0x56, 0x4D, 0x70, 0x56, 0xB7, 0x2B, 0x32, 0x8D,
			0x93, 0x78, 0x45, 0xAF, 0x14, 0xE4, 0xE9, 0x17,
			0x6E, 0x3A, 0xF8, 0xCE, 0x38, 0x95, 0xCB, 0x12,
			0xBF, 0x6A, 0x34, 0x21, 0xA2, 0xF2, 0x27, 0xB2,
			0xD0, 0xCC, 0x98, 0x4E, 0x71, 0x49, 0x40, 0x12,
			0xC1, 0x24, 0x5D, 0xFF, 0xB8, 0x0A, 0x69, 0xBE,
			0x85, 0x8E, 0xF8, 0xBD, 0x81, 0x10, 0x22, 0x40,
			0x22, 0x36, 0x15, 0x93, 0x78, 0x00, 0x0E, 0x56,
			0x46, 0xE0, 0x01, 0x80, 0x64, 0x71, 0xE0, 0xFF,
			0xE7, 0xDC, 0x7E, 0xF9, 0xE4, 0x3A, 0xEE, 0xD8,
			0x23, 0x54, 0x86, 0xA2, 0xA6, 0x1D, 0x2D, 0x60,
			0x22, 0x9D, 0xD4, 0x70, 0x2A, 0x52, 0x40, 0xC4,
			0x37, 0xE4, 0xC2, 0x18, 0x15, 0x32, 0xD3, 0x46,
			0x48, 0x73, 0x29, 0xB8, 0xF2, 0xDF, 0x2E, 0x49,
			0x89, 0xA5, 0xE7, 0x77, 0xAE, 0x1C, 0xBE, 0x03,
			0x94, 0x27, 0xCC, 0xE8, 0x8E, 0xCA, 0x11, 0x85,
			0xFC, 0xD9, 0xE8, 0xD9, 0xC9, 0xA3, 0xEB, 0x77,
			0xAD, 0x3B, 0x69, 0x96, 0x8A, 0xD7, 0x4A, 0x06,
			0x76, 0x05, 0x29, 0xC6, 0xF7, 0xE6, 0xAB, 0xEC,
			0xC5, 0x85, 0x96, 0x76, 0x7A, 0x70, 0x62, 0xEA,
			0xC9, 0x4E, 0x9C, 0x79, 0x10, 0xA4, 0x47, 0x3D,
			0xBA, 0xF6, 0xD1, 0x04, 0xCD, 0x0D, 0x8B, 0xB9,
			0xE2, 0xB6, 0x76, 0xFD, 0xAF, 0x56, 0xE3, 0x9A,
			0xF3, 0x49, 0x14, 0x9E, 0xB9, 0x1F, 0x65, 0x61,
			0xD7, 0x52, 0xC8, 0x46, 0x3E, 0x3F, 0xE0, 0x26,
			0x59, 0x34, 0xA9, 0x1D, 0xC6, 0x1B, 0x20, 0xDC,
			0x9E, 0xB7, 0x8F, 0x8B, 0xBA, 0x90, 0x29, 0xA9,
			0xC1, 0x63, 0x95, 0xFD, 0xA0, 0xC0, 0x9B, 0x5F,
			0x01, 0x40, 0xD4, 0x12, 0x6C, 0x90, 0x24, 0x5F,
			0xBA, 0xD4, 0x85, 0xDE, 0xCD, 0x61, 0x5D, 0x01,
			0x54, 0xC5, 0x01, 0xE7, 0x88, 0xB2, 0x40, 0x7C,
			0xFA, 0xE8, 0xB1, 0x21, 0x09, 0x36, 0xBA, 0x59,
			0x7A, 0x5F, 0xD3, 0x54, 0x1C, 0x83, 0x4F, 0x8C,
			0x65, 0x35, 0xFA, 0x7E, 0x31, 0xC5, 0xD5, 0x05,
			0x2C, 0x61, 0x1C, 0x9F, 0xDE, 0x2C, 0x88, 0x35,
			0x3D, 0x61, 0xA6, 0x3E, 0xA9, 0x3B, 0xE7, 0xBB,
			0x76, 0xAF, 0x81, 0xFC, 0x16, 0x06, 0x4E, 0x08,
			0xA3, 0xDB, 0x58, 0x61, 0xF8, 0xF6, 0x49, 0xFC,
			0x6D, 0x0E, 0xF1, 0x1C, 0xB9, 0x28, 0x8E, 0xD4,
			0xA7, 0xF2, 0x73, 0x16, 0x4B, 0x2E, 0x7B, 0x0D,
			0x1B, 0x9E, 0x90, 0x2D, 0x13, 0xFB, 0xAE, 0xDE,
			0xAD, 0x83, 0x23, 0xC0, 0xE4, 0x3A, 0xD4, 0x43,
			0x05, 0xBE, 0xDC, 0x41, 0xDF, 0x03, 0xE4, 0x1F,
			0xE5, 0x24, 0x3D, 0x8C, 0x26, 0x77, 0xE9, 0x6E,
			0x41, 0x2B, 0x69, 0xD2, 0x4C, 0xC8, 0x14, 0x0F,
			0xF6, 0x6F, 0xC3, 0x01, 0x91, 0x4D, 0x66, 0x7A,
			0xD4, 0xC2, 0x12, 0x34, 0x46, 0xEC, 0x29, 0xA9,
			0x44, 0x3A, 0x0B, 0xF8, 0x32, 0x06, 0xA0, 0xF5,
			0xC0, 0x12, 0x71, 0xFC, 0x29, 0xBC, 0xB6, 0x39,
			0xCC, 0x84, 0x27, 0xC8, 0x1B, 0x83, 0xA7, 0x47,
			0x78, 0x58, 0x5F, 0x83, 0x77, 0x8A, 0x11, 0xCB,
			0x1A, 0x00, 0x75, 0x84, 0xEA, 0x5D, 0x41, 0x9F,
			0xC4, 0x92, 0x1E, 0x91, 0x5B, 0xA4, 0xC0, 0x3E,
			0x61, 0x23, 0xE4, 0xC2, 0x63, 0x33, 0x70, 0xCA,
			0xAA, 0x4D, 0xBA, 0xDB, 0x80, 0xDE, 0x5A, 0x61,
			0x9B, 0x2C, 0xCB, 0xDC, 0x6F, 0x52, 0xDA, 0xDE,
			0xFE, 0xFD, 0x14, 0x2D, 0x19, 0xC6, 0xB5, 0xB8,
			0x97, 0x4B, 0x8F, 0x9F, 0xC9, 0x4B, 0xC3, 0x5A,
			0x50, 0x34, 0x1E, 0x3D, 0x42, 0x7D, 0xF6, 0xDB,
			0xA5, 0xBB, 0x9D, 0x82, 0xDD, 0x6F, 0x23, 0x83,
			0xF3, 0x8C, 0x79, 0x39, 0x68, 0x82, 0x91, 0x61,
			0x58, 0x93, 0x37, 0xCA, 0x66, 0x12, 0xFF, 0xE3,
			0x97, 0x72, 0x65, 0xCB, 0x93, 0x1E, 0xB5, 0x3C,
			0xB8, 0x9C, 0xAB, 0x95, 0x1A, 0x78, 0x2E, 0xDB,
			0xD4, 0x3C, 0xE7, 0xBD, 0x7F, 0x3C, 0x08, 0xD3,
			0xBB, 0xAC, 0xC9, 0xCC, 0x90, 0x60, 0x23, 0xAE,
			0x97, 0x79, 0xAC, 0xE4, 0xA3, 0xAD, 0xAE, 0x3A,
			0x79, 0x94, 0xB3, 0xA9, 0x4C, 0xC4, 0xC5, 0x63,
			0xDA, 0x7D, 0x00, 0xA2, 0x41, 0xB0, 0x35, 0x04,
			0x35, 0xC8, 0xD1, 0x22, 0x99, 0x5D, 0xD0, 0x2A,
			0x04, 0xAD, 0x92, 0xED, 0x06, 0x3A, 0x87, 0x43,
			0x8E, 0xA2, 0x7E, 0xF8, 0xD5, 0xC1, 0x4D, 0x5B,
			0x3D, 0x1A, 0x06, 0x94, 0x8A, 0xA4, 0xA9, 0xFA,
			0x40, 0xA6, 0xBE, 0xDD, 0x96, 0xCA, 0x56, 0x6A,
			0x45, 0xA0, 0x75, 0xF8, 0x81, 0xF5, 0x37, 0x5A,
			0xA8, 0x43, 0x63, 0x1E, 0x12, 0x8C, 0x8E, 0x0A,
			0x98, 0x35, 0x70, 0xCE, 0x20, 0xAE, 0xFA, 0x50,
			0x6C, 0xDB, 0x1B, 0x49, 0xA1, 0xC7, 0x0B, 0x88,
			0x9C, 0x40, 0xBA, 0x1D, 0x99, 0x74, 0x16, 0xFD,
			0xBF, 0xFA, 0xEB, 0xF8, 0x16, 0x15, 0x17, 0x2F,
			0xBE, 0x99, 0x2B, 0x15, 0xC9, 0xCC, 0x85, 0x76,
			0xA6, 0xE9, 0x92, 0x26, 0x59, 0x65, 0xA8, 0x24,
			0x8C, 0x9B, 0x1F, 0xCD, 0x7D, 0x8B, 0x0B, 0xE7,
			0x29, 0x45, 0xB0, 0x8A, 0x0B, 0x08, 0x0F, 0x4B,
			0xE4, 0x03, 0x53, 0x40, 0x16, 0xC1, 0xA4, 0x3D,
			0xF2, 0xD0, 0xD3, 0x31, 0x19, 0x67, 0x87, 0x19,
			0xBB, 0x80, 0x79, 0xA1, 0x98, 0x62, 0xE1, 0xC1,
			0xC1, 0x0A, 0x97, 0x52, 0xAC, 0x39, 0x3D, 0x00,
			0x05, 0xCD, 0xA9, 0x90, 0x29, 0xDA, 0x98, 0xED,
			0x75, 0x33, 0x69, 0x07, 0x43, 0x4D, 0x86, 0x67,
			0x99, 0x06, 0x3C, 0xC9, 0xEC, 0x27, 0x3A, 0x67,
			0xE0, 0x87, 0x30, 0x5F, 0xF0, 0xB1, 0x66, 0xE1,
			0x38, 0x4C, 0xFF, 0x9A, 0x90, 0x69, 0x80, 0xBB,
			0xAC, 0xE1, 0x88, 0xF9, 0x89, 0x3B, 0xE9, 0xF7,
			0xB7, 0x47, 0x52, 0xB1, 0xB4, 0x93, 0x6A, 0xA7,
			0xA5, 0xF3, 0xB9, 0x40, 0x7D, 0x18, 0x54, 0xF3,
			0x7F, 0x02, 0x58, 0x49, 0x47, 0x4D, 0x81, 0xED,
			0xAD, 0xFA, 0xD4, 0x34, 0x02, 0xE2, 0x30, 0xCD,
			0x5D, 0x08, 0xB4, 0x78, 0x1E, 0xD8, 0xD3, 0x6D,
			0x1A, 0x6E, 0x60, 0xE3, 0x11, 0xD7, 0xB8, 0xA6,
			0x62, 0xB7, 0x29, 0x56, 0x54, 0x90, 0xD3, 0x71,
			0x9B, 0x55, 0x13, 0x9C, 0xA5, 0x72, 0xDB, 0xD1,
			0x92, 0x8F, 0x85, 0xC7, 0x07, 0x2E, 0x4A, 0x3A,
			0x4E, 0xCF, 0x65, 0x14, 0x89, 0x36, 0x15, 0x08,
			0xF8, 0x6A, 0xC1, 0x9E, 0xFC, 0x53, 0x64, 0xFD,
			0xFE, 0x17, 0xFD, 0xFB, 0x9F, 0x12, 0xA1, 0xA8,
			0xEB, 0xE6, 0x4F, 0x82, 0x6C, 0x5C, 0xD7, 0x5A,
			0x19, 0x04, 0xE4, 0x64, 0x9A, 0x64, 0xCB, 0x3E,
			0x0C, 0x42, 0x7B, 0x26, 0xFA, 0xCB, 0x65, 0x54,
			0x25, 0x1F, 0xC6, 0xC0, 0x65, 0x69, 0xD0, 0x22,
			0x8F, 0xF8, 0xAD, 0x81, 0xC2, 0x54, 0xA7, 0xDD,
			0x07, 0xAC, 0x6E, 0x09, 0xF0, 0xDD, 0x30, 0x11,
			0x00, 0xBA, 0xFE, 0x18, 0xBE, 0xFC, 0x39, 0x24,
			0xA7, 0x39, 0x99, 0x28, 0xE4, 0xF7, 0x94, 0xF1,
			0xA3, 0x19, 0xD3, 0xEB, 0x10, 0xBB, 0x65, 0x1C,
			0x54, 0xDF, 0xAD, 0x99, 0x32, 0x8F, 0x93, 0x80,
			0xE5, 0x2B, 0xA2, 0x27, 0x9C, 0xAB, 0x01, 0x0A,
			0x83, 0xC2, 0x81, 0x75, 0xC7, 0x05, 0x38, 0xBD,
			0xD9, 0xB7, 0x5B, 0x93, 0xCD, 0x2E, 0xED, 0x13,
			0xAE, 0x8D, 0x88, 0xC2, 0x3B, 0x3B, 0x9F, 0x33,
			0x15, 0x27, 0x3A, 0x68, 0x60, 0xEF, 0xAE, 0x42,
			0xA0, 0x70, 0x08, 0x7D, 0x34, 0x63, 0x53, 0xD7,
			0xAD, 0xCB, 0xB6, 0xFA, 0xC5, 0x7C, 0x24, 0xC7,
			0xD7, 0x6D, 0x11, 0x62, 0xC3, 0xAE, 0x2F, 0xB0,
			0x0E, 0xBC, 0xD0, 0x2E, 0x42, 0x8B, 0x9D, 0xA9,
			0x02, 0x4E, 0xFD, 0xFF, 0x1E, 0xB5, 0xA0, 0xFD,
			0x11, 0x2E, 0xDB, 0xD1, 0x5B, 0x7A, 0x10, 0x20,
			0x4E, 0xDD, 0xD8, 0xCF, 0x8F, 0xA7, 0x67, 0x34,
			0x78, 0xB9, 0x31, 0xC6, 0x90, 0x99, 0xC2, 0x9A,
			0x8D, 0xFE, 0xCC, 0x13, 0xBB, 0x11, 0x03, 0x64,
			0x08, 0x84, 0x26, 0x99, 0xB5, 0xC2, 0x1D, 0xC6,
			0x4A, 0x23, 0x13, 0x8A, 0xD3, 0xA0, 0x22, 0xDA,
			0xCA, 0x2D, 0x0A, 0xE5, 0x88, 0x23, 0x41, 0x48,
			0xD1, 0xD3, 0x4E, 0x2E, 0x47, 0x46, 0x74, 0x06,
			0x12, 0x45, 0x94, 0xEB, 0xCB, 0x15, 0x8E, 0xB4,
			0x8B, 0x12, 0xC4, 0x7A, 0xE8, 0x1C, 0x03, 0x88,
			0x0A, 0x9E, 0x93, 0x87, 0xB1, 0xEA, 0x13, 0xE5,
			0x5D, 0xCE, 0x70, 0x83, 0x11, 0xF2, 0x04, 0x57,
			0xFD, 0x25, 0xA3, 0x77, 0xA6, 0xC1, 0x38, 0x7E,
			0x5E, 0x3A, 0xFE, 0x54, 0x8F, 0x4D, 0xAA, 0x77,
			0x78, 0xA7, 0xFE, 0x9D, 0xA7, 0x05, 0x44, 0x74,
			0xD2, 0x46, 0x6A, 0xB3, 0x75, 0x44, 0x27, 0x95,
			0x98, 0xB9, 0x13, 0x08, 0x28, 0xAD, 0x6F, 0xB1,
			0xEC, 0x85, 0x0E, 0x18, 0xD0, 0x78, 0x7C, 0x96,
			0xBE, 0x1C, 0x58, 0x13, 0x69, 0x46, 0xC0, 0xD5,
			0x2D, 0xAC, 0xBC, 0x9D, 0x69, 0xE7, 0xAD, 0x9D,
			0x86, 0x2D, 0x11, 0xB1, 0x68, 0xF0, 0xE8, 0xF6,
			0x0A, 0x5E, 0xC8, 0x17, 0xAB, 0xAD, 0x12, 0xC5,
			0x3F, 0xC4, 0xE1, 0xBC, 0xBD, 0x24, 0x20, 0xF7,
			0x96, 0xA7, 0x86, 0x25, 0x33, 0x36, 0x21, 0xC2,
			0xF3, 0x93, 0x7D, 0x3E, 0xC8, 0x2F, 0xCB, 0xAC,
			0xF7, 0x2A, 0xCC, 0xD1, 0xA4, 0x57, 0xB1, 0x52,
			0x7C, 0x41, 0x9A, 0xEB, 0xB7, 0x3E, 0x3C, 0xE6,
			0x92, 0x19, 0xDC, 0x4F, 0x28, 0x7E, 0xC2, 0x95,
			0x27, 0xD1, 0xFE, 0x14, 0xB8, 0x85, 0x41, 0x43,
			0x34, 0x83, 0xF0, 0x94, 0xF5, 0x0D, 0x43, 0xB1,
			0xD0, 0xB6, 0xFF, 0x17, 0xDB, 0x28, 0x30, 0xE4,
			0xF5, 0x6B, 0x61, 0x95, 0xA5, 0xF6, 0x34, 0x8B,
			0x5B, 0x2C, 0x26, 0x27, 0xDF, 0x72, 0xB0, 0xBA,
			0x37, 0xB8, 0xAB, 0x50, 0x60, 0xF2, 0xA7, 0x87,
			0x6B, 0x37, 0x09, 0xED, 0x74, 0x00, 0xB5, 0xED,
			0x99, 0x3F, 0x44, 0x56, 0x53, 0x66, 0xC0, 0x56,
			0x39, 0x7B, 0x9C, 0x6E, 0x71, 0xEF, 0x5E, 0xC4,
			0x02, 0xCC, 0x15, 0x7F, 0x35, 0x0D, 0xA4, 0x06,
			0x32, 0xFD, 0xE3, 0xE8, 0xFA, 0x2D, 0x7A, 0x02,
			0x62, 0x21, 0x9F, 0xEB, 0x3B, 0x2E, 0xE2, 0x09,
			0xB8, 0x8B, 0x21, 0x60, 0x36, 0x61, 0xE6, 0x69,
			0xDF, 0xF1, 0x76, 0xC2, 0x67, 0xA8, 0x38, 0x6F,
			0x1F, 0x39, 0x3E, 0x88, 0x0F, 0x84, 0xF5, 0xF1,
			0xF6, 0x01, 0x83, 0x47, 0xD6, 0xD3, 0xC7, 0x64,
			0x65, 0xE1, 0xE1, 0x5B, 0x7F, 0x0B, 0x6F, 0x07,
			0x07, 0x46, 0x6E, 0x41, 0x65, 0x57, 0xA5, 0x1F,
			0xFF, 0xA9, 0xF8, 0xE6, 0x48, 0x94, 0xD4, 0xB6,
			0xDE, 0x92, 0xA8, 0x6A, 0xFA, 0xAB, 0x0D, 0x25,
			0xA2, 0x04, 0x0C, 0x00, 0x57, 0xE9, 0x8D, 0x5A,
			0x54, 0xBC, 0x7E, 0xA3, 0x92, 0xC7, 0xFA, 0x75,
			0x10, 0x3D, 0x87, 0x1E, 0xDE, 0x58, 0xD0, 0x3F,
			0x75, 0x76, 0x6F, 0x7F, 0x0F, 0x16, 0xFD, 0x17,
			0x74, 0xA4, 0xDD, 0x18, 0x79, 0x3E, 0xCA, 0xA6,
			0xCB, 0x2C, 0x24, 0xA1, 0x9A, 0x93, 0x39, 0x9E,
			0xB7, 0x36, 0xF6, 0xD1, 0xE0, 0xA2, 0xB8, 0x53,
			0xBB, 0xB0, 0xDE, 0xCD, 0xDC, 0x16, 0x2E, 0x6C,
			0x24, 0xBA, 0x05, 0x5E, 0x3D, 0x4F, 0x88, 0x44,
			0x62, 0xF4, 0x5B, 0x6F, 0x04, 0x4B, 0x18, 0x2A,
			0x19, 0x53, 0x1A, 0xEC, 0xBA, 0xA3, 0x09, 0x0D,
			0x9D, 0xD9, 0x4A, 0xCB, 0x84, 0x7D, 0xA6, 0x94,
			0x84, 0x05, 0xEC, 0xB3, 0xD6, 0x41, 0x23, 0xAB,
			0xEA, 0xA2, 0xB2, 0xEB, 0x5C, 0xEA, 0x8A, 0x9E,
			0x3D, 0x10, 0x46, 0x81, 0xB9, 0x39, 0x43, 0x59,
			0x97, 0xC4, 0xBB, 0x5A, 0x0C, 0x36, 0x33, 0x2D,
			0x07, 0xBC, 0x68, 0x99, 0xBC, 0x86, 0xF0, 0x51,
			0x01, 0x2F, 0xDE, 0x1A, 0xE5, 0xB0, 0xED, 0x7C,
			0x52, 0xC0, 0x92, 0x17, 0x47, 0x5C, 0x03, 0x1E,
			0x8D, 0x38, 0xAD, 0x3F, 0x31, 0xBC, 0x35, 0x61,
			0xEC, 0x88, 0xF7, 0xA8, 0x94, 0xAE, 0xDD, 0x54,
			0xB9, 0x7F, 0x3D, 0x4C, 0x7E, 0x4F, 0x36, 0x4F,
			0xE6, 0x4E, 0xB4, 0x76, 0x41, 0x16, 0xCF, 0xF2,
			0xEF, 0xEB, 0x8F, 0xD0, 0xBD, 0x4B, 0xFC, 0x20,
			0x6B, 0x72, 0x61, 0x6D, 0x7B, 0x1C, 0xC0, 0x53,
			0x91, 0xC7, 0xA6, 0xB4, 0xEF, 0x0C, 0xA1, 0x23,
			0xC6, 0xF0, 0x02, 0xD9, 0xF2, 0x27, 0xBF, 0x27,
			0xF5, 0x75, 0x4A, 0x8C, 0x54, 0x81, 0x36, 0x90,
			0x52, 0x52, 0xC9, 0xE7, 0x24, 0x2E, 0x33, 0xB6,
			0xC7, 0xFA, 0xDF, 0x7F, 0xCA, 0xFF, 0x9C, 0x57,
			0x2A, 0x06, 0x44, 0x88, 0x8D, 0x30, 0x20, 0x54,
			0x33, 0x8E, 0xA3, 0x78, 0xA8, 0xB2, 0xD6, 0xED,
			0x27, 0x49, 0x66, 0xE1, 0x19, 0xCC, 0x91, 0x46,
			0x47, 0xD9, 0x70, 0x0E, 0xB1, 0x94, 0xFA, 0xDF,
			0x50, 0xAC, 0x4B, 0x75, 0xE5, 0x2A, 0x21, 0x7C,
			0x1A, 0x6F, 0x11, 0xE3, 0x13, 0xEC, 0x9D, 0x00,
			0x7D, 0xE4, 0x93, 0x2F, 0x38, 0x03, 0xDC, 0x28,
			0xB9, 0xCB, 0x99, 0x35, 0xEF, 0xDB, 0xB2, 0xDA,
			0x15, 0x04, 0xDE, 0xEF, 0x1D, 0xDD, 0x22, 0xF5,
			0x39, 0x47, 0xD8, 0x04, 0xE8, 0x0F, 0x8D, 0x82,
			0x9A, 0x65, 0xBE, 0x2A, 0xE1, 0xE6, 0xF4, 0x31,
			0xFD, 0xB5, 0xC4, 0x5B, 0xAB, 0x45, 0xE9, 0xE8,
			0xD5, 0x14, 0x3E, 0x7A, 0xB5, 0xB7, 0x88, 0x51,
			0x0E, 0x16, 0x63, 0x2F, 0x93, 0xD6, 0x42, 0x9E,
			0xEC, 0xF9, 0x39, 0xD3, 0x9C, 0x2D, 0xBF, 0xD8,
			0x4F, 0x42, 0xC8, 0xBA, 0xC6, 0x85, 0x44, 0x02,
			0x02, 0x75, 0x61, 0x18, 0x08, 0xC9, 0x22, 0x90,
			0xDE, 0xAD, 0x66, 0xD0, 0xAC, 0xCE, 0xFE, 0xEB,
			0xF6, 0xCD, 0x68, 0x5B, 0xB2, 0xD9, 0xBA, 0xD6,
			0xA5, 0x5F, 0xC4, 0x42, 0x5D, 0xAA, 0x02, 0xAF,
			0x5B, 0xFB, 0x8A, 0x08, 0x85, 0x14, 0xE2, 0x4A,
			0x4E, 0x3A, 0x2A, 0xFC, 0x85, 0x06, 0x60, 0xFF,
			0xA1, 0x36, 0x68, 0xC7, 0xC7, 0xC2, 0x56, 0x42,
			0x12, 0xD9, 0x21, 0x9D, 0x90, 0xAC, 0x01, 0x64,
			0x1D, 0x05, 0x53, 0xBE, 0xA8, 0x62, 0xEF, 0x8F,
			0x4A, 0x38, 0x31, 0x1D, 0x44, 0x4E, 0x46, 0xEA,
			0xE1, 0x97, 0xFC, 0xF3, 0x70, 0x08, 0xB4, 0x78,
			0xC7, 0xEF, 0xFF, 0xF3, 0x48, 0x1A, 0xA1, 0x3B,
			0xE1, 0xC0, 0x8B, 0x53, 0xF0, 0xA7, 0x98, 0x5E,
			0x87, 0x75, 0x48, 0xFB, 0x6C, 0x57, 0x7D, 0xE4,
			0x11, 0xD1, 0x56, 0x47, 0x70, 0x41, 0x89, 0xE4,
			0xF8, 0xFB, 0x3F, 0xFE, 0x94, 0xEA, 0xDB, 0x4B,
			0x05, 0xFF, 0x5F, 0x1C, 0x40, 0xCB, 0xD9, 0x9D,
			0x9F, 0xD4, 0x08, 0xA5, 0x9B, 0x24, 0x95, 0xB4,
			0x73, 0x92, 0x57, 0x99, 0xFF, 0xFA, 0xD9, 0x30,
			0xEC, 0x2B, 0x91, 0x60, 0x41, 0x5E, 0xA8, 0x09,
			0x52, 0xBA, 0x1B, 0xBC, 0x4A, 0xA2, 0x05, 0x85,
			0xBA, 0x31, 0x81, 0xAF, 0xF4, 0x1F, 0xE2, 0x28,
			0x46, 0xF8, 0x2E, 0xDF, 0x93, 0xF0, 0xD8, 0x28,
			0x06, 0x32, 0x33, 0x52, 0x82, 0x04, 0xF0, 0xD9,
			0xC8, 0x91, 0x61, 0xC2, 0x63, 0xA8, 0x41, 0xB8,
			0xCB, 0xAA, 0xB9, 0x9E, 0xE4, 0x80, 0xDD, 0xD7,
			0x7A, 0x6F, 0x63, 0xC5, 0xB6, 0x1E, 0xFC, 0x8B,
			0xD6, 0x19, 0x09, 0xD5, 0x92, 0x20, 0xC6, 0x96,
			0x26, 0x81, 0x00, 0x24, 0x57, 0x22, 0x09, 0x26,
			0x5C, 0xD7, 0x9D, 0x42, 0xC6, 0xA2, 0x72, 0x9A,
			0x56, 0x1F, 0xF8, 0x86, 0xE6, 0x35, 0x13, 0x65,
			0xB1, 0x72, 0x63, 0x0B, 0xFD, 0xA5, 0x8D, 0x22,
			0x6D, 0xB4, 0xFB, 0xFC, 0x2B, 0x9B, 0x1D, 0xF5,
			0x6F, 0x68, 0x27, 0x1C, 0xC0, 0xD5, 0x8E, 0x74,
			0x61, 0x41, 0x6A, 0x1F, 0x9E, 0xB0, 0xC1, 0x42,
			0x90, 0x75, 0x0F, 0x57, 0x11, 0xF9, 0xD8, 0x0C,
			0x51, 0x60, 0x1D, 0x53, 0x53, 0xE2, 0xFC, 0xB8,
			0x19, 0x46, 0x83, 0x5F, 0x95, 0xF9, 0x11, 0x84,
			0xF1, 0x19, 0x1B, 0x43, 0xD6, 0x03, 0xC9, 0xE3,
			0x79, 0x0F, 0x85, 0x23, 0xDA, 0x06, 0x20, 0xF8,
			0x34, 0xAE, 0x53, 0x17, 0x8E, 0xAB, 0x39, 0x2B,
			0x33, 0x56, 0xC3, 0x01, 0xDA, 0xA8, 0x09, 0xE1,
			0xBE, 0xF0, 0x69, 0x2E, 0x0E, 0xEA, 0x93, 0xF0,
			0x0C, 0xA0, 0x41, 0xA8, 0xD5, 0x6C, 0xE0, 0x64,
			0x6B, 0x2C, 0xF5, 0xF9, 0xEB, 0xE7, 0x8A, 0xBD,
			0x47, 0x99, 0x8F, 0xE9, 0xDB, 0x1F, 0x6C, 0x3C,
			0x9D, 0xC6, 0xD0, 0x1C, 0x0A, 0x61, 0xBC, 0x50,
			0x83, 0x9B, 0xBE, 0xAB, 0x75, 0xF4, 0x72, 0x58,
			0xF8, 0x22, 0x9A, 0xDB, 0xC3, 0x27, 0x89, 0x22,
			0xD4, 0x1D, 0x46, 0xD5, 0xC7, 0x75, 0x58, 0x18,
			0x74, 0xE5, 0x41, 0x46, 0x50, 0x52, 0x06, 0x13,
			0xF6, 0x73, 0x70, 0x23, 0xEE, 0x72, 0x63, 0x15,
			0x7C, 0xE8, 0x03, 0xCC, 0xA9, 0xE6, 0x91, 0xD8,
			0x73, 0x43, 0xB6, 0x83, 0xFC, 0xA5, 0xB2, 0x97,
			0xAF, 0xF9, 0x6B, 0xE2, 0x5D, 0x73, 0xD4, 0x25,
			0xD7, 0xE6, 0xDF, 0x8B, 0x85, 0x40, 0xE1, 0x18,
			0xA5, 0x4E, 0x59, 0xF8, 0x7A, 0x84, 0x7B, 0x9E,
			0x6E, 0x94, 0x84, 0x0E, 0x15, 0x58, 0xB2, 0x39,
			0xD4, 0x5C, 0xCA, 0x30, 0xDF, 0x8C, 0x5E, 0x59,
			0x12, 0xBA, 0x50, 0x35, 0xB9, 0x2C, 0x5C, 0xC7,
			0xD2, 0x17, 0x52, 0xBA, 0xD3, 0x5A, 0x1E, 0x81,
			0x36, 0x0C, 0xB2, 0xA0, 0xCA, 0x0E, 0xE4, 0x93,
			0x71, 0xA8, 0x9A, 0x1D, 0x69, 0x67, 0xD0, 0x5F,
			0xBD, 0x6C, 0x66, 0x00, 0xE6, 0x08, 0xE8, 0x15,
			0x16, 0xCA, 0x09, 0x56, 0x2A, 0xF4, 0xED, 0x48,
			0xA7, 0x8E, 0xC6, 0x76, 0x54, 0xDE, 0x70, 0xFC,
			0xE6, 0xDF, 0x11, 0x12, 0xBB, 0x22, 0x90, 0xE3,
			0x26, 0xAD, 0xBD, 0x5D, 0xE8, 0x65, 0xAE, 0xD7,
			0x8B, 0xC6, 0xE7, 0x42, 0xBE, 0xFE, 0x1E, 0x0D,
			0x0A, 0x04, 0x54, 0xFC, 0x18, 0x1E, 0xD5, 0x5B,
			0x50, 0x5B, 0xCD, 0x03, 0xF1, 0x92, 0x7A, 0x80,
			0x48, 0xA2, 0xB2, 0xE8, 0x36, 0x0D, 0x78, 0xEE,
			0xFB, 0xDF, 0x0F, 0xFB, 0x57, 0x56, 0xDE, 0x64,
			0x1E, 0x5B, 0xDD, 0x9B, 0x42, 0x65, 0x44, 0x71,
			0x48, 0x6C, 0xAD, 0x39, 0x28, 0xA5, 0x28, 0x34,
			0x46, 0x09, 0x8D, 0xA4, 0x9F, 0x69, 0xFD, 0xBE,
			0xA0, 0xAA, 0xBC, 0x7E, 0xB4, 0x5E, 0x07, 0x56,
			0xEA, 0x2E, 0x12, 0x10, 0x35, 0x09, 0x9D, 0xAB,
			0x70, 0xAF, 0x3F, 0x90, 0x8C, 0xDB, 0xCB, 0xEB,
			0xAB, 0x4F, 0x61, 0x7D, 0x9D, 0xF0, 0x3F, 0x3E,
			0x8D, 0x19, 0xEA, 0x12, 0x21, 0x3C, 0x62, 0xD8,
			0xC1, 0xFF, 0xC2, 0x74, 0xFE, 0x59, 0xB1, 0x87,
			0x30, 0xDC, 0xEB, 0x3E, 0x7B, 0x56, 0x92, 0xF5,
			0x81, 0xBF, 0x4B, 0x4B, 0x9E, 0xF5, 0x0F, 0x02,
			0xC4, 0xC5, 0xA5, 0x6A, 0xCE, 0x11, 0x97, 0x6A,
			0x57, 0xE0, 0xD5, 0xC4, 0x02, 0xB4, 0xB4, 0x6A,
			0x42, 0xBA, 0xD7, 0xA4, 0x34, 0x0D, 0xF0, 0xC4,
			0xEC, 0x90, 0xD9, 0xC6, 0x1E, 0x4E, 0xCD, 0x5E,
			0xAC, 0x19, 0xCE, 0x72, 0x0C, 0xD6, 0x3B, 0xC9,
			0x09, 0xF5, 0xE2, 0x84, 0xA7, 0xD0, 0xE6, 0xB3,
			0x18, 0x2C, 0x31, 0x12, 0x35, 0x17, 0x5B, 0xAF,
			0xB7, 0xEE, 0x53, 0x3B, 0x2F, 0x1E, 0x96, 0x84,
			0x33, 0xFE, 0x84, 0x36, 0x45, 0x02, 0xE4, 0xF0,
			0xE1, 0x5B, 0x32, 0x45, 0x16, 0xC5, 0xEC, 0x9C,
			0xA7, 0x43, 0xF2, 0xB6, 0xE4, 0xCD, 0xA1, 0x8D,
			0xDA, 0x2F, 0xAC, 0x29, 0x1D, 0xA4, 0x1C, 0x59,
			0x32, 0xBE, 0x8A, 0x9A, 0x35, 0x93, 0xCF, 0x88,
			0x15, 0x6B, 0xF3, 0x1A, 0x2B, 0xBE, 0x6B, 0x63,
			0x9E, 0x30, 0x85, 0x5F, 0x6D, 0x97, 0x03, 0x7E,
			0xEA, 0x15, 0xE0, 0x1E, 0x41, 0x23, 0x4E, 0xE6,
			0xC4, 0x13, 0x6E, 0xD3, 0x77, 0xEE, 0xDE, 0x22,
			0x5C, 0x80, 0xBB, 0xD6, 0x01, 0x6E, 0xAB, 0xC9,
			0x75, 0x48, 0xEC, 0xAD, 0xDA, 0x8F, 0x94, 0x14,
			0xC9, 0x98, 0x95, 0xBA, 0x46, 0x02, 0x5E, 0x43,
			0x9A, 0x3C, 0x3D, 0x7D, 0x35, 0x40, 0x6B, 0x54,
			0x4A, 0x66, 0xC1, 0xAA, 0xA8, 0x58, 0xD5, 0x3F,
			0xFA, 0xE0, 0x13, 0xEB, 0xB1, 0x53, 0xB6, 0xDD,
			0x15, 0xAC, 0x53, 0xCF, 0x4D, 0x10, 0x2E, 0x3B,
			0xB0, 0x17, 0xFB, 0x89, 0x33, 0x4F, 0x5C, 0x32,
			0x42, 0x86, 0x82, 0x34, 0x15, 0xC9, 0xCF, 0xD4,
			0x2B, 0xFD, 0x59, 0xEE, 0x58, 0xBA, 0x23, 0x46,
			0x8A, 0x8E, 0x98, 0xC4, 0xC8, 0xAD, 0x51, 0x84,
			0xAC, 0xD4, 0x44, 0x41, 0x0A, 0xF9, 0x47, 0xD8,
			0x3C, 0x3D, 0xA9, 0x1D, 0xAC, 0x12, 0x1B, 0xBB,
			0xE6, 0x2D, 0xE6, 0xAF, 0xED, 0xA9, 0xBA, 0x0F,
			0x4A, 0xD6, 0xFE, 0xBF, 0x28, 0xAB, 0x26, 0x3F,
			0xA4, 0x21, 0x26, 0xD6, 0x73, 0x51, 0x32, 0xDB,
			0xE8, 0xC3, 0x21, 0x3B, 0x26, 0xE7, 0xD1, 0x59,
			0x60, 0xE5, 0xE4, 0x68, 0x0A, 0x47, 0xD4, 0x28,
			0x35, 0xCA, 0x79, 0x68, 0xF9, 0xD4, 0xC8, 0x56,
			0x79, 0xCD, 0x58, 0x8A, 0x6F, 0x7A, 0xD8, 0x6E,
			0x67, 0x2F, 0x44, 0x4F, 0x93, 0xEE, 0x78, 0xE8,
			0x91, 0x0D, 0x79, 0xE4, 0xFE, 0x0F, 0xCF, 0xFC,
			0x6A, 0xD0, 0xA0, 0x3B, 0x9F, 0x8B, 0xFB, 0x10,
			0x9F, 0x62, 0xE3, 0xD7, 0xB1, 0xB1, 0xC6, 0x6D,
			0xAA, 0x7D, 0xCE, 0x1F, 0x38, 0x98, 0xA5, 0xA5,
			0xAA, 0x35, 0xFC, 0x64, 0x33, 0xA7, 0x76, 0xFE,
			0x7F, 0x2D, 0x65, 0x3F, 0x5A, 0x49, 0xA4, 0xEF,
			0x93, 0x5E, 0x5B, 0x48, 0x1A, 0x26, 0x7B, 0x4B,
			0x2C, 0xB3, 0x91, 0xCC, 0x9A, 0xCE, 0x24, 0x76,
			0xCD, 0xAB, 0x68, 0x27, 0x73, 0x29, 0x49, 0x25,
			0xB6, 0x66, 0x88, 0x5E, 0x9E, 0xAC, 0x0E, 0x3C,
			0x10, 0xFA, 0x5D, 0xA3, 0x18, 0xFA, 0x4B, 0x9A,
			0xB0, 0x47, 0x4A, 0x40, 0xD8, 0x24, 0xF7, 0x41,
			0x49, 0x41, 0xAF, 0xF6, 0x07, 0x9A, 0x56, 0x3C,
			0x8B, 0x78, 0xAB, 0xB1, 0x63, 0x55, 0x13, 0x0E,
			0x50, 0xD3, 0xEA, 0x7F, 0xC6, 0x3F, 0x4B, 0x15,
			0x57, 0xA8, 0x04, 0x37, 0xC0, 0x35, 0x77, 0xC4,
			0x60, 0x89, 0x80, 0x4E, 0x02, 0x27, 0x6E, 0xF3,
			0x88, 0x82, 0xA6, 0x7F, 0x53, 0x79, 0x74, 0x24,
			0x02, 0xE2, 0x3F, 0xB9, 0xDF, 0xC5, 0xF7, 0x4A,
			0xF4, 0x47, 0x11, 0x00, 0xD6, 0x57, 0x05, 0xEB,
			0x28, 0x5B, 0xE5, 0xDE, 0x50, 0x07, 0x68, 0x71,
			0x7D, 0xFF, 0x77, 0xC0, 0xEB, 0x75, 0x32, 0x4B,
			0xA0, 0xD2, 0x69, 0x6B, 0x71, 0xCF, 0x02, 0x29,
			0x1D, 0xE9, 0x4F, 0x74, 0xFC, 0x69, 0x6A, 0xB4,
			0x4C, 0xBF, 0x4B, 0x7E, 0xFA, 0xD2, 0x08, 0x7E,
			0x53, 0xC0, 0xCD, 0x7A, 0xD0, 0x4A, 0xC6, 0x53,
			0xDC, 0x02, 0xEF, 0x07, 0x5E, 0xED, 0x82, 0xEB,
			0x4A, 0xC3, 0x51, 0xD1, 0x15, 0xCC, 0x06, 0x5D,
			0x31, 0xD7, 0xBA, 0x0A, 0x83, 0x98, 0x23, 0x85,
			0x37, 0xC8, 0x1A, 0x22, 0xBF, 0x73, 0xB1, 0xA8,
			0xEF, 0x4F, 0xAE, 0xD3, 0xD7, 0xB2, 0x03, 0xFD,
			0x87, 0x60, 0x1D, 0x60, 0xAF, 0x38, 0x94, 0x73,
			0xF4, 0x69, 0x6B, 0x43, 0xB2, 0xCC, 0xE1, 0xA8,
			0x3C, 0x26, 0x9D, 0x4B, 0x6B, 0x23, 0xC8, 0xC1,
			0xBC, 0xFA, 0xB4, 0xFF, 0xE0, 0xF1, 0xDA, 0x39,
			0x6B, 0x59, 0xBC, 0x36, 0x8C, 0x73, 0x19, 0x20,
			0x59, 0xC1, 0xFB, 0xFB, 0x21, 0x59, 0x2B, 0x26,
			0x15, 0x8A, 0x83, 0xB9, 0xC1, 0xE6, 0x88, 0x29,
			0xE9, 0xFE, 0x6A, 0x40, 0xE8, 0xD8, 0xEE, 0xC9,
			0xD8, 0x2F, 0x37, 0x94, 0x53, 0x9C, 0x36, 0x68,
			0xE5, 0x64, 0x1E, 0x29, 0xCE, 0xF2, 0x24, 0xCD,
			0xE1, 0xB2, 0x7C, 0xC3, 0x7B, 0x5B, 0x9D, 0xA6,
			0x90, 0x12, 0x99, 0xDA, 0xC5, 0xBC, 0xD5, 0x01,
			0x79, 0xA6, 0x50, 0x5E, 0x56, 0xDA, 0xA7, 0xC2,
			0x23, 0xFE, 0xAC, 0x66, 0xE2, 0xA8, 0x97, 0x24,
			0x73, 0x5A, 0x00, 0xAA, 0x9B, 0x7E, 0xE4, 0x90,
			0x8C, 0x3F, 0x3E, 0xA6, 0x41, 0x05, 0xED, 0x3E,
			0x1B, 0x72, 0xA7, 0x60, 0xA6, 0x27, 0x58, 0x41,
			0x77, 0x6C, 0x66, 0x5B, 0xFC, 0xCA, 0x1D, 0x77,
			0xF2, 0xF4, 0xE8, 0x33, 0x8F, 0x45, 0x00, 0x35,
			0x8B, 0xD4, 0xE8, 0xDC, 0x08, 0x18, 0x95, 0x8F,
			0x71, 0xF6, 0xFF, 0xA5, 0x86, 0x04, 0x0D, 0x16,
			0x67, 0x75, 0x92, 0xB8, 0x29, 0x53, 0xB9, 0x9F,
			0x9A, 0xE1, 0x8F, 0x4F, 0xFC, 0xEB, 0x13, 0xEA,
			0xBC, 0x5C, 0xB4, 0x8A, 0xE0, 0x1D, 0x50, 0x59,
			0x31, 0x39, 0x28, 0x87, 0x3A, 0x54, 0x9D, 0x4D,
			0xAD, 0xD8, 0x1D, 0x43, 0x8A, 0x9F, 0x67, 0x2F,
			0xF2, 0x19, 0x99, 0x9B, 0xB1, 0xB2, 0x80, 0x20,
			0xB6, 0xAA, 0xE2, 0x0A, 0xD3, 0x13, 0xB8, 0xEF,
			0x44, 0x86, 0xDF, 0x5E, 0x6E, 0x78, 0xF4, 0x83,
			0xD1, 0x7B, 0x08, 0x22, 0xC7, 0x56, 0xA0, 0x37,
			0x3B, 0xBA, 0x28, 0xFD, 0x23, 0x35, 0x77, 0x0A,
			0x81, 0x25, 0xF7, 0xD2, 0x96, 0xB9, 0x81, 0xF1,
			0x94, 0x9C, 0x48, 0x19, 0x58, 0x8B, 0xA4, 0x52,
			0x37, 0xAF, 0xBF, 0xFE, 0xE3, 0xB6, 0x39, 0x7E,
			0x42, 0x7B, 0xE3, 0x27, 0x66, 0x3A, 0x30, 0x83,
			0xB6, 0x7A, 0xA6, 0x1B, 0x3C, 0xE7, 0x50, 0xAC,
			0x63, 0x5D, 0x51, 0x08, 0xB1, 0x76, 0xAD, 0xFE,
			0x83, 0x62, 0xA7, 0x5E, 0x26, 0x00, 0x60, 0x7C,
			0xEE, 0x63, 0xA6, 0xAA, 0xAB, 0xA3, 0xBC, 0xBC,
			0xC7, 0x76, 0x38, 0x57, 0xEE, 0xC7, 0x40, 0x80,
			0x2A, 0x5D, 0xFF, 0x07, 0x79, 0x82, 0xE0, 0x29,
			0x13, 0x3B, 0x37, 0xB8, 0x9C, 0x2C, 0x59, 0xEB,
			0xF7, 0x1C, 0x92, 0xC3, 0x20, 0x99, 0x29, 0x06,
			0x82, 0x5C, 0xF5, 0x2F, 0x84, 0xA1, 0x04, 0xBC,
			0x55, 0x69, 0x1D, 0xED, 0x1C, 0x1B, 0xF8, 0x1B,
			0xB6, 0xF4, 0x93, 0x74, 0x90, 0x48, 0xF1, 0x1A,
			0xE6, 0xEE, 0x2B, 0xE6, 0x6E, 0x0D, 0x03, 0x4E,
			0x81, 0x0E, 0x8E, 0x42, 0xC7, 0xDD, 0xD0, 0xDF,
			0x7A, 0xE2, 0x45, 0x4F, 0xF1, 0x6A, 0x28, 0xDD,
			0xA2, 0xC2, 0xB2, 0xCA, 0x1B, 0xD4, 0x5A, 0x81,
			0xDD, 0x6D, 0x08, 0x0D, 0x0B, 0xAB, 0xCA, 0xEB,
			0x44, 0x80, 0x79, 0xBF, 0x13, 0x89, 0xA3, 0x1B,
			0xA5, 0x97, 0xD9, 0x66, 0x7B, 0x1B, 0x3F, 0x04,
			0xE0, 0x5B, 0x8C, 0x01, 0x97, 0xCD, 0x14, 0xA4,
			0x5C, 0x75, 0x3F, 0x77, 0xCB, 0xFF, 0xA6, 0xA4,
			0x31, 0xF7, 0x70, 0xD2, 0xC6, 0x30, 0xE5, 0xE4,
			0xDA, 0x65, 0x31, 0x4F, 0x4C, 0xB9, 0x56, 0xC9,
			0x4E, 0x75, 0x02, 0x3D, 0x91, 0x06, 0x13, 0xA4,
			0xAB, 0xA8, 0x5F, 0x7D, 0x0C, 0x73, 0x00, 0xAE,
			0x9B, 0x30, 0xCE, 0x8A, 0xE4, 0x74, 0x10, 0xB4,
			0x37, 0x94, 0x5C, 0x5C, 0x93, 0x1D, 0x0E, 0xD4,
			0x07, 0x2B, 0x9F, 0x00, 0x04, 0x03, 0x8C, 0x23,
			0x9B, 0x06, 0xF8, 0x98, 0xE0, 0xF1, 0x53, 0x8B,
			0x39, 0x8D, 0xE1, 0x8F, 0x70, 0xC9, 0x24, 0x7F,
			0x43, 0x74, 0x9F, 0xAA, 0xAD, 0x21, 0xF0, 0xC6,
			0x25, 0xB1, 0x44, 0x32, 0x30, 0xC9, 0xE5, 0x8D,
			0x69, 0xD5, 0x96, 0xD7, 0x32, 0x51, 0x77, 0x50,
			0xEB, 0xF6, 0x7D, 0x21, 0x6C, 0x1D, 0x87, 0xFD,
			0x54, 0xB7, 0x1F, 0x00, 0x00, 0x93, 0x20, 0x7D,
			0xE7, 0x35, 0x3F, 0xF0, 0x38, 0xBF, 0xF7, 0x36,
			0x0A, 0x3A, 0x0F, 0x9A, 0x1F, 0xE9, 0x78, 0x25,
			0x03, 0xDF, 0x3A, 0x4B, 0x75, 0x9C, 0xFB, 0x44,
			0xC7, 0xF8, 0x59, 0xCB, 0xAD, 0x58, 0xB9, 0xA7,
			0xBB, 0x85, 0x3A, 0xC4, 0xEC, 0x27, 0x5B, 0x99,
			0x91, 0xF8, 0xF1, 0xB6, 0xCF, 0x48, 0xBA, 0x95,
			0x19, 0x08, 0x2E, 0x99, 0x99, 0x7B, 0xD1, 0x00,
			0x14, 0xDC, 0x7C, 0x5F, 0x82, 0xFB, 0x22, 0x7A,
			0x06, 0x7F, 0x21, 0x43, 0x73, 0x89, 0x93, 0x2A,
			0x54, 0x63, 0x64, 0xDF, 0xC3, 0x4C, 0x0C, 0xEF,
			0x71, 0xD4, 0x8C, 0x66, 0x1F, 0xA2, 0x33, 0x97,
			0xAC, 0x87, 0x32, 0xDC, 0x55, 0xF0, 0x70, 0xA2,
			0xFF, 0x90, 0xEF, 0xE5, 0x0B, 0x0F, 0x01, 0x60,
			0xBB, 0xCD, 0x24, 0xDC, 0x70, 0x90, 0xE1, 0x3C,
			0x40, 0xB0, 0x0D, 0x69, 0xE0, 0xDE, 0x33, 0x31,
			0x38, 0xB8, 0x4F, 0xC7, 0x72, 0x1B, 0x11, 0x2F,
			0x33, 0xC3, 0x4C, 0x12, 0xD3, 0x9F, 0x1A, 0x3F,
			0xB5, 0x4E, 0xD8, 0xB7, 0x2D, 0x90, 0x60, 0x64,
			0x3F, 0x66, 0xFB, 0x1D, 0x44, 0x08, 0x40, 0xA6,
			0x61, 0x0C, 0x2A, 0xC9, 0xE7, 0xEC, 0x7D, 0x69,
			0x2D, 0x14, 0x16, 0x6A, 0xC5, 0xC0, 0x98, 0x83,
			0xEF, 0x4F, 0x1B, 0xBF, 0x48, 0xF9, 0x3D, 0xC2,
			0xD0, 0xEE, 0xE2, 0xB6, 0x2C, 0xF4, 0x57, 0x09,
			0x74, 0xBF, 0x88, 0x70, 0x05, 0xDD, 0xD0, 0x83,
			0x80, 0x2F, 0x64, 0x9A, 0xF3, 0x63, 0xC3, 0x87,
			0xF6, 0xDD, 0xA3, 0x0F, 0xF8, 0x1A, 0x28, 0xF6,
			0xC9, 0xFA, 0xFB, 0x2B, 0x08, 0x13, 0xAC, 0xC9,
			0x37, 0x73, 0x67, 0x97, 0xD8, 0x15, 0xED, 0xF1,
			0x49, 0x0D, 0xC0, 0xE0, 0x0C, 0x1F, 0x19, 0x4D,
			0xEA, 0xBB, 0xBE, 0x56, 0x93, 0x79, 0x1A, 0x31,
			0xA7, 0xF2, 0xA6, 0xBD, 0xCA, 0x81, 0x5F, 0xC7,
			0x1B, 0xD6, 0xB5, 0xF8, 0x4B, 0xAA, 0xCC, 0x68,
			0x69, 0x5A, 0x2C, 0x4C, 0x95, 0xAC, 0x6E, 0xC3,
			0xEE, 0x2A, 0xB5, 0x4F, 0xA9, 0x75, 0x59, 0x57,
			0x30, 0x07, 0xC8, 0x00, 0x81, 0x7F, 0xF4, 0x62,
			0xBF, 0x0C, 0x08, 0x26, 0x82, 0x11, 0x3B, 0x23,
			0xEF, 0x5E, 0xB2, 0xA7, 0x16, 0xB2, 0xE1, 0xF1,
			0x38, 0x79, 0xE1, 0xD2, 0x2C, 0x14, 0x64, 0xCE,
			0xBE, 0x4C, 0xA0, 0x45, 0x83, 0x7B, 0x44, 0xA9,
			0xD5, 0x47, 0x6F, 0x0C, 0x47, 0x8B, 0x0F, 0x9A,
			0xB7, 0xE2, 0x4C, 0xC9, 0x73, 0x2C, 0x04, 0x2D,
			0x2A, 0xAE, 0x62, 0x54, 0xB0, 0x1C, 0x18, 0x10,
			0x04, 0xCA, 0xE7, 0xBD, 0xCF, 0x87, 0xE0, 0xB6,
			0x19, 0x72, 0xE7, 0x2E, 0xA9, 0x7B, 0x96, 0x75,
			0x38, 0x41, 0x65, 0x1E, 0x72, 0xDB, 0x4F, 0xF8,
			0x95, 0x4E, 0x3E, 0x6F, 0x0B, 0x1B, 0x5C, 0xBB,
			0x2F, 0xC9, 0x95, 0x6C, 0x41, 0x4D, 0x8C, 0xA4,
			0x4F, 0xE3, 0xD1, 0x12, 0x35, 0x5E, 0xFD, 0x89,
		},
	}
};

#ifdef __cplusplus
}
#endif

#endif /* SPHINCS_TESTER_VECTORS_SHAKE_128S_H */
