/*
 * Copyright (C) 2025, Stephan Mueller <smueller@chronox.de>
 *
 * License: see LICENSE file in root directory
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ALL OF
 * WHICH ARE HEREBY DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF NOT ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

use leancrypto_sys::lcr_sym::lcr_sym;
use leancrypto_sys::lcr_sym::lcr_sym_type;

fn lc_rust_sym_one(sym_type: lcr_sym_type, key: &[u8], iv: &[u8], pt: &[u8],
		    ct: &mut [u8], exp_ct: &[u8]) {
	let mut sym = lcr_sym::new(sym_type);

	// One-shot encrypt
	let result = sym.setkey(key);
	assert_eq!(result, Ok(()));
	let result = sym.setiv(iv);
	assert_eq!(result, Ok(()));
	let result = sym.encrypt(pt, ct);
	assert_eq!(result, Ok(()));
	assert_eq!(ct, exp_ct);

	// One-shot decrypt
	let result = sym.setkey(key);
	assert_eq!(result, Ok(()));
	let result = sym.setiv(iv);
	assert_eq!(result, Ok(()));
	let mut new_pt = vec!(0u8; pt.len());
	let result = sym.decrypt(ct, &mut new_pt);
	assert_eq!(result, Ok(()));
	assert_eq!(pt, new_pt);
}

fn lc_rust_kw_one(sym_type: lcr_sym_type, key: &[u8], exp_iv: &[u8], pt: &[u8],
		  ct: &mut [u8], exp_ct: &[u8]) {
	let mut sym = lcr_sym::new(sym_type);

	// One-shot encrypt
	let result = sym.setkey(key);
	assert_eq!(result, Ok(()));
	let result = sym.kw_encrypt(pt, ct);
	assert_eq!(result, Ok(()));
	assert_eq!(&ct[0..8], exp_iv);
	assert_eq!(&ct[8..], exp_ct);

	// One-shot decrypt
	let result = sym.setkey(key);
	assert_eq!(result, Ok(()));
	let mut new_pt = vec!(0u8; pt.len());
	let result = sym.kw_decrypt(ct, &mut new_pt);
	assert_eq!(result, Ok(()));
	assert_eq!(pt, new_pt);
}

#[test]
fn lc_rust_aes_cbc() {
	let key: [u8; 32] = [
		0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71,
		0xbe, 0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d,
		0x77, 0x81, 0x1f, 0x35, 0x2c, 0x07, 0x3b,
		0x61, 0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3,
		0x09, 0x14, 0xdf, 0xf4
	];
	let iv: [u8; 16] = [
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
		0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f
	];

	let pt: [u8; 64] = [
		0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
		0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
		0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
		0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
		0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
		0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
		0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
		0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10
	];
	let exp_ct: [u8; 64] = [
		0xf5, 0x8c, 0x4c, 0x04, 0xd6, 0xe5, 0xf1, 0xba, 0x77, 0x9e,
		0xab, 0xfb, 0x5f, 0x7b, 0xfb, 0xd6, 0x9c, 0xfc, 0x4e, 0x96,
		0x7e, 0xdb, 0x80, 0x8d, 0x67, 0x9f, 0x77, 0x7b, 0xc6, 0x70,
		0x2c, 0x7d, 0x39, 0xf2, 0x33, 0x69, 0xa9, 0xd9, 0xba, 0xcf,
		0xa5, 0x30, 0xe2, 0x63, 0x04, 0x23, 0x14, 0x61, 0xb2, 0xeb,
		0x05, 0xe2, 0xc3, 0x9b, 0xe9, 0xfc, 0xda, 0x6c, 0x19, 0x07,
		0x8c, 0x6a, 0x9d, 0x1b
	];
	let mut ct: [u8; 64] = [0; 64];

	lc_rust_sym_one(lcr_sym_type::lcr_aes_cbc, &key, &iv, &pt, &mut ct,
			&exp_ct)
}

#[test]
fn lc_rust_aes_ctr() {
	let key: [u8; 32] = [
		0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71,
		0xbe, 0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d,
		0x77, 0x81, 0x1f, 0x35, 0x2c, 0x07, 0x3b,
		0x61, 0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3,
		0x09, 0x14, 0xdf, 0xf4
	];
	let iv: [u8; 16] = [
		0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5,
		0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb,
		0xfc, 0xfd, 0xfe, 0xff
	];

	let pt: [u8; 63] = [
		0x60, 0x1e, 0xc3, 0x13, 0x77, 0x57, 0x89, 0xa5, 0xb7, 0xa7,
		0xf5, 0x04, 0xbb, 0xf3, 0xd2, 0x28, 0xf4, 0x43, 0xe3, 0xca,
		0x4d, 0x62, 0xb5, 0x9a, 0xca, 0x84, 0xe9, 0x90, 0xca, 0xca,
		0xf5, 0xc5, 0x2b, 0x09, 0x30, 0xda, 0xa2, 0x3d, 0xe9, 0x4c,
		0xe8, 0x70, 0x17, 0xba, 0x2d, 0x84, 0x98, 0x8d, 0xdf, 0xc9,
		0xc5, 0x8d, 0xb6, 0x7a, 0xad, 0xa6, 0x13, 0xc2, 0xdd, 0x08,
		0x45, 0x79, 0x41
	];
	let exp_ct: [u8; 63] = [
		0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d,
		0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a, 0xae, 0x2d, 0x8a, 0x57,
		0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf,
		0x8e, 0x51, 0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
		0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef, 0xf6, 0x9f,
		0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b,
		0xe6, 0x6c, 0x37
	];
	let mut ct: [u8; 63] = [0; 63];

	lc_rust_sym_one(lcr_sym_type::lcr_aes_ctr, &key, &iv, &pt, &mut ct,
			&exp_ct)
}

#[test]
fn lc_rust_aes_kw() {
	let key: [u8; 32] = [
		0x80, 0xaa, 0x99, 0x73, 0x27, 0xa4, 0x80,
		0x6b, 0x6a, 0x7a, 0x41, 0xa5, 0x2b, 0x86,
		0xc3, 0x71, 0x03, 0x86, 0xf9, 0x32, 0x78,
		0x6e, 0xf7, 0x96, 0x76, 0xfa, 0xfb, 0x90,
		0xb8, 0x26, 0x3c, 0x5f
	];
	let iv: [u8; 8] = [
		0x42, 0x3c, 0x96, 0x0d, 0x8a, 0x2a, 0xc4, 0xc1
	];

	let pt: [u8; 16] = [
		0x0a, 0x25, 0x6b, 0xa7, 0x5c, 0xfa, 0x03, 0xaa,
		0xa0, 0x2b, 0xa9, 0x42, 0x03, 0xf1, 0x5b, 0xaa
	];
	let exp_ct: [u8; 16] = [
		0xd3, 0x3d, 0x3d, 0x97, 0x7b, 0xf0, 0xa9, 0x15,
		0x59, 0xf9, 0x9c, 0x8a, 0xcd, 0x29, 0x3d, 0x43
	];
	let mut ct: [u8; 24] = [0; 24];

	lc_rust_kw_one(lcr_sym_type::lcr_aes_kw, &key, &iv, &pt, &mut ct,
		       &exp_ct)
}

#[test]
fn lc_rust_chacha20() {
	let key: [u8; 32] = [
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
		0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d,
		0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14,
		0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b,
		0x1c, 0x1d, 0x1e, 0x1f
	];
	let iv: [u8; 12] = [
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x4a, 0x00, 0x00, 0x00, 0x00
	];

	let pt: &str = "Ladies and Gentlemen of the class of '99: If I could offer you only one tip for the future, sunscreen would be it.";
	let exp_ct: [u8; 114] = [
		0x6e, 0x2e, 0x35, 0x9a, 0x25, 0x68, 0xf9, 0x80, 0x41, 0xba,
		0x07, 0x28, 0xdd, 0x0d, 0x69, 0x81, 0xe9, 0x7e, 0x7a, 0xec,
		0x1d, 0x43, 0x60, 0xc2, 0x0a, 0x27, 0xaf, 0xcc, 0xfd, 0x9f,
		0xae, 0x0b, 0xf9, 0x1b, 0x65, 0xc5, 0x52, 0x47, 0x33, 0xab,
		0x8f, 0x59, 0x3d, 0xab, 0xcd, 0x62, 0xb3, 0x57, 0x16, 0x39,
		0xd6, 0x24, 0xe6, 0x51, 0x52, 0xab, 0x8f, 0x53, 0x0c, 0x35,
		0x9f, 0x08, 0x61, 0xd8, 0x07, 0xca, 0x0d, 0xbf, 0x50, 0x0d,
		0x6a, 0x61, 0x56, 0xa3, 0x8e, 0x08, 0x8a, 0x22, 0xb6, 0x5e,
		0x52, 0xbc, 0x51, 0x4d, 0x16, 0xcc, 0xf8, 0x06, 0x81, 0x8c,
		0xe9, 0x1a, 0xb7, 0x79, 0x37, 0x36, 0x5a, 0xf9, 0x0b, 0xbf,
		0x74, 0xa3, 0x5b, 0xe6, 0xb4, 0x0b, 0x8e, 0xed, 0xf2, 0x78,
		0x5e, 0x42, 0x87, 0x4d
	];
	let mut ct: [u8; 114] = [0; 114];

	lc_rust_sym_one(lcr_sym_type::lcr_chacha20, &key, &iv, pt.as_bytes(),
			&mut ct, &exp_ct)
}
