/*
 * Copyright (C) 2025, Stephan Mueller <smueller@chronox.de>
 *
 * License: see LICENSE file in root directory
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ALL OF
 * WHICH ARE HEREBY DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF NOT ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

use leancrypto_sys::lcr_rng::lcr_rng;
use leancrypto_sys::lcr_rng::lcr_rng_type;
use leancrypto_sys::error::RngError;

#[test]
fn lc_rust_rng_seeded() {
	let mut rng = lcr_rng::new();

	let not_exp: [u8; 15] = [0; 15];

	let mut rngdata = vec!(0u8; not_exp.len());
	let result = rng.generate(&[], &mut rngdata);
	assert_eq!(result, Ok(()));
	assert_eq!(rngdata.len(), not_exp.len());
	assert_ne!(rngdata, not_exp);
}

fn lc_rust_rng_one(rng_type: lcr_rng_type, seed: &[u8], exp: &[u8]) {
	let mut rng = lcr_rng::new();

	let result = rng.set_type(rng_type);
	assert_eq!(result, Ok(()));

	// This should fail as we are not seeded
	let mut rngdata = vec!(0u8; exp.len());
	let result = rng.generate(&[], &mut rngdata);
	assert_eq!(result, Err(RngError::NotSeeded));

	let result = rng.seed(seed, &[]);
	assert_eq!(result, Ok(()));

	let result = rng.generate(&[], &mut rngdata);
	assert_eq!(result, Ok(()));
	assert_eq!(rngdata.len(), exp.len());
	assert_eq!(rngdata, exp);
}

fn lc_rust_drbg_one(rng_type: lcr_rng_type, ent_nonce: &[u8], pers: &[u8],
		    addtl1: &[u8], addtl2: &[u8], exp: &[u8]) {
	let mut rng = lcr_rng::new();

	let result = rng.set_type(rng_type);
	assert_eq!(result, Ok(()));

	let mut rngdata = vec!(0u8; exp.len());

	// This should fail as we are not seeded
	let result = rng.generate(&[], &mut rngdata);
	assert_eq!(result, Err(RngError::NotSeeded));

	let result = rng.seed(ent_nonce, pers);
	assert_eq!(result, Ok(()));

	let result = rng.generate(addtl1, &mut rngdata);
	assert_eq!(result, Ok(()));
	assert_eq!(rngdata.len(), exp.len());

	let result = rng.generate(addtl2, &mut rngdata);
	assert_eq!(result, Ok(()));
	assert_eq!(rngdata.len(), exp.len());
	assert_eq!(rngdata, exp);
}

#[test]
fn lc_rust_rng_xdrbg256() {
	let seed: [u8; 9] = [
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
	];
	let exp: [u8; 345] = [
		0x1a, 0xd2, 0xcb, 0x76, 0x3c, 0x71, 0x6d, 0xf0, 0x79, 0x2c,
		0xc0, 0x69, 0x7d, 0x56, 0x6a, 0x65, 0xb8, 0x36, 0xbe, 0x7d,
		0x09, 0x12, 0x7c, 0x65, 0x47, 0xfc, 0x30, 0x58, 0xaa, 0x24,
		0x39, 0x52, 0x29, 0xea, 0xce, 0x43, 0xdf, 0x16, 0x2c, 0x4f,
		0x1a, 0xed, 0xbd, 0x3f, 0xf5, 0x8e, 0xe6, 0x4d, 0x93, 0x07,
		0x3d, 0x7f, 0x3d, 0xd2, 0x50, 0x3c, 0xae, 0x04, 0x4a, 0x87,
		0x2c, 0x90, 0x30, 0xd4, 0x8e, 0xef, 0x5d, 0x53, 0x0f, 0xb2,
		0xdb, 0xec, 0x16, 0x39, 0x5a, 0xb5, 0x9a, 0xdc, 0x9d, 0x01,
		0x7e, 0xe2, 0xac, 0x7c, 0xe4, 0x3d, 0xfd, 0x93, 0xa6, 0x6c,
		0xc1, 0x22, 0x26, 0x64, 0xa0, 0x43, 0x52, 0x51, 0xf9, 0xb5,
		0xa4, 0x91, 0x54, 0x08, 0xf8, 0x8f, 0x16, 0x85, 0x54, 0xc0,
		0x9d, 0xce, 0xc9, 0xd5, 0xd7, 0xa9, 0x51, 0xc0, 0x06, 0x0c,
		0x04, 0x95, 0xcf, 0x7d, 0x27, 0x00, 0x7e, 0x48, 0x6d, 0x2e,
		0xbc, 0xf8, 0xa3, 0x71, 0x3d, 0xb0, 0x2b, 0x75, 0x2a, 0x48,
		0x1a, 0xd3, 0xed, 0xc9, 0xa3, 0x80, 0x88, 0x03, 0xc0, 0x27,
		0x75, 0xcc, 0xf5, 0xda, 0x56, 0x8d, 0x83, 0x36, 0xe6, 0x90,
		0x9c, 0xd5, 0x82, 0xfa, 0x70, 0xe9, 0xbf, 0x61, 0xec, 0x97,
		0xcc, 0xdd, 0xdc, 0x4e, 0xe1, 0x64, 0x9f, 0x1e, 0xb3, 0xfa,
		0x97, 0xa7, 0x02, 0x0a, 0x28, 0x01, 0x19, 0xd0, 0x45, 0xe9,
		0x21, 0x74, 0x52, 0x1a, 0xac, 0x5f, 0x58, 0x7c, 0x02, 0x47,
		0x45, 0x06, 0x17, 0x71, 0xc5, 0x2b, 0x0f, 0xa9, 0xed, 0x5c,
		0xd1, 0x46, 0x63, 0x57, 0xb5, 0x6a, 0x5c, 0x95, 0xd1, 0xa4,
		0xdf, 0x61, 0x62, 0x39, 0x41, 0x47, 0xb1, 0x4e, 0x91, 0x7c,
		0x50, 0x1f, 0xc0, 0x48, 0x42, 0xb6, 0xea, 0x16, 0x4c, 0x50,
		0x29, 0x12, 0xd0, 0x1c, 0x39, 0x9f, 0x79, 0x23, 0x63, 0x4f,
		0x9e, 0xa2, 0x56, 0x57, 0x26, 0xb2, 0xd3, 0xb3, 0xcc, 0xfe,
		0x58, 0x60, 0x0c, 0x5b, 0x59, 0x00, 0xff, 0xe2, 0xa5, 0x5a,
		0x50, 0x44, 0x02, 0x55, 0x6e, 0x87, 0x4f, 0x01, 0x50, 0x9a,
		0xd4, 0x0b, 0x96, 0x56, 0xb7, 0x6f, 0x13, 0x9d, 0xd7, 0x05,
		0xa4, 0xfb, 0xe7, 0xcc, 0x76, 0x94, 0x4b, 0x19, 0xc6, 0xc7,
		0xda, 0x0f, 0x88, 0x96, 0xb3, 0x9f, 0xbf, 0x6e, 0x46, 0x72,
		0xeb, 0xa9, 0x23, 0xaa, 0xf8, 0x6a, 0xb2, 0xfd, 0x01, 0xef,
		0x76, 0xf8, 0x9d, 0x29, 0x6e, 0xe2, 0x9f, 0xcc, 0xab, 0xc2,
		0x60, 0x7b, 0x8f, 0xae, 0x58, 0xe1, 0x9c, 0x79, 0x20, 0x85,
		0x66, 0xe3, 0x93, 0xd7, 0x7f
	];

	lc_rust_rng_one(lcr_rng_type::lcr_xdrbg256, &seed, &exp)
}

#[test]
fn lc_rust_rng_xdrbg128() {
	let seed: [u8; 9] = [
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
	];
	let exp: [u8; 247] = [
		0x90, 0xfc, 0x06, 0x53, 0xfb, 0x53, 0xac, 0x68, 0xe9, 0x84,
		0xc5, 0x7f, 0x99, 0xdf, 0x5b, 0x33, 0x8d, 0xba, 0xcd, 0xcb,
		0x8b, 0x87, 0x07, 0x94, 0xe5, 0x36, 0x5e, 0x1c, 0xc1, 0x12,
		0x41, 0x9b, 0xc9, 0x78, 0x0a, 0x58, 0xc9, 0xbc, 0x65, 0x9a,
		0xec, 0xd0, 0xd9, 0xad, 0x6b, 0x56, 0xb0, 0xb5, 0xc5, 0x0a,
		0xbc, 0x87, 0x87, 0x5d, 0x94, 0x2b, 0x9e, 0xe3, 0xad, 0x5f,
		0xc3, 0xa8, 0x8c, 0xc6, 0x96, 0xac, 0x38, 0x08, 0x95, 0x58,
		0x41, 0xd3, 0xd8, 0xf0, 0x31, 0xcd, 0x27, 0xe5, 0x23, 0xfd,
		0x54, 0x7d, 0x65, 0x11, 0x1b, 0xa1, 0xdb, 0x09, 0xe1, 0xe9,
		0xb6, 0x47, 0xee, 0xa3, 0x9a, 0x7e, 0x15, 0xfd, 0xcf, 0xa0,
		0x08, 0xba, 0xce, 0x57, 0xa4, 0xa7, 0x03, 0x52, 0xf0, 0x78,
		0xf7, 0x2a, 0x7b, 0xd1, 0xa1, 0xd6, 0x9f, 0xb4, 0xc2, 0x6f,
		0x7d, 0x3e, 0xfc, 0x78, 0xf5, 0x3b, 0x25, 0x51, 0x56, 0x7e,
		0xe9, 0xb3, 0x4c, 0x0d, 0x2b, 0x1f, 0xef, 0xe1, 0xda, 0x13,
		0x2d, 0xd6, 0xf0, 0x32, 0x22, 0x12, 0x8b, 0x59, 0x3b, 0x97,
		0x28, 0x27, 0x09, 0xa1, 0x9c, 0x41, 0xb3, 0x5b, 0x21, 0x53,
		0x70, 0x3d, 0x02, 0xa1, 0x13, 0x81, 0x33, 0x69, 0x71, 0x7d,
		0x3b, 0x19, 0xa2, 0x9e, 0xbf, 0x64, 0xcd, 0xc6, 0x52, 0x9b,
		0xd3, 0x78, 0x6a, 0x29, 0x1a, 0x34, 0x50, 0xc0, 0x92, 0x1b,
		0x4b, 0x4d, 0xa9, 0xc2, 0x47, 0x72, 0xc4, 0xf2, 0xef, 0x32,
		0x0b, 0x4d, 0xb7, 0x4e, 0x78, 0x58, 0x20, 0x1d, 0xbd, 0x0d,
		0x23, 0x29, 0xfe, 0x1c, 0x36, 0x67, 0xa2, 0x8f, 0x2d, 0xba,
		0x4b, 0x69, 0xfd, 0x24, 0x2d, 0x3a, 0x36, 0xc2, 0xea, 0x5e,
		0x65, 0x21, 0x44, 0x23, 0xfc, 0x25, 0x2b, 0x07, 0x1e, 0xcf,
		0x55, 0x92, 0x3e, 0x6a, 0x8a, 0x21, 0xb1
	];

	lc_rust_rng_one(lcr_rng_type::lcr_xdrbg128, &seed, &exp)
}

#[test]
fn lc_rust_rng_hmac_drbg() {
	let ent_nonce: [u8; 64] = [
		0xC5, 0xD9, 0xD7, 0x7B, 0x3E, 0x5C, 0x0E, 0xC8, 0x57, 0x13,
		0xEB, 0x25, 0x12, 0xE8, 0x15, 0x40, 0xBF, 0x65, 0x89, 0x15,
		0xB3, 0xF9, 0xC8, 0x95, 0x22, 0x05, 0xB5, 0xF0, 0x16, 0x0F,
		0xD0, 0xE8, 0xBD, 0xA7, 0xC6, 0x58, 0xE2, 0x4D, 0xB8, 0xBD,
		0xFC, 0xC5, 0x4E, 0x3A, 0xFE, 0xAA, 0xB5, 0x79, 0x71, 0xD4,
		0x95, 0x4D, 0xD9, 0x98, 0x38, 0x34, 0x28, 0xF2, 0x1A, 0x34,
		0x2D, 0xE8, 0xC9, 0x74
	];
	let pers: [u8; 32] = [
		0x6E, 0xCF, 0x0F, 0xCC, 0x7C, 0x6F,
		0xEC, 0x03, 0x25, 0x8B, 0xDB, 0x2D,
		0xC5, 0xC5, 0xEB, 0x39, 0x33, 0x09,
		0x39, 0x53, 0xED, 0xDD, 0xC0, 0x23,
		0x26, 0x8A, 0x38, 0xA0, 0x4C, 0x3F,
		0x33, 0xEE
	];
	let addtl1: [u8; 32] = [
		0xB5, 0x18, 0x81, 0x4D, 0xC6, 0xF8,
		0x71, 0x03, 0x63, 0x5A, 0xCA, 0x88,
		0xE2, 0xB6, 0x57, 0x13, 0x22, 0x0D,
		0xE9, 0x28, 0xAD, 0x86, 0x01, 0x6C,
		0xAE, 0xE9, 0x0C, 0x5C, 0x79, 0x41,
		0x55, 0xB5
	];
	let addtl2: [u8; 32] = [
		0xEB, 0x61, 0x23, 0x14, 0xEA, 0x75,
		0xDE, 0xFB, 0xEA, 0x48, 0x46, 0xA2,
		0x2D, 0x5B, 0x3A, 0xDD, 0xE2, 0x30,
		0x44, 0xFD, 0xD0, 0xB4, 0xD5, 0xE9,
		0xEE, 0xC2, 0xF1, 0x1D, 0x58, 0xF9,
		0x11, 0x9E
	];
	let exp: [u8; 256] = [
		0xb6, 0xb5, 0x4e, 0x7d, 0x8c, 0x62, 0xfe, 0x73, 0x64, 0x0d,
		0x57, 0xb4, 0xb5, 0x87, 0x05, 0x12, 0xe7, 0x62, 0xa8, 0x2d,
		0x86, 0xb8, 0x2b, 0xf3, 0x22, 0x66, 0x7e, 0x93, 0x93, 0x45,
		0x88, 0x30, 0xea, 0x2f, 0xbf, 0xba, 0x8f, 0xe7, 0xed, 0x2a,
		0xb2, 0x08, 0x55, 0x37, 0x63, 0x6e, 0xec, 0x1e, 0xe1, 0xdf,
		0x03, 0x60, 0xf0, 0xc0, 0x92, 0x30, 0x21, 0xdd, 0xff, 0x42,
		0xca, 0x5d, 0x7d, 0x67, 0xce, 0x74, 0xc8, 0x6b, 0xad, 0x20,
		0x75, 0xa8, 0xc3, 0xc0, 0x01, 0x98, 0xa1, 0x38, 0x31, 0x2d,
		0xd0, 0x83, 0x75, 0x17, 0x4c, 0x52, 0x5f, 0xed, 0x8a, 0xbc,
		0xa4, 0x0e, 0xd4, 0x4d, 0x0a, 0x32, 0x44, 0x00, 0xbe, 0x5d,
		0x57, 0xb3, 0x11, 0xa0, 0x32, 0x30, 0x49, 0xd8, 0xd3, 0xe9,
		0x35, 0xb6, 0x3b, 0x27, 0x81, 0xb0, 0x82, 0xf8, 0x19, 0x13,
		0xc5, 0xbc, 0x2f, 0xdc, 0x87, 0xcd, 0x92, 0xd2, 0xa7, 0xb2,
		0xe0, 0x8f, 0xcf, 0x79, 0x20, 0x15, 0x5f, 0x40, 0x47, 0xb6,
		0x7c, 0xe5, 0x6c, 0x7a, 0x6c, 0xe6, 0xb9, 0xba, 0x1b, 0x86,
		0x55, 0x1a, 0xc2, 0xc2, 0xf8, 0x3a, 0xd8, 0xd7, 0xa7, 0xe3,
		0x3c, 0x86, 0xa3, 0xe1, 0x88, 0x0f, 0x5c, 0x1e, 0x79, 0xa6,
		0x14, 0x58, 0x9e, 0x22, 0x22, 0xc2, 0x40, 0xf1, 0x93, 0x4b,
		0xb3, 0x3a, 0x25, 0x4e, 0xc2, 0xeb, 0x0d, 0x04, 0xde, 0xc1,
		0xe2, 0x46, 0x85, 0x6e, 0x65, 0x92, 0xa9, 0x24, 0x44, 0x87,
		0xd1, 0x42, 0x4c, 0x54, 0x71, 0xfb, 0xd7, 0x72, 0x84, 0xa5,
		0x64, 0x41, 0x57, 0x74, 0xb6, 0x01, 0xd7, 0x76, 0x4e, 0x66,
		0x86, 0x60, 0x3a, 0xa5, 0x14, 0x55, 0x5d, 0x5a, 0x56, 0xc4,
		0xb2, 0x82, 0xf9, 0xcd, 0x73, 0x7b, 0xb6, 0xe4, 0xac, 0xe5,
		0x46, 0x74, 0x10, 0xeb, 0x9f, 0x0d, 0x22, 0xf0, 0x94, 0xeb,
		0x09, 0x0f, 0x8e, 0x8d, 0x7f, 0x09
	];

	lc_rust_drbg_one(lcr_rng_type::lcr_hmac_drbg, &ent_nonce, &pers,
			 &addtl1, &addtl2, &exp)
}

#[test]
fn lc_rust_rng_hash_drbg() {
	let ent_nonce: [u8; 64] = [
		0x9E, 0x28, 0x52, 0xF1, 0xD8, 0xB2, 0x3C, 0x1A, 0x80, 0xCA,
		0x75, 0x29, 0x37, 0xAC, 0x58, 0x54, 0x61, 0x98, 0xDB, 0x72,
		0x81, 0xB7, 0x43, 0xDB, 0x37, 0x21, 0x8E, 0x86, 0x40, 0x3B,
		0x74, 0xF9, 0x88, 0x45, 0x49, 0xDC, 0x49, 0x26, 0xBB, 0xAA,
		0x83, 0x3E, 0x50, 0x42, 0xA9, 0x52, 0xAE, 0x97, 0xB2, 0x1B,
		0x72, 0x93, 0x7C, 0xC7, 0x29, 0x5C, 0x47, 0x2B, 0x70, 0xFB,
		0xEC, 0xAC, 0xD9, 0x2C
	];
	let pers: [u8; 32] = [
		0x12, 0x6B, 0xE1, 0x49, 0x3F, 0x41,
		0x28, 0x9A, 0xDC, 0x5C, 0x7F, 0x00,
		0x43, 0x40, 0xFF, 0x21, 0xA7, 0xEC,
		0x4D, 0xAD, 0xFF, 0xDA, 0x64, 0x2D,
		0xE4, 0x65, 0xAB, 0x2E, 0x98, 0x54,
		0x19, 0x1A
	];
	let addtl1: [u8; 32] = [
		0x89, 0x18, 0x8A, 0xB5, 0x82, 0x0B,
		0x05, 0x98, 0xF9, 0x81, 0xB3, 0x34,
		0x44, 0x6D, 0xD4, 0x38, 0x29, 0xCD,
		0x50, 0x4E, 0x06, 0xFE, 0x11, 0xF2,
		0x3C, 0x70, 0x0D, 0xAC, 0xA8, 0x28,
		0x0E, 0x40
	];
	let addtl2: [u8; 32] = [
		0x67, 0x87, 0xEE, 0x02, 0xA6, 0x0F,
		0x2F, 0x8D, 0x8D, 0xF3, 0x4A, 0xBF,
		0xA3, 0x61, 0x7E, 0xD6, 0xB2, 0xB1,
		0x37, 0x61, 0xA5, 0x41, 0xB3, 0x8C,
		0x2A, 0xF9, 0x01, 0x08, 0x3F, 0xC9,
		0x0D, 0xCA
	];
	let exp: [u8; 256] = [
		0x4d, 0xa6, 0x34, 0x92, 0x52, 0x48, 0x31, 0x53, 0x5c, 0x2d,
		0xd8, 0xe9, 0xbd, 0x2f, 0x31, 0x9b, 0x11, 0xc2, 0xda, 0x2f,
		0xd7, 0x21, 0x05, 0xed, 0x2c, 0x67, 0x04, 0x37, 0xbd, 0x53,
		0xb3, 0x4e, 0x9d, 0x0c, 0x16, 0x54, 0x89, 0xca, 0xe3, 0x39,
		0xc0, 0x77, 0xb3, 0xb5, 0xfa, 0xae, 0x9c, 0x59, 0x90, 0x43,
		0x09, 0x43, 0xf1, 0x4c, 0x70, 0x3e, 0x00, 0x02, 0xa7, 0xf3,
		0x13, 0x93, 0x98, 0xba, 0x8b, 0xf4, 0xdf, 0x9e, 0x3f, 0x8d,
		0x65, 0x0f, 0x7a, 0x35, 0xd7, 0xa1, 0x4d, 0x13, 0x70, 0x50,
		0x01, 0xd8, 0x54, 0x26, 0x74, 0x2a, 0xdc, 0x35, 0xb6, 0x59,
		0xc2, 0xfb, 0x75, 0xfa, 0x47, 0x7c, 0x06, 0x26, 0xfc, 0xcc,
		0x20, 0xa0, 0x11, 0xc4, 0xc4, 0xe8, 0xe5, 0x79, 0x33, 0x39,
		0x30, 0x64, 0xb3, 0x75, 0x7b, 0x2f, 0x04, 0x52, 0x0a, 0x60,
		0x41, 0x71, 0xcf, 0x3b, 0x1f, 0x30, 0x5b, 0x81, 0x53, 0x2a,
		0x26, 0xde, 0x3a, 0x4c, 0x5a, 0x64, 0xe2, 0x29, 0x3e, 0x38,
		0x8f, 0x8e, 0x1e, 0x76, 0x08, 0xea, 0x81, 0x9e, 0x5d, 0x7b,
		0x3a, 0xad, 0x64, 0xc7, 0x1c, 0x32, 0x51, 0x9d, 0x67, 0xe3,
		0x75, 0x8f, 0x73, 0x23, 0x55, 0xbd, 0x1b, 0x70, 0x9a, 0x8b,
		0x8f, 0x5d, 0xcf, 0xe5, 0xac, 0x6d, 0xc9, 0xf9, 0x48, 0xfc,
		0xeb, 0xd6, 0x3a, 0x37, 0x01, 0x4e, 0x6a, 0xae, 0x7b, 0x83,
		0xf5, 0x13, 0x22, 0x97, 0x2b, 0xc8, 0xd0, 0x9d, 0xd4, 0x91,
		0x18, 0xa1, 0x4b, 0x36, 0xf3, 0x0d, 0x3f, 0x4e, 0x6d, 0x96,
		0x8d, 0x79, 0xd8, 0xd7, 0xf0, 0x31, 0x57, 0xf8, 0x32, 0x93,
		0x10, 0xf6, 0xba, 0xab, 0x57, 0xa6, 0xec, 0xb8, 0xbc, 0x9b,
		0x0b, 0xef, 0xa5, 0x00, 0x78, 0x7f, 0x63, 0x3e, 0x0f, 0x45,
		0x3b, 0x6d, 0xd9, 0xea, 0x58, 0xee, 0x29, 0x48, 0xad, 0x33,
		0xcb, 0x1b, 0xbf, 0xd1, 0x1d, 0x2a
	];

	lc_rust_drbg_one(lcr_rng_type::lcr_hash_drbg, &ent_nonce, &pers,
			 &addtl1, &addtl2, &exp)
}
