/*
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * Copyright 2012-2024 the original author or authors.
 */
package org.assertj.core.internal.arrays2d;

import static org.assertj.core.api.Assertions.assertThatNullPointerException;
import static org.assertj.core.api.BDDAssertions.then;
import static org.assertj.core.data.Index.atIndex;
import static org.assertj.core.error.ShouldNotBeNull.shouldNotBeNull;
import static org.assertj.core.error.ShouldNotContainAtIndex.shouldNotContainAtIndex;
import static org.assertj.core.test.TestData.someIndex;
import static org.assertj.core.test.TestData.someInfo;
import static org.assertj.core.util.AssertionsUtil.expectAssertionError;

import org.assertj.core.api.AssertionInfo;
import org.assertj.core.api.ThrowableAssert.ThrowingCallable;
import org.assertj.core.data.Index;
import org.assertj.core.internal.Arrays2D;
import org.assertj.core.internal.Failures;
import org.junit.jupiter.api.Test;

/**
 * Tests for <code>{@link Arrays2D#assertDoesNotContain(AssertionInfo, Failures, Object, Object, Index)}</code>.
 *
 * @author Maciej Wajcht
 */
class Arrays2D_assertDoesNotContain_at_Index_Test extends Arrays2D_BaseTest {

  private int[][] actual = new int[][] { { 0, 2, 4 }, { 6, 8, 10 } };

  @Test
  void should_fail_if_actual_is_null() {
    // GIVEN
    int[][] actual = null;
    int[] expectedElement = { 0, 2, 4 };
    // WHEN
    ThrowingCallable assertion = () -> arrays.assertDoesNotContain(someInfo(), failures, actual, expectedElement, someIndex());
    AssertionError assertionError = expectAssertionError(assertion);
    // THEN
    then(assertionError).hasMessage(shouldNotBeNull().create());
  }

  @Test
  void should_pass_if_actual_does_not_contain_value_at_Index() {
    arrays.assertDoesNotContain(someInfo(), failures, actual, new int[] { 0, 2, 4 }, atIndex(1));
  }

  @Test
  void should_pass_if_actual_is_empty() {
    arrays.assertDoesNotContain(someInfo(), failures, new int[][] {}, new int[] { 0, 2, 4 }, someIndex());
  }

  @Test
  void should_throw_error_if_Index_is_null() {
    // GIVEN
    Index nullIndex = null;
    // WHEN
    ThrowingCallable assertion = () -> arrays.assertDoesNotContain(someInfo(), failures, actual, new int[] { 0, 2, 4 },
                                                                   nullIndex);
    // THEN
    assertThatNullPointerException().isThrownBy(assertion)
                                    .withMessage("Index should not be null");
  }

  @Test
  void should_pass_if_Index_is_out_of_bounds() {
    arrays.assertDoesNotContain(someInfo(), failures, actual, new int[] { 0, 2, 4 }, atIndex(6));
  }

  @Test
  void should_fail_if_actual_contains_value_at_index() {
    // GIVEN
    Index index = atIndex(0);
    int[] expectedElement = { 0, 2, 4 };
    // WHEN
    ThrowingCallable assertion = () -> arrays.assertDoesNotContain(someInfo(), failures, actual, expectedElement, index);
    AssertionError assertionError = expectAssertionError(assertion);
    // THEN
    then(assertionError).hasMessage(shouldNotContainAtIndex(actual, new int[] { 0, 2, 4 }, index).create());
  }
}
