import { map } from 'rxjs/operators';
import { getFieldDisplayName } from '../../field/fieldState.js';
import { getValueMatcher } from '../matchers.js';
import { ValueMatcherID } from '../matchers/ids.js';
import { DataTransformerID } from './ids.js';
import { noopTransformer } from './noop.js';
import { transformationsVariableSupport } from './utils.js';

const filterByValueTransformer = {
  id: DataTransformerID.filterByValue,
  name: "Filter data by values",
  description: "select a subset of results based on values",
  defaultOptions: {
    filters: [],
    type: "include" /* include */,
    match: "any" /* any */
  },
  operator: (options, ctx) => (source) => {
    const filters = options.filters;
    const matchAll = options.match === "all" /* all */;
    const include = options.type === "include" /* include */;
    if (!Array.isArray(filters) || filters.length === 0) {
      return source.pipe(noopTransformer.operator({}, ctx));
    }
    const interpolatedFilters = [];
    if (transformationsVariableSupport()) {
      interpolatedFilters.push(
        ...filters.map((filter) => {
          if (filter.config.id === ValueMatcherID.between) {
            let valueFrom = filter.config.options.from;
            let valueTo = filter.config.options.to;
            if (typeof filter.config.options.from === "string") {
              valueFrom = ctx.interpolate(valueFrom);
            }
            if (typeof filter.config.options.to === "string") {
              valueTo = ctx.interpolate(valueTo);
            }
            return {
              ...filter,
              config: {
                ...filter.config,
                options: {
                  ...filter.config.options,
                  to: valueTo,
                  from: valueFrom
                }
              }
            };
          } else if (filter.config.options.value) {
            let value = filter.config.options.value;
            if (typeof filter.config.options.value === "string") {
              value = ctx.interpolate(value);
            }
            return {
              ...filter,
              config: { ...filter.config, options: { ...filter.config.options, value } }
            };
          }
          return filter;
        })
      );
    }
    return source.pipe(
      map((data) => {
        if (data.length === 0) {
          return data;
        }
        const processed = [];
        for (const frame of data) {
          const rows = /* @__PURE__ */ new Set();
          const fieldIndexByName = groupFieldIndexByName(frame, data);
          let matchers;
          if (transformationsVariableSupport()) {
            matchers = createFilterValueMatchers(interpolatedFilters, fieldIndexByName);
          } else {
            matchers = createFilterValueMatchers(filters, fieldIndexByName);
          }
          for (let index = 0; index < frame.length; index++) {
            if (rows.has(index)) {
              continue;
            }
            let matching = true;
            for (const matcher of matchers) {
              const match = matcher(index, frame, data);
              if (!matchAll && match) {
                matching = true;
                break;
              }
              if (matchAll && !match) {
                matching = false;
                break;
              }
              matching = match;
            }
            if (matching) {
              rows.add(index);
            }
          }
          const fields = [];
          const frameLength = include ? rows.size : data[0].length - rows.size;
          for (const field of frame.fields) {
            const buffer = [];
            for (let index = 0; index < frame.length; index++) {
              if (include && rows.has(index)) {
                buffer.push(field.values[index]);
                continue;
              }
              if (!include && !rows.has(index)) {
                buffer.push(field.values[index]);
                continue;
              }
            }
            fields.push({
              ...field,
              values: buffer,
              state: {}
            });
          }
          processed.push({
            ...frame,
            fields,
            length: frameLength
          });
        }
        return processed;
      })
    );
  }
};
const createFilterValueMatchers = (filters, fieldIndexByName) => {
  const noop = () => false;
  return filters.map((filter) => {
    var _a;
    const fieldIndex = (_a = fieldIndexByName[filter.fieldName]) != null ? _a : -1;
    if (fieldIndex < 0) {
      console.warn(`[FilterByValue] Could not find index for field name: ${filter.fieldName}`);
      return noop;
    }
    const matcher = getValueMatcher(filter.config);
    return (index, frame, data) => matcher(index, frame.fields[fieldIndex], frame, data);
  });
};
const groupFieldIndexByName = (frame, data) => {
  const lookup = {};
  frame.fields.forEach((field, fieldIndex) => {
    const fieldName = getFieldDisplayName(field, frame, data);
    lookup[fieldName] = fieldIndex;
  });
  return lookup;
};

export { filterByValueTransformer };
//# sourceMappingURL=filterByValue.js.map
