# Copyright (c) 2023 SUSE Linux LLC.  All rights reserved.
#
# This file is part of suse-migration-services.
#
# suse-migration-services is free software: you can redistribute it and/or
# modify it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# suse-migration-services is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with suse-migration-services. If not, see <http://www.gnu.org/licenses/>
#
"""systemd service to install the shim package and update the bootloader"""
import logging

# project
from suse_migration_services.command import Command
from suse_migration_services.defaults import Defaults
from suse_migration_services.logger import Logger
from suse_migration_services.zypper import Zypper


def main():
    """
    DistMigration install the shim pacakge, enable "Secure boot"
    and update the bootloader
    """
    Logger.setup()
    log = logging.getLogger(Defaults.get_migration_log_name())
    log.info('Running update bootloader service')

    root_path = Defaults.get_system_root_path()

    log.info('Installing the shim package')
    install_shim_package(root_path)
    log.info('Updating the shimbootloader')
    install_secure_bootloader(root_path)
    log.info('Updating the bootloader')
    update_bootloader_config(root_path)


def install_shim_package(root_path):
    """
    Install the shim package
    """
    Zypper.run(
        [
            '--no-cd',
            '--non-interactive',
            '--gpg-auto-import-keys',
            '--root', root_path,
            'install',
            '--auto-agree-with-licenses',
            '--allow-vendor-change',
            '--download', 'in-advance',
            '--replacefiles',
            '--allow-downgrade',
            'shim',
        ]
    )


def install_secure_bootloader(root_path):
    """
    Perform shim-install from inside the upgraded system.
    If the system is not suitable to be setup for shim e.g.
    not EFI based, we don't consider it as a fatal error
    and continue while keeping the log information about the
    attempt.
    """
    Command.run(
        ['chroot', root_path, 'shim-install', '--removable'],
        raise_on_error=False
    )


def update_bootloader_config(root_path):
    """
    Perform update-bootloader from inside of the upgraded system
    """
    Command.run(
        ['chroot', root_path, '/sbin/update-bootloader', '--reinit']
    )
