#define _GNU_SOURCE
#include <stdio.h>
#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <termios.h>
#include <errno.h>
#include <pthread.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <X11/extensions/XTest.h>
#include <ncurses.h>
#include <term.h>

typedef unsigned char uchar;

static FILE* logout;

static KeySym modifiers[] = {
    XK_Shift_L,
    XK_Alt_L,
    XK_Meta_L,
    XK_Control_L,
    0
};

static KeySym keysyms[] = {
    XK_F1,
    XK_F2,
    XK_F3,
    XK_F4,
    XK_F5,
    XK_F6,
    XK_F7,
    XK_F8,
    XK_F9,
    XK_F10,
    XK_F11,
    XK_F12,
#if 0
    XK_BackSpace,
    XK_Print,
    XK_Scroll_Lock,
#endif
    XK_Pause,

    XK_Begin,
    XK_Insert,
    XK_Home,
    XK_Prior,
    XK_Page_Up,

    XK_Delete,
    XK_End,
    XK_Next,
    XK_Page_Down,

    XK_Left,
    XK_Up,
    XK_Right,
    XK_Down,

    XK_Select,
    XK_Execute,

    XK_KP_Space,
    XK_KP_Tab,
    XK_KP_Enter,
    XK_KP_F1,
    XK_KP_F2,
    XK_KP_F3,
    XK_KP_F4,
    XK_KP_Home,
    XK_KP_Left,
    XK_KP_Up,
    XK_KP_Right,
    XK_KP_Down,
    XK_KP_Prior,
    XK_KP_Page_Up,
    XK_KP_Next,
    XK_KP_Page_Down,
    XK_KP_End,
    XK_KP_Begin,
    XK_KP_Insert,
    XK_KP_Delete,
    XK_KP_Equal,
    XK_KP_Multiply,
    XK_KP_Add,
    XK_KP_Separator,
    XK_KP_Subtract,
    XK_KP_Decimal,
    XK_KP_Divide,

    XK_KP_0,
    XK_KP_1,
    XK_KP_2,
    XK_KP_3,
    XK_KP_4,
    XK_KP_5,
    XK_KP_6,
    XK_KP_7,
    XK_KP_8,
    XK_KP_9,
    0
};

/*
 * Determine the possible modifier combinations
 */
#define NUM (sizeof(modifiers)/sizeof(KeySym) - 1)
static uint mask[NUM];
static uint modifier[NUM];
static uint MShift, MAlt, MMeta, MControl;
static int num_mask_bits;
static int setup_modifiers(Display *dpy, Window root)
{
    XSetWindowAttributes xwa;
    Window window;
    int ret, i;
    XEvent ev;

    memset(&xwa, 0, sizeof(xwa));
    xwa.override_redirect = True;

    if ((window = XCreateWindow(dpy, root, 0,0,1,1,0,0,
				InputOnly,CopyFromParent,
				CWOverrideRedirect, &xwa)) == (Window)0) {
	fprintf(stderr, "Can't create Window\n");
	goto out;
    }
    if (XSelectInput(dpy, window, StructureNotifyMask) == 0) {
	fprintf(stderr, "Can't SelectInput\n");
	goto err;
    }
    if (XMapRaised(dpy, window) == 0) {
	fprintf(stderr, "Can't Map Window\n");
	goto err;
    }
    do {
	XSync(dpy, False);
    } while (XCheckTypedWindowEvent(dpy, window, MapNotify, &ev) == False);

    if (XSelectInput(dpy, window, KeyPressMask|KeyReleaseMask) == 0) {
        fprintf(stderr, "Can't SelectInput\n");
        goto map;
    }
    XFlush(dpy);

    if (XSetInputFocus(dpy, window, PointerRoot, CurrentTime) == 0) {
	fprintf(stderr, "Can't set InputFocus\n");
	goto map;
    }
    XFlush(dpy);

    for (i = 0; modifiers[i] != 0; i++) {
	KeySym keysym = modifiers[i];
	KeyCode keycode = XKeysymToKeycode(dpy, keysym);
	int k;

	XTestFakeKeyEvent(dpy, keycode, True, 0);
	do {
	    XSync(dpy, False);
	} while (XCheckTypedWindowEvent(dpy, window, KeyPress, &ev) == False);

	XTestFakeKeyEvent(dpy, keycode, False, 0);
	do {
	    XSync(dpy, False);
	} while (XCheckTypedWindowEvent(dpy, window, KeyRelease, &ev) == False);

	printf("keycode: %#.2x keysym: %s release event state: %#.2x\n",
		ev.xkey.keycode,
		XKeysymToString(keysym),
		ev.xkey.state);
	for (k = 0; k < num_mask_bits; k++) {
	    if (mask[k] == ev.xkey.state)
		break;
	}
	if (k == num_mask_bits) {
	    modifier[num_mask_bits] = keysym;
 	    mask[num_mask_bits++] = ev.xkey.state;
	    switch (keysym) {
	    case XK_Shift_L:
		MShift = (1<<k);
		printf("Mask Shift_L: %#.2x\n", MShift);
		break;
	    case XK_Alt_L:
		MAlt = (1<<k);
		printf("Mask Alt_L: %#.2x\n", MAlt);
		break;
	    case XK_Meta_L:
		MMeta = (1<<k);
		printf("Mask Meta_L: %#.2x\n", MMeta);
		break;
	    case XK_Control_L:
		MControl = (1<<k);
		printf("Mask Control_L: %#.2x\n", MControl);
	    default:
		break;
	    }
	}
    }
    ret = 1;
map:
    XUnmapWindow(dpy, window);
err:
    XDestroyWindow(dpy, window);
out:
    return ret;
}

/*
 * Determine the Window ID of a given Window name
 */
static Window windowbyname(const Display *dpy, const Window top, const char *who)
{
    Window *child, dummy, w = (Window)0;
    unsigned int i, nchild;
    char *wname;

    if (XFetchName((Display *)dpy, top, &wname) && !strcmp(wname, who))
	return(top);

    if (XQueryTree((Display *)dpy, top, &dummy, &dummy, &child, &nchild) == 0)
	return (Window)0;

    for (i = 0; i < nchild; i++) {
	w = windowbyname(dpy, child[i], who);
	if (w)
	    break;
    }
    if (child) XFree ((char *)child);
    return w;
}

/*
 * Mutex used for pthread handling
 */
typedef struct _mutex {
    int locked;
    pthread_mutex_t mutex;
    pthread_t thread;
} mutex_t;

static mutex_t ljoin = {0, PTHREAD_MUTEX_INITIALIZER, 0};
static mutex_t llock = {0, PTHREAD_MUTEX_INITIALIZER, 0};
static pthread_cond_t lcond = PTHREAD_COND_INITIALIZER;
static volatile int running;
static volatile char *Symbol;
static volatile uint ModMask;

/*
 * Locking of a Mutex if not already done
 */
static inline void lock(mutex_t *mutex)
{
    if (mutex->thread != pthread_self() || !mutex->locked) {
	pthread_mutex_lock(&mutex->mutex);
	mutex->thread = pthread_self();
    }
    mutex->locked++;
}

/*
 * Unlocking of a Mutex if not already done
 */
static inline void unlock(mutex_t *mutex)
{
    if (!--mutex->locked) {
	mutex->thread = 0;
	pthread_mutex_unlock(&mutex->mutex);
    }
}

/*
 * Poll on Mutex, uses a condition for wakeup
 */
static inline int ppoll(int msec, mutex_t *outer)
{
    struct timespec abstime;
    struct timeval now;
    int err, ret = 0;

    if (gettimeofday(&now, NULL) < 0) {
	pthread_yield();
	goto out;
    }

    now.tv_usec += msec * 1000;
    while (now.tv_usec >= 1000000) {
	now.tv_sec++;
	now.tv_usec -= 1000000;
    }
    abstime.tv_sec  = now.tv_sec;
    abstime.tv_nsec = now.tv_usec * 1000;

    do {
	int locked = outer->locked;
	/* Note: pthread_cond_timedwait() atomically unlocks the mutex */
	outer->locked = 0;
	err = pthread_cond_timedwait(&lcond, &outer->mutex, &abstime);
	outer->locked = locked;
    } while (err == EINTR);

    ret = 1;
    if (err == ETIMEDOUT || err == EBUSY)
	ret = 0;
out:
    return ret;
}

/*
 * Read out the response of the terminal.
 */
static void *action(void *dummy)
{
    int in = fileno(stdin);
    struct termios tty;

    lock(&ljoin);
    cfmakeraw(&tty);
    cfsetspeed(&tty, B38400);

    tty.c_lflag &= ~ICANON;
    tty.c_lflag |= ISIG;
    tty.c_iflag |= ICRNL;
    tty.c_iflag &= ~INLCR;
    tty.c_iflag &= ~IUCLC;
    tty.c_oflag |= OPOST;
    tty.c_oflag |= ONLCR;
    tty.c_oflag &= ~OCRNL;
    tty.c_oflag &= ~ONOCR;
    tty.c_oflag &= ~ONLRET;
    tty.c_oflag &= ~OLCUC;
    tty.c_oflag &= ~NLDLY;
    tty.c_oflag |= NL0;
    tty.c_cc[VMIN] = 1;
    tty.c_cc[VTIME] = 0;

    if (tcsetattr(in, TCSADRAIN, &tty) < 0) {
	fprintf(stderr, "tcsetattr(%d) failed %s\n", in, strerror(errno));
	goto out;
    }
    if (tcflush(in, TCIFLUSH) < 0) {
	fprintf(stderr, "tcflush(%d, TCIFLUSH) failed %s\n", in, strerror(errno));
	goto out;
    }
    while (running) {
	char buf[256], *ptr = &buf[0];
	ssize_t r = 0;
	int t;

	ppoll(100, &ljoin);

	lock(&llock);
	if (Symbol == (char*)0) {
	    unlock(&llock);
	    pthread_yield();
	    continue;
	}

	if (tcflow(in, TCION) < 0) {	/* enable reads */
	    fprintf(stderr, "tcflow(%d, TCION) failed %s\n", in, strerror(errno));
	}

	do {				/* wait on input, if any */
	    struct timeval zero = {0, 500000};
	    fd_set ins;

	    FD_ZERO(&ins);
	    FD_SET (in, &ins);

	    t = select(in+1, &ins, (fd_set*)0, (fd_set*)0, &zero);

	} while ((t < 0) && (errno == EINTR));

	if (t > 0) {			/* we have input */
	    ssize_t s;

	    while ((s = sizeof(buf) - r) > 0) {
					/* .. but how much do we? */
		if (ioctl(in, TIOCINQ, &t) < 0) {
		    if (errno == EINTR)
			continue;
		    break;
		} else if (t == 0)
		    break;		/* nothing to read, out here */

		if (t > s) t = s;

		while (t > 0) {
		    ssize_t p;

		    if ((p = read(in, ptr, t)) < 0) {
			if (errno == EINTR) {
			    errno = 0;
			    continue;
			}
			break;
		    }

		    ptr += p;
		    r += p;
		    t -= p;

	    	    pthread_yield();
		}
		usleep(10000);		/* KDE konsole requires ... */
	    }				/* ... this: check for more */
	}

	if (tcflow(in, TCIOFF) < 0) {	/* disable reads */
	    fprintf(stderr, "tcflow(%d, TCIOFF) failed %s\n", in, strerror(errno));
	}

	if (r > 0) {
	    if (ModMask) {
		uint mask = ModMask;
		for (t = 0; t < num_mask_bits; t++) {
		    if ((1 << t) & mask) {
			char *keysym = XKeysymToString(modifier[t]);
			if (keysym) fprintf(logout , "%s + ", keysym);
		    }
		}
	    }
	    fprintf(logout , "%s: ", Symbol);
	    for (t = 0; t < r; t++) {
		int c = buf[t];
		switch (c) {
		case 0 ... 26:
		    fprintf(logout, "^%c", (char)c+64);
		    break;
		case 27:
		    fprintf(logout, "\\e");
		    break;
		case 28 ... 31:
		    break;
		case 127:
		    fprintf(logout, "^?");
		    break;
		case 128 ... 160:
		    fprintf(logout, "%#.2x", c);
		    break;
		default:
		    fputc(c, logout);
		    break;
		}
	    } 
	    fputc('\n', logout);
	    fflush(logout);
	}
	Symbol = (char*)0;
	ModMask = (uint)0;
	unlock(&llock);
    }
    unlock(&ljoin);
out:
    return NULL;
}

static int usekp;
static Display *dpy;
static void allcose(void)
{
    if (dpy) {
	XAutoRepeatOn(dpy);
	XFlush(dpy);
	XUngrabKeyboard(dpy, CurrentTime);
	XFlush(dpy);
	XCloseDisplay(dpy);
	dpy = (Display*)0;
    }
    if (usekp) {
	const char* rmkx = tigetstr("rmkx");
	if (rmkx) {
	    fwrite(rmkx, sizeof(char), strlen(rmkx), stdout);
	    fflush(stdout);
	}
    }
}

static int ml, kde, gnome;
static struct option options[] = {
    {"ml",	no_argument,	   &ml,	    1},
    {"kde",	no_argument,	   &kde,    1},
    {"gnome",	no_argument,	   &gnome,  1},
    {"kp",	no_argument,	   &usekp,  1},
    {"id",	required_argument, (int*)0, 2},
    {0,		no_argument,	   (int*)0, 0}
};

int main(int argc, char *argv[])
{
    Window root, term = (Window)0;
    XSetWindowAttributes xwa;
    pthread_attr_t attr;
    pthread_t lthread;
    int m, screen;

    atexit(allcose);

    if (setupterm((char*)0, fileno(stdout), (int*)0) != OK) {
	fprintf(stderr, "Can't use setupterm()\n");
	goto out;
    }

    opterr = 0;
    while ((m = getopt_long_only(argc, argv, "", options, (int*)0)) != -1) {
	switch (m) {
	case 0:
	    break;
	case 2:
	    if (optarg && *optarg) {
		if (*optarg == '0') {
		    if (*(optarg + 1) == 'x')
			term = (Window)strtol(optarg, (char**)0, 16);
		    else
			term = (Window)strtol(optarg, (char**)0, 8);
		} else
		    term = (Window)atol(optarg);
		break;
	    }
	default:
	    fprintf(stderr,
		"%s: Usage:\n"
		"    %s [-kp] [-ml|-kde|-gnome] [-id WinID]\n"
		"with the options:\n"
		"    -kp         Switch terminal into keypad mode if any\n"
		"    -ml         Skip mlterm switches Ctrl+F1 upto Ctrl+F4\n"
		"    -gnome      Skip konsole switches Ctrl+F1, Ctrl+F11\n"
		"    -kde        Skip konsole switches Ctrl+(Shift+)F1/F11\n"
		"    -id WinID   Specify the window ID of e.g. konsole\n",
		basename(argv[0]), argv[0]);
	    goto out;
	}
    }

    argc -= optind;
    argv += optind;

    if (argc < 1 || argc > 3)
	goto out;
    if ((dpy = XOpenDisplay(NULL)) == (Display*)0)
	goto out;
    if ((screen = DefaultScreen(dpy)) < 0)
	goto out;
    if ((root  = RootWindow(dpy, screen)) == (Window)0)
	goto out;
    if (setup_modifiers(dpy, root) == 0)
	goto out;
    if (term == (Window)0) {
	if (*argv == (char*)0)
	    goto out;
	if ((term = windowbyname(dpy, root, *argv)) == (Window)0)
	    goto out;
	argc--;
	argv++;
    }
    printf("WindowID = %#lx\n", (unsigned long)term);

    if (*argv)
	logout = fopen(*argv, "a");
    else
	logout = stdout;

    xwa.override_redirect = True;
    if (XChangeWindowAttributes(dpy, term, CWOverrideRedirect, &xwa) == 0) {
	fprintf(stderr,"Can't change window attributes\n");
	goto out;
    }
    XFlush(dpy);
#if 0
    if (XGrabKeyboard(dpy, term, True, GrabModeAsync, GrabModeAsync, CurrentTime) != GrabSuccess) {
	fprintf(stderr,"Can't grab keyboard\n");
	goto out;
    }
#endif
    if (XSetInputFocus(dpy, term, PointerRoot, CurrentTime) == 0) {
	fprintf(stderr,"Can't set InputFocus\n");
	goto out;
    }
    XFlush(dpy);
    XFlush(dpy);
    if (XAutoRepeatOff(dpy) < 0) {
	fprintf(stderr,"Can't disable auto repeat\n");
	goto out;
    }
    XFlush(dpy);

    lock(&ljoin);
    running = 1;
    unlock(&ljoin);

    lock(&llock);
    Symbol = (char*)0;
    unlock(&llock);

    if (pthread_attr_init(&attr) < 0) {
	fprintf(stderr,"Can't initialize thread attributes\n");
	goto out;
    }
    if (pthread_attr_setschedpolicy(&attr, SCHED_RR) < 0) {
	fprintf(stderr,"Can't set scheduling parameters\n");
	pthread_attr_destroy(&attr);
	goto out;
    }
    if (pthread_create(&lthread, &attr, &action, NULL) < 0) {
	fprintf(stderr,"Can't create reading thread\n");
	pthread_attr_destroy(&attr);
	goto out;
    }

    lock(&ljoin);
    if (usekp) {
	const char* smkx = tigetstr("smkx");
	if (smkx) {
	    fwrite(smkx, sizeof(char), strlen(smkx), stdout);
	    fflush(stdout);
	} else
	    usekp = 0;
    }
    unlock(&ljoin);

    for (m = 0; m < 1<<(num_mask_bits); m++) {
	uint mask = 0;
	int n;

	for (n = 0; n < num_mask_bits; n++) {
	    if ((1 << n) & m)
		mask |= (1 << n);
	}

	for (n = 0; keysyms[n] > 0; n++) {
	    KeyCode keycode;
	    KeySym keysym;
	    int l, timeout;

	    if (!running) {
		fprintf(stderr, "Reading thread broken away\n");
		break;
	    }

	    keysym = keysyms[n];
	    keycode = XKeysymToKeycode(dpy, keysym);
	    XFlush(dpy);

	    if (keycode <= 0)
		continue;

	    if ((mask & MControl) && ((mask & MAlt) || (mask & MMeta))) {
		if (keysym == XK_BackSpace)
		    continue;
		if ((keysym >= XK_F1) && (keysym <= XK_F12))
		    continue;
		if ((keysym >= XK_KP_F1) && (keysym <= XK_KP_F4))
		    continue;
		if ((keysym == XK_KP_Add) || (keysym == XK_KP_Subtract))
		    continue;
		if ((keysym == XK_KP_Multiply) || (keysym == XK_KP_Divide))
		    continue;
	    }

	    if (mask & MShift) {
		if (keysym == XK_KP_Insert)
		    continue;
		if (keysym == XK_Insert)
		    continue;
	    }

	    if (ml && ((mask & 0x0f) == MControl)) {
		if ((keysym >= XK_F1) && (keysym <= XK_F4))
		    continue;
	    }

	    if (gnome) {
		if (((mask & 0xf) == 0) && (keysym == XK_F1))
		    continue;
		if (((mask & 0xf) == 0) && (keysym == XK_F10))
		    continue;
		if (((mask & 0xf) == MShift) && (keysym == XK_F10))
		    continue;
	    }

	    if (kde) {
		if (((mask & 0xf) == 0) && (keysym == XK_F1))
		    continue;
		if (((mask & 0xf) == MShift) && (keysym == XK_F1))
		    continue;
		if (((mask & 0xf) == (MControl|MShift)) && (keysym == XK_F11))
		    continue;
	    }

	    lock(&llock);
	    Symbol = XKeysymToString(keysym);
	    ModMask = mask;
	    timeout = 0;
	    XSetInputFocus(dpy, term, PointerRoot, CurrentTime);
	    XFlush(dpy);
	    for (l = 0; l < num_mask_bits; l++) {
		if ((1 << l) & m) {
		    KeyCode modcode = XKeysymToKeycode(dpy, modifier[l]);
		    XTestFakeKeyEvent(dpy, modcode, True, 0);
		    timeout = 1;
		}
	    }
	    if (timeout) XFlush(dpy);
	    XTestFakeKeyEvent(dpy, keycode, True, 0);
	    XTestFakeKeyEvent(dpy, keycode, False, 0);
	    XFlush(dpy);
	    timeout = 0;
	    for (l = num_mask_bits - 1; l >= 0; l--) {
		if ((1 << l) & m) {
		    KeyCode modcode = XKeysymToKeycode(dpy, modifier[l]);
		    XTestFakeKeyEvent(dpy, modcode, False, 0);
		    timeout = 1;
		}
	    }
	    if (timeout) XFlush(dpy);
	    unlock(&llock);

	    timeout = 50;
	    while (Symbol && (timeout-- > 0)) {
		if (!running) {
		    fprintf(stderr, "Reading thread broken away\n");
		    break;
		}
		usleep(10000);
		pthread_cond_broadcast(&lcond);
	    }
	}
    }

    lock(&ljoin);
    running = 0;
    unlock(&ljoin);
    pthread_join(lthread, (void**)0);
    pthread_attr_destroy(&attr);
out:
    if (logout == stdout)
	sleep(5);
    allcose();
    return 0;
}
