// Copyright 2010-2021, Google Inc.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
//     * Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above
// copyright notice, this list of conditions and the following disclaimer
// in the documentation and/or other materials provided with the
// distribution.
//     * Neither the name of Google Inc. nor the names of its
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

#ifndef MOZC_DICTIONARY_SINGLE_KANJI_DICTIONARY_H_
#define MOZC_DICTIONARY_SINGLE_KANJI_DICTIONARY_H_

#include <memory>
#include <string>
#include <utility>
#include <vector>

#include "absl/strings/string_view.h"
#include "base/container/serialized_string_array.h"
#include "data_manager/data_manager.h"
#include "data_manager/serialized_dictionary.h"

namespace mozc {
namespace dictionary {

class SingleKanjiDictionary {
 public:
  explicit SingleKanjiDictionary(const DataManager &data_manager);

  SingleKanjiDictionary(const SingleKanjiDictionary &) = delete;
  SingleKanjiDictionary &operator=(const SingleKanjiDictionary &) = delete;

  ~SingleKanjiDictionary() = default;

  // Looks up single kanji list from key (reading).  Returns false if not found.
  bool LookupKanjiEntries(absl::string_view key, bool use_svs,
                          std::vector<std::string> *kanji_list) const;

  // Returns the iterator range for noun prefix kanji entries
  // whose keys match the given key.
  // Noun prefix kanji entries are generated by
  // mozc/rewriter/gen_single_kanji_noun_prefix_data.cc
  std::pair<SerializedDictionary::const_iterator,
            SerializedDictionary::const_iterator>
  LookupNounPrefixEntries(absl::string_view key) const {
    return noun_prefix_dictionary_->equal_range(key);
  }

  // Generates kanji variant description.
  // Returns false if the description was not found.
  bool GenerateDescription(absl::string_view kanji_surface,
                           std::string *desc) const;

 private:
  absl::string_view single_kanji_token_array_;
  SerializedStringArray single_kanji_string_array_;
  SerializedStringArray variant_type_array_;
  absl::string_view variant_token_array_;
  SerializedStringArray variant_string_array_;

  // Since noun_prefix_dictionary_ is just a tentative workaround,
  // we copy the SingleKanji structure so that we can remove this workaround
  // easily. Also, the logic of NounPrefix insertion is put independently from
  // the single kanji dictionary. Ideally, we want to regenerate our
  // language model for fixing noun-prefix issue.
  std::unique_ptr<SerializedDictionary> noun_prefix_dictionary_;
};

}  // namespace dictionary
}  // namespace mozc

#endif  // MOZC_DICTIONARY_SINGLE_KANJI_DICTIONARY_H_
