/*
 * Copyright (C) 2024 - 2025, Stephan Mueller <smueller@chronox.de>
 *
 * License: see LICENSE file in root directory
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ALL OF
 * WHICH ARE HEREBY DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF NOT ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

#ifndef SPHINCS_TESTER_VECTORS_SHAKE_128S_H
#define SPHINCS_TESTER_VECTORS_SHAKE_128S_H

#ifdef __cplusplus
extern "C" {
#endif

struct lc_sphincs_test {
	uint8_t seed[LC_SPX_SEEDBYTES];
	uint8_t msg[33];
	uint8_t pk[sizeof(struct lc_sphincs_pk)];
	uint8_t sk[sizeof(struct lc_sphincs_sk)];
	uint8_t sig[sizeof(struct lc_sphincs_sig)];
};

struct lc_sphincs_test tests[] =
{
	{
		.seed = {
			0x7c, 0x99, 0x35, 0xa0, 0xb0, 0x76, 0x94, 0xaa,
			0x0c, 0x6d, 0x10, 0xe4, 0xdb, 0x6b, 0x1a, 0xdd,
			0x2f, 0xd8, 0x1a, 0x25, 0xcc, 0xb1, 0x48, 0x03,
			0x2d, 0xcd, 0x73, 0x99, 0x36, 0x73, 0x7f, 0x2d,
			0xb5, 0x05, 0xd7, 0xcf, 0xad, 0x1b, 0x49, 0x74,
			0x99, 0x32, 0x3c, 0x86, 0x86, 0x32, 0x5e, 0x47,
		},
		.msg = {
			0xD8, 0x1C, 0x4D, 0x8D, 0x73, 0x4F, 0xCB, 0xFB,
			0xEA, 0xDE, 0x3D, 0x3F, 0x8A, 0x03, 0x9F, 0xAA,
			0x2A, 0x2C, 0x99, 0x57, 0xE8, 0x35, 0xAD, 0x55,
			0xB2, 0x2E, 0x75, 0xBF, 0x57, 0xBB, 0x55, 0x6A,
			0xC8
		},
		.pk = {
			0xb5, 0x05, 0xd7, 0xcf, 0xad, 0x1b, 0x49, 0x74,
			0x99, 0x32, 0x3c, 0x86, 0x86, 0x32, 0x5e, 0x47,
			0x8b, 0x8d, 0x23, 0xa7, 0xdf, 0xf7, 0x86, 0x3c,
			0x25, 0xf0, 0x3a, 0x97, 0xcf, 0x76, 0xd4, 0xe0,
		},
		.sk = {
			0x7c, 0x99, 0x35, 0xa0, 0xb0, 0x76, 0x94, 0xaa,
			0x0c, 0x6d, 0x10, 0xe4, 0xdb, 0x6b, 0x1a, 0xdd,
			0x2f, 0xd8, 0x1a, 0x25, 0xcc, 0xb1, 0x48, 0x03,
			0x2d, 0xcd, 0x73, 0x99, 0x36, 0x73, 0x7f, 0x2d,
			0xb5, 0x05, 0xd7, 0xcf, 0xad, 0x1b, 0x49, 0x74,
			0x99, 0x32, 0x3c, 0x86, 0x86, 0x32, 0x5e, 0x47,
			0x8b, 0x8d, 0x23, 0xa7, 0xdf, 0xf7, 0x86, 0x3c,
			0x25, 0xf0, 0x3a, 0x97, 0xcf, 0x76, 0xd4, 0xe0
		},
		.sig = {
			0xf2, 0x0a, 0x44, 0xcd, 0x38, 0xfb, 0xb1, 0xc3,
			0xa7, 0xd2, 0xef, 0xe2, 0xb0, 0xba, 0x10, 0x8f,
			0xcb, 0x7e, 0x74, 0xab, 0x36, 0xbb, 0xa2, 0x53,
			0x29, 0xa0, 0xc2, 0x84, 0x13, 0x50, 0xae, 0xd6,
			0x93, 0x47, 0x89, 0xe6, 0xf4, 0x04, 0x23, 0x0c,
			0x39, 0x9b, 0x44, 0xec, 0x0a, 0xd8, 0xe7, 0xd4,
			0x84, 0xdd, 0x5c, 0x5f, 0x6b, 0x42, 0x14, 0xc6,
			0x0b, 0x56, 0x94, 0xae, 0x93, 0xbc, 0x45, 0x24,
			0x23, 0xc2, 0x71, 0x56, 0xa6, 0x1d, 0xbb, 0x22,
			0x0c, 0xae, 0x2f, 0xc6, 0xc1, 0x8d, 0x9a, 0xe2,
			0x27, 0xc1, 0x41, 0x2f, 0xc7, 0xc2, 0x73, 0xfb,
			0x9e, 0x89, 0x96, 0x2d, 0xa2, 0x37, 0x89, 0xa0,
			0x90, 0x07, 0x3f, 0xeb, 0x7e, 0x80, 0x21, 0x0b,
			0x6e, 0x4e, 0xa7, 0xc5, 0x83, 0x50, 0x04, 0x2d,
			0x40, 0x29, 0xe4, 0x67, 0x4b, 0x1e, 0x00, 0x81,
			0x65, 0xa4, 0x9d, 0x4f, 0xbf, 0x9f, 0xcc, 0x14,
			0x83, 0x4f, 0x97, 0x9c, 0x24, 0xbe, 0x2b, 0x56,
			0x96, 0xc3, 0xe1, 0x55, 0x3e, 0xa0, 0x00, 0xf3,
			0x87, 0xcf, 0x48, 0x52, 0xff, 0x5d, 0x7d, 0x21,
			0x61, 0x7f, 0x7d, 0x87, 0x6f, 0xd1, 0x9e, 0x03,
			0xe0, 0xac, 0x4c, 0x59, 0x7e, 0xeb, 0xb7, 0x40,
			0x14, 0x46, 0x09, 0x01, 0x99, 0xed, 0x4a, 0x79,
			0xe1, 0x8b, 0x3d, 0xb9, 0x06, 0x70, 0x84, 0x36,
			0xb9, 0x5e, 0x59, 0xc1, 0xf3, 0xc4, 0x24, 0x82,
			0x7a, 0x50, 0x40, 0x52, 0xb9, 0x71, 0x5f, 0xa6,
			0xed, 0xa8, 0x47, 0x9f, 0x2c, 0x63, 0x80, 0x16,
			0xe4, 0x53, 0xd5, 0xa4, 0xe6, 0x48, 0x5a, 0x73,
			0x7a, 0x75, 0xcc, 0x90, 0x8d, 0xf1, 0xf5, 0x61,
			0x53, 0x6c, 0x3f, 0xa4, 0x66, 0xd0, 0x50, 0x31,
			0x87, 0x85, 0x3a, 0x24, 0x79, 0xeb, 0x2a, 0xd8,
			0xf6, 0x40, 0xf4, 0xf8, 0xf0, 0xb6, 0xea, 0xa4,
			0x6a, 0x55, 0x70, 0x1d, 0x66, 0xb1, 0x23, 0xc9,
			0x5d, 0x60, 0x7c, 0x45, 0x98, 0xee, 0xcc, 0x3a,
			0x77, 0x4c, 0xb0, 0x38, 0xe3, 0xd0, 0x4a, 0xf1,
			0x78, 0x49, 0x6c, 0x2e, 0x42, 0x90, 0xa8, 0x62,
			0xfa, 0x1d, 0x2e, 0x68, 0x90, 0x32, 0xb9, 0x79,
			0x67, 0x90, 0x56, 0x03, 0xa4, 0x85, 0xe0, 0x51,
			0x55, 0xfd, 0x5b, 0x65, 0xa7, 0xe3, 0x7c, 0x7f,
			0x9c, 0x33, 0xb9, 0x3e, 0xb7, 0xd7, 0x90, 0xac,
			0xeb, 0xa9, 0x63, 0x3d, 0xe9, 0x50, 0xd9, 0x8a,
			0x67, 0x04, 0x80, 0x0a, 0x8f, 0x41, 0x4d, 0xd1,
			0xd3, 0xd3, 0xc3, 0x63, 0x26, 0xf1, 0x3b, 0x0b,
			0x88, 0xa6, 0xe0, 0xe9, 0x1d, 0xf9, 0xc3, 0xf4,
			0x71, 0xb6, 0xbc, 0x7f, 0xb8, 0x98, 0xd4, 0xe4,
			0x66, 0x6b, 0xea, 0xb9, 0xca, 0xf3, 0xf8, 0xcb,
			0x28, 0x3b, 0xd2, 0xae, 0xfe, 0x51, 0xa9, 0xd0,
			0x5d, 0xaa, 0x8e, 0x82, 0x3e, 0x4b, 0xa1, 0x16,
			0x63, 0x62, 0x25, 0xf3, 0x2a, 0xd8, 0x40, 0x37,
			0xb2, 0xd6, 0x5a, 0xf0, 0xb0, 0xe1, 0x92, 0x78,
			0x6b, 0xa0, 0x19, 0xca, 0x0e, 0xf2, 0x5e, 0xa2,
			0x4b, 0x9c, 0xec, 0xa8, 0x4f, 0xaf, 0x31, 0xdc,
			0x3f, 0x37, 0x96, 0x27, 0x32, 0xd0, 0x64, 0x99,
			0x08, 0xa2, 0x38, 0xeb, 0x8d, 0x46, 0xce, 0x44,
			0x8f, 0x4d, 0x9a, 0x17, 0x58, 0xb0, 0x10, 0x6b,
			0xb2, 0x98, 0x81, 0x6b, 0x56, 0x1f, 0xe2, 0x78,
			0x66, 0x4f, 0xce, 0x74, 0x1b, 0x10, 0xbe, 0xc1,
			0xbb, 0xac, 0x97, 0x26, 0x01, 0x0c, 0xa9, 0x82,
			0x48, 0xab, 0x14, 0x27, 0x95, 0xea, 0x5d, 0xc4,
			0x0d, 0xb5, 0xc0, 0xbc, 0x84, 0x58, 0x08, 0xda,
			0x42, 0x5d, 0x89, 0x2e, 0x02, 0x62, 0x0f, 0xfc,
			0xcc, 0xbd, 0xeb, 0x90, 0xf5, 0x45, 0x0a, 0xa5,
			0x71, 0x72, 0xa6, 0xff, 0x12, 0xb0, 0xa9, 0xdb,
			0xd3, 0x22, 0x97, 0xb3, 0xab, 0xac, 0x53, 0xc7,
			0x6c, 0x6e, 0x88, 0x42, 0xc1, 0x24, 0x4e, 0x6f,
			0x50, 0xc1, 0xb6, 0x39, 0x4b, 0x60, 0x86, 0xdb,
			0xfb, 0xec, 0x96, 0x70, 0x69, 0x86, 0x2b, 0x1f,
			0x51, 0x6f, 0xb1, 0x11, 0x6f, 0xd6, 0x05, 0x20,
			0xc0, 0x99, 0x13, 0x6f, 0x1a, 0xe1, 0x3a, 0xd5,
			0x70, 0x3e, 0x5c, 0x06, 0xd7, 0xcd, 0x18, 0x4d,
			0x53, 0x68, 0x5b, 0x7f, 0x41, 0x23, 0x4f, 0xe0,
			0x01, 0xee, 0xb9, 0x7f, 0xdd, 0x2d, 0x52, 0x6b,
			0x77, 0x13, 0xe2, 0x08, 0x4f, 0x72, 0xc8, 0x27,
			0x4e, 0x43, 0x80, 0x9f, 0xf6, 0x00, 0x31, 0x41,
			0x93, 0x4e, 0xbc, 0xf0, 0x50, 0x1a, 0xa8, 0x8d,
			0xa5, 0x04, 0xd6, 0x9b, 0xc9, 0xbf, 0x2c, 0x6d,
			0xa3, 0x3c, 0x05, 0x6a, 0xff, 0x59, 0xea, 0xce,
			0x0c, 0xab, 0xb1, 0x2c, 0xb0, 0x79, 0x7e, 0xbb,
			0x65, 0xcb, 0xb9, 0x85, 0xce, 0x5e, 0x34, 0xc3,
			0x12, 0xdc, 0xdd, 0xbf, 0x40, 0xd5, 0x5b, 0x0f,
			0x69, 0x60, 0x0a, 0xb4, 0xa6, 0xce, 0x16, 0x24,
			0x04, 0xe9, 0xa9, 0x40, 0x5e, 0x10, 0x55, 0xbe,
			0x02, 0xee, 0xcf, 0xef, 0x90, 0xc3, 0xa4, 0x65,
			0x59, 0x6b, 0xbf, 0xba, 0x49, 0x56, 0x8c, 0x5f,
			0x6d, 0x1e, 0xa8, 0xe9, 0xf9, 0x5d, 0xfc, 0xf8,
			0x0e, 0x49, 0x3d, 0xab, 0x8b, 0x71, 0xbf, 0x2c,
			0x4e, 0xfa, 0x5a, 0x75, 0x61, 0x27, 0x10, 0x14,
			0x77, 0x6e, 0xdb, 0x36, 0xa6, 0x01, 0xd4, 0xba,
			0x20, 0xed, 0x2a, 0x0f, 0x48, 0xdd, 0x28, 0xa0,
			0xe8, 0x0d, 0x47, 0xba, 0x53, 0x14, 0x54, 0xee,
			0x83, 0x0b, 0x3d, 0x64, 0x10, 0x22, 0xee, 0xcf,
			0x99, 0x7c, 0x6d, 0xc8, 0x6f, 0x1f, 0x7f, 0xcc,
			0x0f, 0x42, 0x04, 0x79, 0x1b, 0x82, 0xa1, 0xde,
			0x20, 0xea, 0xbf, 0x62, 0x62, 0x9b, 0x80, 0xdf,
			0xc9, 0x6e, 0x71, 0x6e, 0xcd, 0x51, 0x44, 0x06,
			0x16, 0x41, 0xc4, 0x31, 0xe2, 0xdc, 0xa0, 0xf4,
			0xa1, 0xe6, 0x19, 0x98, 0x40, 0x27, 0x06, 0x2e,
			0x30, 0x94, 0x3b, 0x5e, 0x64, 0x7e, 0x91, 0x54,
			0x20, 0x92, 0x5b, 0xa9, 0x04, 0x9d, 0x32, 0x9e,
			0xfb, 0xe7, 0xde, 0x8e, 0x20, 0x6a, 0xcd, 0x17,
			0x1e, 0x1a, 0xd3, 0x7f, 0x83, 0x6f, 0x7f, 0x83,
			0xad, 0x93, 0xd9, 0xfc, 0x96, 0x14, 0x72, 0x45,
			0x73, 0xca, 0xef, 0x1b, 0x0a, 0x75, 0xd1, 0x88,
			0xd7, 0x27, 0x86, 0x81, 0x32, 0x02, 0x6c, 0x6c,
			0x94, 0x4e, 0x05, 0xf8, 0xf3, 0x37, 0xcc, 0xb0,
			0x0c, 0xd1, 0xf2, 0x7d, 0x04, 0x1d, 0xf5, 0x32,
			0xb6, 0x66, 0xbd, 0x0a, 0x77, 0xd2, 0xd4, 0x90,
			0x28, 0xbb, 0xd7, 0x6f, 0x79, 0x99, 0x7a, 0xc3,
			0xc8, 0x05, 0xa1, 0x0f, 0x10, 0x94, 0x07, 0x87,
			0x44, 0xce, 0x34, 0x3d, 0x2b, 0xda, 0x2c, 0x78,
			0xe7, 0xb2, 0xd5, 0x8a, 0xcf, 0xfa, 0xfd, 0x9d,
			0x7c, 0x51, 0xc4, 0xc8, 0x33, 0x18, 0x40, 0x9f,
			0xf4, 0x24, 0xf4, 0x9f, 0x7f, 0x2a, 0x2d, 0xce,
			0x5e, 0xe0, 0x18, 0x7a, 0x1f, 0xe7, 0x27, 0x61,
			0xeb, 0x63, 0xd5, 0x23, 0x36, 0xb7, 0x21, 0x8b,
			0x26, 0x4b, 0xc3, 0x26, 0xc2, 0x0f, 0x0b, 0xf6,
			0xd9, 0x38, 0xe6, 0x96, 0xef, 0xbe, 0xd9, 0xa0,
			0x62, 0x78, 0x6f, 0x86, 0x7e, 0x32, 0xf9, 0x19,
			0x0b, 0x2b, 0x38, 0xb7, 0xc1, 0x8e, 0xa0, 0x1b,
			0x62, 0x00, 0xda, 0xf0, 0xb8, 0xa4, 0xa6, 0x31,
			0x0e, 0xe2, 0x63, 0x2b, 0xc0, 0x65, 0x3e, 0xe2,
			0x8d, 0x55, 0xf7, 0x89, 0xb6, 0x58, 0x8e, 0x0d,
			0xa7, 0x14, 0x1a, 0x52, 0xa9, 0xd9, 0xfd, 0x3d,
			0x4b, 0x49, 0xee, 0x12, 0xe5, 0xac, 0x45, 0xc3,
			0xe3, 0x6e, 0x6d, 0xe3, 0x82, 0x2c, 0x07, 0xc3,
			0x19, 0x43, 0x5e, 0x6f, 0xf3, 0x1f, 0x4e, 0xa6,
			0x26, 0x21, 0xce, 0x44, 0x94, 0x3f, 0x08, 0x1e,
			0x9e, 0xaf, 0xe6, 0x33, 0xd2, 0xe2, 0xc4, 0x8e,
			0xdd, 0x64, 0x06, 0xf0, 0x47, 0x16, 0x26, 0x87,
			0x55, 0xda, 0x6f, 0x2e, 0x12, 0x45, 0x96, 0x84,
			0x11, 0x7a, 0x06, 0x43, 0x8d, 0x91, 0x09, 0x2d,
			0xd3, 0xb2, 0x36, 0xd0, 0xba, 0x17, 0x41, 0x76,
			0x6b, 0x76, 0xef, 0x6d, 0x9b, 0x7f, 0x2b, 0xed,
			0x0d, 0xeb, 0x34, 0x63, 0x24, 0xc5, 0x8d, 0xb8,
			0x31, 0xc5, 0x41, 0xa7, 0xca, 0x84, 0x5d, 0xfa,
			0x1d, 0x03, 0x5b, 0xed, 0xb0, 0x02, 0x24, 0xdd,
			0x02, 0x4b, 0xcd, 0xdf, 0x8b, 0x63, 0x41, 0xe7,
			0x62, 0xa5, 0x9b, 0x2a, 0xa4, 0xed, 0xce, 0x01,
			0xfa, 0xc0, 0x78, 0xef, 0x52, 0x64, 0x4c, 0xbf,
			0x43, 0xeb, 0x37, 0x72, 0xd1, 0x35, 0xc9, 0xdf,
			0x23, 0x60, 0xc4, 0x7a, 0x77, 0x05, 0xf4, 0xe8,
			0xfe, 0x58, 0x0e, 0x0d, 0xb7, 0x68, 0xb9, 0xbe,
			0x8c, 0x1b, 0x0d, 0xdb, 0xf8, 0xb4, 0x8d, 0xa2,
			0xad, 0xf1, 0x11, 0xbb, 0x4b, 0xf8, 0x6e, 0x92,
			0x18, 0xb6, 0xfc, 0xbf, 0x52, 0xbc, 0xc9, 0x1d,
			0x23, 0xa9, 0x4e, 0x73, 0x84, 0x08, 0x1f, 0x14,
			0xff, 0xf8, 0x2f, 0x84, 0xf6, 0xc8, 0x24, 0x89,
			0x62, 0x6f, 0xf5, 0x7d, 0x0e, 0xc9, 0x2e, 0xc8,
			0xef, 0x80, 0xc2, 0xf3, 0xec, 0x45, 0x78, 0x20,
			0xae, 0xf7, 0xa5, 0x63, 0xc0, 0x15, 0x8a, 0x12,
			0x92, 0x32, 0x19, 0x1a, 0xf6, 0x5a, 0xc1, 0x04,
			0x85, 0xcd, 0x17, 0x9b, 0x26, 0xc4, 0xb1, 0x62,
			0xec, 0x70, 0xa6, 0x83, 0xc3, 0x17, 0x9c, 0x2f,
			0xe8, 0x33, 0x49, 0x3c, 0x52, 0xb1, 0x83, 0x21,
			0xe9, 0xc4, 0xfc, 0x5e, 0x7f, 0xa5, 0x74, 0x14,
			0x5f, 0xf8, 0xee, 0xb2, 0xa6, 0x1e, 0x16, 0x39,
			0x37, 0x41, 0xc2, 0x51, 0x45, 0x72, 0x5a, 0x1f,
			0xf3, 0x4e, 0x46, 0x24, 0xff, 0x09, 0x38, 0x50,
			0x83, 0x8e, 0xbe, 0x5e, 0xf8, 0xb5, 0xee, 0xfa,
			0xcd, 0x72, 0x93, 0xb5, 0x14, 0x3a, 0x4e, 0x4f,
			0x06, 0x3d, 0x6e, 0x33, 0xbc, 0x97, 0xa5, 0x4d,
			0xeb, 0xa5, 0x67, 0x42, 0x28, 0x90, 0x22, 0x87,
			0x3a, 0x6d, 0x40, 0x6f, 0xd9, 0xeb, 0xb7, 0x28,
			0xf2, 0x3f, 0x64, 0x9a, 0x76, 0x3e, 0x86, 0xfa,
			0xb9, 0x81, 0xf3, 0xe6, 0x1b, 0x20, 0x0d, 0xd6,
			0x8e, 0xc3, 0x96, 0xd5, 0x5b, 0x6b, 0xe6, 0x10,
			0x70, 0x69, 0xab, 0xad, 0x70, 0x37, 0x78, 0xbc,
			0x4d, 0x05, 0x83, 0x66, 0xad, 0x15, 0x8a, 0x45,
			0xeb, 0x5f, 0x5b, 0xf2, 0x90, 0x75, 0xf7, 0xa9,
			0x2f, 0x8a, 0x51, 0x91, 0x8f, 0x7c, 0xb8, 0x67,
			0xb0, 0x78, 0x2a, 0xb5, 0x14, 0x83, 0x89, 0xd6,
			0x35, 0x38, 0xd3, 0x06, 0x39, 0x2a, 0x3c, 0x45,
			0x99, 0x02, 0x8a, 0x0d, 0xe5, 0x20, 0x4d, 0xf1,
			0x65, 0xad, 0x0f, 0xf3, 0xeb, 0x32, 0x40, 0x7e,
			0x86, 0x06, 0xcd, 0x32, 0x8f, 0x37, 0xad, 0x3b,
			0x28, 0xf1, 0xab, 0x89, 0x47, 0x7d, 0x45, 0x24,
			0x4d, 0x0c, 0xe9, 0x1d, 0xcf, 0xb5, 0x89, 0x35,
			0x9b, 0xc7, 0x83, 0x67, 0xe3, 0xa3, 0x8e, 0xa8,
			0x93, 0x1b, 0x73, 0xfb, 0xf0, 0xe0, 0xcb, 0x25,
			0x8a, 0x73, 0x56, 0xd0, 0x22, 0xbd, 0xbb, 0xce,
			0x08, 0x72, 0x9d, 0x9c, 0x18, 0xa0, 0x97, 0x98,
			0x41, 0x50, 0xfd, 0xd8, 0x35, 0x22, 0xc1, 0xcb,
			0x48, 0x84, 0x74, 0xb6, 0xee, 0x14, 0x72, 0xc9,
			0xb6, 0xec, 0xac, 0xad, 0x07, 0xcb, 0xcf, 0xf2,
			0x63, 0x56, 0xab, 0xfc, 0xae, 0x48, 0x37, 0x59,
			0xc5, 0xf0, 0x0b, 0xce, 0xe3, 0xec, 0x6e, 0x56,
			0x8b, 0xbf, 0xba, 0x54, 0x76, 0x44, 0x96, 0x70,
			0x6b, 0x94, 0x21, 0x0c, 0x3a, 0xfa, 0x08, 0x40,
			0xef, 0xde, 0xfd, 0x14, 0x40, 0x50, 0x99, 0xe8,
			0x6a, 0xa1, 0x83, 0x2f, 0x4d, 0x02, 0x0f, 0xf6,
			0xe9, 0xb2, 0x58, 0xff, 0xbb, 0xd2, 0x7a, 0xbb,
			0xba, 0xa3, 0x28, 0x90, 0x32, 0x89, 0x29, 0xf8,
			0xc3, 0x94, 0x81, 0xc4, 0xe0, 0x88, 0x78, 0x64,
			0xb5, 0x07, 0x11, 0xdc, 0x41, 0x17, 0x79, 0x29,
			0x44, 0x75, 0x54, 0x2c, 0x57, 0x39, 0xea, 0xff,
			0x8e, 0x37, 0xc0, 0x01, 0xff, 0xcf, 0xa8, 0x30,
			0x9c, 0x9a, 0xc0, 0xe6, 0xeb, 0x18, 0x82, 0x6b,
			0xb0, 0x2b, 0xc6, 0xba, 0x28, 0xaa, 0xfb, 0xe6,
			0xe3, 0xc9, 0x87, 0x2b, 0x19, 0xc4, 0x4a, 0x48,
			0xde, 0x08, 0xbb, 0x20, 0x52, 0x12, 0x3d, 0x31,
			0x82, 0xe1, 0xc7, 0x96, 0x86, 0x4a, 0xd5, 0x27,
			0xba, 0xf6, 0x55, 0xd3, 0x54, 0x41, 0x11, 0xac,
			0xa9, 0x0c, 0x1f, 0x2f, 0x43, 0xad, 0xff, 0x4c,
			0x16, 0x61, 0x19, 0xbe, 0xfa, 0xff, 0x38, 0x17,
			0xec, 0x3a, 0x4e, 0x0d, 0xe2, 0x23, 0x35, 0x56,
			0x63, 0x24, 0x9f, 0x00, 0xbd, 0xe8, 0x78, 0x89,
			0xa5, 0x91, 0x18, 0x63, 0x13, 0x76, 0x07, 0xbe,
			0x53, 0x73, 0xe7, 0x7d, 0x52, 0x77, 0xa5, 0x67,
			0x81, 0xfb, 0xec, 0x7b, 0x71, 0x2b, 0x2c, 0x80,
			0x52, 0x05, 0xf6, 0x26, 0xf6, 0xfc, 0xea, 0x88,
			0xff, 0x52, 0xeb, 0xb6, 0x45, 0xc2, 0x0b, 0x1c,
			0x4f, 0x08, 0xfb, 0x61, 0x81, 0x42, 0x66, 0xb6,
			0xa8, 0xf4, 0xc8, 0xa8, 0xdc, 0x8a, 0x9e, 0x4f,
			0x49, 0x83, 0xe4, 0xa5, 0x01, 0xf8, 0x22, 0x2d,
			0x41, 0xdb, 0x84, 0x72, 0x78, 0xe6, 0x80, 0x99,
			0xf9, 0x99, 0x45, 0x9a, 0x1f, 0xe4, 0x02, 0x21,
			0x0b, 0xd3, 0x45, 0xcb, 0x67, 0x94, 0xc4, 0x77,
			0x4b, 0x4b, 0xbb, 0xc3, 0x2f, 0x93, 0x7e, 0xda,
			0x01, 0x91, 0x0b, 0x74, 0x03, 0xab, 0x48, 0x0c,
			0x35, 0x6c, 0x41, 0x16, 0x0d, 0x5b, 0xd3, 0xf6,
			0x29, 0x3f, 0x8e, 0x05, 0xbf, 0xcd, 0x55, 0x53,
			0x95, 0x9f, 0xfe, 0x29, 0x2b, 0x07, 0x46, 0xaa,
			0x25, 0x94, 0x56, 0xa3, 0x84, 0xce, 0x56, 0x6d,
			0xf3, 0x3a, 0xbc, 0xa1, 0x6a, 0x82, 0x40, 0xdf,
			0xc8, 0x4a, 0xc8, 0xd9, 0x3e, 0xe4, 0x0a, 0xda,
			0xbc, 0x1b, 0x77, 0x70, 0xb9, 0x21, 0x19, 0x8c,
			0x94, 0x93, 0x98, 0xf9, 0xca, 0x03, 0x21, 0xdc,
			0xe1, 0x78, 0x24, 0x9d, 0x4f, 0x9e, 0x3d, 0xe8,
			0xa1, 0x02, 0xf6, 0xeb, 0xfb, 0x67, 0x70, 0x5a,
			0x2e, 0x2c, 0xa4, 0xfa, 0xc8, 0xa0, 0xaf, 0xc6,
			0x11, 0xa5, 0x86, 0xb2, 0xac, 0x54, 0x7c, 0x94,
			0x53, 0xec, 0xd7, 0x0b, 0xe2, 0xc1, 0x76, 0x5b,
			0x45, 0x86, 0x85, 0x78, 0x17, 0xce, 0x89, 0x15,
			0xf2, 0x37, 0x43, 0xcf, 0xb9, 0xbe, 0x63, 0x70,
			0xe1, 0xc7, 0x66, 0x2e, 0x02, 0x96, 0x54, 0x9c,
			0xf6, 0x64, 0x27, 0x06, 0x26, 0x71, 0xc7, 0x13,
			0x93, 0x08, 0xdd, 0x20, 0x60, 0x0d, 0x63, 0xe2,
			0x83, 0x1a, 0x37, 0x7f, 0x28, 0x66, 0xca, 0xc1,
			0x13, 0x7a, 0x43, 0xc6, 0x2a, 0xd8, 0x75, 0x83,
			0x60, 0x18, 0x24, 0x14, 0x5d, 0x10, 0x9b, 0x38,
			0x99, 0xb0, 0xb0, 0xfc, 0xcf, 0xeb, 0xdc, 0xb6,
			0x2b, 0xc9, 0xc9, 0x5d, 0xe6, 0x8f, 0xfd, 0xf9,
			0x35, 0x9f, 0x29, 0x69, 0x89, 0x7e, 0xac, 0x3a,
			0xff, 0x70, 0xda, 0x9c, 0xf7, 0xe8, 0x92, 0xfc,
			0x57, 0xda, 0x6b, 0x3d, 0x10, 0xba, 0x4e, 0x16,
			0x7d, 0x51, 0x7a, 0x86, 0xbf, 0x1e, 0xcb, 0xb5,
			0x3d, 0x17, 0xbd, 0xf3, 0x1e, 0x6f, 0x30, 0x51,
			0xf5, 0x16, 0x8b, 0x9d, 0x1b, 0x7b, 0xf4, 0x7e,
			0x87, 0x4b, 0xd1, 0x59, 0x91, 0xb8, 0x06, 0x10,
			0xad, 0x06, 0x7f, 0x75, 0x92, 0xda, 0x68, 0xe9,
			0x0d, 0x61, 0xec, 0x60, 0x8a, 0xc2, 0x27, 0x0e,
			0x19, 0xc6, 0xd3, 0xe6, 0x01, 0xef, 0x82, 0x29,
			0xa8, 0xc9, 0x22, 0x9f, 0x7e, 0x6f, 0x3e, 0x9d,
			0x2d, 0xf6, 0xce, 0xf3, 0x1c, 0xb7, 0x80, 0x05,
			0xc3, 0x7d, 0xb4, 0x1b, 0xb9, 0xbd, 0x45, 0x20,
			0x53, 0x80, 0x37, 0x80, 0xf3, 0xf8, 0xe4, 0x88,
			0x45, 0xc1, 0x61, 0x7a, 0x85, 0x08, 0x94, 0x87,
			0x56, 0xa4, 0xc2, 0x4f, 0x2e, 0x02, 0xce, 0xc2,
			0x2b, 0x16, 0x0a, 0x58, 0x56, 0x60, 0x0b, 0xb6,
			0xc3, 0x87, 0x74, 0x80, 0x2b, 0x07, 0x18, 0x6e,
			0x39, 0x17, 0xe2, 0xb1, 0x44, 0xde, 0xb1, 0x61,
			0x4d, 0xc4, 0x71, 0x49, 0x89, 0xf7, 0x88, 0x65,
			0xc4, 0xbe, 0x5e, 0x84, 0xe9, 0xef, 0x1e, 0xa5,
			0xb8, 0x6c, 0x41, 0xbf, 0xdf, 0xc3, 0x95, 0x36,
			0x3a, 0xa3, 0x32, 0xf7, 0xbc, 0xca, 0xd0, 0x70,
			0x70, 0x00, 0xba, 0x3a, 0xb1, 0x11, 0xe4, 0x6a,
			0xff, 0x94, 0x9f, 0x90, 0x46, 0x7f, 0x34, 0x3a,
			0xec, 0x55, 0x23, 0x5c, 0x38, 0xf2, 0x93, 0xc0,
			0xd4, 0xfc, 0x14, 0xe8, 0x2f, 0x84, 0xdd, 0x45,
			0x49, 0x93, 0xbf, 0x26, 0x14, 0x5a, 0x09, 0xaa,
			0xbe, 0x9a, 0x90, 0x99, 0xad, 0xf9, 0x73, 0xbf,
			0xf5, 0x0e, 0xa6, 0x6e, 0x7a, 0xe3, 0x4d, 0x7e,
			0xdd, 0x03, 0xca, 0x43, 0x7b, 0x53, 0x63, 0xe2,
			0xf0, 0x36, 0x9c, 0xc1, 0xeb, 0x63, 0x2c, 0xbd,
			0x29, 0xe9, 0xd0, 0x14, 0xf3, 0x2e, 0xa1, 0xd3,
			0x00, 0xa9, 0xca, 0xeb, 0x53, 0x54, 0xf9, 0x55,
			0x92, 0x8a, 0xa4, 0xcc, 0xce, 0x5d, 0x51, 0x7f,
			0x9c, 0x63, 0xe7, 0x27, 0x07, 0xa6, 0x34, 0x2c,
			0x1c, 0x27, 0x5f, 0x64, 0x4e, 0xa3, 0x30, 0xc4,
			0xaa, 0xbb, 0xe6, 0x56, 0x1d, 0x0d, 0xb2, 0x7d,
			0xe0, 0x8a, 0x7e, 0xd0, 0x28, 0x47, 0x8a, 0xe2,
			0x16, 0x73, 0x60, 0x63, 0xb5, 0x5a, 0x53, 0xb0,
			0x49, 0x2b, 0x72, 0x19, 0xe7, 0x16, 0x15, 0xae,
			0x20, 0x9f, 0x6c, 0x2b, 0xb6, 0xd5, 0x6a, 0x0c,
			0xa8, 0x09, 0x1a, 0x15, 0xa6, 0x83, 0x87, 0xc6,
			0x50, 0xfe, 0x63, 0xd2, 0xbb, 0xbd, 0x86, 0x4f,
			0xfb, 0x8b, 0x50, 0x5c, 0xc4, 0xe5, 0x8c, 0x80,
			0xaa, 0xb7, 0x63, 0x7f, 0x28, 0x65, 0x55, 0x07,
			0x15, 0x91, 0x07, 0x67, 0x70, 0x3d, 0x94, 0x94,
			0x75, 0xc1, 0x11, 0xa1, 0x77, 0xde, 0xbc, 0x02,
			0xa3, 0x34, 0xb7, 0x5e, 0xe1, 0x2a, 0xe1, 0x35,
			0xd8, 0x26, 0x11, 0x16, 0x3c, 0x96, 0xd9, 0x1c,
			0xa3, 0x64, 0x65, 0x2f, 0xa2, 0x07, 0xbe, 0x21,
			0x74, 0x27, 0xae, 0xa4, 0xcb, 0x40, 0x0b, 0xa8,
			0xab, 0x47, 0xd6, 0x80, 0x48, 0xdd, 0x81, 0x8c,
			0x79, 0x33, 0x6a, 0x18, 0x29, 0x25, 0x3f, 0x47,
			0x5d, 0xac, 0x36, 0x6e, 0x4f, 0x60, 0xb6, 0xda,
			0x66, 0xae, 0xf4, 0x0b, 0xe7, 0x9b, 0xfe, 0x6d,
			0x8b, 0x74, 0xc8, 0x7c, 0x2c, 0x4b, 0x77, 0x14,
			0x4c, 0x82, 0x57, 0xfe, 0xe4, 0xa5, 0xac, 0x38,
			0x79, 0xa9, 0x29, 0x3a, 0xe1, 0x77, 0xe3, 0xc5,
			0x34, 0xd7, 0x51, 0xb8, 0x33, 0x71, 0x98, 0x32,
			0x04, 0xbd, 0xa7, 0xb0, 0xb7, 0xc9, 0x73, 0x52,
			0x00, 0x56, 0x19, 0x94, 0xde, 0xa0, 0xe2, 0xac,
			0x4e, 0x36, 0xaa, 0x12, 0x1e, 0x6e, 0xf2, 0x3d,
			0x52, 0x5b, 0xa5, 0x7a, 0x55, 0x69, 0x9c, 0x04,
			0x9a, 0x6a, 0x9f, 0xb0, 0xea, 0x24, 0xda, 0xca,
			0xdf, 0x29, 0x53, 0xea, 0x60, 0x04, 0x0e, 0x75,
			0x1c, 0x35, 0x3f, 0x8c, 0x36, 0x65, 0x60, 0x2e,
			0x3e, 0x65, 0xa7, 0xbc, 0x6c, 0x7f, 0x76, 0x5e,
			0x28, 0x59, 0xb3, 0x70, 0xd0, 0xe1, 0x10, 0x82,
			0x3d, 0xff, 0x0e, 0x15, 0x33, 0xc3, 0x12, 0x57,
			0xa2, 0x4e, 0xd0, 0x29, 0x07, 0x23, 0x9f, 0xaa,
			0x36, 0x56, 0xd2, 0xfe, 0xe6, 0x47, 0xbf, 0x6f,
			0x02, 0x4f, 0xb0, 0xbe, 0x1b, 0xa6, 0x1b, 0x85,
			0xb6, 0x16, 0x59, 0x7c, 0xbb, 0x6c, 0xf3, 0x24,
			0x0f, 0xbf, 0x41, 0x2a, 0xe5, 0xa5, 0xf6, 0xa2,
			0x28, 0x33, 0xa6, 0xee, 0x7c, 0xfa, 0xea, 0xf8,
			0x19, 0xa5, 0x89, 0xf8, 0xf5, 0x67, 0x39, 0xb8,
			0x67, 0x94, 0xb6, 0x6a, 0xb3, 0xf3, 0x44, 0x88,
			0xd7, 0x49, 0x28, 0xc1, 0xd1, 0x36, 0xd5, 0xd3,
			0xf4, 0x04, 0xce, 0x38, 0x88, 0x83, 0xeb, 0x34,
			0x96, 0x60, 0xa1, 0x31, 0xf7, 0xdd, 0x1d, 0xc6,
			0xa1, 0x68, 0x8a, 0x98, 0x1d, 0xc2, 0x1e, 0xe2,
			0xbe, 0xc3, 0x5e, 0xbc, 0x20, 0xde, 0x67, 0x2f,
			0xca, 0x36, 0x48, 0xde, 0x8a, 0x5d, 0x9a, 0xb6,
			0x97, 0x05, 0xdf, 0xc0, 0x7e, 0xb0, 0xef, 0xb7,
			0xd1, 0x60, 0xbb, 0x95, 0x35, 0xc9, 0x6c, 0xd8,
			0xd7, 0xd6, 0x0d, 0xf6, 0xa4, 0x8c, 0x25, 0x9a,
			0xcc, 0x4a, 0x9f, 0x51, 0xd0, 0xc7, 0x98, 0x2f,
			0x83, 0x97, 0x4f, 0x9f, 0x2e, 0xe7, 0x7e, 0x09,
			0xa2, 0x99, 0xd1, 0x18, 0x4f, 0x8d, 0x38, 0x57,
			0x37, 0x35, 0xad, 0xcf, 0xdb, 0xe1, 0x7f, 0x1e,
			0x43, 0x01, 0x23, 0x35, 0xe4, 0xae, 0x56, 0x1c,
			0x65, 0x92, 0x72, 0xcb, 0x7f, 0x40, 0xa5, 0xa0,
			0xd1, 0x48, 0xfd, 0xb3, 0xc6, 0xd3, 0x50, 0xf8,
			0x5c, 0x5c, 0xd0, 0xd3, 0x9d, 0xac, 0xf5, 0x33,
			0xaf, 0x0c, 0xcf, 0x02, 0x95, 0xc3, 0x58, 0xd6,
			0x45, 0x22, 0x65, 0x96, 0x42, 0xec, 0x12, 0xab,
			0x16, 0x33, 0x78, 0xcd, 0x14, 0x41, 0xdb, 0x60,
			0x35, 0x49, 0x58, 0xb3, 0x47, 0x27, 0xd4, 0x30,
			0x43, 0x7f, 0x55, 0x24, 0x2b, 0xf1, 0x79, 0x59,
			0x79, 0x1d, 0x35, 0xc1, 0x82, 0x71, 0x94, 0x6e,
			0x1f, 0x21, 0x2c, 0x5c, 0x11, 0xed, 0xca, 0x19,
			0x2b, 0x8c, 0x3c, 0xd5, 0x6c, 0x33, 0x19, 0x4f,
			0x1a, 0xcc, 0x04, 0x2b, 0x96, 0xda, 0xcd, 0xd5,
			0x12, 0x37, 0xe0, 0x63, 0x3c, 0x63, 0x49, 0x60,
			0x6f, 0xed, 0x03, 0x3f, 0x24, 0xc9, 0xa8, 0xe1,
			0x44, 0x50, 0x0e, 0xfc, 0xbc, 0xeb, 0x39, 0x0d,
			0x18, 0xa1, 0xf8, 0xf0, 0xff, 0x5e, 0xe8, 0xf2,
			0x70, 0x81, 0xe9, 0x68, 0x16, 0xa5, 0xd0, 0x47,
			0xec, 0xe0, 0x58, 0x54, 0x0e, 0x23, 0x53, 0xe7,
			0x49, 0xe3, 0xc3, 0xdc, 0x77, 0x3f, 0x36, 0xe1,
			0x56, 0x9d, 0x28, 0x18, 0x55, 0x64, 0x4a, 0x60,
			0x11, 0xf9, 0x1b, 0xfb, 0x09, 0xa9, 0x33, 0xab,
			0x63, 0xc0, 0xee, 0x24, 0x14, 0x9f, 0x3f, 0x87,
			0x19, 0xd7, 0x9b, 0x4c, 0x9a, 0xfe, 0xd1, 0xb0,
			0xb6, 0xbc, 0x3f, 0xf2, 0xc5, 0xc4, 0x13, 0x84,
			0xc1, 0xd5, 0x37, 0x4e, 0x6a, 0x15, 0xd1, 0x96,
			0xdd, 0x15, 0x94, 0xf5, 0xf2, 0x01, 0xa5, 0x09,
			0x0e, 0x90, 0x65, 0x84, 0x8d, 0x66, 0x3f, 0x7b,
			0x8e, 0xf3, 0x15, 0x5d, 0x32, 0xe6, 0xce, 0xc1,
			0x95, 0xa0, 0x02, 0xd1, 0xdf, 0xb6, 0xf4, 0x5d,
			0x84, 0xb5, 0xc8, 0x94, 0x97, 0x7f, 0x94, 0x66,
			0xcf, 0x4d, 0x15, 0xd1, 0xdb, 0xfc, 0x73, 0xc5,
			0x49, 0x79, 0x42, 0x00, 0x6a, 0xaa, 0xa3, 0x68,
			0x46, 0x1c, 0x19, 0x44, 0xbd, 0x42, 0x91, 0xb2,
			0x66, 0xda, 0x9e, 0x73, 0x08, 0x1f, 0xb1, 0x7a,
			0x78, 0x73, 0xa4, 0x63, 0x1a, 0xf6, 0x36, 0x5f,
			0x84, 0x02, 0x8c, 0x20, 0x6d, 0x1c, 0x03, 0xa4,
			0x1d, 0x18, 0x4e, 0xcc, 0x7f, 0x59, 0xf4, 0xcb,
			0x6f, 0x77, 0xd4, 0x50, 0x71, 0xfa, 0xe3, 0x1e,
			0x3f, 0xc1, 0xf4, 0x04, 0xc5, 0xf6, 0x2e, 0xd1,
			0x91, 0xc0, 0x11, 0x0e, 0x41, 0x01, 0x92, 0x4a,
			0x60, 0x68, 0x6c, 0x39, 0x74, 0x0b, 0xd0, 0x1c,
			0x9b, 0x42, 0xb7, 0xcc, 0xe2, 0xb9, 0x18, 0xe7,
			0x25, 0x8c, 0x78, 0xea, 0x4c, 0x58, 0xcb, 0xb1,
			0xa8, 0x01, 0x36, 0x74, 0x82, 0x0b, 0x32, 0xd6,
			0x38, 0xc2, 0xe9, 0xe5, 0x27, 0x63, 0xe6, 0x4b,
			0x85, 0x21, 0xd4, 0xd4, 0x90, 0x29, 0x19, 0x37,
			0x68, 0xe4, 0xd5, 0x05, 0x41, 0xfc, 0xb3, 0x99,
			0x33, 0x6a, 0x4b, 0xf2, 0x14, 0xd6, 0x57, 0xa4,
			0x3e, 0x04, 0x6d, 0xc7, 0xa6, 0x01, 0x0b, 0xb5,
			0x04, 0x4f, 0xf7, 0xfc, 0xc9, 0x9c, 0x20, 0xbb,
			0x25, 0xa1, 0xf1, 0x54, 0x51, 0x00, 0xd6, 0x51,
			0xe7, 0x64, 0x43, 0xcb, 0xcc, 0xd5, 0x46, 0xca,
			0x1d, 0x38, 0x2d, 0xa5, 0xfb, 0x1d, 0xfc, 0x53,
			0x88, 0xec, 0x29, 0x69, 0xce, 0x76, 0x1e, 0x5a,
			0x3d, 0x92, 0x1b, 0x12, 0xf4, 0x13, 0xef, 0x30,
			0xdb, 0xf0, 0x5e, 0x1e, 0x22, 0xb6, 0x52, 0x35,
			0x89, 0xe2, 0xd8, 0xbb, 0x99, 0x58, 0x02, 0xdd,
			0xe5, 0xd8, 0x01, 0x6a, 0x2a, 0xd9, 0x9c, 0xd8,
			0x70, 0x98, 0x62, 0x8b, 0x7b, 0xc8, 0x78, 0x27,
			0xed, 0xed, 0x65, 0xbf, 0x89, 0xf1, 0x06, 0x13,
			0x4c, 0x5a, 0x6f, 0x30, 0xbf, 0xca, 0x48, 0x31,
			0xbb, 0xcf, 0x5e, 0xb4, 0x49, 0x9e, 0xd2, 0xaf,
			0x05, 0xbb, 0x36, 0x2a, 0xee, 0x35, 0xce, 0x27,
			0x70, 0x12, 0x6e, 0xc5, 0x97, 0xd8, 0x1d, 0x54,
			0x4e, 0xe7, 0xbc, 0x2e, 0x4a, 0x8a, 0x94, 0xdd,
			0x3b, 0xa6, 0x38, 0x88, 0xdd, 0x6b, 0xe5, 0xfe,
			0xed, 0xc8, 0x21, 0x32, 0xe1, 0x24, 0x6a, 0x9c,
			0x92, 0x16, 0x83, 0x6b, 0x6c, 0xd0, 0x0a, 0xb7,
			0x96, 0x78, 0xf3, 0x4a, 0xa7, 0x83, 0x3e, 0x1f,
			0x76, 0xb1, 0x66, 0x4e, 0xee, 0xb8, 0x1a, 0x83,
			0x3b, 0x00, 0x8f, 0x64, 0x7e, 0x63, 0x8e, 0x1d,
			0x82, 0xc3, 0x1f, 0xf3, 0x75, 0x98, 0xe1, 0xf1,
			0x55, 0x68, 0x8c, 0x37, 0xf5, 0x0b, 0xbd, 0x11,
			0xec, 0xe4, 0x78, 0x15, 0xa9, 0x87, 0x6c, 0x54,
			0x32, 0x44, 0x77, 0x27, 0xe1, 0x33, 0x42, 0x65,
			0x44, 0xf7, 0x8a, 0xfb, 0x2e, 0x58, 0x9a, 0x3e,
			0x60, 0x74, 0xa4, 0x51, 0x4b, 0x10, 0xef, 0x25,
			0xf6, 0x7a, 0xa4, 0x74, 0x29, 0xa7, 0xce, 0x01,
			0x6d, 0x5e, 0x71, 0xf9, 0xf4, 0x25, 0xa9, 0xa8,
			0x6c, 0xfd, 0x79, 0x33, 0xcb, 0x52, 0xe3, 0xc6,
			0xfa, 0x83, 0x14, 0x58, 0x7b, 0xb9, 0xe7, 0x38,
			0x79, 0xa9, 0x70, 0x1a, 0xfa, 0x97, 0x4e, 0x1a,
			0x1d, 0x77, 0x7e, 0x58, 0xd3, 0x47, 0x7c, 0x14,
			0xa9, 0x6e, 0x12, 0x96, 0x51, 0x88, 0xdd, 0xe4,
			0xc4, 0xb7, 0xc3, 0x8b, 0x91, 0xf4, 0xe6, 0xee,
			0xe9, 0x3a, 0x2f, 0x33, 0xa8, 0x84, 0x58, 0x81,
			0xbb, 0x0d, 0x98, 0xb5, 0xb2, 0x1e, 0xbf, 0x0c,
			0xe2, 0x2c, 0xc6, 0x2f, 0xff, 0x8a, 0x79, 0x77,
			0x59, 0x59, 0x8a, 0xd6, 0x7c, 0xf4, 0x75, 0x22,
			0x19, 0x1b, 0xbe, 0x6e, 0xa2, 0x76, 0x81, 0x57,
			0xed, 0x08, 0x5e, 0x44, 0x2d, 0x1d, 0xdf, 0x53,
			0x43, 0xc0, 0xbf, 0x07, 0xad, 0x75, 0x71, 0x84,
			0x6b, 0x86, 0x53, 0x49, 0x9a, 0x34, 0x58, 0xdb,
			0x24, 0xc9, 0x61, 0x66, 0x33, 0x2a, 0x62, 0x52,
			0x47, 0x26, 0x50, 0x72, 0x5c, 0x50, 0x25, 0xf9,
			0xb7, 0x79, 0x59, 0xc6, 0x13, 0xa1, 0xd1, 0xbe,
			0xa0, 0x7b, 0xf1, 0x1e, 0x07, 0x1d, 0xef, 0x4c,
			0xc7, 0xb7, 0x79, 0x34, 0x40, 0x92, 0xff, 0xdc,
			0xc7, 0x25, 0xe5, 0x08, 0xa8, 0xe9, 0x66, 0xd0,
			0x62, 0x07, 0xe4, 0x14, 0x42, 0xae, 0x14, 0x96,
			0x08, 0x6c, 0xfd, 0x35, 0xb0, 0x6e, 0x82, 0x01,
			0x53, 0xdb, 0x74, 0x2a, 0xd3, 0x03, 0xda, 0x15,
			0x60, 0xd8, 0xa0, 0x43, 0xc1, 0xd4, 0xd2, 0xe1,
			0x6d, 0x1e, 0x34, 0x6d, 0x0b, 0x45, 0x8e, 0x79,
			0x56, 0x53, 0x7b, 0x8e, 0xaf, 0xd9, 0x72, 0xe7,
			0x43, 0x9c, 0x33, 0x70, 0x33, 0x2f, 0x73, 0x2a,
			0xc5, 0xe9, 0x07, 0x8e, 0x50, 0xe8, 0x3a, 0x2a,
			0xc8, 0xfb, 0x31, 0x5e, 0xdf, 0x55, 0x13, 0x9e,
			0x71, 0x96, 0x6c, 0x13, 0xbf, 0x1d, 0x1a, 0x83,
			0xb4, 0xdb, 0xa0, 0xae, 0x59, 0xcd, 0xa4, 0xb7,
			0xbe, 0x6f, 0x1b, 0x23, 0xb8, 0x0b, 0x87, 0xa2,
			0xb8, 0x0b, 0x17, 0x2a, 0x02, 0xb6, 0xb0, 0xb7,
			0xd5, 0x94, 0xeb, 0x08, 0x2b, 0x74, 0xd5, 0x72,
			0x01, 0xa1, 0x41, 0x0e, 0x59, 0xe5, 0x51, 0x34,
			0x1b, 0xe0, 0xc0, 0xec, 0x79, 0xe2, 0x6f, 0xfd,
			0x99, 0xd5, 0x78, 0xb7, 0x08, 0x89, 0xb0, 0x72,
			0x78, 0xc0, 0x5c, 0x2f, 0x96, 0x5b, 0x92, 0xa2,
			0xe2, 0x00, 0x0b, 0x00, 0xcb, 0x37, 0xb5, 0x02,
			0xd8, 0x80, 0x27, 0x3e, 0x06, 0x35, 0x67, 0xbb,
			0x0c, 0xaf, 0x46, 0x34, 0x59, 0x6e, 0x90, 0xd3,
			0x05, 0xe2, 0xd6, 0x38, 0x18, 0x9f, 0x4e, 0xf2,
			0x63, 0xb2, 0xf0, 0x10, 0x47, 0xfa, 0x3c, 0x1a,
			0xa6, 0x57, 0x3a, 0x3a, 0xe1, 0x19, 0xfd, 0x10,
			0x90, 0x16, 0x88, 0x3e, 0x4c, 0x67, 0xf9, 0x02,
			0xa0, 0x66, 0x82, 0x50, 0x21, 0xf3, 0xbd, 0x00,
			0xf0, 0xf1, 0xc6, 0xc1, 0x98, 0x75, 0x6a, 0x2c,
			0x81, 0x3d, 0xc7, 0x16, 0x15, 0x19, 0xa6, 0x8d,
			0x54, 0x49, 0x80, 0x26, 0x9a, 0x40, 0x17, 0x5c,
			0x38, 0x90, 0x4e, 0x21, 0x53, 0xa7, 0xba, 0x9f,
			0x53, 0x34, 0xf8, 0xb7, 0x3c, 0xd5, 0xf2, 0xc5,
			0xb8, 0xdd, 0xdb, 0xc5, 0x47, 0x7d, 0xb7, 0x44,
			0xc5, 0xc6, 0xf5, 0x4a, 0x68, 0x13, 0xf5, 0x09,
			0xdd, 0x7b, 0xb5, 0x9e, 0x1c, 0x4d, 0x44, 0x25,
			0x2b, 0x92, 0xee, 0xb6, 0x5b, 0xcf, 0xa5, 0x57,
			0xa9, 0xd8, 0x6e, 0x1b, 0x76, 0x07, 0xdd, 0xa3,
			0x5c, 0x6c, 0xb5, 0x63, 0x84, 0xf6, 0xb2, 0x62,
			0xf3, 0x9d, 0x77, 0x99, 0x6b, 0x0b, 0x6a, 0x53,
			0xd0, 0xe5, 0xd2, 0xb0, 0x13, 0x0a, 0x5b, 0xd3,
			0x0e, 0xf7, 0x82, 0x91, 0xa2, 0x53, 0x97, 0x12,
			0xf5, 0xbc, 0xb7, 0x5e, 0x04, 0xeb, 0xf7, 0xef,
			0x11, 0xd8, 0x06, 0xf2, 0x32, 0x2b, 0x2a, 0x40,
			0x3d, 0xad, 0x09, 0x66, 0x09, 0xcf, 0x3c, 0x34,
			0x28, 0x0b, 0x8d, 0x27, 0x72, 0xbd, 0x9c, 0xe5,
			0x80, 0xf7, 0xbc, 0x63, 0x2f, 0x4e, 0xd9, 0x5d,
			0xd5, 0x15, 0x24, 0xf7, 0xd3, 0xfb, 0xe5, 0x16,
			0xd2, 0x9b, 0x5f, 0xa3, 0x9c, 0xec, 0xb1, 0xcd,
			0x7f, 0xbb, 0xb4, 0x97, 0x61, 0xa3, 0x84, 0x55,
			0x77, 0x56, 0x79, 0x8a, 0x55, 0x68, 0x54, 0x0f,
			0x7e, 0xe4, 0x39, 0xa6, 0xbf, 0xe9, 0x3a, 0xde,
			0x41, 0x06, 0x3e, 0x1e, 0xa0, 0x06, 0x9b, 0x8d,
			0x5f, 0x05, 0xc8, 0xb6, 0x4a, 0xdf, 0xc2, 0xc3,
			0xc5, 0x67, 0xdf, 0x71, 0x9d, 0x88, 0x65, 0x7b,
			0xcf, 0x07, 0xa6, 0xc9, 0xb5, 0x7b, 0xf2, 0x8b,
			0x94, 0x76, 0x2a, 0xd1, 0x12, 0xa3, 0xf8, 0x34,
			0xcf, 0xfb, 0xdc, 0xf7, 0xbb, 0xb9, 0xff, 0x0d,
			0x7f, 0xc1, 0x6d, 0xa3, 0x61, 0xf1, 0x69, 0xa5,
			0x92, 0x5a, 0x0e, 0x70, 0x49, 0x99, 0x0a, 0x02,
			0x17, 0xd0, 0xa5, 0xe4, 0x63, 0x05, 0x34, 0x11,
			0x55, 0x42, 0x3c, 0xa0, 0xe4, 0xc0, 0x8b, 0xf9,
			0xa5, 0x74, 0x51, 0x42, 0xbb, 0xa7, 0x82, 0x70,
			0x7c, 0x6d, 0x37, 0xe4, 0xd5, 0xae, 0xd0, 0xeb,
			0xa2, 0x4f, 0x6b, 0x47, 0x2e, 0x50, 0x23, 0xf5,
			0xc6, 0x90, 0xd6, 0xe9, 0x66, 0xce, 0xba, 0x61,
			0x3c, 0xe6, 0x13, 0xb3, 0xb6, 0x48, 0x8f, 0x2e,
			0x28, 0x82, 0xdb, 0xa2, 0x54, 0x15, 0x66, 0x91,
			0x5a, 0x09, 0x16, 0x02, 0x98, 0xb5, 0x1d, 0xc5,
			0x1a, 0x87, 0x13, 0x55, 0x9d, 0xa5, 0x80, 0x61,
			0xa0, 0x9c, 0x9e, 0x14, 0x06, 0x91, 0x4f, 0x28,
			0x80, 0xb5, 0xe2, 0x9b, 0xdb, 0x89, 0x55, 0xe0,
			0x8b, 0x32, 0x02, 0xc0, 0x17, 0x52, 0x70, 0xbf,
			0x3e, 0x09, 0x18, 0xba, 0xa2, 0x53, 0xaa, 0x9e,
			0x74, 0x01, 0x2c, 0x1a, 0x83, 0x39, 0x70, 0x7c,
			0x0b, 0xff, 0xbb, 0xf3, 0x65, 0x8a, 0x06, 0x74,
			0x91, 0xef, 0xb4, 0xf1, 0xd6, 0xeb, 0xb2, 0xd7,
			0x0e, 0x4c, 0xcd, 0xf3, 0x99, 0xad, 0xf7, 0xe2,
			0x70, 0xc7, 0xa7, 0xbe, 0x32, 0xe5, 0x33, 0x9b,
			0x9f, 0xff, 0x2c, 0x53, 0x0f, 0x3d, 0xc4, 0x0f,
			0x78, 0xa4, 0x14, 0x05, 0xce, 0xcd, 0x5a, 0xe9,
			0x6f, 0x16, 0x68, 0x69, 0x30, 0x76, 0x9b, 0xdb,
			0x51, 0xa2, 0xaa, 0xd8, 0x01, 0x4a, 0xb9, 0xfb,
			0x49, 0x36, 0xd4, 0xbd, 0xd8, 0xb9, 0xfa, 0x87,
			0x2f, 0x7d, 0x18, 0x39, 0xeb, 0x7d, 0x65, 0xd2,
			0x20, 0x69, 0x8f, 0x3a, 0x0c, 0x5b, 0x25, 0x93,
			0xab, 0x4c, 0xc8, 0x0c, 0x12, 0x32, 0x19, 0xea,
			0x8e, 0x35, 0x4c, 0x02, 0x6f, 0xc8, 0x02, 0xaa,
			0x75, 0x0b, 0xe3, 0x32, 0x59, 0x67, 0x4f, 0x56,
			0x9e, 0xd6, 0x68, 0xa0, 0xc3, 0xd0, 0x06, 0x55,
			0x3c, 0x72, 0x0e, 0x63, 0xe2, 0xb1, 0xe3, 0xa3,
			0x7b, 0x19, 0xc6, 0x20, 0xa9, 0x53, 0xe8, 0x57,
			0xd4, 0xb0, 0x22, 0xd7, 0xb6, 0xad, 0x2f, 0x6f,
			0x36, 0x3c, 0xde, 0x2b, 0x3d, 0xe7, 0xf6, 0xc9,
			0x7d, 0x4c, 0xa6, 0x77, 0x62, 0x36, 0xe9, 0x95,
			0x40, 0xa0, 0x1d, 0x8d, 0x9b, 0xb7, 0x30, 0x11,
			0x2e, 0x5d, 0xc3, 0x67, 0xca, 0x49, 0x26, 0x94,
			0x44, 0x6c, 0x8b, 0x51, 0x4b, 0xb5, 0xc3, 0x96,
			0xb9, 0xf0, 0xfe, 0xc6, 0x58, 0xbc, 0xa1, 0x12,
			0xdc, 0x30, 0x2a, 0x9e, 0x78, 0xb1, 0xe7, 0x73,
			0xac, 0x8c, 0x04, 0x42, 0x31, 0x36, 0x6b, 0x6e,
			0x92, 0xf0, 0xb4, 0xd2, 0xd6, 0x78, 0xaf, 0x8f,
			0xbc, 0xac, 0x8b, 0x1e, 0x4e, 0x28, 0x14, 0x1b,
			0xcd, 0xf8, 0x86, 0x48, 0xb4, 0xd9, 0x47, 0x84,
			0x50, 0x18, 0x77, 0xe8, 0x02, 0x0a, 0xaf, 0x89,
			0xa6, 0x21, 0x81, 0xd1, 0x06, 0xa1, 0x37, 0xd1,
			0xb2, 0x27, 0x47, 0xc8, 0xcf, 0x5e, 0x5f, 0x31,
			0x13, 0xe0, 0x8b, 0xf4, 0x73, 0x45, 0x65, 0x93,
			0x49, 0x47, 0x94, 0xd1, 0xe2, 0x6e, 0xf9, 0x1d,
			0x18, 0x1c, 0xe6, 0x2f, 0x7d, 0xbd, 0x40, 0xff,
			0x71, 0x0e, 0xde, 0xe4, 0x7c, 0x2f, 0xc3, 0x9f,
			0x56, 0xdf, 0x58, 0x99, 0xa9, 0x45, 0x37, 0x72,
			0x15, 0x0f, 0x67, 0x8b, 0xf8, 0x84, 0x56, 0x6c,
			0x1d, 0x10, 0x4c, 0x6f, 0x4e, 0x61, 0x8e, 0xc2,
			0xc5, 0x33, 0xe6, 0x18, 0x1a, 0x81, 0xb5, 0x66,
			0xfa, 0xda, 0x25, 0x57, 0x0b, 0x6a, 0x3a, 0x20,
			0x2e, 0x1a, 0x15, 0xc9, 0x24, 0x82, 0x21, 0xfd,
			0xa2, 0x2f, 0x1b, 0x52, 0x50, 0x54, 0xbd, 0x17,
			0xc4, 0xea, 0x1b, 0x7c, 0xe6, 0x41, 0xe9, 0x91,
			0xdd, 0x23, 0xf2, 0x7d, 0x7d, 0xa5, 0x29, 0xd2,
			0x87, 0x38, 0x64, 0x28, 0x15, 0x44, 0x4c, 0x3a,
			0x7f, 0xce, 0xe3, 0xf7, 0xee, 0x1c, 0xb1, 0xac,
			0x44, 0xd5, 0xab, 0x18, 0x47, 0x94, 0x4e, 0xd1,
			0xad, 0xdd, 0x2f, 0xc9, 0x33, 0xfd, 0x56, 0x54,
			0x34, 0xfd, 0x13, 0x16, 0x7d, 0x2a, 0x4d, 0xd8,
			0xfd, 0xcb, 0x5e, 0x39, 0x69, 0xb6, 0xb9, 0x4b,
			0xb8, 0x60, 0xa2, 0x4c, 0xac, 0x85, 0x98, 0xf7,
			0x34, 0xb2, 0x0c, 0x3e, 0xfa, 0xbc, 0xd7, 0xdc,
			0xf0, 0xe2, 0xf0, 0x2d, 0x6e, 0x5d, 0xc8, 0xdf,
			0xb1, 0xe5, 0x9b, 0x7d, 0xee, 0x06, 0x5a, 0xb0,
			0xca, 0xa9, 0xd3, 0x19, 0x49, 0x98, 0x7f, 0xcc,
			0x66, 0x0b, 0xbb, 0xf6, 0x67, 0x2d, 0xfb, 0xe9,
			0x47, 0x95, 0x14, 0x75, 0xc9, 0x0d, 0x2c, 0xa6,
			0xba, 0x7e, 0x1c, 0xd9, 0x4a, 0xc2, 0x8e, 0x0a,
			0x03, 0x2f, 0x6b, 0xc8, 0x33, 0x4e, 0x77, 0xb8,
			0xe2, 0xea, 0xf1, 0x01, 0xc1, 0x8e, 0xb9, 0x10,
			0xc5, 0xfe, 0xf6, 0x58, 0xba, 0x0a, 0x4a, 0x57,
			0xb9, 0x2a, 0x8d, 0x45, 0x30, 0xd1, 0x70, 0x4d,
			0xd9, 0x5f, 0x9f, 0xed, 0x95, 0xdd, 0x2b, 0x3b,
			0x05, 0x9c, 0x7e, 0x39, 0xa1, 0xef, 0x3a, 0x0a,
			0x7a, 0x6f, 0x24, 0x31, 0x17, 0x6d, 0xa9, 0x06,
			0x5f, 0xf9, 0xd5, 0xcf, 0xe8, 0xbb, 0x53, 0x83,
			0x6a, 0x54, 0xff, 0xac, 0xad, 0xfc, 0xa6, 0x14,
			0x12, 0xc2, 0xb9, 0xa7, 0x8e, 0xf1, 0x2a, 0x92,
			0x53, 0x59, 0x14, 0x56, 0x10, 0x03, 0x24, 0xe2,
			0x9a, 0x4d, 0x5c, 0x1b, 0xbd, 0x52, 0x14, 0xb0,
			0x6b, 0x61, 0x79, 0x45, 0xf6, 0x9c, 0xdd, 0x25,
			0xdc, 0xcc, 0x3e, 0x97, 0x0c, 0xb0, 0x9d, 0xca,
			0xdb, 0x17, 0x13, 0x4b, 0xbb, 0x44, 0x75, 0x00,
			0xd7, 0xe2, 0x7c, 0xc8, 0x7d, 0xef, 0xa6, 0x30,
			0x82, 0x44, 0x0c, 0xec, 0x53, 0x12, 0xb1, 0xe0,
			0xba, 0x7f, 0xd7, 0xdf, 0x40, 0xf7, 0x8d, 0x00,
			0x0a, 0x1b, 0x7a, 0xa9, 0x87, 0xfd, 0x7a, 0xe6,
			0x61, 0x56, 0xe5, 0xe0, 0xcb, 0x37, 0xbd, 0x1c,
			0x29, 0xc6, 0xea, 0x07, 0xaa, 0xe0, 0x7d, 0x38,
			0x96, 0xee, 0x30, 0x63, 0x28, 0x1e, 0x2a, 0x4b,
			0x98, 0x4c, 0x8b, 0x4e, 0x88, 0xed, 0xe9, 0x19,
			0x5c, 0xa3, 0x8a, 0xbe, 0x27, 0x7c, 0x2e, 0x92,
			0x16, 0x60, 0x84, 0xb3, 0x53, 0x85, 0x32, 0xae,
			0xb6, 0x77, 0xb2, 0x55, 0x7c, 0xac, 0x98, 0x18,
			0x84, 0xf5, 0xd8, 0x9f, 0x21, 0x06, 0x5a, 0x81,
			0xe4, 0x33, 0x4d, 0x1e, 0xc4, 0x57, 0xfe, 0x06,
			0x73, 0x0a, 0x3a, 0x58, 0xc8, 0xf8, 0xd9, 0x5c,
			0xdb, 0x3b, 0x1f, 0x20, 0x8f, 0xbd, 0x22, 0x0e,
			0xc1, 0x4d, 0xf4, 0x50, 0xd5, 0x8e, 0x95, 0xb3,
			0x26, 0x61, 0xf8, 0xe5, 0x33, 0xc8, 0x10, 0x47,
			0x48, 0x93, 0x8a, 0x85, 0xf8, 0x35, 0x8c, 0x1a,
			0x28, 0xc5, 0x64, 0x6a, 0xa0, 0x3f, 0xe7, 0x66,
			0x38, 0xc6, 0xdf, 0x3f, 0xdf, 0xed, 0x5b, 0x8e,
			0x81, 0xdf, 0x20, 0xe4, 0xf8, 0x89, 0xc9, 0xe3,
			0x9e, 0xa6, 0x1d, 0x47, 0x49, 0x8a, 0x3e, 0x53,
			0xac, 0x18, 0x17, 0x6d, 0x80, 0xdf, 0x61, 0x23,
			0x2a, 0x0f, 0x31, 0x9e, 0x26, 0xb6, 0x4b, 0x27,
			0x77, 0x9e, 0x6c, 0x50, 0x94, 0xd1, 0x22, 0x6f,
			0x87, 0x95, 0x47, 0x27, 0x0e, 0xce, 0xa5, 0x3b,
			0x0c, 0xe2, 0x65, 0x8e, 0x12, 0x38, 0x22, 0x81,
			0x30, 0x52, 0xaf, 0x35, 0x97, 0x0b, 0x31, 0xe8,
			0x9a, 0x30, 0xfe, 0x17, 0x18, 0xae, 0x93, 0xbc,
			0xb7, 0x0a, 0xac, 0x6e, 0xd1, 0x07, 0x97, 0x71,
			0x5a, 0x90, 0x95, 0xf3, 0x63, 0x62, 0x09, 0xcb,
			0x76, 0x95, 0xe2, 0xc2, 0x21, 0xd1, 0x28, 0xb7,
			0xbc, 0xcf, 0x81, 0x13, 0x0b, 0x1b, 0xcb, 0x4f,
			0x46, 0x1b, 0x97, 0xb0, 0xe3, 0x39, 0x14, 0x93,
			0x6b, 0xf5, 0x19, 0x93, 0xef, 0x63, 0xb4, 0x74,
			0x33, 0x14, 0x9c, 0xbd, 0x94, 0xf5, 0x58, 0x2a,
			0x3d, 0x39, 0x0c, 0x04, 0x51, 0xcf, 0x88, 0xe6,
			0x94, 0xb5, 0xc3, 0x62, 0x97, 0xe6, 0x72, 0x60,
			0x1b, 0x12, 0x7c, 0xdd, 0x1e, 0xe0, 0x29, 0xfb,
			0xb8, 0xf5, 0xe1, 0x88, 0x1c, 0x35, 0x02, 0xda,
			0xe1, 0x48, 0x11, 0x2c, 0xbc, 0x72, 0x5f, 0x03,
			0x52, 0x7c, 0x60, 0xf7, 0xe6, 0x2f, 0x17, 0x46,
			0xf5, 0x48, 0xfa, 0x33, 0x96, 0x04, 0xad, 0xa7,
			0x31, 0xec, 0x7c, 0xb2, 0x56, 0x46, 0x81, 0x25,
			0xca, 0x40, 0x16, 0x30, 0x27, 0x2b, 0x6a, 0xe0,
			0xc0, 0xb9, 0xbe, 0xa8, 0x41, 0x5f, 0x8e, 0x57,
			0xc6, 0x2d, 0x68, 0xb8, 0xe0, 0xe3, 0xdb, 0xb9,
			0x70, 0xa6, 0x3d, 0x36, 0xe6, 0x25, 0x2d, 0x5b,
			0x92, 0x41, 0xdb, 0xfc, 0x8e, 0x39, 0x26, 0x68,
			0x1c, 0x82, 0xdd, 0xe3, 0xad, 0x3a, 0x38, 0x58,
			0x02, 0xef, 0x25, 0x92, 0xf1, 0x4b, 0xac, 0x66,
			0xd9, 0x68, 0xc8, 0x76, 0x80, 0x6e, 0x4d, 0x6d,
			0xb5, 0x75, 0x48, 0x80, 0x02, 0xd7, 0x70, 0x3f,
			0xb1, 0xf2, 0x6f, 0xce, 0x22, 0x61, 0xf3, 0x0c,
			0xac, 0xb2, 0xa9, 0x0c, 0x30, 0x3f, 0x6a, 0xcb,
			0x13, 0x37, 0x09, 0x3e, 0xe7, 0x8d, 0x87, 0x60,
			0xa4, 0x8d, 0x38, 0x77, 0x38, 0xdb, 0x07, 0x5e,
			0xc8, 0xe7, 0x83, 0xf6, 0xaf, 0x1d, 0xa5, 0x42,
			0x75, 0xf4, 0x5d, 0x53, 0xf6, 0x25, 0x14, 0x9d,
			0xa5, 0x18, 0x9c, 0x81, 0x9b, 0xdf, 0xc8, 0xb6,
			0x56, 0x87, 0xcd, 0xa0, 0xec, 0x9b, 0x3c, 0xf9,
			0xa9, 0xce, 0x42, 0xa3, 0xf3, 0x2d, 0x56, 0xb5,
			0xde, 0x24, 0x49, 0x5d, 0x9c, 0x80, 0x8f, 0xb1,
			0x4e, 0x65, 0x9f, 0x09, 0x21, 0x86, 0xbd, 0x29,
			0xc3, 0xca, 0xf0, 0x8c, 0x21, 0x2b, 0x04, 0x74,
			0x8c, 0xa9, 0x9f, 0x43, 0x42, 0x4a, 0xbf, 0x3c,
			0x68, 0x77, 0x35, 0xaa, 0x00, 0x44, 0xb0, 0x10,
			0x1c, 0x30, 0x90, 0x5b, 0x41, 0xc2, 0xfd, 0xbd,
			0x3f, 0xe5, 0x69, 0x5c, 0xfe, 0x7a, 0x7a, 0x28,
			0x60, 0xc4, 0x6f, 0x9e, 0x28, 0x2f, 0xc0, 0x6a,
			0x0d, 0x04, 0x22, 0x95, 0xbb, 0x68, 0xdb, 0x4c,
			0xbf, 0x2f, 0xfc, 0xf2, 0x65, 0x8f, 0x33, 0xbf,
			0xf9, 0x81, 0x8a, 0x4e, 0xb8, 0xe8, 0x1a, 0x06,
			0x6e, 0x2f, 0x52, 0x96, 0x12, 0x19, 0x30, 0x60,
			0xb8, 0x6a, 0x6c, 0xec, 0x00, 0xa3, 0xf5, 0x19,
			0xad, 0x0b, 0xeb, 0xae, 0x0b, 0xfd, 0xbb, 0x05,
			0xdd, 0x71, 0x74, 0x52, 0x1e, 0x7d, 0xd5, 0xa8,
			0xc1, 0x5f, 0x76, 0x46, 0x66, 0x94, 0x87, 0x8b,
			0xa6, 0x7b, 0xf9, 0xbb, 0xc0, 0x08, 0xc5, 0x38,
			0x4f, 0x21, 0xea, 0xa0, 0x5a, 0xd6, 0x41, 0x0b,
			0xa4, 0x43, 0x5f, 0xda, 0x17, 0xa7, 0x88, 0x2b,
			0xe5, 0xd1, 0x1f, 0xdf, 0x86, 0x3f, 0x09, 0x50,
			0xbd, 0xd7, 0xbd, 0x7e, 0x73, 0x2c, 0x55, 0x80,
			0x1f, 0xf7, 0x55, 0x43, 0x63, 0x73, 0x55, 0x66,
			0xd8, 0xf9, 0x44, 0x0d, 0xa1, 0x35, 0xca, 0x3f,
			0xf4, 0xd0, 0x5b, 0xa2, 0xcc, 0x9b, 0x4e, 0x98,
			0x36, 0xbe, 0xc6, 0x03, 0x14, 0x2a, 0x39, 0x44,
			0x66, 0xdb, 0xb1, 0x30, 0x57, 0x0f, 0xe3, 0x6b,
			0x6d, 0x97, 0xe4, 0xe7, 0x21, 0xa9, 0xae, 0x9c,
			0x97, 0x4f, 0xe7, 0x2b, 0x2d, 0x88, 0xd7, 0x14,
			0x5d, 0x1a, 0x21, 0x10, 0x36, 0x59, 0x6b, 0x6a,
			0x25, 0x34, 0xfa, 0x54, 0x3b, 0x8f, 0xc6, 0xdd,
			0xe8, 0xa8, 0x30, 0x3b, 0xb1, 0xab, 0x11, 0x6a,
			0x9e, 0x67, 0xcb, 0xa0, 0xd3, 0x4a, 0x82, 0x30,
			0xcf, 0xf8, 0x9d, 0x12, 0x4d, 0xd4, 0x5d, 0x9c,
			0x09, 0xee, 0x4c, 0x26, 0xfb, 0x87, 0x3a, 0xd7,
			0xf0, 0x4d, 0xe2, 0xae, 0x98, 0x95, 0xce, 0xcc,
			0x70, 0x1c, 0xd5, 0xf3, 0x4f, 0x2a, 0x4c, 0x47,
			0x60, 0x3b, 0xa7, 0x75, 0x56, 0xa2, 0x47, 0x31,
			0x82, 0x7f, 0x4a, 0x80, 0x0c, 0xab, 0xd0, 0x97,
			0xd7, 0xe3, 0x78, 0xf2, 0x6b, 0x83, 0xb6, 0x2f,
			0x9b, 0x1e, 0xb1, 0x64, 0xf1, 0x19, 0x33, 0x12,
			0x5d, 0x94, 0xac, 0x6c, 0xa3, 0xf8, 0x66, 0xb0,
			0x95, 0x5f, 0x67, 0xb4, 0x15, 0x2f, 0x90, 0x58,
			0x29, 0x48, 0x86, 0xc5, 0x9d, 0xf0, 0xa4, 0x86,
			0xb7, 0xe2, 0x6e, 0xdf, 0x3c, 0x7a, 0x6f, 0x7c,
			0x4f, 0xe8, 0x95, 0x13, 0x81, 0x7b, 0xb3, 0xba,
			0xd3, 0x05, 0x82, 0x26, 0xf0, 0xb9, 0xd7, 0x30,
			0x54, 0xb4, 0xbf, 0x6a, 0x68, 0xbf, 0x37, 0x82,
			0xa9, 0x48, 0x2b, 0x3a, 0x00, 0x9d, 0xab, 0x6c,
			0x4f, 0x18, 0xf1, 0xed, 0xab, 0x36, 0x40, 0xac,
			0x58, 0x1e, 0xf3, 0xf7, 0x4d, 0xde, 0xc4, 0x98,
			0x6e, 0x8d, 0xf4, 0xa9, 0x04, 0xd4, 0x1f, 0x9d,
			0x9b, 0x92, 0x6b, 0xbf, 0x44, 0x1f, 0x4c, 0xec,
			0x7e, 0xcb, 0x4d, 0x65, 0xf5, 0x4c, 0x37, 0x5c,
			0x84, 0xd8, 0x07, 0x87, 0x1b, 0xa8, 0xa3, 0xa2,
			0x31, 0xf8, 0x63, 0x31, 0xd5, 0x82, 0x75, 0x4d,
			0x80, 0x73, 0xe7, 0x5a, 0x59, 0xd7, 0x0d, 0x59,
			0x6b, 0xbb, 0x85, 0x51, 0xb9, 0xbe, 0x7e, 0x61,
			0xa1, 0xc7, 0xb3, 0x52, 0xdd, 0x36, 0x3a, 0xbe,
			0xfe, 0xcd, 0xeb, 0xe1, 0x6b, 0x4d, 0x3b, 0x8a,
			0x36, 0x38, 0x67, 0x27, 0xf4, 0x50, 0x90, 0x97,
			0xfe, 0x93, 0x6d, 0xd3, 0x7e, 0xb7, 0xfc, 0x8f,
			0x7b, 0xbd, 0xb1, 0x3d, 0x33, 0x0c, 0x51, 0x86,
			0x5a, 0x48, 0x0c, 0x19, 0xf3, 0x6c, 0x60, 0x03,
			0x92, 0xcb, 0xbc, 0x0a, 0x17, 0x0a, 0x16, 0xff,
			0x73, 0x68, 0x04, 0xa8, 0xf0, 0xc9, 0x97, 0x1d,
			0x10, 0x4e, 0xc3, 0x1a, 0x30, 0x22, 0x58, 0x1a,
			0x34, 0xca, 0x7f, 0x6c, 0xec, 0xda, 0xb3, 0xb3,
			0xe4, 0x48, 0x6b, 0x3c, 0x3f, 0xd9, 0xcc, 0xde,
			0x5f, 0xf6, 0x77, 0xd1, 0xea, 0xd0, 0x6e, 0xb6,
			0x43, 0xf6, 0x92, 0x48, 0xef, 0x78, 0x79, 0xdb,
			0xa4, 0xce, 0x70, 0x4a, 0xbb, 0xe1, 0xba, 0xf5,
			0x0e, 0x3b, 0x1f, 0x1f, 0x2e, 0x27, 0x5a, 0x38,
			0x63, 0x40, 0x9f, 0x66, 0x46, 0x90, 0x05, 0x2e,
			0xbf, 0xaa, 0xab, 0xcc, 0x5c, 0xdf, 0x19, 0x8e,
			0xb3, 0x82, 0x6c, 0xf8, 0xf6, 0xf6, 0x79, 0xf4,
			0x9a, 0x97, 0x95, 0xf0, 0x70, 0x07, 0xf7, 0xc5,
			0x06, 0xd0, 0x9c, 0xc9, 0x03, 0x31, 0x5e, 0x2a,
			0xf5, 0x2b, 0x90, 0xd4, 0xdd, 0x9c, 0x79, 0xff,
			0x6d, 0x31, 0x11, 0xbb, 0xab, 0x74, 0x0e, 0x69,
			0xdf, 0x0a, 0xbb, 0x09, 0x95, 0xae, 0xc3, 0x63,
			0x07, 0x97, 0x19, 0x99, 0x33, 0x8c, 0x6b, 0x85,
			0x2b, 0x1f, 0x90, 0xf3, 0x34, 0x7d, 0xff, 0x0c,
			0x0f, 0x2b, 0x9c, 0x7c, 0x7b, 0x8c, 0x2b, 0xd0,
			0x18, 0x4c, 0xd4, 0x98, 0xf8, 0x06, 0x14, 0x0b,
			0x75, 0xf1, 0x34, 0x0c, 0x3d, 0xc7, 0x87, 0x34,
			0x88, 0x92, 0xe7, 0xaa, 0x6f, 0x91, 0xab, 0x8d,
			0x85, 0xc6, 0x03, 0xef, 0xf7, 0xc8, 0xec, 0x33,
			0xb4, 0x31, 0x10, 0x04, 0xaa, 0x81, 0x3a, 0x93,
			0x5b, 0x31, 0xa9, 0x29, 0xb7, 0x60, 0x63, 0x25,
			0x55, 0x16, 0xd9, 0xe2, 0xde, 0x7f, 0x3a, 0x9b,
			0x54, 0xc0, 0x24, 0x23, 0x8c, 0x9b, 0xca, 0x1e,
			0x17, 0xc3, 0x9d, 0x68, 0x74, 0xf0, 0x00, 0xb8,
			0xa3, 0xec, 0xbf, 0xf4, 0x00, 0x84, 0xf4, 0x84,
			0x21, 0x26, 0xda, 0x1c, 0xcd, 0xfa, 0x93, 0x65,
			0xd5, 0x53, 0xd6, 0x7f, 0x1b, 0x46, 0x5e, 0x94,
			0x3f, 0xfc, 0xd4, 0x33, 0x73, 0x3b, 0xc6, 0x96,
			0x95, 0x91, 0xc0, 0x80, 0x0e, 0x4d, 0x00, 0x30,
			0xa2, 0xb9, 0x21, 0xbe, 0x43, 0x5f, 0x46, 0xf2,
			0x6a, 0xfc, 0xf6, 0xdc, 0x1e, 0x2b, 0xa3, 0xed,
			0xa2, 0x19, 0x7e, 0xe7, 0x2a, 0x3e, 0xa0, 0xbe,
			0x0e, 0xab, 0xd8, 0xae, 0xd8, 0xd1, 0xd8, 0x49,
			0xa6, 0x26, 0xf7, 0xf3, 0x62, 0xe3, 0x8f, 0xd9,
			0x94, 0x06, 0xa3, 0x82, 0x17, 0x70, 0x4b, 0x00,
			0xf7, 0xa7, 0xe6, 0x3d, 0xe8, 0x8d, 0x2a, 0xec,
			0xc1, 0x0c, 0xcf, 0x50, 0x79, 0x43, 0x67, 0x07,
			0xbb, 0xaa, 0xda, 0x7d, 0x2f, 0xdd, 0x9f, 0x5c,
			0x2c, 0xf5, 0x94, 0xa4, 0xc2, 0x57, 0x15, 0x74,
			0xaf, 0x30, 0x6e, 0xef, 0xc7, 0x70, 0x6a, 0x20,
			0x1c, 0x43, 0x3d, 0x22, 0xc4, 0x64, 0x1d, 0x9c,
			0xdf, 0xd6, 0x86, 0xf7, 0xa3, 0x16, 0xc5, 0x92,
			0xa5, 0x35, 0x2d, 0xe0, 0x7a, 0x1a, 0xaa, 0xb1,
			0x43, 0x15, 0x17, 0xb5, 0x51, 0xbe, 0x55, 0x8b,
			0x82, 0x90, 0x01, 0x2f, 0x83, 0xd2, 0xaa, 0xe3,
			0x42, 0x8d, 0x95, 0x7c, 0x28, 0x23, 0x3a, 0xe5,
			0x08, 0x8f, 0xb3, 0x41, 0xa3, 0x0b, 0x1f, 0x47,
			0x3b, 0xc7, 0xe9, 0xcd, 0x8c, 0x98, 0x1c, 0xaf,
			0x61, 0x99, 0xc1, 0x38, 0x8e, 0x67, 0x8a, 0x3d,
			0x3d, 0xc3, 0x6b, 0xc5, 0x91, 0xc0, 0x25, 0xab,
			0x5e, 0x11, 0x05, 0xa0, 0x01, 0x3b, 0xcf, 0x12,
			0x7b, 0x78, 0xc9, 0x97, 0x90, 0x2a, 0x89, 0xee,
			0x70, 0x3b, 0xde, 0x64, 0xcf, 0x50, 0x41, 0x61,
			0x7b, 0x5c, 0x16, 0xdf, 0x95, 0xaf, 0x84, 0x98,
			0x4c, 0x52, 0x7e, 0x0d, 0x43, 0x83, 0xc5, 0x31,
			0xf1, 0x8a, 0xde, 0xe4, 0x0a, 0xfa, 0x8a, 0x5b,
			0x41, 0x0c, 0x94, 0xdb, 0x71, 0x79, 0xd8, 0xef,
			0xfb, 0x04, 0xa5, 0x7e, 0x8f, 0xed, 0xa9, 0x6a,
			0x2f, 0x55, 0xfb, 0xf3, 0xea, 0x7c, 0xd7, 0x14,
			0xbd, 0x4e, 0x6c, 0x2c, 0x1a, 0x0a, 0x4d, 0xdd,
			0x37, 0x1a, 0x39, 0xa5, 0x1b, 0x39, 0xf0, 0xaf,
			0xec, 0xf6, 0x82, 0xf1, 0x5e, 0x1c, 0x9a, 0xfb,
			0x3b, 0x7b, 0xe5, 0x93, 0xb2, 0xd8, 0xde, 0x16,
			0x0b, 0x56, 0x5b, 0x7d, 0x0e, 0xc9, 0xe4, 0x41,
			0x61, 0xa9, 0xe8, 0xf9, 0x62, 0x47, 0xd5, 0x8a,
			0xe6, 0x21, 0xaf, 0xb3, 0x6e, 0x14, 0x75, 0xed,
			0xf7, 0x18, 0x59, 0xcc, 0x47, 0x07, 0x3e, 0x66,
			0x55, 0x4c, 0xbe, 0x8e, 0x45, 0x6b, 0xeb, 0x6d,
			0x25, 0xc8, 0x29, 0xe6, 0x4f, 0x1d, 0x8b, 0x8f,
			0x1c, 0x50, 0x7c, 0x25, 0xda, 0xbb, 0xc1, 0x52,
			0x84, 0x5c, 0xad, 0x3b, 0x0e, 0xc7, 0xe0, 0xe3,
			0x91, 0xe5, 0x8f, 0x56, 0x1c, 0x68, 0x2b, 0xa7,
			0xcd, 0x87, 0xf8, 0xb4, 0x95, 0x8b, 0xfa, 0x35,
			0xca, 0x59, 0x4e, 0x55, 0x24, 0xbd, 0x21, 0xf7,
			0x60, 0x66, 0xa8, 0x64, 0x53, 0xa7, 0x1a, 0xda,
			0xa0, 0x0a, 0xf5, 0x37, 0xad, 0x04, 0x44, 0xde,
			0x54, 0x0f, 0x16, 0x60, 0xcd, 0x5a, 0x65, 0xd2,
			0xe7, 0xd7, 0x4b, 0xf3, 0x29, 0x5b, 0xfa, 0xbf,
			0xfc, 0x4f, 0x68, 0xd8, 0x16, 0xea, 0x12, 0x68,
			0xb0, 0x30, 0x55, 0xb0, 0x69, 0x52, 0x92, 0xec,
			0x58, 0x0e, 0x47, 0x8d, 0x42, 0x81, 0xab, 0xb8,
			0xdc, 0x22, 0x28, 0x7a, 0xb0, 0x2d, 0xb4, 0x73,
			0xab, 0xa2, 0x33, 0x12, 0x87, 0x29, 0xd2, 0x3e,
			0x56, 0x8d, 0x9a, 0x0f, 0x34, 0x4b, 0x4b, 0xfa,
			0x64, 0x5c, 0xe6, 0xca, 0xa3, 0x71, 0x70, 0x96,
			0x97, 0x62, 0xca, 0x22, 0x50, 0x63, 0xd5, 0x3e,
			0x14, 0xf5, 0x89, 0x06, 0x09, 0xef, 0x34, 0xc0,
			0xed, 0x6d, 0x6b, 0xd0, 0x0b, 0x93, 0x1e, 0x35,
			0x06, 0xac, 0x68, 0x8f, 0x6a, 0x25, 0x48, 0xdf,
			0x1d, 0x0e, 0x73, 0x0f, 0x77, 0xbd, 0xa5, 0xb1,
			0x77, 0xd3, 0xdc, 0xd6, 0x2e, 0x4d, 0x74, 0xc5,
			0x43, 0x10, 0x28, 0xba, 0xdd, 0x6d, 0x82, 0x5d,
			0x6d, 0xdd, 0x78, 0xb6, 0x68, 0x62, 0x4b, 0x3f,
			0xd9, 0xfc, 0xcc, 0x9e, 0x6e, 0x7e, 0x9d, 0x2b,
			0x0a, 0xc8, 0xbf, 0xfb, 0x13, 0xba, 0xce, 0x7f,
			0x41, 0x3c, 0x1f, 0x5a, 0x87, 0x00, 0x36, 0x0a,
			0xba, 0xbc, 0x4a, 0x80, 0x6f, 0x6c, 0x9b, 0x9a,
			0xbc, 0x50, 0x62, 0x17, 0x62, 0xcf, 0xf5, 0x25,
			0x15, 0xc1, 0xfd, 0xe2, 0x8b, 0xf2, 0xec, 0xf3,
			0x08, 0xee, 0x25, 0x9b, 0x89, 0xed, 0x73, 0xbc,
			0x0d, 0xfe, 0xc4, 0x45, 0x28, 0x6f, 0x64, 0xc4,
			0x15, 0x7a, 0xa7, 0x05, 0xee, 0x0b, 0x3b, 0x10,
			0x54, 0x9a, 0x61, 0x51, 0x5f, 0xb5, 0x02, 0x4e,
			0x54, 0xef, 0x3e, 0xf7, 0x9c, 0x8e, 0x93, 0xb8,
			0xf7, 0xc7, 0x05, 0xd7, 0x8d, 0x82, 0xdf, 0x03,
			0x63, 0x73, 0x2c, 0xe1, 0xb8, 0xa4, 0x25, 0x86,
			0xc9, 0x79, 0xe6, 0xd3, 0x18, 0xbd, 0x1c, 0x46,
			0x55, 0xb9, 0x5a, 0x91, 0x90, 0x67, 0x7d, 0xda,
			0xae, 0xb8, 0x97, 0xa0, 0x0e, 0x7b, 0xd8, 0x14,
			0x98, 0x32, 0x61, 0x37, 0x80, 0xfe, 0x65, 0xce,
			0xdf, 0x3a, 0x06, 0x3b, 0x28, 0xfd, 0xad, 0xc6,
			0x6b, 0xb0, 0xb3, 0xae, 0x96, 0x76, 0xb6, 0x52,
			0xa4, 0xa1, 0x76, 0xa4, 0xc5, 0x06, 0xc9, 0x7b,
			0x16, 0xc3, 0x55, 0x0a, 0xae, 0x6a, 0x5c, 0xac,
			0xcb, 0x08, 0xf2, 0x8c, 0x26, 0x20, 0x09, 0x43,
			0xe0, 0xb7, 0x9f, 0xc4, 0x0f, 0x2f, 0x5f, 0xb5,
			0x29, 0x78, 0x7c, 0xf1, 0x7f, 0xfa, 0xb1, 0xfa,
			0x9b, 0x71, 0xdd, 0x1f, 0x0f, 0x1c, 0xd4, 0xf4,
			0xb5, 0x60, 0x7a, 0x2c, 0xe8, 0xcb, 0x33, 0xd6,
			0xf9, 0x9c, 0x34, 0x94, 0x7c, 0xf6, 0xd3, 0xee,
			0x2f, 0xeb, 0xe6, 0xa7, 0x7e, 0x4a, 0x27, 0xf7,
			0xf1, 0x98, 0xfc, 0x51, 0xff, 0x02, 0x5c, 0x1d,
			0xeb, 0xd4, 0xea, 0xa8, 0x08, 0x48, 0xb8, 0x95,
			0x7f, 0xc3, 0x84, 0x44, 0x98, 0x42, 0x16, 0x54,
			0xfd, 0x46, 0xd9, 0x27, 0x71, 0xba, 0x36, 0xd0,
			0x72, 0x6c, 0x64, 0x9b, 0x32, 0x61, 0x80, 0xbc,
			0x35, 0x8d, 0xe1, 0x4b, 0x2b, 0x84, 0x05, 0x51,
			0xcc, 0x7c, 0xf3, 0xea, 0xf7, 0xc5, 0xf4, 0xca,
			0x68, 0x02, 0x6e, 0x82, 0x81, 0x14, 0xc9, 0x73,
			0xd0, 0xb5, 0x4b, 0x2e, 0x0c, 0x9c, 0x50, 0x50,
			0xdd, 0xc9, 0xaf, 0x5d, 0x31, 0x9c, 0xab, 0xfa,
			0xed, 0x97, 0x29, 0x58, 0xcc, 0x97, 0x85, 0x30,
			0xf2, 0x60, 0x48, 0xe3, 0xba, 0x39, 0xeb, 0x51,
			0xb3, 0x0b, 0x76, 0x94, 0xae, 0xf4, 0x35, 0xdc,
			0x3a, 0xf7, 0xcb, 0x38, 0x83, 0x3c, 0xcc, 0xff,
			0xba, 0x6b, 0x49, 0x4a, 0x5e, 0x03, 0xf7, 0xe2,
			0xb2, 0x79, 0xa5, 0x91, 0x9a, 0xae, 0x39, 0x0b,
			0x4c, 0x1a, 0xc4, 0xa8, 0xab, 0xd8, 0xac, 0x2d,
			0x1f, 0x18, 0x2f, 0xdb, 0x08, 0x25, 0x63, 0x7c,
			0x39, 0xa0, 0x69, 0x25, 0x02, 0x29, 0x34, 0xb5,
			0xee, 0xd9, 0xbd, 0xd4, 0xf9, 0xe7, 0xcb, 0x64,
			0xb0, 0x49, 0x2b, 0x04, 0x7b, 0xae, 0x69, 0x7e,
			0x04, 0x82, 0xda, 0x19, 0x7a, 0x8b, 0x51, 0x4c,
			0xf1, 0x1e, 0x6f, 0xf3, 0x84, 0x70, 0x67, 0x99,
			0xd7, 0x20, 0x93, 0xdb, 0x1b, 0x80, 0x81, 0x62,
			0xe0, 0xec, 0x58, 0xdd, 0xc7, 0xd2, 0x52, 0x43,
			0xb2, 0x75, 0x8c, 0xc7, 0x13, 0x0d, 0x24, 0xc9,
			0xd5, 0x91, 0x3b, 0x68, 0x44, 0x0e, 0x16, 0xe5,
			0xe8, 0xc8, 0x3c, 0x38, 0x9f, 0x3f, 0x46, 0xe3,
			0x51, 0x09, 0x84, 0x52, 0xcc, 0x27, 0xfd, 0x49,
			0x7c, 0x26, 0xa5, 0x47, 0xe6, 0x97, 0x95, 0xed,
			0x96, 0x1b, 0x78, 0x45, 0x62, 0x6b, 0xbb, 0xc9,
			0x63, 0x78, 0xd6, 0x07, 0xae, 0xa8, 0x5e, 0x9c,
			0xb2, 0x98, 0xa1, 0xc5, 0xb3, 0xcf, 0x8b, 0x84,
			0x81, 0x6d, 0xc8, 0x2d, 0x87, 0xec, 0x16, 0x7a,
			0xfb, 0xd2, 0x67, 0x3e, 0xf4, 0x8d, 0xab, 0xdd,
			0x1b, 0x40, 0x3c, 0xe0, 0xa8, 0x3d, 0x30, 0xc4,
			0xc6, 0xf2, 0x19, 0xa3, 0xd4, 0xe2, 0x7c, 0x1b,
			0x5a, 0x8f, 0x5e, 0x39, 0x64, 0x56, 0x21, 0xf9,
			0x9d, 0x9a, 0x14, 0x35, 0x81, 0xba, 0x00, 0xb9,
			0x60, 0x7f, 0xc3, 0x0f, 0xaf, 0xda, 0x10, 0x2e,
			0xe0, 0x90, 0x98, 0xc6, 0xc8, 0x10, 0x42, 0x5b,
			0x06, 0x3b, 0xf4, 0x84, 0x26, 0xbe, 0xe9, 0x25,
			0x29, 0x7d, 0x80, 0x70, 0x21, 0x46, 0x1b, 0x0f,
			0x7c, 0x25, 0xa3, 0x1a, 0x9b, 0x31, 0x45, 0x6c,
			0x1f, 0xdb, 0x04, 0x57, 0x08, 0x0c, 0x2e, 0xc3,
			0x91, 0xdd, 0x09, 0x4c, 0xb2, 0x09, 0x52, 0x63,
			0xc5, 0xb9, 0x2a, 0x70, 0xd3, 0x42, 0x97, 0xc5,
			0x60, 0x65, 0x6c, 0xfd, 0xb2, 0x6c, 0x0d, 0xf5,
			0x26, 0x0e, 0xfa, 0xa5, 0x56, 0xb0, 0x6a, 0x59,
			0xb1, 0x6e, 0xab, 0x35, 0x8d, 0x3d, 0x4c, 0xa7,
			0xbe, 0x03, 0x79, 0x84, 0x63, 0x9c, 0xb6, 0x9e,
			0x95, 0x9a, 0x3f, 0xfe, 0xac, 0xe4, 0x66, 0xf5,
			0x84, 0x4a, 0x99, 0x2b, 0x1b, 0xfa, 0x6f, 0x90,
			0x52, 0x4d, 0x5f, 0xc3, 0xab, 0xb0, 0xb3, 0x19,
			0x7d, 0x77, 0x8c, 0x9f, 0xc7, 0xc2, 0xc9, 0xe6,
			0xd5, 0xb8, 0xe0, 0x9c, 0xef, 0x03, 0x1c, 0xed,
			0x20, 0x13, 0xf3, 0x29, 0x52, 0x0c, 0x9d, 0x39,
			0xad, 0xba, 0x25, 0xb5, 0x76, 0xc0, 0x79, 0x83,
			0x41, 0x14, 0xf7, 0xf9, 0x06, 0x15, 0x59, 0xb9,
			0x33, 0xf2, 0x7c, 0x54, 0x86, 0xe0, 0x5d, 0xe8,
			0xce, 0x75, 0x5d, 0xa8, 0x21, 0x30, 0xe3, 0x00,
			0xe7, 0x44, 0x25, 0x19, 0x02, 0x74, 0x12, 0x0b,
			0xa0, 0x68, 0x0e, 0x0a, 0x09, 0xe1, 0xff, 0x64,
			0x74, 0x90, 0x98, 0x8f, 0x1d, 0x6c, 0x7a, 0x04,
			0x9d, 0xc9, 0x2c, 0x23, 0xb7, 0xd7, 0x57, 0xe6,
			0x60, 0xd7, 0x0a, 0xdb, 0x7f, 0xa8, 0x8e, 0xe0,
			0x52, 0x3a, 0x26, 0x46, 0x02, 0xe3, 0x8c, 0x46,
			0x34, 0x5b, 0xc7, 0xd7, 0x2f, 0xc3, 0xfd, 0x1f,
			0xbc, 0x75, 0x9a, 0xf7, 0x52, 0x00, 0x01, 0x5a,
			0x3c, 0xaa, 0x98, 0x58, 0xa2, 0xe0, 0xca, 0xc7,
			0x0f, 0xc2, 0x0b, 0xd0, 0xb2, 0x09, 0x7a, 0x67,
			0x34, 0xdb, 0x82, 0x6e, 0xeb, 0x76, 0x61, 0xee,
			0x5b, 0xa9, 0xbe, 0xd9, 0x8b, 0xd5, 0x4a, 0x64,
			0xbd, 0x52, 0x65, 0xdc, 0xad, 0xdd, 0x41, 0x2f,
			0x72, 0x7d, 0x9f, 0xe5, 0xc4, 0x0e, 0x07, 0x3e,
			0x68, 0xf7, 0x5e, 0xf1, 0xe0, 0x25, 0x99, 0x6a,
			0x22, 0x60, 0x57, 0x57, 0x51, 0x86, 0x44, 0xe6,
			0xc0, 0x10, 0xfa, 0xf5, 0x66, 0x82, 0xae, 0x9a,
			0xb5, 0x46, 0x67, 0xae, 0x54, 0xa9, 0x56, 0xe2,
			0x5b, 0x8a, 0xd9, 0xed, 0xe9, 0x16, 0x5c, 0xfd,
			0x4b, 0xbd, 0xd3, 0x07, 0x97, 0x6b, 0x97, 0x9e,
			0xe9, 0xf2, 0xd2, 0xa3, 0x9b, 0x43, 0x6e, 0x78,
			0x3d, 0xf7, 0x22, 0x14, 0x6a, 0x9b, 0x45, 0x2c,
			0x27, 0x40, 0x04, 0x49, 0x81, 0x19, 0x66, 0x56,
			0x27, 0xae, 0x87, 0x42, 0x08, 0x31, 0xad, 0xf9,
			0x38, 0xcb, 0x68, 0x5f, 0xf6, 0x7a, 0x88, 0x0f,
			0x21, 0x7b, 0x99, 0xba, 0x9d, 0xf3, 0x89, 0xc4,
			0x18, 0x5d, 0x51, 0x5d, 0x62, 0x9b, 0x90, 0xb1,
			0x1d, 0x47, 0x88, 0xa3, 0x32, 0xf6, 0xfe, 0x3f,
			0x9f, 0xff, 0x09, 0xdc, 0xd3, 0xe9, 0xce, 0x73,
			0x3b, 0x53, 0x1c, 0xef, 0x42, 0x25, 0xc6, 0xdf,
			0xef, 0xe1, 0x10, 0xe5, 0xdd, 0x99, 0x9c, 0x48,
			0x6d, 0xc7, 0x4d, 0x7d, 0xeb, 0xb5, 0xeb, 0x2b,
			0xcd, 0x97, 0xd1, 0x2c, 0xb4, 0xc1, 0x14, 0x19,
			0x13, 0x44, 0xa6, 0x7e, 0xb1, 0xd3, 0x9e, 0x67,
			0x97, 0x50, 0xda, 0x0b, 0x81, 0x19, 0x8c, 0x17,
			0xf2, 0xd8, 0xac, 0xc2, 0x17, 0xa2, 0xd4, 0xd8,
			0x56, 0x82, 0x2f, 0x34, 0xd4, 0x44, 0x26, 0x63,
			0x3c, 0xde, 0xec, 0xb3, 0x45, 0xf3, 0xdb, 0x4b,
			0xaf, 0x58, 0x3c, 0x9d, 0xe4, 0x6e, 0xa3, 0x6a,
			0x69, 0x46, 0xb7, 0x95, 0x7b, 0x69, 0xd3, 0x16,
			0x43, 0x83, 0x7e, 0xb0, 0x87, 0x07, 0x84, 0x75,
			0xae, 0x1f, 0x6d, 0x7c, 0x8e, 0x70, 0x20, 0x0c,
			0x8e, 0x15, 0x30, 0x94, 0xd2, 0x28, 0x38, 0x69,
			0x24, 0xc0, 0xda, 0xcb, 0x38, 0x1d, 0x64, 0x72,
			0x5d, 0xb4, 0x11, 0x29, 0xdd, 0xc9, 0xfe, 0x4e,
			0x90, 0x33, 0x43, 0x08, 0xdf, 0x9f, 0xce, 0xe1,
			0x22, 0x4b, 0xf0, 0x25, 0xbe, 0xf8, 0x16, 0x79,
			0x76, 0xcc, 0xa5, 0xc9, 0x76, 0x24, 0x8d, 0x1e,
			0xa4, 0xa7, 0xf8, 0x7e, 0x61, 0xaf, 0xc4, 0x59,
			0xc4, 0x6d, 0x4a, 0x8f, 0x1c, 0x75, 0xee, 0x1d,
			0x2c, 0x14, 0x1c, 0x68, 0xe3, 0x12, 0xfd, 0x09,
			0x6f, 0xa7, 0x8d, 0x45, 0x2b, 0xdd, 0x4e, 0x7f,
			0xf5, 0x80, 0xa7, 0x5f, 0xd3, 0xbe, 0x76, 0x04,
			0x36, 0x3a, 0x42, 0xc9, 0x5d, 0xca, 0x02, 0x00,
			0x29, 0x1a, 0x74, 0x95, 0xdf, 0xd5, 0x1a, 0xba,
			0xb1, 0x56, 0xc8, 0xfd, 0x52, 0x00, 0x67, 0x93,
			0x0a, 0x09, 0xe4, 0x34, 0x0c, 0x8c, 0x6b, 0xa9,
			0x06, 0xe0, 0xf8, 0xd3, 0x10, 0x94, 0x1a, 0x46,
			0xda, 0xf5, 0x38, 0x43, 0xb9, 0xd6, 0xd1, 0xbb,
			0x68, 0xcd, 0xbd, 0x2f, 0x5b, 0x44, 0x53, 0x06,
			0x63, 0x7b, 0xba, 0xbd, 0xcf, 0x92, 0x10, 0xbd,
			0xc2, 0x6f, 0x72, 0x94, 0xd6, 0x9b, 0xd0, 0xc8,
			0x85, 0x2d, 0xae, 0x8c, 0xa3, 0x88, 0x9b, 0x76,
			0x39, 0xfe, 0x37, 0x9e, 0xb5, 0xbf, 0xed, 0x47,
			0xe5, 0x93, 0x2d, 0xc9, 0x08, 0x72, 0x49, 0x87,
			0xaf, 0x8d, 0x6e, 0x22, 0x17, 0x22, 0xaa, 0x23,
			0xe5, 0x7f, 0x56, 0x7e, 0x8e, 0xab, 0x0f, 0xe0,
			0x00, 0x78, 0x98, 0x26, 0x0d, 0x69, 0xe5, 0x8d,
			0x19, 0xb5, 0x6a, 0xbb, 0xf3, 0xda, 0x20, 0x3e,
			0x1a, 0xcf, 0xb4, 0xc2, 0x2d, 0x2b, 0xca, 0x4a,
			0xad, 0x31, 0x4a, 0x8f, 0x5d, 0x48, 0xf7, 0xa5,
			0xd6, 0x24, 0xd0, 0x05, 0x58, 0xe3, 0xdd, 0xa8,
			0x3f, 0xb6, 0x93, 0xee, 0x31, 0xee, 0xcd, 0x35,
			0x12, 0x33, 0xd5, 0x50, 0x75, 0x54, 0xd6, 0x29,
			0x62, 0xa0, 0xc3, 0xc4, 0xbd, 0x20, 0x5a, 0x85,
			0xdc, 0x76, 0xdb, 0xcc, 0xc3, 0xc6, 0xe3, 0xeb,
			0xb6, 0x04, 0xe5, 0x77, 0x57, 0x01, 0x37, 0x4e,
			0xee, 0x5d, 0xfe, 0x24, 0x2a, 0x87, 0xc8, 0x14,
			0x5a, 0x22, 0x53, 0x9f, 0xee, 0xd5, 0x37, 0x49,
			0xba, 0x8b, 0xb0, 0x3a, 0xaa, 0xc1, 0xdf, 0x3b,
			0x8c, 0xb7, 0xb4, 0xe4, 0x65, 0x3b, 0x48, 0x18,
			0x91, 0xff, 0xe1, 0xc8, 0x8b, 0xc2, 0x77, 0x24,
			0xbb, 0x9f, 0xc1, 0xb6, 0x26, 0x30, 0xaf, 0xac,
			0x3c, 0x01, 0xf9, 0x59, 0x7a, 0xa9, 0xd2, 0x6d,
			0x77, 0x3f, 0x5e, 0x94, 0x87, 0xb5, 0x05, 0xcb,
			0xce, 0x35, 0xe6, 0x1c, 0x1a, 0xa2, 0x4b, 0x5b,
			0x1f, 0x0d, 0x85, 0xcb, 0x22, 0xd4, 0xc5, 0x7d,
			0x87, 0x89, 0x36, 0x56, 0x2d, 0xda, 0xac, 0x5b,
			0x70, 0x1c, 0xb5, 0xd2, 0xd8, 0x51, 0xa3, 0x07,
			0x2f, 0x7b, 0x1e, 0xb1, 0x0a, 0xd1, 0x8e, 0x7f,
			0x88, 0x79, 0xc4, 0x19, 0xf0, 0x10, 0x77, 0xf8,
		},
	}
};

#ifdef __cplusplus
}
#endif

#endif /* SPHINCS_TESTER_VECTORS_SHAKE_128S_H */
