// Copyright 2022 Contributors to the Parsec project.
// SPDX-License-Identifier: Apache-2.0
use crate::{
    constants::BaseError, error::return_code::BaseReturnCode, Error, Result, WrapperErrorKind,
};
use log::error;
use std::convert::TryFrom;

/// Enum representing the error codes generated by the FAPI layer
/// in TSS.
#[derive(Copy, Clone, Debug, PartialEq, Eq, Hash)]
pub struct FapiReturnCode {
    base_error: BaseError,
}

impl FapiReturnCode {
    /// Returns the [BaseError] associated with the FAPI return code.
    pub const fn base_error(&self) -> BaseError {
        self.base_error
    }
}

impl From<FapiReturnCode> for BaseReturnCode {
    fn from(fapi_return_code: FapiReturnCode) -> Self {
        fapi_return_code.base_error.into()
    }
}

impl TryFrom<BaseReturnCode> for FapiReturnCode {
    type Error = Error;

    fn try_from(base_return_code: BaseReturnCode) -> Result<Self> {
        FapiReturnCode::try_from(BaseError::from(base_return_code))
    }
}

impl TryFrom<u16> for FapiReturnCode {
    type Error = Error;

    fn try_from(value: u16) -> Result<Self> {
        FapiReturnCode::try_from(BaseError::try_from(value)?)
    }
}

impl From<FapiReturnCode> for u16 {
    fn from(fapi_error_code: FapiReturnCode) -> Self {
        BaseReturnCode::from(fapi_error_code).into()
    }
}

impl TryFrom<BaseError> for FapiReturnCode {
    type Error = Error;

    fn try_from(base_error: BaseError) -> Result<Self> {
        match base_error {
            BaseError::GeneralFailure
            | BaseError::NotImplemented
            | BaseError::BadReference
            | BaseError::BadSequence
            | BaseError::IoError
            | BaseError::BadValue
            | BaseError::NoDecryptParam
            | BaseError::NoEncryptParam
            | BaseError::Memory
            | BaseError::BadContext
            | BaseError::NoConfig
            | BaseError::BadPath
            | BaseError::NotDeletable
            | BaseError::PathAlreadyExists
            | BaseError::KeyNotFound
            | BaseError::SignatureVerificationFailed
            | BaseError::HashMismatch
            | BaseError::KeyNotDuplicable
            | BaseError::PathNotFound
            | BaseError::NoCert
            | BaseError::NoPcr
            | BaseError::PcrNotResettable
            | BaseError::BadTemplate
            | BaseError::AuthorizationFailed
            | BaseError::AuthorizationUnknown
            | BaseError::NvNotReadable
            | BaseError::NvTooSmall
            | BaseError::NvNotWriteable
            | BaseError::PolicyUnknown
            | BaseError::NvWrongType
            | BaseError::NameAlreadyExists
            | BaseError::NoTpm
            | BaseError::TryAgain
            | BaseError::BadKey
            | BaseError::NoHandle
            | BaseError::NotProvisioned
            | BaseError::AlreadyProvisioned => Ok(FapiReturnCode { base_error }),
            _ => {
                error!(
                    "{} is not a valid FapiReturnCode base error",
                    u16::from(base_error)
                );
                Err(Error::local_error(WrapperErrorKind::InvalidParam))
            }
        }
    }
}

impl From<FapiReturnCode> for BaseError {
    fn from(fapi_return_code: FapiReturnCode) -> Self {
        fapi_return_code.base_error
    }
}

impl std::error::Error for FapiReturnCode {}

impl std::fmt::Display for FapiReturnCode {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        write!(f, "{}", BaseReturnCode::from(*self))
    }
}
