import { TwolayerOperator } from ".";
/**
 * An interface for aggregating numbers
 */
export interface Aggregator {
    /** add another value to the aggregator */
    add(inp: number): void;
    /** return the aggregate of all added values */
    val(): number | undefined;
}
/**
 * A way to create aggregators
 */
export declare type AggFactory<A extends Aggregator = Aggregator> = () => A;
/**
 * A {@link AggFactory | factory} that creates mean {@link Aggregator}s,
 * bundled as {@link aggMeanFactory}.
 */
export declare const meanFactory: () => Aggregator;
/**
 * A {@link AggFactory | factory} that creates median {@link Aggregator}s,
 * bundled as {@link aggMedianFactory}.
 */
export declare const medianFactory: () => Aggregator;
/**
 * A {@link AggFactory | factory} that creates weighted median {@link Aggregator}s,
 * bundled as {@link aggWeightedMedianFactory}.
 *
 * @remarks
 * This is slightly slower than the {@link medianFactory}.
 */
export declare const weightedMedianFactory: () => Aggregator;
/**
 * A {@link TwolayerOperator} that orders nodes based off the aggregation of their
 * parents' or children's indices.
 *
 * This is much faster than {@link OptOperator}, and often produces comparable
 * or better layouts. If memory is an issue then {@link meanFactory} uses a
 * little less memory, but there is little reason to use it. Nodes without
 * parents or children respectively will be placed first to minimize the
 * distance between nodes with common parents, and then to minimize rank
 * inversions with respect to the initial ordering.
 *
 * Create with {@link agg}.
 *
 * <img alt="two layer agg example" src="media://sugi-simplex-twolayer-quad.png" width="400">
 */
export interface AggOperator<Factory extends AggFactory = AggFactory> extends TwolayerOperator<unknown, unknown> {
    /**
     * Set the {@link AggFactory} for this operator.
     *
     * The aggregators that this produces are used to fuse the indices of parents
     * or children of an node into it's target index for ordering. The provided
     * {@link medianFactory} works very well, but {@link meanFactory} works too,
     * as will any user provided method. (default: {@link medianFactory})
     */
    aggregator<NewFactory extends AggFactory>(val: NewFactory): AggOperator<NewFactory>;
    /**
     * Get the current aggregator factory.
     */
    aggregator(): Factory;
}
/**
 * Create a default {@link AggOperator}, bundled as {@link twolayerAgg}.
 */
export declare function agg(...args: never[]): AggOperator;
