/**
 * The {@link QuadOperator} positions nodes to minimize a quadratic
 * optimization.
 *
 * @packageDocumentation
 */
import { CoordOperator } from ".";
import { DagLink, DagNode } from "../../dag";
import { Up } from "../../utils";
/**
 * an accessor to get the optimization of the weight for a node
 *
 * Currently this is only used to set the {@link QuadOperator.nodeCurve}.
 */
export interface NodeWeightAccessor<NodeDatum = never, LinkDatum = never> {
    (node: DagNode<NodeDatum, LinkDatum>): number;
}
/**
 * an accessor to get the optimization of the weight for a link
 *
 * Currently this is only used to set the following accessors: {@link
 * QuadOperator.linkCurve}, {@link QuadOperator.vertWeak}, {@link
 * QuadOperator.vertStrong}.
 */
export interface LinkWeightAccessor<NodeDatum = never, LinkDatum = never> {
    (link: DagLink<NodeDatum, LinkDatum>): number;
}
/**
 * a {@link NodeWeightAccessor} or {@link LinkWeightAccessor} that returns a constant value
 *
 * If using a constant value, this provides some small memory and time savings
 * over a regular accessor.
 */
export interface ConstAccessor<T extends number = number> extends NodeWeightAccessor<unknown, unknown>, LinkWeightAccessor<unknown, unknown> {
    (): T;
    /** the constant value */
    value: T;
}
/**
 * a function for creating a {@link ConstAccessor}
 */
export declare function createConstAccessor<T extends number>(value: T): ConstAccessor<T>;
/** the operators for the quad operator */
export interface Operators<N = never, L = never> {
    /** the vert weak accessor */
    vertWeak: LinkWeightAccessor<N, L>;
    /** the vert strong accessor */
    vertStrong: LinkWeightAccessor<N, L>;
    /** the link weight accessor */
    linkCurve: LinkWeightAccessor<N, L>;
    /** the node weight accessor */
    nodeCurve: NodeWeightAccessor<N, L>;
}
/** node datum for operators */
declare type OpNodeDatum<O extends Operators> = O extends Operators<infer N, never> ? N : never;
/** link datum for operators */
declare type OpLinkDatum<O extends Operators> = O extends Operators<never, infer L> ? L : never;
/**
 * A {@link CoordOperator} that places nodes to minimize a quadratic function
 *
 * The minimization involves minimizing the distance between {@link vertical |
 * connected nodes}, the {@link curve | curvature of edges}, and the distance
 * between {@link component | disconnected components}.
 *
 * This operators generally takes the longest of all built-in operators but
 * produces the most pleasing layout.
 *
 * Create with {@link quad}.
 *
 * <img alt="quad example" src="media://sugi-simplex-opt-quad.png" width="400">
 */
export interface QuadOperator<Ops extends Operators> extends CoordOperator<OpNodeDatum<Ops>, OpLinkDatum<Ops>> {
    /**
     * Set the weight for verticality. Higher weights mean connected nodes should
     * be closer together, or corollarily edges should be closer to vertical
     * lines. There are two different weights, [ *regular nodes*, *dummy nodes*
     * ], the weight for a pair of connected nodes the sum of the weight value
     * for each node depending on whether not that node is a dummy node. Setting
     * them both to positive means all lines should be roughly vertical, while
     * setting a weight to zero doesn't peanalize edges between those types of
     * nodes. (default: [1, 0])
     *
     * @remarks
     * `.vertical([a, b])` is the same as `.vertWeak(() => a).vertStrong(() => b)`
     */
    vertical(val: readonly [number, number]): QuadOperator<Up<Ops, {
        /** new vert weak */
        vertWeak: ConstAccessor;
        /** new vert strong */
        vertStrong: ConstAccessor;
    }>>;
    /**
     * Get the current vertical weights if they're constant. If {@link
     * QuadOperator.vertWeak} or {@link QuadOperator.vertStrong} is not constant,
     * then null is returned. By setting the weight of dummy nodes to zero,
     * longer edges aren't penalized to be straighter than short edges.
     */
    vertical(): Ops extends {
        /** const vert weak */
        vertWeak: ConstAccessor;
        /** const vert strong */
        vertStrong: ConstAccessor;
    } ? [number, number] : null;
    /**
     * Set the weak vertical accessor.
     *
     * The weak vertical accessor adds a penalty to make edges vertical. It's
     * weak in that it applies to all edges equally regardless of length, and
     * while it penalized non vertical edges, it allows curving in the middle of
     * long edges. (default: () =\> 1)
     */
    vertWeak<NewVertWeak extends LinkWeightAccessor>(val: NewVertWeak): QuadOperator<Up<Ops, {
        /** new vert weak */
        vertWeak: NewVertWeak;
    }>>;
    /**
     * Get the current vertWeak accessor
     */
    vertWeak(): Ops["vertWeak"];
    /**
     * Set the strong vertical accessor.
     *
     * The strong vertical accessor adds a penalty to make edges vertical. It
     * penealizes any section of an edge that isn't vertical, making longer edges
     * contribute more to the overall impact on the objective. (default: () =\> 0)
     */
    vertStrong<NewVertStrong extends LinkWeightAccessor>(val: NewVertStrong): QuadOperator<Up<Ops, {
        /** new vert strong */
        vertStrong: NewVertStrong;
    }>>;
    /**
     * Get the current vertStrong accessor
     */
    vertStrong(): Ops["vertStrong"];
    /**
     * Set the weight for curviness. Higher weights mean an edge going through a
     * node type should be roughly straight.  There are two different weights, [
     * *regular nodes*, *dummy nodes* ], that impact the curvature through those
     * node types. Setting regular nodes to positive will create a type of flow
     * of edges going through a node, while setting dummy nodes will enforce the
     * longer edges should try to be straight. (default: [0, 1])
     *
     * @remarks
     * `.curve([a, b])` is the same as `.nodeCurve(() =\> a).linkCurve(() =\> b)`
     */
    curve(val: readonly [number, number]): QuadOperator<Up<Ops, {
        /** new link curve */
        linkCurve: ConstAccessor;
        /** new node curve */
        nodeCurve: ConstAccessor;
    }>>;
    /**
     * Get the current curve weights if they're constant, otherwise return null.
     * By setting the weight of non-dummy nodes to zero, we only care about the
     * curvature of edges, not lines that pass through nodes.
     */
    curve(): Ops extends {
        /** constant link curve */
        linkCurve: ConstAccessor;
        /** constant node curve */
        nodeCurve: ConstAccessor;
    } ? [number, number] : null;
    /**
     * Set the link curve weight accessor
     *
     * The link curve weight penalizes links to reduce their curving, in
     * dependent of their verticality. If using strongVert for an edge, it
     * probably won't need a strong link curve weight. (default: () =\> 1)
     */
    linkCurve<NewLinkCurve extends LinkWeightAccessor>(val: NewLinkCurve): QuadOperator<Up<Ops, {
        /** new link curve */
        linkCurve: NewLinkCurve;
    }>>;
    /**
     * Get the current link curve weight accessor
     */
    linkCurve(): Ops["linkCurve"];
    /**
     * Set the node curve weight accessor
     *
     * The node curve weight penalizes curves through nodes. If a node only has
     * one incoming and one outgoing edge, it will try to make them match in
     * angle. Note that it does it for all possible "through edges" so multiple
     * incoming and multiple outgoing will get counted several times. It's not
     * clear why this would ever be desirable, but it's possible to specify.
     * (default: () =\> 0)
     */
    nodeCurve<NewNodeCurve extends NodeWeightAccessor>(val: NewNodeCurve): QuadOperator<Up<Ops, {
        /** new node curve */
        nodeCurve: NewNodeCurve;
    }>>;
    /**
     * Get the current node curve accessor
     */
    nodeCurve(): Ops["nodeCurve"];
    /**
     * Set the weight for how close different disconnected components should be.
     * The higher the weight, the more different components will be close to each
     * other at the expense of other objectives. This needs to be greater than
     * zero to make the objective sound when there are disconnected components,
     * but otherwise should probably be very small. (default: 1)
     */
    component(val: number): QuadOperator<Ops>;
    /** Get the current component weight. */
    component(): number;
}
/** default quad operator */
export declare type DefaultQuadOperator = QuadOperator<{
    /** default vert weak */
    vertWeak: ConstAccessor<1>;
    /** default vert strong */
    vertStrong: ConstAccessor<0>;
    /** default link curve */
    linkCurve: ConstAccessor<1>;
    /** default node curve */
    nodeCurve: ConstAccessor<0>;
}>;
/**
 * Create a default {@link QuadOperator}, bundled as {@link coordQuad}.
 */
export declare function quad(...args: never[]): DefaultQuadOperator;
export {};
