import { jsxs, jsx } from 'react/jsx-runtime';
import { css } from '@emotion/css';
import { DragDropContext, Droppable } from '@hello-pangea/dnd';
import { cloneDeep } from 'lodash';
import { useState, useEffect } from 'react';
import '@grafana/data';
import { useStyles2 } from '../../../themes/ThemeContext.js';
import 'micro-memoize';
import '@emotion/react';
import 'tinycolor2';
import '../../../utils/skeleton.js';
import { Trans } from '../../../utils/i18n.js';
import { Button } from '../../Button/Button.js';
import '../../Button/ButtonGroup.js';
import { Modal } from '../../Modal/Modal.js';
import { DataLinkEditorModalContent } from './DataLinkEditorModalContent.js';
import { DataLinksListItem } from './DataLinksListItem.js';

const DataLinksInlineEditor = ({
  links,
  onChange,
  getSuggestions,
  data,
  showOneClick = false
}) => {
  const [editIndex, setEditIndex] = useState(null);
  const [isNew, setIsNew] = useState(false);
  const [linksSafe, setLinksSafe] = useState([]);
  useEffect(() => {
    setLinksSafe(links != null ? links : []);
  }, [links]);
  const styles = useStyles2(getDataLinksInlineEditorStyles);
  const isEditing = editIndex !== null;
  const onDataLinkChange = (index, link) => {
    if (isNew) {
      if (link.title.trim() === "" && link.url.trim() === "") {
        setIsNew(false);
        setEditIndex(null);
        return;
      } else {
        setEditIndex(null);
        setIsNew(false);
      }
    }
    const update = cloneDeep(linksSafe);
    update[index] = link;
    onChange(update);
    setEditIndex(null);
  };
  const onDataLinkAdd = () => {
    let update = cloneDeep(linksSafe);
    setEditIndex(update.length);
    setIsNew(true);
  };
  const onDataLinkCancel = (index) => {
    if (isNew) {
      setIsNew(false);
    }
    setEditIndex(null);
  };
  const onDataLinkRemove = (index) => {
    const update = cloneDeep(linksSafe);
    update.splice(index, 1);
    onChange(update);
  };
  const onDragEnd = (result) => {
    if (!links || !result.destination) {
      return;
    }
    const update = cloneDeep(linksSafe);
    const link = update[result.source.index];
    update.splice(result.source.index, 1);
    update.splice(result.destination.index, 0, link);
    setLinksSafe(update);
    onChange(update);
  };
  return /* @__PURE__ */ jsxs("div", { className: styles.container, children: [
    showOneClick && linksSafe.length > 0 && /* @__PURE__ */ jsx("div", { className: styles.oneClickOverlay, children: /* @__PURE__ */ jsx("span", { className: styles.oneClickSpan, children: /* @__PURE__ */ jsx(Trans, { i18nKey: "grafana-ui.data-links-inline-editor.one-click-link", children: "One-click link" }) }) }),
    /* @__PURE__ */ jsx(DragDropContext, { onDragEnd, children: /* @__PURE__ */ jsx(Droppable, { droppableId: "sortable-links", direction: "vertical", children: (provided) => /* @__PURE__ */ jsxs(
      "div",
      {
        className: styles.wrapper,
        ref: provided.innerRef,
        ...provided.droppableProps,
        style: { paddingTop: showOneClick && linksSafe.length > 0 ? "28px" : "0px" },
        children: [
          linksSafe.map((link, idx) => {
            const key = `${link.title}/${idx}`;
            return /* @__PURE__ */ jsx(
              DataLinksListItem,
              {
                index: idx,
                link,
                onChange: onDataLinkChange,
                onEdit: () => setEditIndex(idx),
                onRemove: () => onDataLinkRemove(idx),
                data,
                itemKey: key
              },
              key
            );
          }),
          provided.placeholder
        ]
      }
    ) }) }),
    isEditing && editIndex !== null && /* @__PURE__ */ jsx(
      Modal,
      {
        title: "Edit link",
        isOpen: true,
        closeOnBackdropClick: false,
        onDismiss: () => {
          onDataLinkCancel();
        },
        children: /* @__PURE__ */ jsx(
          DataLinkEditorModalContent,
          {
            index: editIndex,
            link: isNew ? { title: "", url: "" } : linksSafe[editIndex],
            data,
            onSave: onDataLinkChange,
            onCancel: onDataLinkCancel,
            getSuggestions
          }
        )
      }
    ),
    /* @__PURE__ */ jsx(Button, { size: "sm", icon: "plus", onClick: onDataLinkAdd, variant: "secondary", className: styles.button, children: /* @__PURE__ */ jsx(Trans, { i18nKey: "grafana-ui.data-links-inline-editor.add-link", children: "Add link" }) })
  ] });
};
const getDataLinksInlineEditorStyles = (theme) => ({
  container: css({
    position: "relative"
  }),
  wrapper: css({
    marginBottom: theme.spacing(2),
    display: "flex",
    flexDirection: "column"
  }),
  oneClickOverlay: css({
    border: `2px dashed ${theme.colors.text.link}`,
    fontSize: 10,
    color: theme.colors.text.primary,
    marginBottom: theme.spacing(1),
    position: "absolute",
    width: "100%",
    height: "92px"
  }),
  oneClickSpan: css({
    padding: 10,
    // Negates the padding on the span from moving the underlying link
    marginBottom: -10,
    display: "inline-block"
  }),
  button: css({
    marginLeft: theme.spacing(1)
  })
});

export { DataLinksInlineEditor };
//# sourceMappingURL=DataLinksInlineEditor.js.map
