import { map } from 'rxjs/operators';
import { TIME_SERIES_VALUE_FIELD_NAME } from '../../types/dataFrame.js';
import { DataTransformerID } from './ids.js';

const concatenateTransformer = {
  id: DataTransformerID.concatenate,
  name: "Concatenate fields",
  description: "Combine all fields into a single frame.  Values will be appended with undefined values if not the same length.",
  defaultOptions: {
    frameNameMode: "field" /* FieldName */,
    frameNameLabel: "frame"
  },
  operator: (options) => (source) => source.pipe(
    map((dataFrames) => {
      if (!Array.isArray(dataFrames) || dataFrames.length < 2) {
        return dataFrames;
      }
      return [concatenateFields(dataFrames, options)];
    })
  )
};
function concatenateFields(data, opts) {
  var _a;
  let sameLength = true;
  let maxLength = data[0].length;
  const frameNameLabel = (_a = opts.frameNameLabel) != null ? _a : "frame";
  let fields = [];
  for (const frame of data) {
    if (maxLength !== frame.length) {
      sameLength = false;
      maxLength = Math.max(maxLength, frame.length);
    }
    for (const f of frame.fields) {
      const copy = { ...f };
      copy.state = void 0;
      if (frame.name) {
        if (opts.frameNameMode === "drop" /* Drop */) ; else if (opts.frameNameMode === "label" /* Label */) {
          copy.labels = { ...f.labels };
          copy.labels[frameNameLabel] = frame.name;
        } else if (!copy.name || copy.name === TIME_SERIES_VALUE_FIELD_NAME) {
          copy.name = frame.name;
        } else {
          copy.name = `${frame.name} \xB7 ${f.name}`;
        }
      }
      fields.push(copy);
    }
  }
  if (!sameLength) {
    fields = fields.map((f) => {
      if (f.values.length === maxLength) {
        return f;
      }
      const values = f.values.slice();
      values.length = maxLength;
      return {
        ...f,
        values
      };
    });
  }
  return {
    fields,
    length: maxLength
  };
}

export { concatenateFields, concatenateTransformer };
//# sourceMappingURL=concat.js.map
