package tlsh

// bitPairsDiffTable is pre-calculated table that represents approximation
// to the Hamming distance. It's generated using Jonathan Oliver's algorithm.
//
// Original implementation and algorithm for generation can be found on
// following urls:
// https://github.com/trendmicro/tlsh/blob/master/src/tlsh_util.cpp#L84-L4694
// https://github.com/trendmicro/tlsh/blob/master/src/gen_arr2.cpp#L1-L91
//
// Details about distance score can be also found in Trendmicro TLSH paper:
// https://github.com/trendmicro/tlsh/blob/master/TLSH_CTC_final.pdf

var bitPairsDiffTable = [256][256]int{
	{
		0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8, 6, 7, 8, 12,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14, 12, 13, 14, 18,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15, 13, 14, 15, 19,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12, 10, 11, 12, 16,
		8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16, 14, 15, 16, 20,
		6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14, 12, 13, 14, 18,
		7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15, 13, 14, 15, 19,
		8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16, 14, 15, 16, 20,
		12, 13, 14, 18, 13, 14, 15, 19, 14, 15, 16, 20, 18, 19, 20, 24,
	},
	{
		1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4, 7, 6, 7, 8,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10, 13, 12, 13, 14,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11, 14, 13, 14, 15,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8, 11, 10, 11, 12,
		9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12, 15, 14, 15, 16,
		7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10, 13, 12, 13, 14,
		8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11, 14, 13, 14, 15,
		9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12, 15, 14, 15, 16,
		13, 12, 13, 14, 14, 13, 14, 15, 15, 14, 15, 16, 19, 18, 19, 20,
	},
	{
		2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3, 8, 7, 6, 7,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9, 14, 13, 12, 13,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10, 15, 14, 13, 14,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7, 12, 11, 10, 11,
		10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11, 16, 15, 14, 15,
		8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9, 14, 13, 12, 13,
		9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10, 15, 14, 13, 14,
		10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11, 16, 15, 14, 15,
		14, 13, 12, 13, 15, 14, 13, 14, 16, 15, 14, 15, 20, 19, 18, 19,
	},
	{
		6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2, 12, 8, 7, 6,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8, 18, 14, 13, 12,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9, 19, 15, 14, 13,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6, 16, 12, 11, 10,
		14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10, 20, 16, 15, 14,
		12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8, 18, 14, 13, 12,
		13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9, 19, 15, 14, 13,
		14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10, 20, 16, 15, 14,
		18, 14, 13, 12, 19, 15, 14, 13, 20, 16, 15, 14, 24, 20, 19, 18,
	},
	{
		1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12,
		9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16,
		7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14,
		8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15,
		9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16,
		13, 14, 15, 19, 12, 13, 14, 18, 13, 14, 15, 19, 14, 15, 16, 20,
	},
	{
		2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8,
		10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12,
		8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10,
		9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11,
		10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12,
		14, 13, 14, 15, 13, 12, 13, 14, 14, 13, 14, 15, 15, 14, 15, 16,
	},
	{
		3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7,
		11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11,
		9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9,
		10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10,
		11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11,
		15, 14, 13, 14, 14, 13, 12, 13, 15, 14, 13, 14, 16, 15, 14, 15,
	},
	{
		7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6,
		15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10,
		13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8,
		14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9,
		15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10,
		19, 15, 14, 13, 18, 14, 13, 12, 19, 15, 14, 13, 20, 16, 15, 14,
	},
	{
		2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11,
		10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15,
		8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13,
		9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14,
		10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15,
		14, 15, 16, 20, 13, 14, 15, 19, 12, 13, 14, 18, 13, 14, 15, 19,
	},
	{
		3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7,
		11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11,
		9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9,
		10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10,
		11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11,
		15, 14, 15, 16, 14, 13, 14, 15, 13, 12, 13, 14, 14, 13, 14, 15,
	},
	{
		4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6,
		12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10,
		10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8,
		11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9,
		12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10,
		16, 15, 14, 15, 15, 14, 13, 14, 14, 13, 12, 13, 15, 14, 13, 14,
	},
	{
		8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5,
		16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9,
		14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7,
		15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8,
		16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9,
		20, 16, 15, 14, 19, 15, 14, 13, 18, 14, 13, 12, 19, 15, 14, 13,
	},
	{
		6, 7, 8, 12, 2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		12, 13, 14, 18, 8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		13, 14, 15, 19, 9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		10, 11, 12, 16, 6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10,
		14, 15, 16, 20, 10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14,
		12, 13, 14, 18, 8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12,
		13, 14, 15, 19, 9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13,
		14, 15, 16, 20, 10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14,
		18, 19, 20, 24, 14, 15, 16, 20, 13, 14, 15, 19, 12, 13, 14, 18,
	},
	{
		7, 6, 7, 8, 3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		13, 12, 13, 14, 9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		14, 13, 14, 15, 10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		11, 10, 11, 12, 7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6,
		15, 14, 15, 16, 11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10,
		13, 12, 13, 14, 9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8,
		14, 13, 14, 15, 10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9,
		15, 14, 15, 16, 11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10,
		19, 18, 19, 20, 15, 14, 15, 16, 14, 13, 14, 15, 13, 12, 13, 14,
	},
	{
		8, 7, 6, 7, 4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		14, 13, 12, 13, 10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		15, 14, 13, 14, 11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		12, 11, 10, 11, 8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5,
		16, 15, 14, 15, 12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9,
		14, 13, 12, 13, 10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7,
		15, 14, 13, 14, 11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8,
		16, 15, 14, 15, 12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9,
		20, 19, 18, 19, 16, 15, 14, 15, 15, 14, 13, 14, 14, 13, 12, 13,
	},
	{
		12, 8, 7, 6, 8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		18, 14, 13, 12, 14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		19, 15, 14, 13, 15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		16, 12, 11, 10, 12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4,
		20, 16, 15, 14, 16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8,
		18, 14, 13, 12, 14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6,
		19, 15, 14, 13, 15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7,
		20, 16, 15, 14, 16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8,
		24, 20, 19, 18, 20, 16, 15, 14, 19, 15, 14, 13, 18, 14, 13, 12,
	},
	{
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8, 6, 7, 8, 12,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12, 10, 11, 12, 16,
		7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15, 13, 14, 15, 19,
		6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14, 12, 13, 14, 18,
		7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15, 13, 14, 15, 19,
		8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16, 14, 15, 16, 20,
	},
	{
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4, 7, 6, 7, 8,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8, 11, 10, 11, 12,
		8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11, 14, 13, 14, 15,
		7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10, 13, 12, 13, 14,
		8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11, 14, 13, 14, 15,
		9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12, 15, 14, 15, 16,
	},
	{
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3, 8, 7, 6, 7,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7, 12, 11, 10, 11,
		9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10, 15, 14, 13, 14,
		8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9, 14, 13, 12, 13,
		9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10, 15, 14, 13, 14,
		10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11, 16, 15, 14, 15,
	},
	{
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2, 12, 8, 7, 6,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6, 16, 12, 11, 10,
		13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9, 19, 15, 14, 13,
		12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8, 18, 14, 13, 12,
		13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9, 19, 15, 14, 13,
		14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10, 20, 16, 15, 14,
	},
	{
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12,
		8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15,
		7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14,
		8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15,
		9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16,
	},
	{
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8,
		9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11,
		8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10,
		9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11,
		10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12,
	},
	{
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7,
		10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10,
		9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9,
		10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10,
		11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11,
	},
	{
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6,
		14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9,
		13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8,
		14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9,
		15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10,
	},
	{
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11,
		9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14,
		8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13,
		9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14,
		10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15,
	},
	{
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7,
		10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10,
		9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9,
		10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10,
		11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11,
	},
	{
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6,
		11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9,
		10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8,
		11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9,
		12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10,
	},
	{
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5,
		15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8,
		14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7,
		15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8,
		16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9,
	},
	{
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		6, 7, 8, 12, 2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		10, 11, 12, 16, 6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10,
		13, 14, 15, 19, 9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13,
		12, 13, 14, 18, 8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12,
		13, 14, 15, 19, 9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13,
		14, 15, 16, 20, 10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14,
	},
	{
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		7, 6, 7, 8, 3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		11, 10, 11, 12, 7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6,
		14, 13, 14, 15, 10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9,
		13, 12, 13, 14, 9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8,
		14, 13, 14, 15, 10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9,
		15, 14, 15, 16, 11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10,
	},
	{
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		8, 7, 6, 7, 4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		12, 11, 10, 11, 8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5,
		15, 14, 13, 14, 11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8,
		14, 13, 12, 13, 10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7,
		15, 14, 13, 14, 11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8,
		16, 15, 14, 15, 12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9,
	},
	{
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		12, 8, 7, 6, 8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		16, 12, 11, 10, 12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4,
		19, 15, 14, 13, 15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7,
		18, 14, 13, 12, 14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6,
		19, 15, 14, 13, 15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7,
		20, 16, 15, 14, 16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8,
	},
	{
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8, 6, 7, 8, 12,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12, 10, 11, 12, 16,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16, 14, 15, 16, 20,
		7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15, 13, 14, 15, 19,
		6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14, 12, 13, 14, 18,
		7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15, 13, 14, 15, 19,
	},
	{
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4, 7, 6, 7, 8,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8, 11, 10, 11, 12,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12, 15, 14, 15, 16,
		8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11, 14, 13, 14, 15,
		7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10, 13, 12, 13, 14,
		8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11, 14, 13, 14, 15,
	},
	{
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3, 8, 7, 6, 7,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7, 12, 11, 10, 11,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11, 16, 15, 14, 15,
		9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10, 15, 14, 13, 14,
		8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9, 14, 13, 12, 13,
		9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10, 15, 14, 13, 14,
	},
	{
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2, 12, 8, 7, 6,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6, 16, 12, 11, 10,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10, 20, 16, 15, 14,
		13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9, 19, 15, 14, 13,
		12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8, 18, 14, 13, 12,
		13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9, 19, 15, 14, 13,
	},
	{
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16,
		8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15,
		7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14,
		8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15,
	},
	{
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12,
		9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11,
		8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10,
		9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11,
	},
	{
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11,
		10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10,
		9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9,
		10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10,
	},
	{
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10,
		14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9,
		13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8,
		14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9,
	},
	{
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15,
		9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14,
		8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13,
		9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14,
	},
	{
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11,
		10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10,
		9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9,
		10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10,
	},
	{
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10,
		11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9,
		10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8,
		11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9,
	},
	{
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9,
		15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8,
		14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7,
		15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8,
	},
	{
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		6, 7, 8, 12, 2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		10, 11, 12, 16, 6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		14, 15, 16, 20, 10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14,
		13, 14, 15, 19, 9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13,
		12, 13, 14, 18, 8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12,
		13, 14, 15, 19, 9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13,
	},
	{
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		7, 6, 7, 8, 3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		11, 10, 11, 12, 7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		15, 14, 15, 16, 11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10,
		14, 13, 14, 15, 10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9,
		13, 12, 13, 14, 9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8,
		14, 13, 14, 15, 10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9,
	},
	{
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		8, 7, 6, 7, 4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		12, 11, 10, 11, 8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		16, 15, 14, 15, 12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9,
		15, 14, 13, 14, 11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8,
		14, 13, 12, 13, 10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7,
		15, 14, 13, 14, 11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8,
	},
	{
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		12, 8, 7, 6, 8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		16, 12, 11, 10, 12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		20, 16, 15, 14, 16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8,
		19, 15, 14, 13, 15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7,
		18, 14, 13, 12, 14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6,
		19, 15, 14, 13, 15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7,
	},
	{
		6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14, 12, 13, 14, 18,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8, 6, 7, 8, 12,
		7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15, 13, 14, 15, 19,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16, 14, 15, 16, 20,
		4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12, 10, 11, 12, 16,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		12, 13, 14, 18, 13, 14, 15, 19, 14, 15, 16, 20, 18, 19, 20, 24,
		8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16, 14, 15, 16, 20,
		7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15, 13, 14, 15, 19,
		6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14, 12, 13, 14, 18,
	},
	{
		7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10, 13, 12, 13, 14,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4, 7, 6, 7, 8,
		8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11, 14, 13, 14, 15,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12, 15, 14, 15, 16,
		5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8, 11, 10, 11, 12,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		13, 12, 13, 14, 14, 13, 14, 15, 15, 14, 15, 16, 19, 18, 19, 20,
		9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12, 15, 14, 15, 16,
		8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11, 14, 13, 14, 15,
		7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10, 13, 12, 13, 14,
	},
	{
		8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9, 14, 13, 12, 13,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3, 8, 7, 6, 7,
		9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10, 15, 14, 13, 14,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11, 16, 15, 14, 15,
		6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7, 12, 11, 10, 11,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		14, 13, 12, 13, 15, 14, 13, 14, 16, 15, 14, 15, 20, 19, 18, 19,
		10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11, 16, 15, 14, 15,
		9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10, 15, 14, 13, 14,
		8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9, 14, 13, 12, 13,
	},
	{
		12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8, 18, 14, 13, 12,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2, 12, 8, 7, 6,
		13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9, 19, 15, 14, 13,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10, 20, 16, 15, 14,
		10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6, 16, 12, 11, 10,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		18, 14, 13, 12, 19, 15, 14, 13, 20, 16, 15, 14, 24, 20, 19, 18,
		14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10, 20, 16, 15, 14,
		13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9, 19, 15, 14, 13,
		12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8, 18, 14, 13, 12,
	},
	{
		7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8,
		8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16,
		5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		13, 14, 15, 19, 12, 13, 14, 18, 13, 14, 15, 19, 14, 15, 16, 20,
		9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16,
		8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15,
		7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14,
	},
	{
		8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4,
		9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12,
		6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		14, 13, 14, 15, 13, 12, 13, 14, 14, 13, 14, 15, 15, 14, 15, 16,
		10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12,
		9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11,
		8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10,
	},
	{
		9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3,
		10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11,
		7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		15, 14, 13, 14, 14, 13, 12, 13, 15, 14, 13, 14, 16, 15, 14, 15,
		11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11,
		10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10,
		9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9,
	},
	{
		13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2,
		14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10,
		11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		19, 15, 14, 13, 18, 14, 13, 12, 19, 15, 14, 13, 20, 16, 15, 14,
		15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10,
		14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9,
		13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8,
	},
	{
		8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7,
		9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15,
		6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		14, 15, 16, 20, 13, 14, 15, 19, 12, 13, 14, 18, 13, 14, 15, 19,
		10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15,
		9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14,
		8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13,
	},
	{
		9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3,
		10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11,
		7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		15, 14, 15, 16, 14, 13, 14, 15, 13, 12, 13, 14, 14, 13, 14, 15,
		11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11,
		10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10,
		9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9,
	},
	{
		10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2,
		11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10,
		8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		16, 15, 14, 15, 15, 14, 13, 14, 14, 13, 12, 13, 15, 14, 13, 14,
		12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10,
		11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9,
		10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8,
	},
	{
		14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1,
		15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9,
		12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		20, 16, 15, 14, 19, 15, 14, 13, 18, 14, 13, 12, 19, 15, 14, 13,
		16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9,
		15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8,
		14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7,
	},
	{
		12, 13, 14, 18, 8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		6, 7, 8, 12, 2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6,
		13, 14, 15, 19, 9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		14, 15, 16, 20, 10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14,
		10, 11, 12, 16, 6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		18, 19, 20, 24, 14, 15, 16, 20, 13, 14, 15, 19, 12, 13, 14, 18,
		14, 15, 16, 20, 10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14,
		13, 14, 15, 19, 9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13,
		12, 13, 14, 18, 8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12,
	},
	{
		13, 12, 13, 14, 9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		7, 6, 7, 8, 3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2,
		14, 13, 14, 15, 10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		15, 14, 15, 16, 11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10,
		11, 10, 11, 12, 7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		19, 18, 19, 20, 15, 14, 15, 16, 14, 13, 14, 15, 13, 12, 13, 14,
		15, 14, 15, 16, 11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10,
		14, 13, 14, 15, 10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9,
		13, 12, 13, 14, 9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8,
	},
	{
		14, 13, 12, 13, 10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		8, 7, 6, 7, 4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1,
		15, 14, 13, 14, 11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		16, 15, 14, 15, 12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9,
		12, 11, 10, 11, 8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		20, 19, 18, 19, 16, 15, 14, 15, 15, 14, 13, 14, 14, 13, 12, 13,
		16, 15, 14, 15, 12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9,
		15, 14, 13, 14, 11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8,
		14, 13, 12, 13, 10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7,
	},
	{
		18, 14, 13, 12, 14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		12, 8, 7, 6, 8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0,
		19, 15, 14, 13, 15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		20, 16, 15, 14, 16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8,
		16, 12, 11, 10, 12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		24, 20, 19, 18, 20, 16, 15, 14, 19, 15, 14, 13, 18, 14, 13, 12,
		20, 16, 15, 14, 16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8,
		19, 15, 14, 13, 15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7,
		18, 14, 13, 12, 14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6,
	},
	{
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15, 13, 14, 15, 19,
		0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8, 6, 7, 8, 12,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14, 12, 13, 14, 18,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15, 13, 14, 15, 19,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12, 10, 11, 12, 16,
		8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16, 14, 15, 16, 20,
	},
	{
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11, 14, 13, 14, 15,
		1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4, 7, 6, 7, 8,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10, 13, 12, 13, 14,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11, 14, 13, 14, 15,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8, 11, 10, 11, 12,
		9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12, 15, 14, 15, 16,
	},
	{
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10, 15, 14, 13, 14,
		2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3, 8, 7, 6, 7,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9, 14, 13, 12, 13,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10, 15, 14, 13, 14,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7, 12, 11, 10, 11,
		10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11, 16, 15, 14, 15,
	},
	{
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9, 19, 15, 14, 13,
		6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2, 12, 8, 7, 6,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8, 18, 14, 13, 12,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9, 19, 15, 14, 13,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6, 16, 12, 11, 10,
		14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10, 20, 16, 15, 14,
	},
	{
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15,
		1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12,
		9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16,
	},
	{
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11,
		2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8,
		10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12,
	},
	{
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10,
		3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7,
		11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11,
	},
	{
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9,
		7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6,
		15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10,
	},
	{
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14,
		2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11,
		10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15,
	},
	{
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10,
		3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7,
		11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11,
	},
	{
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9,
		4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6,
		12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10,
	},
	{
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8,
		8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5,
		16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9,
	},
	{
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		13, 14, 15, 19, 9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13,
		6, 7, 8, 12, 2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		12, 13, 14, 18, 8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		13, 14, 15, 19, 9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		10, 11, 12, 16, 6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10,
		14, 15, 16, 20, 10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14,
	},
	{
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		14, 13, 14, 15, 10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9,
		7, 6, 7, 8, 3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		13, 12, 13, 14, 9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		14, 13, 14, 15, 10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		11, 10, 11, 12, 7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6,
		15, 14, 15, 16, 11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10,
	},
	{
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		15, 14, 13, 14, 11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8,
		8, 7, 6, 7, 4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		14, 13, 12, 13, 10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		15, 14, 13, 14, 11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		12, 11, 10, 11, 8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5,
		16, 15, 14, 15, 12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9,
	},
	{
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		19, 15, 14, 13, 15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7,
		12, 8, 7, 6, 8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		18, 14, 13, 12, 14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		19, 15, 14, 13, 15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		16, 12, 11, 10, 12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4,
		20, 16, 15, 14, 16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8,
	},
	{
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8, 6, 7, 8, 12,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12, 10, 11, 12, 16,
	},
	{
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4, 7, 6, 7, 8,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8, 11, 10, 11, 12,
	},
	{
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3, 8, 7, 6, 7,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7, 12, 11, 10, 11,
	},
	{
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2, 12, 8, 7, 6,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6, 16, 12, 11, 10,
	},
	{
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12,
	},
	{
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8,
	},
	{
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7,
	},
	{
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6,
	},
	{
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11,
	},
	{
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7,
	},
	{
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6,
	},
	{
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5,
	},
	{
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		6, 7, 8, 12, 2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		10, 11, 12, 16, 6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10,
	},
	{
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		7, 6, 7, 8, 3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		11, 10, 11, 12, 7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6,
	},
	{
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		8, 7, 6, 7, 4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		12, 11, 10, 11, 8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5,
	},
	{
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		12, 8, 7, 6, 8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		16, 12, 11, 10, 12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4,
	},
	{
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8, 6, 7, 8, 12,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12, 10, 11, 12, 16,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
	},
	{
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4, 7, 6, 7, 8,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8, 11, 10, 11, 12,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
	},
	{
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3, 8, 7, 6, 7,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7, 12, 11, 10, 11,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
	},
	{
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2, 12, 8, 7, 6,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6, 16, 12, 11, 10,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
	},
	{
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
	},
	{
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
	},
	{
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
	},
	{
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
	},
	{
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
	},
	{
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
	},
	{
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
	},
	{
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
	},
	{
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		6, 7, 8, 12, 2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		10, 11, 12, 16, 6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
	},
	{
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		7, 6, 7, 8, 3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		11, 10, 11, 12, 7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
	},
	{
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		8, 7, 6, 7, 4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		12, 11, 10, 11, 8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
	},
	{
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		12, 8, 7, 6, 8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		16, 12, 11, 10, 12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
	},
	{
		7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15, 13, 14, 15, 19,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14, 12, 13, 14, 18,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8, 6, 7, 8, 12,
		7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15, 13, 14, 15, 19,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16, 14, 15, 16, 20,
		4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12, 10, 11, 12, 16,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
	},
	{
		8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11, 14, 13, 14, 15,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10, 13, 12, 13, 14,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4, 7, 6, 7, 8,
		8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11, 14, 13, 14, 15,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12, 15, 14, 15, 16,
		5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8, 11, 10, 11, 12,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
	},
	{
		9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10, 15, 14, 13, 14,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9, 14, 13, 12, 13,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3, 8, 7, 6, 7,
		9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10, 15, 14, 13, 14,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11, 16, 15, 14, 15,
		6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7, 12, 11, 10, 11,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
	},
	{
		13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9, 19, 15, 14, 13,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8, 18, 14, 13, 12,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2, 12, 8, 7, 6,
		13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9, 19, 15, 14, 13,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10, 20, 16, 15, 14,
		10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6, 16, 12, 11, 10,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
	},
	{
		8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8,
		8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16,
		5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
	},
	{
		9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4,
		9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12,
		6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
	},
	{
		10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3,
		10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11,
		7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
	},
	{
		14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2,
		14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10,
		11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
	},
	{
		9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7,
		9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15,
		6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
	},
	{
		10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3,
		10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11,
		7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
	},
	{
		11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2,
		11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10,
		8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
	},
	{
		15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1,
		15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9,
		12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
	},
	{
		13, 14, 15, 19, 9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		12, 13, 14, 18, 8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		6, 7, 8, 12, 2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6,
		13, 14, 15, 19, 9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		14, 15, 16, 20, 10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14,
		10, 11, 12, 16, 6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
	},
	{
		14, 13, 14, 15, 10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		13, 12, 13, 14, 9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		7, 6, 7, 8, 3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2,
		14, 13, 14, 15, 10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		15, 14, 15, 16, 11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10,
		11, 10, 11, 12, 7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
	},
	{
		15, 14, 13, 14, 11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		14, 13, 12, 13, 10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		8, 7, 6, 7, 4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1,
		15, 14, 13, 14, 11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		16, 15, 14, 15, 12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9,
		12, 11, 10, 11, 8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
	},
	{
		19, 15, 14, 13, 15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		18, 14, 13, 12, 14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		12, 8, 7, 6, 8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0,
		19, 15, 14, 13, 15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		20, 16, 15, 14, 16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8,
		16, 12, 11, 10, 12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
	},
	{
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12, 10, 11, 12, 16,
		8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16, 14, 15, 16, 20,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15, 13, 14, 15, 19,
		0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8, 6, 7, 8, 12,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14, 12, 13, 14, 18,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15, 13, 14, 15, 19,
	},
	{
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8, 11, 10, 11, 12,
		9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12, 15, 14, 15, 16,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11, 14, 13, 14, 15,
		1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4, 7, 6, 7, 8,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10, 13, 12, 13, 14,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11, 14, 13, 14, 15,
	},
	{
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7, 12, 11, 10, 11,
		10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11, 16, 15, 14, 15,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10, 15, 14, 13, 14,
		2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3, 8, 7, 6, 7,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9, 14, 13, 12, 13,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10, 15, 14, 13, 14,
	},
	{
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6, 16, 12, 11, 10,
		14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10, 20, 16, 15, 14,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9, 19, 15, 14, 13,
		6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2, 12, 8, 7, 6,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8, 18, 14, 13, 12,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9, 19, 15, 14, 13,
	},
	{
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12,
		9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15,
		1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15,
	},
	{
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8,
		10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11,
		2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11,
	},
	{
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7,
		11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10,
		3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10,
	},
	{
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6,
		15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9,
		7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9,
	},
	{
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11,
		10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14,
		2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14,
	},
	{
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7,
		11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10,
		3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10,
	},
	{
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6,
		12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9,
		4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9,
	},
	{
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5,
		16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8,
		8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8,
	},
	{
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		10, 11, 12, 16, 6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10,
		14, 15, 16, 20, 10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		13, 14, 15, 19, 9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13,
		6, 7, 8, 12, 2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		12, 13, 14, 18, 8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		13, 14, 15, 19, 9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13,
	},
	{
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		11, 10, 11, 12, 7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6,
		15, 14, 15, 16, 11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		14, 13, 14, 15, 10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9,
		7, 6, 7, 8, 3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		13, 12, 13, 14, 9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		14, 13, 14, 15, 10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9,
	},
	{
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		12, 11, 10, 11, 8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5,
		16, 15, 14, 15, 12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		15, 14, 13, 14, 11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8,
		8, 7, 6, 7, 4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		14, 13, 12, 13, 10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		15, 14, 13, 14, 11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8,
	},
	{
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		16, 12, 11, 10, 12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4,
		20, 16, 15, 14, 16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		19, 15, 14, 13, 15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7,
		12, 8, 7, 6, 8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		18, 14, 13, 12, 14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		19, 15, 14, 13, 15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7,
	},
	{
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12, 10, 11, 12, 16,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8, 6, 7, 8, 12,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
	},
	{
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8, 11, 10, 11, 12,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4, 7, 6, 7, 8,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
	},
	{
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7, 12, 11, 10, 11,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3, 8, 7, 6, 7,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
	},
	{
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6, 16, 12, 11, 10,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2, 12, 8, 7, 6,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
	},
	{
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
	},
	{
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
	},
	{
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
	},
	{
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
	},
	{
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
	},
	{
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
	},
	{
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
	},
	{
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
	},
	{
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		10, 11, 12, 16, 6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		6, 7, 8, 12, 2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
	},
	{
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		11, 10, 11, 12, 7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		7, 6, 7, 8, 3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
	},
	{
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		12, 11, 10, 11, 8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		8, 7, 6, 7, 4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
	},
	{
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		16, 12, 11, 10, 12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		12, 8, 7, 6, 8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
	},
	{
		4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12, 10, 11, 12, 16,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8, 6, 7, 8, 12,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
	},
	{
		5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8, 11, 10, 11, 12,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4, 7, 6, 7, 8,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
	},
	{
		6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7, 12, 11, 10, 11,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3, 8, 7, 6, 7,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
	},
	{
		10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6, 16, 12, 11, 10,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2, 12, 8, 7, 6,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
	},
	{
		5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
	},
	{
		6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
	},
	{
		7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
	},
	{
		11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
	},
	{
		6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
	},
	{
		7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
	},
	{
		8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
	},
	{
		12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
	},
	{
		10, 11, 12, 16, 6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		6, 7, 8, 12, 2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
	},
	{
		11, 10, 11, 12, 7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		7, 6, 7, 8, 3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
	},
	{
		12, 11, 10, 11, 8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		8, 7, 6, 7, 4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
	},
	{
		16, 12, 11, 10, 12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		12, 8, 7, 6, 8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
	},
	{
		8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16, 14, 15, 16, 20,
		4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12, 10, 11, 12, 16,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15, 13, 14, 15, 19,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14, 12, 13, 14, 18,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8, 6, 7, 8, 12,
		7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15, 13, 14, 15, 19,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
	},
	{
		9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12, 15, 14, 15, 16,
		5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8, 11, 10, 11, 12,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11, 14, 13, 14, 15,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10, 13, 12, 13, 14,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4, 7, 6, 7, 8,
		8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11, 14, 13, 14, 15,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
	},
	{
		10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11, 16, 15, 14, 15,
		6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7, 12, 11, 10, 11,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10, 15, 14, 13, 14,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9, 14, 13, 12, 13,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3, 8, 7, 6, 7,
		9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10, 15, 14, 13, 14,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
	},
	{
		14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10, 20, 16, 15, 14,
		10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6, 16, 12, 11, 10,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9, 19, 15, 14, 13,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8, 18, 14, 13, 12,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2, 12, 8, 7, 6,
		13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9, 19, 15, 14, 13,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
	},
	{
		9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16,
		5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8,
		8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
	},
	{
		10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12,
		6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4,
		9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
	},
	{
		11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11,
		7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3,
		10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
	},
	{
		15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10,
		11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2,
		14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
	},
	{
		10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15,
		6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7,
		9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
	},
	{
		11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11,
		7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3,
		10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
	},
	{
		12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10,
		8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2,
		11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
	},
	{
		16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9,
		12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1,
		15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
	},
	{
		14, 15, 16, 20, 10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14,
		10, 11, 12, 16, 6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		13, 14, 15, 19, 9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		12, 13, 14, 18, 8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		6, 7, 8, 12, 2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6,
		13, 14, 15, 19, 9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
	},
	{
		15, 14, 15, 16, 11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10,
		11, 10, 11, 12, 7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		14, 13, 14, 15, 10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		13, 12, 13, 14, 9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		7, 6, 7, 8, 3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2,
		14, 13, 14, 15, 10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
	},
	{
		16, 15, 14, 15, 12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9,
		12, 11, 10, 11, 8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		15, 14, 13, 14, 11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		14, 13, 12, 13, 10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		8, 7, 6, 7, 4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1,
		15, 14, 13, 14, 11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
	},
	{
		20, 16, 15, 14, 16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8,
		16, 12, 11, 10, 12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		19, 15, 14, 13, 15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		18, 14, 13, 12, 14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		12, 8, 7, 6, 8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0,
		19, 15, 14, 13, 15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
	},
	{
		6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14, 12, 13, 14, 18,
		7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15, 13, 14, 15, 19,
		8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16, 14, 15, 16, 20,
		12, 13, 14, 18, 13, 14, 15, 19, 14, 15, 16, 20, 18, 19, 20, 24,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12, 10, 11, 12, 16,
		8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16, 14, 15, 16, 20,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15, 13, 14, 15, 19,
		0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8, 6, 7, 8, 12,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14, 12, 13, 14, 18,
	},
	{
		7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10, 13, 12, 13, 14,
		8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11, 14, 13, 14, 15,
		9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12, 15, 14, 15, 16,
		13, 12, 13, 14, 14, 13, 14, 15, 15, 14, 15, 16, 19, 18, 19, 20,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8, 11, 10, 11, 12,
		9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12, 15, 14, 15, 16,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11, 14, 13, 14, 15,
		1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4, 7, 6, 7, 8,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10, 13, 12, 13, 14,
	},
	{
		8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9, 14, 13, 12, 13,
		9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10, 15, 14, 13, 14,
		10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11, 16, 15, 14, 15,
		14, 13, 12, 13, 15, 14, 13, 14, 16, 15, 14, 15, 20, 19, 18, 19,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7, 12, 11, 10, 11,
		10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11, 16, 15, 14, 15,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10, 15, 14, 13, 14,
		2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3, 8, 7, 6, 7,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9, 14, 13, 12, 13,
	},
	{
		12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8, 18, 14, 13, 12,
		13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9, 19, 15, 14, 13,
		14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10, 20, 16, 15, 14,
		18, 14, 13, 12, 19, 15, 14, 13, 20, 16, 15, 14, 24, 20, 19, 18,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6, 16, 12, 11, 10,
		14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10, 20, 16, 15, 14,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9, 19, 15, 14, 13,
		6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2, 12, 8, 7, 6,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8, 18, 14, 13, 12,
	},
	{
		7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14,
		8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15,
		9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16,
		13, 14, 15, 19, 12, 13, 14, 18, 13, 14, 15, 19, 14, 15, 16, 20,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12,
		9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15,
		1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14,
	},
	{
		8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10,
		9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11,
		10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12,
		14, 13, 14, 15, 13, 12, 13, 14, 14, 13, 14, 15, 15, 14, 15, 16,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8,
		10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11,
		2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10,
	},
	{
		9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9,
		10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10,
		11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11,
		15, 14, 13, 14, 14, 13, 12, 13, 15, 14, 13, 14, 16, 15, 14, 15,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7,
		11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10,
		3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9,
	},
	{
		13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8,
		14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9,
		15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10,
		19, 15, 14, 13, 18, 14, 13, 12, 19, 15, 14, 13, 20, 16, 15, 14,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6,
		15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9,
		7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8,
	},
	{
		8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13,
		9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14,
		10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15,
		14, 15, 16, 20, 13, 14, 15, 19, 12, 13, 14, 18, 13, 14, 15, 19,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11,
		10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14,
		2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13,
	},
	{
		9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9,
		10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10,
		11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11,
		15, 14, 15, 16, 14, 13, 14, 15, 13, 12, 13, 14, 14, 13, 14, 15,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7,
		11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10,
		3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9,
	},
	{
		10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8,
		11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9,
		12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10,
		16, 15, 14, 15, 15, 14, 13, 14, 14, 13, 12, 13, 15, 14, 13, 14,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6,
		12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9,
		4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8,
	},
	{
		14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7,
		15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8,
		16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9,
		20, 16, 15, 14, 19, 15, 14, 13, 18, 14, 13, 12, 19, 15, 14, 13,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5,
		16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8,
		8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7,
	},
	{
		12, 13, 14, 18, 8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12,
		13, 14, 15, 19, 9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13,
		14, 15, 16, 20, 10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14,
		18, 19, 20, 24, 14, 15, 16, 20, 13, 14, 15, 19, 12, 13, 14, 18,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		10, 11, 12, 16, 6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10,
		14, 15, 16, 20, 10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		13, 14, 15, 19, 9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13,
		6, 7, 8, 12, 2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		12, 13, 14, 18, 8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12,
	},
	{
		13, 12, 13, 14, 9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8,
		14, 13, 14, 15, 10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9,
		15, 14, 15, 16, 11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10,
		19, 18, 19, 20, 15, 14, 15, 16, 14, 13, 14, 15, 13, 12, 13, 14,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		11, 10, 11, 12, 7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6,
		15, 14, 15, 16, 11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		14, 13, 14, 15, 10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9,
		7, 6, 7, 8, 3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		13, 12, 13, 14, 9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8,
	},
	{
		14, 13, 12, 13, 10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7,
		15, 14, 13, 14, 11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8,
		16, 15, 14, 15, 12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9,
		20, 19, 18, 19, 16, 15, 14, 15, 15, 14, 13, 14, 14, 13, 12, 13,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		12, 11, 10, 11, 8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5,
		16, 15, 14, 15, 12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		15, 14, 13, 14, 11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8,
		8, 7, 6, 7, 4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		14, 13, 12, 13, 10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7,
	},
	{
		18, 14, 13, 12, 14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6,
		19, 15, 14, 13, 15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7,
		20, 16, 15, 14, 16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8,
		24, 20, 19, 18, 20, 16, 15, 14, 19, 15, 14, 13, 18, 14, 13, 12,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		16, 12, 11, 10, 12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4,
		20, 16, 15, 14, 16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		19, 15, 14, 13, 15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7,
		12, 8, 7, 6, 8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		18, 14, 13, 12, 14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6,
	},
	{
		7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15, 13, 14, 15, 19,
		6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14, 12, 13, 14, 18,
		7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15, 13, 14, 15, 19,
		8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16, 14, 15, 16, 20,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12, 10, 11, 12, 16,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8, 6, 7, 8, 12,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
	},
	{
		8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11, 14, 13, 14, 15,
		7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10, 13, 12, 13, 14,
		8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11, 14, 13, 14, 15,
		9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12, 15, 14, 15, 16,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8, 11, 10, 11, 12,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4, 7, 6, 7, 8,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
	},
	{
		9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10, 15, 14, 13, 14,
		8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9, 14, 13, 12, 13,
		9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10, 15, 14, 13, 14,
		10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11, 16, 15, 14, 15,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7, 12, 11, 10, 11,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3, 8, 7, 6, 7,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
	},
	{
		13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9, 19, 15, 14, 13,
		12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8, 18, 14, 13, 12,
		13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9, 19, 15, 14, 13,
		14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10, 20, 16, 15, 14,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6, 16, 12, 11, 10,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2, 12, 8, 7, 6,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
	},
	{
		8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15,
		7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14,
		8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15,
		9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
	},
	{
		9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11,
		8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10,
		9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11,
		10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
	},
	{
		10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10,
		9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9,
		10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10,
		11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
	},
	{
		14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9,
		13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8,
		14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9,
		15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
	},
	{
		9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14,
		8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13,
		9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14,
		10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
	},
	{
		10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10,
		9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9,
		10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10,
		11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
	},
	{
		11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9,
		10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8,
		11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9,
		12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
	},
	{
		15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8,
		14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7,
		15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8,
		16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
	},
	{
		13, 14, 15, 19, 9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13,
		12, 13, 14, 18, 8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12,
		13, 14, 15, 19, 9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13,
		14, 15, 16, 20, 10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		10, 11, 12, 16, 6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		6, 7, 8, 12, 2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
	},
	{
		14, 13, 14, 15, 10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9,
		13, 12, 13, 14, 9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8,
		14, 13, 14, 15, 10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9,
		15, 14, 15, 16, 11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		11, 10, 11, 12, 7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		7, 6, 7, 8, 3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
	},
	{
		15, 14, 13, 14, 11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8,
		14, 13, 12, 13, 10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7,
		15, 14, 13, 14, 11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8,
		16, 15, 14, 15, 12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		12, 11, 10, 11, 8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		8, 7, 6, 7, 4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
	},
	{
		19, 15, 14, 13, 15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7,
		18, 14, 13, 12, 14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6,
		19, 15, 14, 13, 15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7,
		20, 16, 15, 14, 16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		16, 12, 11, 10, 12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		12, 8, 7, 6, 8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
	},
	{
		8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16, 14, 15, 16, 20,
		7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15, 13, 14, 15, 19,
		6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14, 12, 13, 14, 18,
		7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15, 13, 14, 15, 19,
		4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12, 10, 11, 12, 16,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8, 6, 7, 8, 12,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
	},
	{
		9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12, 15, 14, 15, 16,
		8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11, 14, 13, 14, 15,
		7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10, 13, 12, 13, 14,
		8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11, 14, 13, 14, 15,
		5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8, 11, 10, 11, 12,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4, 7, 6, 7, 8,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
	},
	{
		10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11, 16, 15, 14, 15,
		9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10, 15, 14, 13, 14,
		8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9, 14, 13, 12, 13,
		9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10, 15, 14, 13, 14,
		6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7, 12, 11, 10, 11,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3, 8, 7, 6, 7,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
	},
	{
		14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10, 20, 16, 15, 14,
		13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9, 19, 15, 14, 13,
		12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8, 18, 14, 13, 12,
		13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9, 19, 15, 14, 13,
		10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6, 16, 12, 11, 10,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2, 12, 8, 7, 6,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
	},
	{
		9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16,
		8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15,
		7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14,
		8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15,
		5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
	},
	{
		10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12,
		9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11,
		8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10,
		9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11,
		6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
	},
	{
		11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11,
		10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10,
		9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9,
		10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10,
		7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
	},
	{
		15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10,
		14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9,
		13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8,
		14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9,
		11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
	},
	{
		10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15,
		9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14,
		8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13,
		9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14,
		6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
	},
	{
		11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11,
		10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10,
		9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9,
		10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10,
		7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
	},
	{
		12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10,
		11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9,
		10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8,
		11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9,
		8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
	},
	{
		16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9,
		15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8,
		14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7,
		15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8,
		12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
	},
	{
		14, 15, 16, 20, 10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14,
		13, 14, 15, 19, 9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13,
		12, 13, 14, 18, 8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12,
		13, 14, 15, 19, 9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13,
		10, 11, 12, 16, 6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		6, 7, 8, 12, 2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
	},
	{
		15, 14, 15, 16, 11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10,
		14, 13, 14, 15, 10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9,
		13, 12, 13, 14, 9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8,
		14, 13, 14, 15, 10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9,
		11, 10, 11, 12, 7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		7, 6, 7, 8, 3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
	},
	{
		16, 15, 14, 15, 12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9,
		15, 14, 13, 14, 11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8,
		14, 13, 12, 13, 10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7,
		15, 14, 13, 14, 11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8,
		12, 11, 10, 11, 8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		8, 7, 6, 7, 4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
	},
	{
		20, 16, 15, 14, 16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8,
		19, 15, 14, 13, 15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7,
		18, 14, 13, 12, 14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6,
		19, 15, 14, 13, 15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7,
		16, 12, 11, 10, 12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		12, 8, 7, 6, 8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
	},
	{
		12, 13, 14, 18, 13, 14, 15, 19, 14, 15, 16, 20, 18, 19, 20, 24,
		8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16, 14, 15, 16, 20,
		7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15, 13, 14, 15, 19,
		6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14, 12, 13, 14, 18,
		8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16, 14, 15, 16, 20,
		4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12, 10, 11, 12, 16,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15, 13, 14, 15, 19,
		3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11, 9, 10, 11, 15,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14, 12, 13, 14, 18,
		2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10, 8, 9, 10, 14,
		1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9, 7, 8, 9, 13,
		0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8, 6, 7, 8, 12,
	},
	{
		13, 12, 13, 14, 14, 13, 14, 15, 15, 14, 15, 16, 19, 18, 19, 20,
		9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12, 15, 14, 15, 16,
		8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11, 14, 13, 14, 15,
		7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10, 13, 12, 13, 14,
		9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12, 15, 14, 15, 16,
		5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8, 11, 10, 11, 12,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11, 14, 13, 14, 15,
		4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7, 10, 9, 10, 11,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10, 13, 12, 13, 14,
		3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6, 9, 8, 9, 10,
		2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5, 8, 7, 8, 9,
		1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4, 7, 6, 7, 8,
	},
	{
		14, 13, 12, 13, 15, 14, 13, 14, 16, 15, 14, 15, 20, 19, 18, 19,
		10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11, 16, 15, 14, 15,
		9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10, 15, 14, 13, 14,
		8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9, 14, 13, 12, 13,
		10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11, 16, 15, 14, 15,
		6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7, 12, 11, 10, 11,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10, 15, 14, 13, 14,
		5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6, 11, 10, 9, 10,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9, 14, 13, 12, 13,
		4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5, 10, 9, 8, 9,
		3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4, 9, 8, 7, 8,
		2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3, 8, 7, 6, 7,
	},
	{
		18, 14, 13, 12, 19, 15, 14, 13, 20, 16, 15, 14, 24, 20, 19, 18,
		14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10, 20, 16, 15, 14,
		13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9, 19, 15, 14, 13,
		12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8, 18, 14, 13, 12,
		14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10, 20, 16, 15, 14,
		10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6, 16, 12, 11, 10,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9, 19, 15, 14, 13,
		9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5, 15, 11, 10, 9,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8, 18, 14, 13, 12,
		8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4, 14, 10, 9, 8,
		7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3, 13, 9, 8, 7,
		6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2, 12, 8, 7, 6,
	},
	{
		13, 14, 15, 19, 12, 13, 14, 18, 13, 14, 15, 19, 14, 15, 16, 20,
		9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16,
		8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15,
		7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14,
		9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15, 10, 11, 12, 16,
		5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11, 6, 7, 8, 12,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14, 9, 10, 11, 15,
		4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10, 5, 6, 7, 11,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13, 8, 9, 10, 14,
		3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9, 4, 5, 6, 10,
		2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8, 3, 4, 5, 9,
		1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7, 2, 3, 4, 8,
	},
	{
		14, 13, 14, 15, 13, 12, 13, 14, 14, 13, 14, 15, 15, 14, 15, 16,
		10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12,
		9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11,
		8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10,
		10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11, 11, 10, 11, 12,
		6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7, 7, 6, 7, 8,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10, 10, 9, 10, 11,
		5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6, 6, 5, 6, 7,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9, 9, 8, 9, 10,
		4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5, 5, 4, 5, 6,
		3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4, 4, 3, 4, 5,
		2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3, 3, 2, 3, 4,
	},
	{
		15, 14, 13, 14, 14, 13, 12, 13, 15, 14, 13, 14, 16, 15, 14, 15,
		11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11,
		10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10,
		9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9,
		11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10, 12, 11, 10, 11,
		7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6, 8, 7, 6, 7,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9, 11, 10, 9, 10,
		6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5, 7, 6, 5, 6,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8, 10, 9, 8, 9,
		5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4, 6, 5, 4, 5,
		4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3, 5, 4, 3, 4,
		3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2, 4, 3, 2, 3,
	},
	{
		19, 15, 14, 13, 18, 14, 13, 12, 19, 15, 14, 13, 20, 16, 15, 14,
		15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10,
		14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9,
		13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8,
		15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9, 16, 12, 11, 10,
		11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5, 12, 8, 7, 6,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8, 15, 11, 10, 9,
		10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4, 11, 7, 6, 5,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7, 14, 10, 9, 8,
		9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3, 10, 6, 5, 4,
		8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2, 9, 5, 4, 3,
		7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1, 8, 4, 3, 2,
	},
	{
		14, 15, 16, 20, 13, 14, 15, 19, 12, 13, 14, 18, 13, 14, 15, 19,
		10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15,
		9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14,
		8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13,
		10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14, 9, 10, 11, 15,
		6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10, 5, 6, 7, 11,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13, 8, 9, 10, 14,
		5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9, 4, 5, 6, 10,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12, 7, 8, 9, 13,
		4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8, 3, 4, 5, 9,
		3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7, 2, 3, 4, 8,
		2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6, 1, 2, 3, 7,
	},
	{
		15, 14, 15, 16, 14, 13, 14, 15, 13, 12, 13, 14, 14, 13, 14, 15,
		11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11,
		10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10,
		9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9,
		11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10, 10, 9, 10, 11,
		7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6, 6, 5, 6, 7,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9, 9, 8, 9, 10,
		6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5, 5, 4, 5, 6,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8, 8, 7, 8, 9,
		5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4, 4, 3, 4, 5,
		4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3, 3, 2, 3, 4,
		3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2, 2, 1, 2, 3,
	},
	{
		16, 15, 14, 15, 15, 14, 13, 14, 14, 13, 12, 13, 15, 14, 13, 14,
		12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10,
		11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9,
		10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8,
		12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9, 11, 10, 9, 10,
		8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5, 7, 6, 5, 6,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8, 10, 9, 8, 9,
		7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4, 6, 5, 4, 5,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7, 9, 8, 7, 8,
		6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3, 5, 4, 3, 4,
		5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2, 4, 3, 2, 3,
		4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1, 3, 2, 1, 2,
	},
	{
		20, 16, 15, 14, 19, 15, 14, 13, 18, 14, 13, 12, 19, 15, 14, 13,
		16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9,
		15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8,
		14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7,
		16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8, 15, 11, 10, 9,
		12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4, 11, 7, 6, 5,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7, 14, 10, 9, 8,
		11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3, 10, 6, 5, 4,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6, 13, 9, 8, 7,
		10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2, 9, 5, 4, 3,
		9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1, 8, 4, 3, 2,
		8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0, 7, 3, 2, 1,
	},
	{
		18, 19, 20, 24, 14, 15, 16, 20, 13, 14, 15, 19, 12, 13, 14, 18,
		14, 15, 16, 20, 10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14,
		13, 14, 15, 19, 9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13,
		12, 13, 14, 18, 8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12,
		14, 15, 16, 20, 10, 11, 12, 16, 9, 10, 11, 15, 8, 9, 10, 14,
		10, 11, 12, 16, 6, 7, 8, 12, 5, 6, 7, 11, 4, 5, 6, 10,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		13, 14, 15, 19, 9, 10, 11, 15, 8, 9, 10, 14, 7, 8, 9, 13,
		9, 10, 11, 15, 5, 6, 7, 11, 4, 5, 6, 10, 3, 4, 5, 9,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		12, 13, 14, 18, 8, 9, 10, 14, 7, 8, 9, 13, 6, 7, 8, 12,
		8, 9, 10, 14, 4, 5, 6, 10, 3, 4, 5, 9, 2, 3, 4, 8,
		7, 8, 9, 13, 3, 4, 5, 9, 2, 3, 4, 8, 1, 2, 3, 7,
		6, 7, 8, 12, 2, 3, 4, 8, 1, 2, 3, 7, 0, 1, 2, 6,
	},
	{
		19, 18, 19, 20, 15, 14, 15, 16, 14, 13, 14, 15, 13, 12, 13, 14,
		15, 14, 15, 16, 11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10,
		14, 13, 14, 15, 10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9,
		13, 12, 13, 14, 9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8,
		15, 14, 15, 16, 11, 10, 11, 12, 10, 9, 10, 11, 9, 8, 9, 10,
		11, 10, 11, 12, 7, 6, 7, 8, 6, 5, 6, 7, 5, 4, 5, 6,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		14, 13, 14, 15, 10, 9, 10, 11, 9, 8, 9, 10, 8, 7, 8, 9,
		10, 9, 10, 11, 6, 5, 6, 7, 5, 4, 5, 6, 4, 3, 4, 5,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		13, 12, 13, 14, 9, 8, 9, 10, 8, 7, 8, 9, 7, 6, 7, 8,
		9, 8, 9, 10, 5, 4, 5, 6, 4, 3, 4, 5, 3, 2, 3, 4,
		8, 7, 8, 9, 4, 3, 4, 5, 3, 2, 3, 4, 2, 1, 2, 3,
		7, 6, 7, 8, 3, 2, 3, 4, 2, 1, 2, 3, 1, 0, 1, 2,
	},
	{
		20, 19, 18, 19, 16, 15, 14, 15, 15, 14, 13, 14, 14, 13, 12, 13,
		16, 15, 14, 15, 12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9,
		15, 14, 13, 14, 11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8,
		14, 13, 12, 13, 10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7,
		16, 15, 14, 15, 12, 11, 10, 11, 11, 10, 9, 10, 10, 9, 8, 9,
		12, 11, 10, 11, 8, 7, 6, 7, 7, 6, 5, 6, 6, 5, 4, 5,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		15, 14, 13, 14, 11, 10, 9, 10, 10, 9, 8, 9, 9, 8, 7, 8,
		11, 10, 9, 10, 7, 6, 5, 6, 6, 5, 4, 5, 5, 4, 3, 4,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		14, 13, 12, 13, 10, 9, 8, 9, 9, 8, 7, 8, 8, 7, 6, 7,
		10, 9, 8, 9, 6, 5, 4, 5, 5, 4, 3, 4, 4, 3, 2, 3,
		9, 8, 7, 8, 5, 4, 3, 4, 4, 3, 2, 3, 3, 2, 1, 2,
		8, 7, 6, 7, 4, 3, 2, 3, 3, 2, 1, 2, 2, 1, 0, 1,
	},
	{
		24, 20, 19, 18, 20, 16, 15, 14, 19, 15, 14, 13, 18, 14, 13, 12,
		20, 16, 15, 14, 16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8,
		19, 15, 14, 13, 15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7,
		18, 14, 13, 12, 14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6,
		20, 16, 15, 14, 16, 12, 11, 10, 15, 11, 10, 9, 14, 10, 9, 8,
		16, 12, 11, 10, 12, 8, 7, 6, 11, 7, 6, 5, 10, 6, 5, 4,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		19, 15, 14, 13, 15, 11, 10, 9, 14, 10, 9, 8, 13, 9, 8, 7,
		15, 11, 10, 9, 11, 7, 6, 5, 10, 6, 5, 4, 9, 5, 4, 3,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		18, 14, 13, 12, 14, 10, 9, 8, 13, 9, 8, 7, 12, 8, 7, 6,
		14, 10, 9, 8, 10, 6, 5, 4, 9, 5, 4, 3, 8, 4, 3, 2,
		13, 9, 8, 7, 9, 5, 4, 3, 8, 4, 3, 2, 7, 3, 2, 1,
		12, 8, 7, 6, 8, 4, 3, 2, 7, 3, 2, 1, 6, 2, 1, 0,
	},
}
