"use strict";
/**
 * npm_install_proxy -- localhost NPM registry to `npm install` without network
 *
 * Copyright (C) 2020  SUSE LLC
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.mainEntryFunction = void 0;
const process_1 = require("process");
const child_process_1 = require("child_process");
const url_1 = require("url");
const process_2 = require("process");
const registry_1 = require("./registry");
const service_1 = require("./service");
const fs_registry_1 = require("./fs_registry");
const dir_registry_1 = require("./dir_registry");
const concurrent_processes = 100;
const install_options = [];
function registerTarballsFromCommandline(registry) {
    const processes = [];
    for (let i = 0; i < concurrent_processes; i++)
        processes.push(Promise.resolve(0));
    for (let i = 2; i < process_1.argv.length; ++i) {
        const fn = process_1.argv[i];
        processes[i % concurrent_processes] = processes[i % concurrent_processes].then((processed) => registry.register(fn).then(n => {
            if (n == 0)
                install_options.push(process_1.argv[i]);
            return processed + n;
        }));
    }
    return Promise.all(processes).then((vals) => {
        let total = 0;
        for (let i = 0; i < vals.length; ++i)
            total += vals[i];
        console.log(`Serving ${total} packages`);
        return total;
    });
}
function setupServerAndGetPort(service, registry) {
    return new Promise(accepted => {
        const server = service.run(registry).on("listening", () => {
            const addr = server.address();
            console.log(addr);
            if (typeof addr === 'object')
                accepted(addr.port);
        });
    });
}
function configureNpmToSpecificLocalhostPort(service, port) {
    return new Promise((accept, reject) => {
        (0, child_process_1.spawn)("npm", ['config', 'set', 'registry', service.url.toString()], { stdio: 'inherit' })
            .on("exit", (code) => {
            code === 0 ? accept() : reject();
        });
    });
}
function cleanupNpmLocalhostConfig() {
    return new Promise((accept, reject) => {
        (0, child_process_1.spawn)("npm", ['config', 'delete', 'registry'])
            .on("exit", (code) => {
            code === 0 ? accept() : reject();
        });
    });
}
function runNpmInstall() {
    if (install_options.length === 0) {
        console.log("npm install skipped");
        return Promise.reject("npm install skipped");
    }
    return new Promise((accept, reject) => {
        (0, child_process_1.spawn)("npm", install_options.concat("--no-package-lock"), { stdio: 'inherit' })
            .on("exit", (code) => {
            code === 0 ? accept() : reject("NPM returned code: " + code);
        });
    });
}
function printHelpInformation() {
    console.log("   usage: index [ npm files | npm tarball directories ] ... [npm run options] ");
    console.log("--help    prints this help message");
    console.log("--debug   starts service and listens on localhost until killed");
}
function mainEntryFunction() {
    if (process_2.env.QUILT_COMMAND === "setup") {
        console.log("Run in quilt setup mode. 'npm install' skipped. Run 'npm ci' manually.");
        return;
    }
    if (process_1.argv.includes("--help")) {
        printHelpInformation();
        return;
    }
    const isDebug = process_1.argv.includes("--debug");
    const registry = new registry_1.Registry();
    registry.addBackend(new fs_registry_1.TarballRegistryBackend);
    registry.addBackend(new dir_registry_1.DirRegistryBackend);
    const service = new service_1.Service({ url: new url_1.URL("http://localhost") });
    registry.serviceProvider = service;
    const s = registerTarballsFromCommandline(registry)
        .then(() => setupServerAndGetPort(service, registry));
    if (isDebug)
        return;
    return s.then(port => configureNpmToSpecificLocalhostPort(service, port))
        .then(() => runNpmInstall())
        .then(() => {
        console.log("npm done. Shutting down proxy");
    })
        .catch(msg => {
        process.exitCode = 1;
        console.log("An error occurred: " + msg);
    })
        .finally(() => {
        return service.stop()
            .then(cleanupNpmLocalhostConfig);
    });
}
exports.mainEntryFunction = mainEntryFunction;
if (require.main === module) {
    mainEntryFunction();
}
//# sourceMappingURL=index.js.map