/*
 * Copyright (C) 2025, Stephan Mueller <smueller@chronox.de>
 *
 * License: see LICENSE file in root directory
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ALL OF
 * WHICH ARE HEREBY DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF NOT ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

use leancrypto_sys::lcr_kmac::lcr_kmac;
use leancrypto_sys::lcr_kmac::lcr_kmac_type;

fn lc_rust_kmac_xof_squeeze_more(kmac_type: lcr_kmac_type) {
	let data: [u8; 3] = [ 0x00, 0x01, 0x02 ];
	let mut kmac = lcr_kmac::new(kmac_type);

	let mut out_full = vec![0u8; 20];
	let result = kmac.kmac_xof(&data, &data, &data, &mut out_full);
	assert_eq!(result, Ok(()));

	let result = kmac.init(&data, &data);
	assert_eq!(result, Ok(()));
	let result = kmac.update(&data);
	assert_eq!(result, Ok(()));
	let mut out_small1 = vec![0u8; 5];
	let result = kmac.fini_xof(&mut out_small1);
	assert_eq!(result, Ok(()));

	let mut out_small2 = vec![0u8; 15];
	let result = kmac.fini_xof(&mut out_small2);
	assert_eq!(result, Ok(()));

	// Concatenate both values
	let mut out_small_concat: Vec<u8> = vec![];
	out_small_concat.extend(out_small1);
	out_small_concat.extend(out_small2);

	// Check that value with one squeeze matches that with two squeezes
	assert_eq!(out_small_concat, out_full);
}

fn lc_rust_kmac_one(kmac_type: lcr_kmac_type, key: &[u8], s: &[u8], msg: &[u8], exp: &[u8]) {
	let mut kmac = lcr_kmac::new(kmac_type);

	let mut mac = vec![0u8; exp.len()];
	let result = kmac.kmac(key, s, msg, &mut mac);
	assert_eq!(result, Ok(()));
	assert_eq!(mac, exp);

	let result = kmac.init(key, s);
	assert_eq!(result, Ok(()));
	let result = kmac.update(msg);
	assert_eq!(result, Ok(()));
	let result = kmac.fini(&mut mac);
	assert_eq!(result, Ok(()));
	assert_eq!(mac, exp);
}

#[test]
fn lc_rust_kmac_xof_squeeze_more_256() {
	lc_rust_kmac_xof_squeeze_more(lcr_kmac_type::lcr_kmac_256)
}

#[test]
fn lc_rust_kmac_xof_squeeze_more_128() {
	lc_rust_kmac_xof_squeeze_more(lcr_kmac_type::lcr_kmac_128)
}

#[test]
fn lc_rust_kmac_256() {
	let msg: [u8; 8192] = [
		0x89, 0xBA, 0x9E, 0x35, 0xD8, 0xCA, 0x69, 0xFB, 0x8A, 0x46,
		0x52, 0x7D, 0x8F, 0xD0, 0x73, 0x00, 0x21, 0x2C, 0x5C, 0xC2,
		0xDD, 0xEC, 0x8D, 0x18, 0xD2, 0xEB, 0x05, 0x45, 0xF1, 0xD3,
		0x2B, 0x86, 0x9F, 0x0F, 0x6D, 0xB5, 0xDE, 0x75, 0x5F, 0x55,
		0x31, 0xE4, 0x49, 0x94, 0x04, 0x52, 0xD5, 0x05, 0x98, 0x5E,
		0xBB, 0x4D, 0x97, 0xB2, 0x74, 0xF6, 0x6F, 0x7C, 0xAA, 0x38,
		0x1F, 0x9B, 0xE0, 0xD3, 0xDB, 0x57, 0xD0, 0x65, 0xD6, 0x0C,
		0x5E, 0xDD, 0x7A, 0xF4, 0x74, 0x52, 0xE3, 0xFC, 0x94, 0x7D,
		0x23, 0x25, 0xE0, 0xD2, 0xA6, 0x06, 0xDD, 0x6F, 0x75, 0x43,
		0xC1, 0x3D, 0x52, 0xDA, 0x49, 0xA6, 0x37, 0x2C, 0x91, 0x99,
		0xD0, 0xF9, 0x2A, 0x79, 0x30, 0x62, 0x54, 0x75, 0x22, 0xEA,
		0x39, 0x31, 0xB9, 0xFD, 0x12, 0x68, 0x41, 0x17, 0x00, 0x6C,
		0xF0, 0xA1, 0x5F, 0x00, 0x85, 0x8F, 0x3F, 0xCE, 0x69, 0x97,
		0x3C, 0x1D, 0x2E, 0x4D, 0x7B, 0x36, 0x8D, 0x22, 0xE9, 0xA7,
		0x73, 0xCE, 0x11, 0xD4, 0x44, 0x47, 0xCC, 0x92, 0xA9, 0xB2,
		0x3D, 0x74, 0x49, 0xBF, 0xE7, 0x4B, 0x30, 0xB8, 0x9B, 0x1A,
		0x09, 0xC7, 0x43, 0x33, 0xBE, 0x7E, 0x78, 0x86, 0x8F, 0xBB,
		0x1B, 0x7F, 0x8B, 0x49, 0x2C, 0xD8, 0xED, 0x8F, 0xB9, 0x21,
		0xAA, 0x49, 0x96, 0xAD, 0xCB, 0xFB, 0x9B, 0xA6, 0xD3, 0x0A,
		0xF2, 0xA8, 0xA1, 0x75, 0xF0, 0x9F, 0xA6, 0xFE, 0x63, 0xFD,
		0x68, 0x76, 0xDB, 0x3F, 0x5F, 0x08, 0x70, 0xE5, 0x65, 0xBB,
		0x56, 0x22, 0x61, 0x3D, 0x90, 0xFA, 0x67, 0xEA, 0x32, 0x22,
		0x7C, 0xE0, 0xFC, 0x6B, 0xFB, 0x7A, 0x88, 0xFC, 0x2E, 0x92,
		0x93, 0x43, 0xB1, 0x1A, 0x5E, 0x3A, 0x43, 0xFE, 0x97, 0xFE,
		0x1E, 0x17, 0x87, 0xD0, 0x75, 0xED, 0x23, 0x29, 0x72, 0x1F,
		0x31, 0x39, 0x42, 0x15, 0x18, 0xC7, 0xB1, 0x1E, 0x3E, 0xF7,
		0x06, 0x8F, 0xFC, 0xEC, 0x8C, 0xCC, 0x10, 0x85, 0x66, 0x02,
		0x19, 0x99, 0x24, 0x75, 0x38, 0x94, 0xA7, 0xAE, 0x8A, 0x39,
		0x72, 0x8E, 0x8B, 0x2B, 0x7F, 0x1F, 0x10, 0xC1, 0xA0, 0xC5,
		0x3D, 0x5C, 0x97, 0xBB, 0x73, 0x57, 0xAC, 0x2E, 0x7F, 0xFF,
		0x27, 0x95, 0xB8, 0xFD, 0x4A, 0xB1, 0x59, 0x52, 0xD7, 0xB8,
		0x8C, 0xEF, 0x7B, 0xD5, 0xB3, 0x7A, 0xE6, 0xA9, 0x1B, 0x0B,
		0xCC, 0x37, 0x1B, 0x1E, 0x00, 0x64, 0x4A, 0x7D, 0xC8, 0x0F,
		0x4D, 0x33, 0x9D, 0x05, 0x0F, 0x25, 0x09, 0x72, 0xD4, 0x9A,
		0x2A, 0xDC, 0xF9, 0xBB, 0xE4, 0x3D, 0xC1, 0xE2, 0x39, 0x3B,
		0x67, 0xF9, 0x62, 0x1D, 0x05, 0x37, 0xBB, 0xC1, 0x6F, 0x1F,
		0x4C, 0x84, 0x2F, 0xD1, 0x73, 0x70, 0x2D, 0x3D, 0xB8, 0xEC,
		0xB5, 0x4E, 0xA3, 0x7E, 0x28, 0xD2, 0x6F, 0xD6, 0xE0, 0x6F,
		0x83, 0x96, 0x01, 0xF7, 0xE1, 0xC1, 0xA4, 0xCB, 0x43, 0xC8,
		0x11, 0xBF, 0xC0, 0x77, 0x19, 0x53, 0xAE, 0x69, 0x91, 0xC4,
		0xAD, 0x44, 0x78, 0x3B, 0x33, 0x48, 0xBB, 0x06, 0xE8, 0x48,
		0xFF, 0xCC, 0x80, 0x2F, 0x96, 0x65, 0xD7, 0xDE, 0x3A, 0x05,
		0x34, 0xDB, 0x81, 0x31, 0x27, 0xE8, 0x96, 0x5E, 0x85, 0x11,
		0x1B, 0x6F, 0xA2, 0x61, 0x9F, 0x1A, 0xFC, 0xD9, 0xD6, 0x1B,
		0xC6, 0xD9, 0xA6, 0xC4, 0xF9, 0x39, 0xA8, 0x56, 0xD6, 0x9F,
		0x65, 0xCA, 0xD8, 0x34, 0x26, 0x4D, 0x5E, 0x75, 0x14, 0x09,
		0x64, 0x81, 0xE0, 0x8E, 0x0D, 0xF9, 0x06, 0x28, 0x86, 0x6C,
		0x04, 0x3D, 0xA8, 0x13, 0x39, 0x5B, 0x28, 0xDF, 0xA6, 0x3C,
		0xF2, 0xFA, 0xE0, 0x65, 0xA0, 0x28, 0x68, 0xA9, 0x8A, 0x25,
		0x87, 0x00, 0x16, 0x7E, 0x2F, 0xFF, 0x34, 0xCE, 0xA4, 0x9F,
		0x75, 0x9A, 0x51, 0xB6, 0x04, 0x05, 0xF2, 0x41, 0xDD, 0xB0,
		0x72, 0xE5, 0x76, 0x2A, 0x87, 0x93, 0x81, 0x14, 0x5D, 0xB1,
		0x99, 0x7C, 0xC3, 0xD7, 0x23, 0x09, 0x42, 0x9B, 0x55, 0x16,
		0x0B, 0x9B, 0xC5, 0x1C, 0x67, 0x7A, 0x8E, 0x61, 0x79, 0xFD,
		0xDA, 0x01, 0x25, 0x4D, 0xFD, 0xAA, 0x0A, 0x59, 0x1A, 0x84,
		0x16, 0x3F, 0x69, 0x69, 0xBC, 0xDC, 0x14, 0xDF, 0x18, 0x7E,
		0x03, 0xCD, 0x43, 0x03, 0xFD, 0x6F, 0x90, 0x80, 0x84, 0xA2,
		0x17, 0x98, 0x53, 0xC7, 0x1B, 0x55, 0x58, 0xD7, 0xB6, 0x3B,
		0x87, 0x45, 0x68, 0x98, 0x19, 0x08, 0x0B, 0x55, 0x9D, 0xEB,
		0x1C, 0xA6, 0xF9, 0xCD, 0xC4, 0xF6, 0x57, 0x7E, 0x24, 0xB3,
		0xE5, 0x92, 0x9E, 0x83, 0x1D, 0x30, 0x47, 0x74, 0xBF, 0x59,
		0xC7, 0x30, 0x85, 0x11, 0x4C, 0xC7, 0xC3, 0x9A, 0x81, 0x21,
		0x23, 0x76, 0x12, 0x22, 0x25, 0x34, 0xC8, 0x9B, 0x3C, 0xDB,
		0xAE, 0x20, 0x08, 0x3C, 0x4D, 0xD4, 0x0D, 0x2C, 0x33, 0x9B,
		0x2A, 0x2E, 0xCB, 0xAF, 0x0D, 0x41, 0xDA, 0xC2, 0x69, 0x00,
		0xD1, 0x06, 0x9E, 0x2C, 0xF0, 0x33, 0x66, 0xAC, 0xAF, 0x50,
		0xCB, 0x72, 0xA3, 0x8B, 0x09, 0xD5, 0x71, 0x60, 0xD4, 0xB5,
		0xC8, 0x94, 0xA0, 0x20, 0x4E, 0x13, 0xA9, 0x78, 0xCF, 0xD4,
		0x85, 0x94, 0x0D, 0xE0, 0x66, 0xB7, 0xF6, 0x79, 0x9C, 0x21,
		0x89, 0xD6, 0xC1, 0xD3, 0xBB, 0x78, 0x42, 0x75, 0x28, 0xDD,
		0x15, 0x5F, 0x56, 0x77, 0xE1, 0x37, 0xBE, 0x95, 0xA6, 0x8C,
		0x55, 0xEF, 0x82, 0x70, 0x24, 0x87, 0xEA, 0xE6, 0x00, 0x31,
		0xB2, 0x87, 0xC0, 0x2A, 0x2C, 0xA9, 0xCC, 0x5E, 0xB5, 0xF4,
		0xA0, 0xFE, 0x68, 0x51, 0x45, 0xA9, 0x1B, 0xF8, 0xC6, 0x66,
		0xE9, 0xA9, 0x01, 0x5C, 0x4C, 0xB6, 0x80, 0x8D, 0x32, 0x71,
		0x6C, 0xE2, 0x73, 0x90, 0x2F, 0x5B, 0x09, 0x71, 0x6F, 0xA7,
		0x9D, 0xA7, 0x3F, 0xA8, 0xE9, 0xA1, 0xA6, 0x03, 0xFE, 0x56,
		0xF8, 0x57, 0xC9, 0x73, 0xC2, 0xE1, 0xE2, 0xC9, 0x9C, 0x9C,
		0x51, 0x6A, 0x06, 0xF4, 0x9C, 0x89, 0xA0, 0x0E, 0x55, 0x43,
		0x00, 0x13, 0x4F, 0x90, 0xDA, 0xAB, 0x8B, 0xE4, 0x26, 0xE8,
		0x52, 0xD2, 0xF7, 0x39, 0x65, 0x09, 0x6C, 0x13, 0x9F, 0x84,
		0x9E, 0x09, 0x85, 0xBE, 0x37, 0x78, 0x53, 0xCA, 0x8C, 0x0F,
		0x8A, 0xF9, 0xD5, 0xF6, 0x90, 0x5D, 0x60, 0x36, 0x87, 0x62,
		0x66, 0x81, 0xD2, 0x21, 0x5B, 0x89, 0xE2, 0x59, 0xBF, 0x29,
		0x98, 0x2A, 0x07, 0x2A, 0x39, 0x90, 0x71, 0xB0, 0xC6, 0x61,
		0x0D, 0xF1, 0x66, 0xF9, 0x0A, 0xAD, 0x5B, 0xDA, 0x92, 0x25,
		0x32, 0x05, 0x3E, 0xA7, 0xAE, 0xD8, 0x99, 0xF6, 0x6F, 0xFE,
		0xD9, 0x61, 0xBC, 0xF2, 0x08, 0x94, 0xA6, 0x31, 0xD0, 0x23,
		0x83, 0x93, 0x3B, 0xC5, 0x09, 0x15, 0x5A, 0x6C, 0x17, 0xCA,
		0x87, 0xBB, 0xBA, 0x7B, 0x1C, 0xA1, 0xBF, 0xAE, 0x5B, 0x0A,
		0xB3, 0xF4, 0x44, 0x01, 0x27, 0x17, 0x06, 0xC0, 0x65, 0x22,
		0x78, 0xE3, 0x96, 0x4D, 0xA0, 0x4F, 0xA7, 0xFD, 0x56, 0x0A,
		0xE3, 0x4D, 0xEA, 0xB3, 0x00, 0x00, 0x78, 0xC3, 0xB3, 0x6E,
		0xB7, 0x1F, 0x14, 0x66, 0x70, 0x24, 0xE3, 0xE2, 0xC2, 0x60,
		0x1F, 0x3D, 0xBA, 0x02, 0x7E, 0x88, 0xBC, 0x33, 0x3E, 0xCF,
		0x10, 0x76, 0xCE, 0x70, 0x71, 0x23, 0x18, 0x27, 0xE4, 0xE9,
		0x78, 0x26, 0xFE, 0x20, 0x25, 0xF6, 0x6B, 0x52, 0x80, 0xDF,
		0x1E, 0xB7, 0x73, 0x24, 0x7B, 0x54, 0x71, 0x03, 0xDB, 0xE4,
		0xEA, 0xF0, 0xE9, 0xB1, 0x8E, 0xDC, 0x16, 0x80, 0xB9, 0xF1,
		0x03, 0x01, 0x99, 0x35, 0x85, 0x9A, 0x91, 0x2B, 0x06, 0x59,
		0x6D, 0xA7, 0x24, 0x28, 0x81, 0x9E, 0x52, 0xE4, 0xF1, 0x2A,
		0x15, 0x1C, 0x29, 0x7E, 0x1F, 0xFD, 0xC1, 0x8F, 0x4B, 0x2B,
		0x50, 0xD9, 0xF9, 0xA4, 0x08, 0xA0, 0x55, 0xC5, 0x09, 0x6C,
		0x7F, 0x86, 0x20, 0x04, 0x63, 0x0A, 0xEE, 0xAA, 0x41, 0xD9,
		0x08, 0x66, 0x59, 0xB3, 0x95, 0x0C, 0xA2, 0xEF, 0xFB, 0x8E,
		0x07, 0x11, 0xB1, 0x26, 0x5D, 0xD0, 0xB1, 0x12, 0xD2, 0xC2,
		0x78, 0x5A, 0x4A, 0x4B, 0xDE, 0xAA, 0x28, 0xCB, 0x6D, 0x1B,
		0x8B, 0xC8, 0xA1, 0xFD, 0x52, 0xAD, 0xA9, 0xAB, 0xB6, 0xDA,
		0x62, 0xCD, 0x20, 0x53, 0xB8, 0x5D, 0x31, 0x49, 0xCF, 0xD1,
		0x25, 0x7A, 0x73, 0xE2, 0xD7, 0x5A, 0x14, 0x03, 0x5E, 0x4C,
		0xE6, 0x04, 0xBD, 0x1B, 0xE2, 0xBF, 0x6F, 0xE3, 0x6F, 0x2D,
		0xA1, 0xD6, 0x8C, 0x45, 0xA5, 0x4F, 0xB5, 0x5E, 0x4F, 0x3D,
		0xF7, 0x0F, 0xD0, 0x2B, 0x37, 0x6E, 0x88, 0x10, 0xE0, 0xC2,
		0xDB, 0x4B, 0x82, 0x43, 0x20, 0x08, 0x1D, 0xD5, 0xB2, 0x07,
		0x9A, 0x28, 0x58, 0xA0, 0xFC, 0x5C, 0xE8, 0xFB, 0xF3, 0xB7,
		0x1D, 0xEF, 0x4E, 0xE0, 0x8F, 0x07, 0x0D, 0xDA, 0xF1, 0x95,
		0xC7, 0x0A, 0x0B, 0xC8, 0xCD, 0x9D, 0x33, 0x74, 0x41, 0x99,
		0x06, 0xD3, 0x4E, 0x9D, 0x1D, 0x87, 0x17, 0xDD, 0x05, 0x28,
		0x03, 0x4F, 0x1A, 0xEC, 0x8B, 0xA9, 0xE8, 0xB5, 0xB7, 0x61,
		0xA1, 0x24, 0xA8, 0x01, 0xA5, 0xB8, 0xB0, 0x70, 0x7D, 0x01,
		0x85, 0xEC, 0x4A, 0x0E, 0xA3, 0xA9, 0x99, 0x37, 0x96, 0x47,
		0x69, 0xAE, 0xFF, 0x9A, 0x97, 0x6C, 0xC2, 0xBC, 0x38, 0x4B,
		0x9F, 0xB9, 0x60, 0x8B, 0xA7, 0xAA, 0x7C, 0xE7, 0xD5, 0xB4,
		0xCE, 0x80, 0x8C, 0x04, 0xC3, 0x9A, 0xBB, 0xDD, 0xA8, 0xC4,
		0xF5, 0x88, 0x6F, 0x18, 0xED, 0x49, 0xCE, 0x25, 0xE9, 0x7A,
		0x6D, 0x7F, 0xF0, 0x0A, 0x06, 0x48, 0x87, 0x06, 0xEC, 0x4B,
		0x6D, 0x8E, 0xBF, 0x06, 0x60, 0x56, 0xF8, 0xCD, 0xBF, 0x80,
		0xE6, 0x30, 0x3B, 0x29, 0x56, 0xA6, 0x02, 0xE7, 0x92, 0x93,
		0xF5, 0xE9, 0x76, 0x95, 0xBA, 0x2E, 0xD7, 0x84, 0xD2, 0x83,
		0x1B, 0xA2, 0x0D, 0xE3, 0xED, 0x4B, 0x2F, 0x71, 0x00, 0x7F,
		0xDC, 0xC4, 0x0F, 0x5F, 0x0F, 0x27, 0xAE, 0xC8, 0x8F, 0xD9,
		0xCB, 0xA2, 0xA8, 0xFA, 0x2E, 0x5A, 0xD9, 0xEF, 0x78, 0x60,
		0xD6, 0xD4, 0x92, 0xCF, 0xE0, 0xF5, 0x57, 0xE9, 0xF1, 0xE6,
		0x80, 0xCE, 0x8A, 0x3A, 0x8D, 0x24, 0x44, 0x4F, 0x84, 0x32,
		0x8F, 0xAE, 0x59, 0x65, 0x52, 0xFB, 0x97, 0x65, 0xCC, 0xDF,
		0x31, 0xAF, 0x40, 0x8B, 0xBA, 0x6A, 0xE9, 0x53, 0xAF, 0x93,
		0x5A, 0x3D, 0x76, 0xE9, 0x3F, 0x1E, 0x9E, 0x40, 0x60, 0xFB,
		0x39, 0x0B, 0x9E, 0xB8, 0xEB, 0x90, 0x3F, 0x7B, 0x22, 0x8F,
		0x89, 0xB2, 0x26, 0x67, 0xCF, 0xBF, 0xEC, 0x29, 0x7A, 0xED,
		0x20, 0x7E, 0x5B, 0x22, 0xEB, 0x7B, 0x3E, 0x98, 0xE1, 0x74,
		0x8B, 0x21, 0x18, 0xF3, 0x46, 0x01, 0x6F, 0x2C, 0xB4, 0x34,
		0x58, 0x09, 0xD2, 0x0B, 0xB6, 0x45, 0xB9, 0xB5, 0x82, 0x20,
		0x92, 0x2A, 0x61, 0x78, 0xEE, 0x80, 0x0C, 0x0D, 0xE5, 0xA0,
		0xC5, 0x97, 0x85, 0xF4, 0x9A, 0x3F, 0xF8, 0xDF, 0x1C, 0x51,
		0xE0, 0x7F, 0x53, 0xDB, 0x66, 0x0E, 0xFF, 0x68, 0x30, 0xD0,
		0x1B, 0x04, 0x30, 0x66, 0x69, 0x26, 0x60, 0x10, 0xD3, 0x63,
		0x52, 0x4F, 0xAC, 0x07, 0x10, 0xBE, 0x18, 0xD1, 0x73, 0xE6,
		0x15, 0xD4, 0xB9, 0xB2, 0x87, 0xFC, 0x50, 0xC1, 0x89, 0xC1,
		0xE4, 0x5F, 0x5D, 0x91, 0xA8, 0xD6, 0x56, 0x9D, 0xC5, 0x0E,
		0x26, 0x0D, 0xB4, 0xD6, 0x03, 0x67, 0x7B, 0xA4, 0x9D, 0xEF,
		0xD0, 0xE3, 0xC7, 0x8D, 0x61, 0xC3, 0xDA, 0x86, 0x04, 0x2F,
		0x34, 0x86, 0x1E, 0x87, 0x3A, 0x56, 0xB6, 0x27, 0xAB, 0xAE,
		0xAE, 0xE8, 0xB5, 0x3A, 0x74, 0x76, 0x98, 0x93, 0x39, 0x8B,
		0x2B, 0x85, 0x48, 0xA7, 0x49, 0x1E, 0xB5, 0x13, 0xB8, 0x53,
		0x84, 0x04, 0x85, 0x7A, 0x49, 0x9A, 0x84, 0x4D, 0x69, 0x3D,
		0xF3, 0xA7, 0x29, 0xE5, 0x96, 0x19, 0x02, 0x7D, 0xA1, 0x19,
		0x17, 0x5E, 0x8B, 0x8A, 0xAA, 0x7E, 0x20, 0xA2, 0xD4, 0xA3,
		0x93, 0xAA, 0xCF, 0xD5, 0x94, 0x44, 0x02, 0xB7, 0xB8, 0x15,
		0x90, 0xAF, 0xF1, 0xC4, 0x15, 0xD6, 0x04, 0x68, 0x49, 0x1E,
		0xC2, 0x49, 0xBD, 0x88, 0x29, 0x18, 0x49, 0x88, 0x32, 0x01,
		0xAF, 0x15, 0xFD, 0x5B, 0x02, 0x09, 0x7E, 0x2D, 0x4E, 0xE0,
		0x38, 0xC5, 0xC6, 0x5F, 0xA2, 0x41, 0x77, 0xD4, 0x12, 0xDD,
		0x7F, 0xD0, 0xB7, 0xD1, 0xFE, 0x96, 0xB3, 0x4D, 0x03, 0xE2,
		0x25, 0x3A, 0x7D, 0x7F, 0x74, 0x33, 0xF1, 0x68, 0x99, 0x27,
		0xB3, 0x84, 0x38, 0x01, 0xF0, 0xC1, 0xDF, 0x71, 0xD5, 0x2D,
		0xFB, 0x6E, 0xC2, 0xB0, 0x80, 0x23, 0x48, 0xEE, 0xC4, 0x9C,
		0x31, 0x31, 0xB1, 0x87, 0xE1, 0x29, 0x86, 0xA6, 0xD0, 0x78,
		0x6E, 0x72, 0x43, 0xD5, 0x8D, 0x00, 0xF3, 0xFF, 0xC0, 0xC2,
		0x32, 0xE4, 0x7F, 0xB5, 0x6A, 0x06, 0x63, 0x04, 0x25, 0x44,
		0xAB, 0x5F, 0xDB, 0xDD, 0xFD, 0x41, 0x84, 0x2C, 0x01, 0xB0,
		0x84, 0x12, 0x55, 0x55, 0xE2, 0x66, 0x75, 0x26, 0xB4, 0xA6,
		0x68, 0xF3, 0x4D, 0x9F, 0x79, 0x2E, 0x73, 0x36, 0xB4, 0xE2,
		0xD9, 0x0B, 0xB8, 0x25, 0xCD, 0x50, 0xD8, 0xB8, 0xFC, 0x41,
		0xA7, 0x5C, 0xF3, 0x0D, 0x62, 0xAC, 0x12, 0xF2, 0xEC, 0x1E,
		0xE9, 0x3D, 0x0F, 0xDE, 0x6A, 0x79, 0xB0, 0xB4, 0x95, 0xF9,
		0x83, 0x2C, 0x89, 0xF4, 0xBE, 0xE6, 0xBE, 0x67, 0x47, 0xCE,
		0x84, 0x5E, 0x64, 0xC3, 0x85, 0xD2, 0x6C, 0xFD, 0x53, 0x0F,
		0x27, 0xF6, 0x62, 0x3D, 0xD7, 0x97, 0xF1, 0x77, 0x8C, 0x15,
		0x1F, 0x5D, 0x16, 0x60, 0x05, 0x2C, 0x2F, 0x60, 0x0B, 0xB6,
		0xBB, 0x39, 0xE8, 0x4E, 0x5C, 0xC0, 0xD5, 0x00, 0x55, 0xF6,
		0x8E, 0xCF, 0x11, 0xAC, 0x59, 0x88, 0x58, 0xB9, 0x54, 0xC9,
		0x9D, 0x7A, 0xA1, 0x30, 0x1D, 0xE1, 0x1E, 0x3A, 0xC4, 0xDB,
		0xC9, 0xFB, 0xCC, 0x65, 0xBC, 0x81, 0x7E, 0xBB, 0xF7, 0x2C,
		0x61, 0x8E, 0xF1, 0x2D, 0x46, 0x3C, 0x3F, 0x35, 0x9A, 0xC9,
		0x07, 0xF9, 0x5B, 0xD7, 0x1B, 0x77, 0xBA, 0x67, 0x0A, 0x54,
		0xEC, 0x36, 0x8F, 0x47, 0x24, 0xD4, 0x0B, 0x51, 0xE1, 0xF4,
		0x51, 0x60, 0xB2, 0xB0, 0xA1, 0xB8, 0xF1, 0x5B, 0x98, 0x27,
		0x35, 0x0A, 0x70, 0x54, 0x22, 0xEE, 0xD0, 0xD8, 0xB6, 0x9D,
		0x75, 0xD0, 0x1B, 0xA9, 0xDC, 0x02, 0x47, 0xE3, 0x89, 0xDE,
		0x64, 0x75, 0x40, 0x0B, 0x1D, 0x2A, 0xE8, 0x2B, 0xB0, 0xD2,
		0x15, 0x30, 0x38, 0x26, 0xFD, 0xC8, 0x39, 0xD7, 0x2A, 0xAE,
		0xB3, 0x70, 0xC6, 0x21, 0xFF, 0x6F, 0x8A, 0x9B, 0xDC, 0x51,
		0x8B, 0x07, 0x8B, 0xD0, 0x4E, 0x33, 0xD3, 0xA9, 0x2B, 0x4C,
		0x9D, 0x83, 0x49, 0x7D, 0xBF, 0x74, 0x40, 0xA6, 0x45, 0xB8,
		0x53, 0xD3, 0xEA, 0x15, 0xAE, 0xB2, 0xA9, 0x13, 0xB5, 0x49,
		0x76, 0x86, 0xAF, 0xF9, 0x50, 0xD5, 0xA4, 0x78, 0xCC, 0x43,
		0x81, 0x8C, 0xAA, 0x15, 0xC4, 0x62, 0x23, 0x70, 0x35, 0xB4,
		0xB9, 0x01, 0x21, 0x8B, 0x4B, 0xDB, 0xDD, 0x30, 0x49, 0x1A,
		0x76, 0x55, 0x36, 0xD6, 0x62, 0x62, 0xE5, 0x6C, 0xED, 0xF5,
		0x29, 0xF9, 0xA7, 0x66, 0x43, 0x9F, 0xAE, 0xC8, 0x46, 0xF8,
		0xCD, 0x20, 0x69, 0xEB, 0x20, 0xD9, 0xE8, 0x94, 0x66, 0x3E,
		0xCB, 0xCE, 0x71, 0x74, 0xA9, 0xC5, 0x2C, 0x59, 0xDD, 0x0B,
		0x5A, 0xF0, 0x90, 0x9F, 0xA0, 0xD9, 0xDA, 0x09, 0x1F, 0xB1,
		0x19, 0xA8, 0xFE, 0xBA, 0x8E, 0x20, 0x1A, 0x62, 0x30, 0x3F,
		0x6D, 0x3F, 0x5C, 0x7E, 0x94, 0xC9, 0xD2, 0x81, 0x6F, 0x63,
		0x50, 0xC8, 0x70, 0x45, 0xB2, 0x5B, 0x40, 0x83, 0x4E, 0x50,
		0x19, 0x55, 0xD4, 0xC2, 0xBC, 0x60, 0x97, 0x44, 0xF6, 0x19,
		0x28, 0x7C, 0x21, 0x4E, 0x50, 0x0D, 0x4C, 0xD1, 0x22, 0xEE,
		0x29, 0xF3, 0x5F, 0xF4, 0x74, 0xD6, 0x08, 0x96, 0xFF, 0x84,
		0x39, 0x43, 0xDF, 0x52, 0x8D, 0xBD, 0x4D, 0xA9, 0x50, 0xB7,
		0xB8, 0xFB, 0x75, 0xCF, 0x57, 0xC4, 0x9E, 0x79, 0xDB, 0xB2,
		0xE0, 0x7A, 0xC6, 0x2D, 0xA2, 0xA6, 0x13, 0x21, 0x6C, 0x0C,
		0x17, 0x50, 0x3F, 0x6B, 0xE8, 0x80, 0x41, 0xC0, 0xC7, 0x2B,
		0xBF, 0x2C, 0x4C, 0xE9, 0x1C, 0xCB, 0x9A, 0x72, 0x15, 0xE0,
		0xE3, 0x20, 0xE6, 0x3F, 0xF0, 0x22, 0xC3, 0x20, 0x12, 0x55,
		0xE9, 0x0C, 0xD5, 0x9C, 0xA0, 0xD5, 0xC7, 0x0F, 0x41, 0x6D,
		0xA5, 0x82, 0x9A, 0xC2, 0xF1, 0xD0, 0x9C, 0x8A, 0x4E, 0xB5,
		0xFC, 0xFC, 0x2D, 0x42, 0xA9, 0xB8, 0x25, 0x06, 0x7B, 0x29,
		0xEE, 0xE6, 0x43, 0x62, 0xED, 0xB0, 0xFD, 0xE8, 0x73, 0x6B,
		0x80, 0x97, 0x6E, 0x43, 0x22, 0x8C, 0xD9, 0x0B, 0x45, 0x6B,
		0x4C, 0xB1, 0x07, 0x56, 0xB7, 0xD7, 0xBD, 0xD3, 0x84, 0x13,
		0x40, 0x47, 0xA7, 0xAF, 0xB1, 0xC7, 0x0E, 0x73, 0xCD, 0x37,
		0x3C, 0xE8, 0x3B, 0x56, 0x0B, 0xB4, 0xD3, 0xEA, 0x15, 0x2D,
		0xCB, 0x65, 0x4E, 0x22, 0xD9, 0x9F, 0xAE, 0x52, 0xD5, 0xBB,
		0xE7, 0x39, 0xD1, 0xAE, 0xC1, 0x8B, 0x4B, 0x42, 0x2E, 0x38,
		0xBA, 0xA5, 0x71, 0x94, 0x8D, 0x25, 0x50, 0xB5, 0xA8, 0x8D,
		0x92, 0xA4, 0x0D, 0x56, 0xC2, 0x01, 0xD3, 0xF2, 0xF1, 0xDF,
		0xFF, 0xC9, 0x18, 0x3E, 0xCB, 0xCA, 0x0E, 0x8C, 0x0B, 0xB3,
		0x42, 0x66, 0x8F, 0x22, 0x84, 0x8E, 0x67, 0x21, 0x15, 0xCB,
		0x7E, 0xAE, 0x1B, 0xEE, 0xFA, 0xB3, 0x03, 0xE0, 0x3B, 0xCD,
		0x3F, 0x8E, 0xA8, 0xBE, 0x5A, 0x3E, 0x33, 0xD8, 0xB6, 0xA3,
		0xF1, 0xCF, 0x64, 0xC4, 0x7C, 0x8C, 0x58, 0x10, 0x36, 0xFE,
		0xCE, 0x13, 0x80, 0x9A, 0x9D, 0x5A, 0x08, 0x4C, 0x17, 0x81,
		0x6D, 0x00, 0x5C, 0x6F, 0x77, 0xBE, 0xF2, 0xE7, 0x33, 0xBA,
		0x8C, 0x35, 0x34, 0x95, 0x65, 0x1C, 0x08, 0x24, 0x3A, 0x11,
		0x35, 0x70, 0xE0, 0x98, 0x3D, 0xB7, 0x4C, 0x9A, 0xA6, 0x8E,
		0xF9, 0xC0, 0x0E, 0xEB, 0x39, 0x57, 0xEB, 0x6C, 0xE8, 0xB6,
		0x9E, 0x8D, 0x81, 0x16, 0x96, 0x0B, 0xCA, 0xCD, 0x35, 0x43,
		0xE8, 0x01, 0x0C, 0x25, 0x6C, 0xA6, 0x57, 0x48, 0xDC, 0x60,
		0x74, 0x91, 0x75, 0xB5, 0xD1, 0x23, 0x41, 0xFF, 0x63, 0x9A,
		0x60, 0xF1, 0x25, 0x46, 0x36, 0xA2, 0x17, 0x37, 0xE0, 0x19,
		0x17, 0xC1, 0x2C, 0x1B, 0x43, 0xCA, 0xE3, 0x50, 0xEA, 0x9C,
		0xF2, 0x4C, 0x5E, 0x0E, 0xFD, 0xD4, 0x2F, 0x98, 0x4E, 0x77,
		0x25, 0xE2, 0x8E, 0x61, 0x5F, 0x2C, 0xCC, 0xF0, 0x75, 0x57,
		0x42, 0x49, 0x94, 0x3A, 0x6E, 0x71, 0x66, 0xA4, 0x4E, 0x1F,
		0x9A, 0x67, 0xE1, 0xF1, 0xFB, 0xBF, 0x1D, 0xF2, 0x36, 0xAC,
		0xF9, 0x59, 0x81, 0x74, 0x54, 0x85, 0x33, 0x92, 0xBF, 0x91,
		0xEA, 0x7E, 0xB8, 0xCC, 0x8D, 0xED, 0x0B, 0x7B, 0x00, 0x34,
		0x81, 0x29, 0xF2, 0xCF, 0xEC, 0x7A, 0x68, 0xC1, 0xF3, 0x1F,
		0xBD, 0xB7, 0x6E, 0x2D, 0x24, 0xCF, 0x37, 0x9F, 0xB6, 0x6F,
		0xA5, 0xE7, 0xCD, 0x40, 0xEE, 0x86, 0xE2, 0x4A, 0xB2, 0xEF,
		0xDE, 0xDB, 0xAA, 0xA5, 0xDA, 0x1E, 0x29, 0xB8, 0x14, 0x0B,
		0xF4, 0xD9, 0x79, 0x8D, 0xD1, 0xD8, 0x04, 0x9A, 0x16, 0x40,
		0xDD, 0xE9, 0x56, 0xAA, 0xD9, 0x5B, 0x47, 0x1B, 0x87, 0x00,
		0x85, 0x5D, 0x9A, 0x6C, 0xAC, 0x8F, 0x90, 0x72, 0xC8, 0x3B,
		0x0F, 0x14, 0x89, 0x0D, 0x19, 0x7F, 0x43, 0x14, 0x5B, 0x76,
		0x86, 0x68, 0xA7, 0x4C, 0x5C, 0x8B, 0x6B, 0x3A, 0x17, 0xA2,
		0x59, 0x38, 0xCD, 0x9D, 0x18, 0x0E, 0x58, 0xBC, 0xA2, 0x2D,
		0x4F, 0x8B, 0x03, 0xBE, 0x8C, 0x39, 0x74, 0xC1, 0xE5, 0xF0,
		0xFE, 0xA0, 0x54, 0x54, 0x9E, 0x9E, 0x6B, 0xF2, 0x29, 0x4F,
		0xBD, 0xDF, 0xFE, 0xCB, 0xF9, 0x9C, 0x4D, 0xFD, 0xCF, 0xFF,
		0x6F, 0x41, 0xB5, 0xAF, 0xCA, 0x25, 0x07, 0x13, 0x3F, 0x85,
		0xDA, 0x28, 0x19, 0x15, 0x6D, 0x53, 0xD5, 0x1A, 0x9B, 0xE8,
		0x0D, 0xC7, 0xFE, 0x58, 0xDD, 0x1A, 0xB0, 0x0A, 0xD1, 0xCC,
		0xBE, 0x4E, 0xEA, 0xFF, 0x06, 0x6B, 0xF3, 0x29, 0x6F, 0x39,
		0x86, 0x79, 0xBA, 0x27, 0xA7, 0x9A, 0xF9, 0xAF, 0xD5, 0xCA,
		0x69, 0x0C, 0x4C, 0xB9, 0xFE, 0x76, 0xAD, 0xA9, 0xD9, 0x3A,
		0x5F, 0x8E, 0x8C, 0xF9, 0xAD, 0x82, 0xC3, 0x13, 0xC5, 0x43,
		0x1D, 0x3E, 0x27, 0xE8, 0xA2, 0x0C, 0x3E, 0xAB, 0x02, 0xB6,
		0x54, 0x0F, 0x98, 0x5D, 0xC6, 0x6C, 0xD0, 0xAD, 0x12, 0xCB,
		0xAF, 0x11, 0x67, 0x96, 0x21, 0x29, 0x85, 0xF8, 0xD2, 0xA9,
		0x51, 0x4E, 0xBE, 0x6E, 0x28, 0x2F, 0x79, 0x1D, 0x67, 0x10,
		0x42, 0x5C, 0xBB, 0x40, 0xD1, 0x80, 0xB7, 0x12, 0xF2, 0x80,
		0x2B, 0x14, 0x61, 0x83, 0xB1, 0x46, 0x6D, 0xA0, 0x05, 0x09,
		0x1C, 0x80, 0x07, 0xBD, 0xF7, 0x25, 0x90, 0x4F, 0x70, 0xB8,
		0xED, 0xFC, 0xC2, 0x74, 0x4F, 0x60, 0x57, 0xD5, 0x37, 0x27,
		0x32, 0xA2, 0x79, 0xDA, 0x67, 0x48, 0x73, 0x4F, 0xBD, 0x99,
		0xE7, 0x66, 0x1A, 0x2A, 0x84, 0x92, 0xBC, 0x12, 0x16, 0x08,
		0xA8, 0xEA, 0xB5, 0x6B, 0x5A, 0x93, 0x5D, 0xD5, 0xD3, 0x6F,
		0xDF, 0x55, 0x07, 0x20, 0x33, 0xA1, 0x12, 0xC4, 0xCF, 0x0E,
		0x2E, 0x39, 0x39, 0x26, 0x24, 0x47, 0x90, 0x1F, 0x49, 0x23,
		0xCF, 0x36, 0x2A, 0x3E, 0x48, 0x66, 0x44, 0x8C, 0x9E, 0x37,
		0x56, 0xA2, 0x89, 0xF7, 0xBF, 0x0C, 0xAE, 0xAA, 0x49, 0xCE,
		0xB5, 0x8C, 0x49, 0x41, 0x47, 0xF0, 0x50, 0xF9, 0xF8, 0xF8,
		0xCA, 0xB4, 0x78, 0x5C, 0x66, 0x9E, 0x61, 0xB5, 0xCB, 0x7D,
		0x70, 0xC8, 0x50, 0x19, 0x4F, 0x76, 0x2A, 0x8A, 0x20, 0x1C,
		0x40, 0x5A, 0xD4, 0xC2, 0x80, 0x36, 0xD5, 0x8C, 0xF4, 0x13,
		0xE4, 0xB5, 0x54, 0xEE, 0x50, 0xA6, 0x18, 0xB3, 0x82, 0xDF,
		0x28, 0xEE, 0x05, 0x1E, 0x91, 0x35, 0xC3, 0x1E, 0xCE, 0x3B,
		0x04, 0x35, 0xAF, 0x4C, 0xE7, 0x1A, 0x5A, 0x91, 0x10, 0xE9,
		0x7D, 0x8A, 0xA4, 0xD0, 0x9C, 0x01, 0xFD, 0x13, 0x37, 0x1D,
		0xB2, 0x2E, 0xC0, 0xCF, 0x69, 0x28, 0x74, 0x2F, 0x53, 0x90,
		0x20, 0x53, 0x17, 0x98, 0xE3, 0x80, 0xB6, 0x51, 0x02, 0x87,
		0xC3, 0xCA, 0xE1, 0x42, 0xAE, 0x92, 0x17, 0x79, 0x4D, 0xE4,
		0xC5, 0xE3, 0x71, 0xE5, 0xD4, 0xE7, 0xB5, 0x65, 0x9E, 0xEA,
		0xA1, 0x1E, 0x5C, 0xF1, 0x2B, 0x2C, 0x1C, 0xBB, 0x49, 0xE9,
		0xE6, 0xC2, 0xF7, 0xA8, 0x02, 0x86, 0x16, 0x75, 0x35, 0x07,
		0x12, 0x16, 0x4D, 0xAF, 0x82, 0x4C, 0x7F, 0x34, 0x54, 0x2D,
		0x6A, 0x9D, 0x14, 0xFA, 0x30, 0xC6, 0x51, 0xF8, 0xB8, 0xE3,
		0x99, 0x2A, 0x8F, 0x9B, 0x93, 0x47, 0x2F, 0xF1, 0x19, 0x29,
		0x15, 0x20, 0x02, 0xB3, 0xE4, 0xD1, 0xE5, 0xAD, 0xEA, 0xE4,
		0x10, 0xB6, 0xE6, 0xD9, 0x84, 0x7A, 0x0A, 0x27, 0x9A, 0x1C,
		0xB0, 0x2C, 0x95, 0x38, 0xBA, 0xF7, 0xE7, 0x32, 0x5D, 0x6C,
		0x30, 0x90, 0xEA, 0x3A, 0x07, 0xEE, 0x17, 0x1F, 0x21, 0xCA,
		0xE2, 0x02, 0x24, 0x4E, 0x1B, 0x54, 0x22, 0x77, 0xCD, 0xF1,
		0xB3, 0x5F, 0x81, 0x04, 0x63, 0x1C, 0x0E, 0x1A, 0xD2, 0x06,
		0x9C, 0xC8, 0xAF, 0x0E, 0x32, 0x2B, 0x0A, 0x08, 0x51, 0x52,
		0x6B, 0xBE, 0x86, 0x1B, 0x20, 0x13, 0x48, 0xA4, 0x52, 0x8C,
		0xFD, 0x84, 0xFA, 0x0D, 0x7B, 0x62, 0xBC, 0xF5, 0x42, 0x58,
		0x41, 0x82, 0xDD, 0xA8, 0xE5, 0x03, 0x15, 0x6D, 0xF3, 0x6D,
		0xE0, 0x1F, 0xFB, 0xC8, 0xFE, 0x2F, 0xC2, 0x3D, 0xF9, 0xA6,
		0x75, 0x50, 0x9C, 0xAF, 0xAE, 0x81, 0xCB, 0xC5, 0x7C, 0x25,
		0x99, 0xEB, 0x27, 0x4C, 0x1B, 0x69, 0xED, 0x49, 0x58, 0x19,
		0xBA, 0xBD, 0xF5, 0xEE, 0x3C, 0xAC, 0x05, 0xC6, 0xD2, 0xF7,
		0x88, 0x55, 0xE0, 0x6A, 0xA5, 0x5C, 0xEB, 0xCA, 0xF1, 0x3D,
		0x1D, 0xD0, 0x2B, 0xE8, 0x5C, 0x1D, 0xCC, 0x01, 0x8E, 0xD1,
		0x26, 0x4B, 0x93, 0xD9, 0x10, 0x1A, 0xAC, 0x87, 0x7A, 0xA0,
		0xBF, 0xE9, 0x96, 0xA7, 0x0E, 0xF6, 0x08, 0xF4, 0x4E, 0xAA,
		0x3A, 0x13, 0x94, 0xEB, 0xF3, 0xCB, 0xD8, 0xA2, 0x76, 0xB7,
		0x4C, 0xC5, 0xAF, 0x69, 0x9E, 0x96, 0x4C, 0xE8, 0x5E, 0xF3,
		0x06, 0x25, 0xDE, 0x86, 0x29, 0x10, 0x9B, 0x03, 0x8A, 0xD3,
		0xD4, 0x18, 0xC6, 0x3E, 0x3A, 0xC5, 0xE2, 0xDF, 0xC2, 0x6F,
		0x0D, 0xB3, 0xAD, 0x59, 0x26, 0xE9, 0xFA, 0x31, 0xAA, 0x98,
		0xC9, 0xDC, 0x0C, 0x14, 0xE8, 0x74, 0xD9, 0x76, 0xCB, 0xB5,
		0x3A, 0xE5, 0x55, 0x24, 0x10, 0x73, 0xAE, 0xA9, 0x9A, 0x49,
		0x2F, 0x15, 0xC5, 0x6B, 0x07, 0xEE, 0x9A, 0xB8, 0xFD, 0xF4,
		0xC6, 0xD6, 0xE8, 0xDF, 0xF7, 0xE5, 0x8C, 0x74, 0xE7, 0xD4,
		0x0E, 0x11, 0xF5, 0x87, 0xDB, 0x15, 0xA2, 0x47, 0x11, 0x24,
		0xDD, 0xC0, 0x07, 0x93, 0x78, 0xC0, 0xC2, 0x98, 0x89, 0xD0,
		0x49, 0xE8, 0x88, 0xC0, 0x3D, 0x15, 0xFA, 0xF7, 0x6B, 0x22,
		0x98, 0x8C, 0x6A, 0xA2, 0xDC, 0xA7, 0xD6, 0x87, 0x04, 0x91,
		0xBF, 0xE3, 0xA0, 0x85, 0xB4, 0x04, 0xEC, 0xE3, 0xD7, 0x63,
		0x07, 0x24, 0x01, 0x52, 0x77, 0xEB, 0xB8, 0xCB, 0x0F, 0xDF,
		0xA6, 0x61, 0x2A, 0xB1, 0xAA, 0x92, 0xC5, 0xF4, 0x76, 0x50,
		0x23, 0xC9, 0xBD, 0x8A, 0x22, 0xC0, 0xD3, 0x40, 0x8B, 0x4D,
		0xF5, 0xA1, 0x59, 0x96, 0xD4, 0x42, 0x1E, 0x2F, 0x96, 0x53,
		0xF7, 0xFA, 0x51, 0xC1, 0x1B, 0xE1, 0xA4, 0x73, 0xC6, 0x40,
		0x51, 0xE1, 0x84, 0x31, 0xF9, 0xDC, 0x66, 0x1F, 0x81, 0x0E,
		0x3F, 0xB0, 0xC5, 0x60, 0xA1, 0xC2, 0x43, 0xCF, 0x25, 0x71,
		0xD2, 0x17, 0xD5, 0xFB, 0x81, 0x1D, 0xA5, 0x23, 0xAA, 0xA4,
		0xCF, 0x13, 0xF1, 0x51, 0xAE, 0x96, 0x63, 0x9C, 0x65, 0xFA,
		0xB3, 0xF1, 0x46, 0x6C, 0x4F, 0x79, 0x17, 0xF7, 0xE5, 0xC6,
		0x85, 0x73, 0x87, 0xB8, 0x91, 0x00, 0x78, 0xE8, 0xCD, 0x5F,
		0xD6, 0x1A, 0x11, 0xC8, 0x7A, 0x8F, 0x29, 0xDD, 0x58, 0x86,
		0x88, 0xC8, 0x42, 0xAD, 0x29, 0x64, 0x17, 0x4D, 0xE3, 0x4F,
		0x7D, 0x7C, 0x3D, 0xBB, 0x6C, 0x49, 0xA2, 0x1B, 0x2E, 0x06,
		0x1D, 0xEC, 0x65, 0xA7, 0xA7, 0xDD, 0x56, 0x5E, 0xBA, 0x33,
		0xC8, 0x95, 0x73, 0x23, 0x81, 0xF2, 0xBC, 0x29, 0xD3, 0xC0,
		0x06, 0x95, 0xD4, 0x33, 0x07, 0x7D, 0xF7, 0x67, 0x26, 0x34,
		0xC2, 0xF2, 0x72, 0xCB, 0xA9, 0x09, 0xBA, 0xFE, 0xD7, 0xFC,
		0x24, 0x1E, 0xE3, 0xB8, 0x83, 0x85, 0xFA, 0x35, 0x22, 0x93,
		0x7E, 0x65, 0x0D, 0xA0, 0xD2, 0xC6, 0xF6, 0x20, 0xA6, 0x7E,
		0xD9, 0x7A, 0x7C, 0x9D, 0xEB, 0x4B, 0x1A, 0x6D, 0x58, 0xA4,
		0x85, 0x3A, 0x8F, 0xE0, 0x60, 0xD9, 0x5C, 0xCA, 0x54, 0xAE,
		0xFC, 0xD1, 0xE2, 0xA3, 0xD1, 0xAE, 0x3D, 0x8C, 0x39, 0x01,
		0xAD, 0x40, 0x39, 0x66, 0xDD, 0x85, 0xF7, 0x3F, 0x83, 0xD7,
		0x33, 0xF6, 0x06, 0x04, 0x36, 0x50, 0x13, 0x4B, 0x2D, 0x0C,
		0xC8, 0xA4, 0x60, 0xDF, 0xE4, 0xB7, 0x00, 0x74, 0xDD, 0x6F,
		0x22, 0x4D, 0x64, 0x5E, 0xA8, 0xBB, 0x8A, 0xA3, 0x2C, 0x8D,
		0xE5, 0x25, 0x6E, 0xB5, 0x8E, 0xE0, 0x88, 0xE2, 0x17, 0xAF,
		0x1D, 0x60, 0xD9, 0x9D, 0x60, 0x56, 0xFB, 0xC3, 0x09, 0x0A,
		0x13, 0xDC, 0x65, 0xF2, 0x8C, 0xC1, 0xD7, 0x8E, 0xE7, 0x4D,
		0xFF, 0x31, 0xAC, 0x06, 0xA9, 0x28, 0xFA, 0x5C, 0xA3, 0xED,
		0xC2, 0x13, 0x51, 0x42, 0x61, 0xAF, 0x0F, 0x3B, 0xED, 0x8F,
		0x08, 0x89, 0xD4, 0xD8, 0x94, 0x8F, 0x4E, 0x69, 0x11, 0x31,
		0xCE, 0x4A, 0xF6, 0x68, 0xF1, 0x7F, 0x10, 0xE9, 0xD9, 0x4D,
		0xE6, 0x04, 0x4D, 0xDD, 0xE3, 0xA8, 0x6C, 0xB3, 0x5C, 0x9D,
		0xF3, 0xA6, 0xD9, 0xDD, 0xB5, 0xF1, 0xB0, 0xBC, 0xEF, 0x82,
		0x76, 0xE1, 0x61, 0xF1, 0xCB, 0xA5, 0xBB, 0xBD, 0xF5, 0xBD,
		0x4D, 0x02, 0xE1, 0xC7, 0x6D, 0xBE, 0x7A, 0x85, 0x95, 0x84,
		0x12, 0xC4, 0x4B, 0x72, 0x23, 0x9A, 0xF0, 0xB2, 0x25, 0xE7,
		0x3C, 0x5D, 0xEB, 0xBA, 0x14, 0x54, 0xC3, 0xC8, 0x63, 0x78,
		0x2C, 0x7E, 0x77, 0xA7, 0x10, 0x16, 0x40, 0xC9, 0x8F, 0xDB,
		0xD8, 0xED, 0x1A, 0xDB, 0x3D, 0x8A, 0x35, 0x24, 0xBA, 0x2F,
		0x1D, 0xE8, 0x3B, 0x6D, 0x64, 0x5D, 0x18, 0x22, 0x91, 0xFD,
		0x9A, 0x53, 0x73, 0xAF, 0x60, 0x4F, 0x4F, 0x81, 0x97, 0xDD,
		0xCA, 0x0D, 0x9B, 0x60, 0x55, 0xA1, 0xBB, 0x04, 0xDD, 0xA5,
		0x51, 0x5D, 0x1C, 0xD6, 0x28, 0x8C, 0xCD, 0xC7, 0x28, 0xA7,
		0xF3, 0x0B, 0xD7, 0x2B, 0x3B, 0x2D, 0x8F, 0xD5, 0xDF, 0x1B,
		0x30, 0x87, 0x81, 0xD6, 0x06, 0x46, 0x75, 0x08, 0x34, 0x8D,
		0x05, 0x0C, 0x2D, 0x68, 0x28, 0xD5, 0x81, 0x02, 0x85, 0x3F,
		0x6A, 0x7F, 0x08, 0x19, 0xB3, 0xD0, 0x54, 0x11, 0x69, 0x89,
		0x2F, 0xC0, 0x10, 0x0B, 0x0E, 0x7A, 0x33, 0x9D, 0xD9, 0xAD,
		0xCC, 0x38, 0x46, 0x31, 0xC0, 0xA7, 0xDB, 0x2B, 0x3C, 0xA5,
		0xB7, 0xBE, 0x96, 0xEF, 0x26, 0x3E, 0x02, 0xB5, 0x74, 0xE0,
		0xF3, 0x42, 0x41, 0x7B, 0xBE, 0xC6, 0xB3, 0x09, 0x94, 0x65,
		0x7A, 0x39, 0xC1, 0x1B, 0x2F, 0x90, 0xBE, 0x0A, 0xCE, 0x27,
		0xB0, 0x47, 0x53, 0xAA, 0x05, 0xED, 0x86, 0xC4, 0x14, 0xA2,
		0xDF, 0xAE, 0xD6, 0x9A, 0xC9, 0xFB, 0xC9, 0x04, 0x3C, 0x8E,
		0xCF, 0x8C, 0x73, 0x8F, 0x50, 0xBC, 0xAC, 0x1D, 0xB4, 0x3A,
		0xE1, 0xE1, 0x63, 0x4A, 0x7A, 0xBD, 0x3D, 0x0A, 0x80, 0xFB,
		0x54, 0x83, 0x96, 0x08, 0xF4, 0xCB, 0xF5, 0x6D, 0x38, 0xE3,
		0xCB, 0x9F, 0xEC, 0x31, 0x72, 0x2B, 0x04, 0x90, 0xB5, 0x85,
		0xE9, 0xDB, 0x47, 0xAB, 0xAB, 0xD4, 0x79, 0xCC, 0x08, 0x02,
		0xDF, 0x95, 0xA4, 0x29, 0xC0, 0x41, 0xB8, 0x78, 0x20, 0x05,
		0x42, 0x95, 0xF8, 0xFB, 0xE5, 0x8C, 0xB6, 0xDC, 0x17, 0x82,
		0xBF, 0x1D, 0x9F, 0x24, 0x24, 0x15, 0xC4, 0x4E, 0x80, 0x14,
		0xD0, 0x37, 0x02, 0x31, 0xDF, 0xB1, 0x2C, 0xA3, 0x0D, 0x8A,
		0xAE, 0xC4, 0xAE, 0x55, 0xA9, 0x0D, 0xF1, 0x0C, 0xF9, 0x2B,
		0xE9, 0x4A, 0x04, 0x0A, 0x33, 0xBE, 0x46, 0xD1, 0xCC, 0xCE,
		0xCC, 0x67, 0x3C, 0xCE, 0x5B, 0xEB, 0xA2, 0xE9, 0x08, 0xDF,
		0xB8, 0x9F, 0x26, 0x1C, 0x8D, 0xF3, 0xDB, 0xE6, 0xF6, 0xF0,
		0xE3, 0x90, 0xB5, 0xBD, 0xE3, 0x2C, 0x23, 0xA5, 0x1B, 0xE7,
		0x91, 0xE5, 0x42, 0x34, 0xA6, 0x3C, 0xB7, 0x89, 0x3C, 0xA0,
		0xFD, 0xD4, 0x9D, 0xAF, 0xE9, 0xCC, 0x76, 0x28, 0xB0, 0x4F,
		0xA6, 0xD8, 0xB6, 0x0E, 0x02, 0x72, 0xA3, 0xF4, 0x58, 0x98,
		0x8C, 0xA0, 0xBE, 0xA5, 0x8F, 0xA1, 0x6B, 0x9D, 0x44, 0x3D,
		0x43, 0x82, 0xBF, 0xAD, 0xFF, 0xE6, 0x03, 0x59, 0xB1, 0x3B,
		0xC4, 0xB0, 0x45, 0xDA, 0x8C, 0x21, 0x87, 0xAE, 0xD1, 0xEB,
		0x79, 0xFA, 0x0C, 0xFB, 0x4D, 0xFD, 0xBB, 0xA0, 0x08, 0x9A,
		0x08, 0x3B, 0xD8, 0xF5, 0x2A, 0x33, 0xFA, 0x3E, 0xBD, 0xD4,
		0x5D, 0x1D, 0xEE, 0x64, 0x94, 0xD9, 0x9D, 0x79, 0x46, 0x98,
		0x3C, 0x73, 0xD8, 0x1D, 0x33, 0x6D, 0xB5, 0xB9, 0xEB, 0xBC,
		0xBA, 0x61, 0x76, 0xA0, 0xA0, 0x59, 0x89, 0xE3, 0x54, 0xD3,
		0xBA, 0xC4, 0x22, 0xAA, 0x08, 0xAD, 0x67, 0xC1, 0xB5, 0x38,
		0xB9, 0x75, 0x01, 0x7F, 0x3B, 0xA8, 0x94, 0x62, 0xD9, 0x7E,
		0x06, 0xB4, 0x02, 0xDF, 0x7E, 0x66, 0xCA, 0x56, 0x26, 0x49,
		0x12, 0x57, 0xEF, 0x48, 0x17, 0x78, 0x1C, 0x6D, 0x7F, 0x0D,
		0xEB, 0xBB, 0x6D, 0xED, 0x9C, 0x39, 0xC7, 0x41, 0xF7, 0xC6,
		0xA2, 0x9B, 0x3B, 0x44, 0x1B, 0xD2, 0x3C, 0x3E, 0x17, 0x87,
		0xBA, 0x53, 0x7C, 0x22, 0x26, 0x56, 0xAF, 0xC9, 0xC0, 0x32,
		0x6E, 0x36, 0xFE, 0x14, 0xC2, 0x21, 0x86, 0x3E, 0x1E, 0x96,
		0xED, 0x85, 0xAA, 0xA2, 0x6C, 0x38, 0xAD, 0x9F, 0x11, 0xD8,
		0xAA, 0x8B, 0xA4, 0x7D, 0x22, 0xAE, 0xDC, 0xF4, 0x2C, 0xBA,
		0xCF, 0xB5, 0xC6, 0xA1, 0x69, 0x0E, 0xD4, 0xE9, 0xE3, 0x2C,
		0xAA, 0xE7, 0xC9, 0xBB, 0x05, 0xDC, 0x02, 0x59, 0x17, 0x52,
		0x10, 0x7F, 0x7E, 0x86, 0xD4, 0xD7, 0x45, 0xD2, 0xFE, 0xA6,
		0x4D, 0x31, 0x26, 0xE8, 0x52, 0xF3, 0x62, 0xAC, 0xFC, 0x83,
		0x8B, 0xBE, 0x1E, 0x8F, 0x5F, 0x7F, 0x63, 0xE9, 0x4F, 0x20,
		0x55, 0x2A, 0x25, 0x52, 0xA2, 0x21, 0xA8, 0x29, 0x4E, 0x65,
		0x91, 0x5F, 0xA7, 0x02, 0xBB, 0x09, 0x66, 0x2C, 0xA5, 0x55,
		0xFD, 0x57, 0x84, 0x3C, 0xA5, 0xB2, 0xA1, 0x26, 0xB0, 0x22,
		0x3D, 0xD7, 0x32, 0xA5, 0xA8, 0x09, 0x3B, 0x8C, 0x14, 0xF7,
		0x48, 0x5F, 0x04, 0x80, 0xBF, 0x30, 0x70, 0xBF, 0xB5, 0x0F,
		0xA1, 0xA0, 0xC0, 0x82, 0xC5, 0xFD, 0x33, 0x60, 0xDF, 0xB7,
		0x19, 0xF0, 0x88, 0x52, 0xE6, 0x51, 0x90, 0x17, 0xC2, 0x71,
		0x19, 0x08, 0xCD, 0x66, 0xD7, 0xC2, 0x55, 0xB8, 0x9C, 0x84,
		0x69, 0xBA, 0x16, 0x38, 0x75, 0x24, 0x28, 0xC3, 0xBD, 0x6A,
		0x7A, 0x54, 0x94, 0xE2, 0x5E, 0x20, 0x22, 0x59, 0x39, 0x5C,
		0xB5, 0xDA, 0x7A, 0xCC, 0x47, 0x16, 0xC5, 0xE9, 0x45, 0x82,
		0x68, 0xAE, 0xB4, 0x74, 0x99, 0x77, 0x10, 0xAA, 0xAC, 0x20,
		0xAE, 0x46, 0xF2, 0x3D, 0x3D, 0x60, 0x1E, 0x93, 0x50, 0x1D,
		0x45, 0xBE, 0x66, 0x51, 0xF9, 0x03, 0x32, 0x27, 0xE3, 0x21,
		0x44, 0xD6, 0x09, 0xAF, 0xA7, 0xB8, 0xB5, 0xB3, 0x75, 0xAE,
		0x3D, 0x67, 0x17, 0xA0, 0xAD, 0xAE, 0xDD, 0xAD, 0xF4, 0x2B,
		0x38, 0x0A, 0xC0, 0x4F, 0x32, 0x10, 0x67, 0x69, 0x28, 0x0F,
		0x25, 0x4E, 0x12, 0xC3, 0xC0, 0x5B, 0xF5, 0xD0, 0xEA, 0x32,
		0x8C, 0x57, 0x9D, 0x11, 0x6C, 0x78, 0xCE, 0xCE, 0xBB, 0x21,
		0x12, 0x58, 0x76, 0xF5, 0xE5, 0x04, 0x32, 0xFF, 0x80, 0x92,
		0xEA, 0x41, 0x0B, 0x13, 0xBF, 0xAA, 0xE0, 0xE6, 0x6E, 0x68,
		0xD0, 0xF0, 0x05, 0x6A, 0x20, 0x1D, 0x0F, 0xF3, 0x4E, 0xB2,
		0xA3, 0x52, 0xAA, 0x33, 0x23, 0xE5, 0xF8, 0xF2, 0xAA, 0x2E,
		0xC4, 0xC5, 0xC1, 0xEF, 0x9C, 0xAC, 0x74, 0xAD, 0x04, 0xBB,
		0x2B, 0x71, 0x20, 0x65, 0x54, 0x36, 0x3D, 0x6F, 0xE8, 0x8F,
		0x33, 0xF7, 0xC4, 0x41, 0x82, 0x87, 0xEE, 0x7F, 0xC1, 0xC4,
		0x65, 0x0B, 0x58, 0x8F, 0xCE, 0x24, 0x27, 0x43, 0xD9, 0x09,
		0xAD, 0x42, 0xEB, 0x12, 0xF3, 0x25, 0xD9, 0x98, 0xB3, 0xE4,
		0xA9, 0x5E, 0xFD, 0x02, 0xBD, 0x93, 0xE9, 0xE8, 0x33, 0x75,
		0xB6, 0x22, 0xF0, 0xF0, 0xCE, 0x1F, 0x97, 0xFA, 0xFE, 0x07,
		0x1B, 0xE7, 0xCD, 0xE6, 0x92, 0xF1, 0x8B, 0xAE, 0x29, 0xB5,
		0xFD, 0xA4, 0xA2, 0xB1, 0xB4, 0x34, 0x2A, 0xEF, 0x89, 0xCD,
		0x2C, 0x42, 0x2A, 0xA7, 0xEF, 0x40, 0x34, 0x77, 0xC1, 0x1D,
		0xDA, 0xD6, 0x3D, 0x55, 0x96, 0x5D, 0x92, 0x28, 0xF2, 0x47,
		0x61, 0x15, 0x2B, 0x5D, 0x1D, 0xA5, 0x62, 0x24, 0x67, 0x8F,
		0x3C, 0x71, 0xD0, 0x3C, 0x5C, 0xB1, 0x44, 0x60, 0x0E, 0x84,
		0x38, 0xED, 0xC4, 0x54, 0x12, 0x1A, 0x4A, 0x04, 0xD6, 0x1B,
		0xC3, 0x66, 0x60, 0x5A, 0x0A, 0x88, 0xF4, 0x6E, 0x07, 0xFE,
		0xC4, 0x6D, 0x64, 0x85, 0x72, 0xED, 0xF0, 0x87, 0x5A, 0x6C,
		0x79, 0x56, 0x8F, 0x53, 0xA1, 0x6B, 0x9E, 0x1F, 0xCC, 0x24,
		0xA4, 0x27, 0x91, 0xCB, 0x14, 0xAF, 0x5E, 0xC1, 0xC3, 0xB6,
		0x1E, 0xBD, 0xB6, 0x9D, 0xD7, 0xDA, 0x58, 0x63, 0x74, 0xDE,
		0x3E, 0xEE, 0x2C, 0x68, 0xE7, 0xD8, 0x15, 0xBF, 0x1C, 0xCE,
		0x3F, 0x13, 0x2C, 0xBE, 0x9B, 0x9C, 0x47, 0xBD, 0xE8, 0xE7,
		0xCC, 0x2F, 0x76, 0x86, 0xD5, 0x60, 0x2E, 0xA5, 0x13, 0xBE,
		0x39, 0xDE, 0x33, 0xF8, 0xE2, 0x40, 0xC4, 0xD9, 0x89, 0xFD,
		0x96, 0x2B, 0xF1, 0x3A, 0x3D, 0xF3, 0x1E, 0xE5, 0x42, 0xEA,
		0x98, 0x92, 0x41, 0xA7, 0xD7, 0x21, 0x2E, 0xF0, 0x41, 0x70,
		0xD6, 0xAC, 0xEB, 0x0D, 0x08, 0x7D, 0x07, 0x4B, 0xA3, 0xDE,
		0x87, 0xE0, 0x50, 0x2A, 0x8F, 0xA4, 0x50, 0xA4, 0x76, 0x16,
		0xD0, 0x04, 0x51, 0xFE, 0xCB, 0x41, 0x5B, 0x3A, 0x43, 0x85,
		0x1E, 0x4B, 0xB9, 0xD9, 0x27, 0x25, 0xCF, 0x42, 0xDE, 0xB4,
		0x0A, 0xEF, 0xE9, 0x93, 0xE9, 0x64, 0xD4, 0x9D, 0xE2, 0x1B,
		0xA7, 0x4C, 0x3C, 0xFB, 0x49, 0xFD, 0xD1, 0xBA, 0x4F, 0xE7,
		0x31, 0x12, 0x3A, 0x8D, 0x72, 0x70, 0x44, 0x24, 0xC7, 0x71,
		0xBF, 0xF7, 0xF1, 0xB3, 0x11, 0x1A, 0x02, 0x22, 0xB8, 0x1B,
		0x29, 0x8D, 0x4C, 0x7E, 0x12, 0x90, 0xBC, 0xFE, 0xFD, 0xEF,
		0x97, 0x8F, 0x39, 0xD0, 0x67, 0x1C, 0x09, 0xEC, 0xB6, 0xA1,
		0x7F, 0xFD, 0x2A, 0x2F, 0x9A, 0xB9, 0xDD, 0xF9, 0xF9, 0x47,
		0xED, 0x68, 0x49, 0x89, 0xB7, 0x99, 0x2E, 0x43, 0x79, 0x28,
		0x12, 0xC3, 0x22, 0xF2, 0xA1, 0x5A, 0xB1, 0xCF, 0xAC, 0x0B,
		0x61, 0xF0, 0x8B, 0x02, 0xA4, 0xE0, 0x26, 0x7B, 0xC1, 0x8C,
		0xB8, 0xCE, 0xC2, 0x1F, 0xE2, 0x38, 0xE9, 0xEB, 0x9F, 0x14,
		0x27, 0x6A, 0xDC, 0x41, 0xA4, 0x23, 0xDE, 0x51, 0xAB, 0x5B,
		0x46, 0x96, 0xC8, 0xBB, 0x66, 0xD0, 0xA1, 0x33, 0x74, 0x57,
		0xAD, 0x55, 0xAE, 0x56, 0x69, 0xB6, 0xE7, 0x00, 0x43, 0x4C,
		0xB6, 0xF7, 0x65, 0x74, 0xC6, 0xC8, 0xC0, 0x2C, 0x35, 0x19,
		0x38, 0xD3, 0xD0, 0x70, 0xE2, 0x22, 0x85, 0xA9, 0x20, 0x26,
		0x85, 0x9F, 0xE3, 0x05, 0x88, 0xED, 0x32, 0x9D, 0x6E, 0xD9,
		0x3C, 0x17, 0x24, 0x7B, 0xCC, 0x28, 0xB8, 0x4D, 0x65, 0x87,
		0xF6, 0x27, 0xD2, 0x57, 0x5C, 0x36, 0x43, 0xE1, 0xF4, 0xD7,
		0x90, 0x24, 0x17, 0x68, 0xC8, 0x82, 0x20, 0x1D, 0xC2, 0x2B,
		0x4F, 0xB7, 0x53, 0x08, 0x70, 0x95, 0x09, 0x13, 0x51, 0xD6,
		0xFA, 0x15, 0xCD, 0x37, 0x8E, 0x4E, 0x21, 0x03, 0x22, 0x4A,
		0x19, 0x82, 0x55, 0xC2, 0xE1, 0xAD, 0xAC, 0x2E, 0x8F, 0x1D,
		0xFF, 0xB3, 0x63, 0x33, 0xBF, 0x85, 0x75, 0xF4, 0x5B, 0x7F,
		0x81, 0x5D, 0x63, 0x4A, 0x9A, 0x72, 0xD1, 0xC3, 0x96, 0x58,
		0x9F, 0x20, 0xD1, 0x25, 0x5F, 0x0E, 0x3D, 0xC9, 0xFC, 0xCC,
		0x15, 0x8A, 0x71, 0xB5, 0x21, 0x94, 0x87, 0x9F, 0xFB, 0x25,
		0xA6, 0xC1, 0x23, 0xF4, 0x70, 0x78, 0x91, 0x93, 0x8B, 0xDE,
		0x79, 0xC1, 0xB8, 0x00, 0xD2, 0x20, 0x47, 0xEB, 0xD2, 0xBB,
		0xAB, 0x1B, 0xB5, 0x7F, 0x73, 0xF1, 0xD6, 0xF4, 0xDD, 0x25,
		0x50, 0x4B, 0x85, 0x4F, 0x58, 0x72, 0x40, 0x19, 0x10, 0x31,
		0xC9, 0xE9, 0xE5, 0xB2, 0x4B, 0x88, 0x7C, 0x47, 0xA9, 0xFF,
		0xBA, 0xD8, 0xEB, 0x81, 0x75, 0xFF, 0x34, 0x64, 0x0D, 0x81,
		0xAA, 0x5A, 0x7C, 0x71, 0x90, 0xC5, 0x4A, 0x35, 0x48, 0x7C,
		0x1F, 0x02, 0xC4, 0xBE, 0x3C, 0x20, 0x1B, 0xE9, 0xE3, 0x81,
		0x73, 0x81, 0x4F, 0x5A, 0x96, 0xE1, 0x11, 0xCF, 0x84, 0xA0,
		0xEF, 0x60, 0x1B, 0xD4, 0xF0, 0x06, 0x09, 0x00, 0x9B, 0x2E,
		0x0A, 0xEF, 0xC8, 0xF9, 0x59, 0x51, 0xB9, 0xFC, 0x50, 0xF9,
		0xA0, 0x23, 0xC5, 0x28, 0x56, 0x9A, 0x61, 0x86, 0x71, 0x2C,
		0xF4, 0x54, 0x24, 0x10, 0x76, 0x79, 0x53, 0x7D, 0xBA, 0x8D,
		0x1A, 0xBB, 0x0D, 0xA8, 0xA9, 0xC9, 0x1B, 0xB1, 0xE1, 0x08,
		0x0D, 0x05, 0x24, 0xA9, 0xB1, 0x65, 0xDA, 0x6F, 0x52, 0xAA,
		0x93, 0x24, 0xBC, 0xA9, 0xD7, 0xB4, 0xB8, 0xE4, 0xAA, 0x3B,
		0x5A, 0xC7, 0x01, 0x75, 0xA2, 0x51, 0x89, 0x9F, 0x97, 0xA9,
		0x45, 0xF4, 0xAA, 0x2E, 0xD5, 0xCB, 0x08, 0xDD, 0xF5, 0x21,
		0xB4, 0xAE, 0x45, 0xBC, 0x02, 0xBD, 0x0D, 0x96, 0x48, 0x41,
		0x72, 0x61, 0xA5, 0x8C, 0x37, 0xE3, 0xD1, 0x2C, 0x86, 0xA6,
		0x86, 0x8E, 0x1D, 0x98, 0x67, 0x95, 0xE1, 0xC9, 0x40, 0x1F,
		0xF5, 0x2B, 0x46, 0x50, 0xC7, 0x91, 0xAB, 0xD3, 0xA1, 0x29,
		0xF8, 0x10, 0x2E, 0xB5, 0x41, 0xAD, 0x8A, 0x8C, 0xF7, 0x16,
		0xEB, 0xFE, 0x08, 0x3B, 0x71, 0x7C, 0x3F, 0xF0, 0x8F, 0xA4,
		0xE6, 0x05, 0xA7, 0x76, 0xF7, 0xB4, 0x0F, 0x30, 0x18, 0x2E,
		0xFA, 0xBC, 0x2B, 0x8E, 0xE1, 0xF8, 0x0A, 0x5C, 0x89, 0xF7,
		0x3A, 0xC8, 0xCE, 0xE7, 0x95, 0xED, 0xDF, 0x55, 0x81, 0xE0,
		0xBF, 0xA5, 0x8F, 0xCD, 0xB7, 0xDA, 0x08, 0x39, 0x12, 0xCA,
		0xA5, 0xC5, 0x54, 0xC5, 0xFE, 0x81, 0xEB, 0xE7, 0x90, 0xF2,
		0x5C, 0xFF, 0x96, 0x1F, 0x71, 0x9E, 0xDF, 0xB0, 0xAF, 0x09,
		0x7E, 0xB4, 0xF5, 0x6D, 0xF3, 0xF8, 0x3D, 0x16, 0x76, 0x72,
		0xEB, 0x67, 0x5C, 0x1F, 0x17, 0xAB, 0x66, 0xAD, 0x97, 0x45,
		0xF6, 0x5E, 0x0A, 0x76, 0x4E, 0xB3, 0x9D, 0xCD, 0xE4, 0x2E,
		0x4C, 0xF4, 0xA0, 0x6E, 0xEA, 0xBC, 0x70, 0xBB, 0x73, 0x96,
		0xC6, 0xE1, 0x46, 0x9A, 0x69, 0xAA, 0xB2, 0x34, 0xEC, 0x55,
		0xCC, 0x2B, 0xA4, 0x50, 0x77, 0x69, 0x70, 0x66, 0x29, 0x67,
		0x8E, 0x04, 0xF8, 0xBC, 0x0D, 0x54, 0x12, 0x0F, 0x53, 0x03,
		0x01, 0x33, 0xD9, 0x3C, 0x96, 0x6D, 0xC7, 0x7A, 0xDD, 0x77,
		0x1B, 0x2B, 0xD1, 0xDC, 0xEB, 0x5A, 0x49, 0xB3, 0x7D, 0x3E,
		0xC0, 0x71, 0xEE, 0x4C, 0xD2, 0x5C, 0x4F, 0x0D, 0x15, 0x46,
		0x22, 0xCE, 0x60, 0xBE, 0x32, 0xB7, 0x85, 0xED, 0x03, 0xF4,
		0xA7, 0x31, 0xD8, 0x18, 0xB4, 0x60, 0x61, 0x48, 0xCA, 0x23,
		0x9B, 0x34, 0xDD, 0xB4, 0xC2, 0x33, 0xEB, 0x5F, 0x2A, 0x99,
		0x62, 0x6F, 0xAB, 0xB5, 0x8F, 0xCF, 0x8C, 0x33, 0x25, 0xB5,
		0x7D, 0xE5, 0xE7, 0x5A, 0xF2, 0x82, 0xA5, 0xD9, 0xAC, 0xFC,
		0xF6, 0xB1, 0x7D, 0xC1, 0xFC, 0xB2, 0x9D, 0xEC, 0xF1, 0x57,
		0x6D, 0x03, 0xD2, 0xEB, 0x4F, 0x9A, 0xB4, 0x7D, 0xF2, 0x60,
		0x5D, 0x57, 0xFA, 0x47, 0xE4, 0x98, 0xE7, 0x7A, 0xA1, 0xB4,
		0x8B, 0x58, 0x62, 0xFD, 0x84, 0xB1, 0x09, 0x5F, 0x33, 0x82,
		0x30, 0xEA, 0x8A, 0x1F, 0xFD, 0x62, 0x7D, 0xCE, 0xD4, 0x93,
		0x37, 0x30, 0x32, 0x0F, 0xFA, 0x7A, 0x63, 0x86, 0x21, 0x2F,
		0x34, 0xCD, 0xD7, 0x7C, 0x76, 0x6A, 0x65, 0x50, 0xBA, 0x23,
		0x2D, 0xEB, 0xAE, 0x24, 0x70, 0x57, 0xD2, 0x5D, 0xCE, 0x06,
		0xC7, 0x1B, 0x48, 0x78, 0x31, 0x01, 0xC3, 0xC8, 0x4D, 0x3E,
		0xD5, 0x36, 0x4B, 0xE1, 0xA6, 0xA0, 0x42, 0x5D, 0x7C, 0xCB,
		0x68, 0x44, 0xEA, 0x1C, 0x0C, 0x68, 0x32, 0xF5, 0xD7, 0x47,
		0xB9, 0x38, 0xBD, 0x6C, 0x10, 0xDB, 0x6D, 0x19, 0xDA, 0xF2,
		0x36, 0xCD, 0x94, 0xA4, 0x40, 0x39, 0x53, 0xE0, 0x1A, 0x47,
		0x9F, 0x8C, 0x95, 0x8B, 0x73, 0xD8, 0xF6, 0xB5, 0x92, 0x0B,
		0xF4, 0xF9, 0x95, 0xE2, 0xE3, 0x15, 0x70, 0x9E, 0xC2, 0xC0,
		0xE7, 0xE1, 0xE9, 0x11, 0x57, 0xDB, 0x1D, 0xE1, 0xD2, 0x9E,
		0x73, 0x26, 0xEF, 0x6D, 0x29, 0x9C, 0x4E, 0x2E, 0x0B, 0xD3,
		0x9C, 0x55, 0xFB, 0xC3, 0x04, 0x8D, 0xA9, 0xB2, 0x86, 0x13,
		0xD8, 0xC6, 0x07, 0x09, 0xB4, 0xC9, 0x28, 0xE3, 0xB9, 0x7F,
		0xAB, 0x93, 0x0B, 0x8E, 0xCE, 0xF9, 0x86, 0x3F, 0x94, 0x69,
		0x29, 0x49, 0x65, 0xC9, 0xCA, 0x9C, 0xAB, 0x12, 0xF5, 0x11,
		0x5A, 0xC8, 0x74, 0xDE, 0x2C, 0x41, 0x20, 0x75, 0xAD, 0xBE,
		0x05, 0x77, 0x77, 0x71, 0x9C, 0x66, 0xF1, 0xA7, 0xD5, 0xFA,
		0x9E, 0x04, 0x7B, 0xFA, 0xF1, 0x05, 0x08, 0x27, 0x7F, 0x61,
		0x87, 0x2E, 0x5B, 0x31, 0xF0, 0x47, 0xF9, 0xFD, 0x35, 0xA3,
		0xF3, 0x25, 0x44, 0x0F, 0x64, 0x1A, 0x4E, 0x6E, 0xF0, 0x49,
		0xFE, 0x33, 0x40, 0x51, 0x42, 0x4B, 0x10, 0x6E, 0xDB, 0x62,
		0xE5, 0x13, 0x9B, 0xF1, 0xCE, 0x6A, 0x92, 0xC5, 0xDF, 0x8C,
		0xAF, 0xEF, 0xDA, 0x16, 0x4B, 0xE0, 0x50, 0x8B, 0x69, 0xCD,
		0x67, 0xEE, 0x3B, 0x1A, 0x53, 0x55, 0x98, 0xED, 0xF7, 0xCF,
		0xD4, 0xA9, 0xED, 0x10, 0xBA, 0x3F, 0x06, 0x94, 0x9B, 0xCD,
		0xE7, 0xB6, 0x5C, 0x2B, 0x63, 0x51, 0x89, 0xD9, 0x3E, 0x39,
		0x1A, 0xE5, 0x17, 0xC8, 0x72, 0x9F, 0x53, 0xA1, 0x86, 0x69,
		0x37, 0x7A, 0x7C, 0x3B, 0xB8, 0x13, 0x41, 0x63, 0xEF, 0x8A,
		0x1A, 0x6B, 0x71, 0x80, 0x95, 0xDA, 0x8E, 0x8E, 0xBB, 0xDA,
		0xBF, 0xD0, 0x67, 0x73, 0x0B, 0x8D, 0xBC, 0x2E, 0x73, 0xA9,
		0xC9, 0x20, 0xFF, 0x0C, 0x4E, 0x7D, 0x90, 0xEA, 0xBA, 0x1E,
		0x2C, 0x48, 0xA9, 0xA6, 0xAD, 0x26, 0x72, 0xD3, 0xF7, 0xE6,
		0x49, 0x1C, 0xC1, 0x86, 0x75, 0x85, 0xF7, 0x0D, 0x26, 0x72,
		0x31, 0x6D, 0xAC, 0x99, 0x30, 0xC7, 0x91, 0xE3, 0x54, 0x8D,
		0xC0, 0xD7, 0xAB, 0x7D, 0x61, 0x34, 0x55, 0x5F, 0x81, 0x31,
		0x00, 0xEE, 0x9C, 0x20, 0x42, 0xFA, 0xE8, 0x91, 0xF0, 0xA2,
		0x0A, 0x51, 0xC2, 0x65, 0x9C, 0x8D, 0x7D, 0xFF, 0x06, 0x2E,
		0x57, 0x72, 0xC1, 0x94, 0x06, 0x8C, 0x17, 0xAA, 0x17, 0x65,
		0x28, 0x02, 0x5F, 0xE8, 0xD7, 0xCE, 0xFA, 0x29, 0x98, 0x36,
		0xF6, 0xE9, 0x56, 0x16, 0x21, 0xC7, 0x64, 0xC2, 0x27, 0x78,
		0x9A, 0x7D, 0x62, 0xE0, 0x19, 0xD7, 0xBC, 0x97, 0x4A, 0xDB,
		0x31, 0xDE, 0x7A, 0x46, 0xB8, 0x2C, 0x19, 0x26, 0x27, 0xC3,
		0x05, 0x18, 0x7D, 0x68, 0x0E, 0x71, 0x4A, 0x7D, 0x2B, 0x51,
		0x67, 0x48, 0x89, 0x91, 0xAC, 0xE0, 0x2E, 0x78, 0x0E, 0xFE,
		0x91, 0x00, 0x42, 0x15, 0x9E, 0x05, 0x5A, 0xA1, 0x94, 0x4D,
		0x18, 0xCB, 0x0A, 0xB0, 0x0A, 0x47, 0x5E, 0x91, 0x59, 0x28,
		0xD6, 0x4D, 0x98, 0x8C, 0xF1, 0xD0, 0xE4, 0xD4, 0xE8, 0x4C,
		0x79, 0x32, 0xF2, 0x31, 0x74, 0x10, 0x5F, 0xDB, 0xF9, 0x68,
		0x99, 0x21, 0xCB, 0xC4, 0xDD, 0x59, 0x18, 0x6D, 0x66, 0x94,
		0x2D, 0xA8, 0xAB, 0x00, 0x30, 0x05, 0x11, 0xA1, 0xF1, 0x98,
		0x6D, 0xF4, 0xC2, 0x5F, 0x11, 0xF6, 0x25, 0x22, 0x1E, 0x71,
		0xDC, 0x8B, 0x39, 0x04, 0x28, 0x06, 0x53, 0xAA, 0x2D, 0xE2,
		0x68, 0x51, 0xC9, 0xAA, 0xAC, 0x8B, 0x28, 0x17, 0x53, 0x92,
		0x82, 0x3B, 0x67, 0x32, 0x6C, 0xC2, 0x27, 0x1F, 0x6E, 0x0B,
		0x3B, 0x80, 0xBF, 0x30, 0x3F, 0xDA, 0x34, 0xD3, 0x17, 0x83,
		0x98, 0x94, 0xDD, 0x58, 0xCB, 0xDA, 0xB6, 0xEA, 0x7F, 0x80,
		0x01, 0x60, 0xA2, 0xDD, 0xA1, 0x1B, 0x65, 0x6E, 0xD8, 0x40,
		0x8A, 0xEC, 0x3E, 0x0B, 0x3B, 0x33, 0x2B, 0x80, 0x94, 0x2C,
		0x0B, 0x29, 0x77, 0xF6, 0x6F, 0x24, 0x57, 0xCD, 0xA1, 0x45,
		0x02, 0xB2, 0x0B, 0xAC, 0x16, 0x77, 0xB3, 0xEA, 0xB0, 0xD3,
		0x06, 0x40, 0xFD, 0x8F, 0x7A, 0x08, 0xB8, 0x34, 0x26, 0x6C,
		0x1F, 0xB9, 0x9C, 0x8B, 0x2D, 0x18, 0xD7, 0x56, 0xC8, 0x52,
		0xEF, 0x6C, 0xAE, 0xD6, 0xCF, 0x04, 0xF4, 0xF6, 0x40, 0x61,
		0x9F, 0x4B, 0x39, 0x31, 0xB7, 0x39, 0x42, 0x4F, 0x4C, 0x38,
		0x76, 0x46, 0x39, 0x4F, 0x1B, 0xD1, 0xB6, 0x00, 0xA3, 0x57,
		0xD8, 0x52, 0x36, 0xE9, 0x05, 0xBE, 0x79, 0x3F, 0xDF, 0xC6,
		0x40, 0x2F, 0x7B, 0x60, 0xBB, 0x3B, 0x8B, 0x53, 0x8B, 0xD0,
		0x6F, 0xB0, 0xA8, 0xFE, 0x3B, 0xF4, 0x2A, 0x27, 0x16, 0xAE,
		0xCB, 0x34, 0x25, 0xBC, 0xC1, 0x5A, 0x86, 0xC4, 0x84, 0x43,
		0x09, 0xB1, 0xFF, 0xF5, 0x36, 0xB2, 0x3B, 0x64, 0x76, 0x32,
		0x80, 0x81, 0x34, 0xC9, 0x13, 0x9A, 0xAD, 0xBC, 0xAE, 0x89,
		0x01, 0x83, 0xF1, 0xDE, 0xBB, 0x76, 0xE1, 0xB7, 0xC5, 0x32,
		0x6D, 0x0B, 0xD5, 0xC0, 0x9A, 0x82, 0x4A, 0x56, 0xBA, 0xEA,
		0x7E, 0x2C, 0xAA, 0x07, 0x8D, 0x86, 0xAB, 0x07, 0x73, 0x86,
		0x8C, 0xF3, 0xD0, 0x59, 0x04, 0xA4, 0x34, 0x4A, 0x4B, 0xF1,
		0x9C, 0xB9, 0x91, 0x21, 0x3A, 0x0D, 0x08, 0xE0, 0xD1, 0x86,
		0x21, 0x39, 0xB0, 0x7E, 0xE3, 0x06, 0x27, 0x25, 0x29, 0x07,
		0x68, 0x7E, 0x70, 0x56, 0x53, 0xA7, 0x19, 0xDE, 0xA0, 0x6B,
		0x09, 0x6B, 0x90, 0x6B, 0x69, 0x61, 0x32, 0xD1, 0x2D, 0xB7,
		0x8E, 0x13, 0x86, 0xE3, 0xDD, 0xDF, 0x9B, 0xEA, 0x38, 0xA5,
		0x10, 0x3E, 0xB6, 0xDB, 0x5C, 0x06, 0x1A, 0xC4, 0xF3, 0x67,
		0xC5, 0x97, 0xF3, 0x57, 0x6A, 0xFE, 0x19, 0x4B, 0x69, 0x1B,
		0xAB, 0x37, 0x63, 0xE0, 0xE3, 0xC5, 0x84, 0x8B, 0x1B, 0x46,
		0x0F, 0xDE, 0x75, 0x94, 0xC3, 0x74, 0x6A, 0xAA, 0xB7, 0xD2,
		0x7D, 0xDD, 0x09, 0x91, 0x97, 0x71, 0xD8, 0x0A, 0x6E, 0xF8,
		0x26, 0xAA, 0xC9, 0xAD, 0x39, 0xD3, 0xB2, 0x21, 0x85, 0x14,
		0xD8, 0x98, 0xA2, 0xFC, 0xF9, 0x5C, 0x09, 0xCC, 0x62, 0xD9,
		0xF7, 0x40, 0xF6, 0x32, 0xC9, 0xE4, 0x0A, 0x16, 0x01, 0x69,
		0xD9, 0x5D, 0x23, 0xBD, 0x43, 0x08, 0x3B, 0x7E, 0x60, 0x25,
		0x4D, 0x62, 0xEA, 0xAD, 0x04, 0x46, 0xC2, 0x14, 0xA7, 0x99,
		0xDA, 0xB1, 0x68, 0x91, 0xB6, 0x69, 0xB2, 0x04, 0x2C, 0x9A,
		0xF9, 0xFF, 0x83, 0x70, 0xC1, 0xDD, 0xDE, 0x64, 0x11, 0xA0,
		0xF5, 0xE4, 0x7D, 0xCA, 0x6B, 0xAD, 0xA9, 0xA8, 0x36, 0xF5,
		0x7F, 0x04, 0x16, 0x85, 0xD9, 0x9B, 0x3A, 0x02, 0x22, 0x54,
		0xA9, 0x30, 0x62, 0xDE, 0xF0, 0xB8, 0x0F, 0x93, 0x56, 0x3B,
		0x3C, 0x41, 0x18, 0xBE, 0x1E, 0xB4, 0x4C, 0xA2, 0x93, 0xD8,
		0xBD, 0x5B, 0x35, 0x8F, 0x79, 0xE9, 0xB8, 0x92, 0x03, 0x18,
		0x18, 0x1B, 0xA6, 0x29, 0x5E, 0x13, 0x98, 0xCA, 0x6D, 0xFF,
		0x86, 0x66, 0x04, 0x6C, 0x84, 0xF8, 0xED, 0xCE, 0x17, 0xDC,
		0xB3, 0x52, 0x1D, 0xC9, 0x7D, 0x07, 0x44, 0xFA, 0xA9, 0x5D,
		0xB7, 0x95, 0xFA, 0x5B, 0xA7, 0xC0, 0x3A, 0x06, 0x8C, 0x36,
		0x91, 0x00, 0xF5, 0xB1, 0xA1, 0x3B, 0x63, 0x79, 0x8A, 0x23,
		0x47, 0x70, 0x28, 0xFE, 0x73, 0xBF, 0xDC, 0xE1, 0x30, 0xEB,
		0x49, 0x29, 0xCB, 0x62, 0xB0, 0x27, 0xA1, 0x68, 0xE6, 0x0E,
		0x1B, 0x35, 0x3E, 0x40, 0xDB, 0x27, 0x12, 0xF9, 0xAA, 0xAE,
		0xF3, 0x8E, 0x24, 0x74, 0xC0, 0x67, 0xBD, 0x9C, 0xC3, 0x8D,
		0xCF, 0x55, 0x3C, 0xCB, 0x76, 0x6C, 0x27, 0x08, 0xAA, 0x4C,
		0xE9, 0x57, 0x08, 0xAE, 0x97, 0x08, 0xD7, 0xE5, 0x9A, 0x37,
		0x4D, 0xAB, 0x37, 0x6C, 0xB6, 0xC0, 0x95, 0xF2, 0x6A, 0x5A,
		0x74, 0x8B, 0xBF, 0xEE, 0xBD, 0xB3, 0x3D, 0x16, 0x26, 0x75,
		0x5E, 0x60, 0xA4, 0xE1, 0x1A, 0xAA, 0x4E, 0xBA, 0x67, 0x2B,
		0xC2, 0x31, 0x27, 0x82, 0x7C, 0x8C, 0x09, 0x7A, 0xE1, 0x9C,
		0xE2, 0x86, 0x68, 0x4E, 0x2D, 0x03, 0x01, 0x56, 0x68, 0xFF,
		0x33, 0x99, 0x17, 0xC2, 0xB9, 0xC9, 0x38, 0x40, 0xE6, 0x98,
		0x5D, 0x78, 0x78, 0xD2, 0xA9, 0xA2, 0x8E, 0xC0, 0x45, 0x25,
		0xB2, 0x71, 0xB0, 0x3D, 0x38, 0x18, 0x29, 0xDD, 0x22, 0x01,
		0x43, 0xAB, 0x32, 0xE0, 0xFE, 0x6B, 0xD3, 0xB6, 0x43, 0xC1,
		0xB2, 0xBB, 0xEB, 0x92, 0xDA, 0x1A, 0x9E, 0x91, 0x2C, 0x4F,
		0x05, 0x44, 0x02, 0x9A, 0xDF, 0x31, 0x04, 0xEA, 0x78, 0xFD,
		0x8A, 0xEB, 0xFC, 0xD2, 0x2A, 0x37, 0x92, 0x17, 0x64, 0xD5,
		0xF1, 0x93, 0x04, 0x47, 0xC0, 0x91, 0xA2, 0xB5, 0x28, 0xB0,
		0x15, 0xB4, 0xC8, 0x1E, 0xD3, 0x6E, 0xE2, 0xC4, 0xF7, 0x62,
		0x40, 0xC5, 0x49, 0x48, 0x09, 0x73, 0x4D, 0x8E, 0x47, 0x02,
		0xCA, 0xB4, 0x46, 0xF4, 0x6E, 0xD8, 0xB2, 0xE5, 0x5F, 0xD6,
		0x94, 0x45, 0xB0, 0xC2, 0x45, 0x94, 0x5E, 0x49, 0x60, 0x72,
		0x3A, 0x5D, 0xCB, 0xAA, 0xC9, 0x60, 0x6C, 0xB7, 0x2D, 0x20,
		0x68, 0x37, 0xFA, 0xD8, 0x72, 0x5B, 0x48, 0x26, 0x67, 0x33,
		0xE0, 0x1D, 0x21, 0xC3, 0x0D, 0x09, 0x25, 0xB5, 0xFF, 0x36,
		0x93, 0xB5, 0xC5, 0x41, 0xDB, 0x3D, 0x0F, 0x8E, 0x1B, 0xD5,
		0x0B, 0x93, 0x02, 0x81, 0x9A, 0xA1, 0x4E, 0xD5, 0x78, 0x44,
		0xCA, 0xF2, 0xFF, 0x26, 0xB4, 0x15, 0x9A, 0xDB, 0x39, 0xE4,
		0x35, 0x77, 0x4A, 0x2F, 0xA2, 0x02, 0xD9, 0x48, 0xAE, 0xEB,
		0x9F, 0x12, 0x19, 0x66, 0x59, 0x95, 0x32, 0xBC, 0x9E, 0x46,
		0xEE, 0x5B, 0x2D, 0x49, 0x66, 0x35, 0xB8, 0x9F, 0x8A, 0x89,
		0x28, 0xAF, 0xA1, 0x34, 0x3C, 0x65, 0x28, 0x34, 0x3E, 0x39,
		0xE8, 0x3C, 0x31, 0xA6, 0x8E, 0x30, 0x28, 0xB7, 0xC5, 0x3F,
		0x9F, 0xC3, 0x8A, 0xDA, 0x19, 0x72, 0xDD, 0x72, 0x64, 0xE0,
		0x3C, 0xC1, 0x85, 0xE0, 0x91, 0x2E, 0x43, 0x41, 0x1E, 0x1F,
		0xD4, 0xD1, 0x7F, 0x29, 0x20, 0x07, 0xCE, 0x0B, 0xBC, 0xCB,
		0x16, 0xF4, 0x60, 0x10, 0x6A, 0xB9, 0xDD, 0xE5, 0x75, 0x5B,
		0x84, 0xFC, 0x06, 0xBD, 0x62, 0x09, 0x44, 0xFF, 0xB1, 0x4A,
		0xFB, 0x3E, 0xAC, 0x26, 0x4F, 0x09, 0x7C, 0x68, 0xBF, 0x8E,
		0xCD, 0x62, 0x6F, 0x6B, 0x9E, 0x9E, 0xCA, 0xC9, 0xDF, 0x43,
		0x79, 0x3F, 0x7F, 0xCB, 0x45, 0x24, 0xDB, 0xFC, 0x91, 0xE0,
		0x96, 0x9C, 0x6E, 0x2B, 0x36, 0xB4, 0x93, 0x2B, 0xDA, 0x49,
		0xFC, 0x56, 0x24, 0xF5, 0x26, 0xCA, 0x21, 0xE2, 0xBA, 0xF4,
		0xE9, 0xDA, 0x9B, 0x20, 0x75, 0x32, 0xF2, 0x40, 0x51, 0x2A,
		0x53, 0x79, 0xBE, 0xE4, 0x84, 0xF4, 0x2C, 0x90, 0x4D, 0x58,
		0x92, 0xB4, 0xA4, 0x7B, 0x63, 0xEE, 0x72, 0x72, 0x0B, 0x97,
		0x4E, 0x2E, 0x76, 0x50, 0xC5, 0x1A, 0x90, 0x9B, 0xBB, 0x84,
		0x0C, 0x26, 0xE0, 0x7E, 0xC5, 0x60, 0x45, 0x97, 0x67, 0x32,
		0x61, 0x70, 0x10, 0x30, 0x44, 0xAE, 0xF4, 0x2C, 0xBB, 0x31,
		0xB7, 0xFE, 0x66, 0x32, 0xA8, 0x18, 0xD9, 0x04, 0x7B, 0xEB,
		0x27, 0x8E, 0x00, 0x5E, 0x89, 0xD7, 0x63, 0x98, 0xA8, 0x1E,
		0x20, 0x29, 0x70, 0x72, 0xC9, 0xF7, 0x47, 0x50, 0x54, 0x41,
		0x06, 0xB8, 0x29, 0xBC, 0xE3, 0x95, 0xFD, 0x43, 0x17, 0x6D,
		0x7E, 0x66, 0xC1, 0xD8, 0xA4, 0x2B, 0x67, 0x58, 0x81, 0x99,
		0xEF, 0xFB, 0xE5, 0xE4, 0xED, 0x9A, 0x94, 0x64, 0x75, 0xC0,
		0x86, 0x58, 0x6A, 0x3C, 0xC4, 0xCB, 0xB9, 0x68, 0xB7, 0x0A,
		0x71, 0x53, 0x94, 0x8D, 0x03, 0x0E, 0xC4, 0x05, 0xD6, 0x3F,
		0xB7, 0xCD, 0xA8, 0xBF, 0x7D, 0xCA, 0x84, 0xE3, 0x24, 0x9A,
		0x9F, 0x6F, 0x97, 0xAB, 0xFB, 0x82, 0x74, 0x85, 0xDD, 0x7B,
		0x6A, 0x1F, 0x26, 0x2A, 0x59, 0x6D, 0xC5, 0x69, 0xC8, 0x11,
		0x35, 0x34, 0xD3, 0x0C, 0x6B, 0x81, 0xDB, 0xDF, 0xC2, 0xA2,
		0x47, 0x24, 0x87, 0xDD, 0x34, 0xCA, 0xC1, 0x5A, 0xF6, 0xCB,
		0x20, 0xF9, 0x39, 0xAA, 0x4C, 0x23, 0x4F, 0x83, 0x9F, 0x40,
		0x6D, 0xCB, 0x07, 0xAB, 0x78, 0xD4, 0x3A, 0x58, 0x05, 0xF2,
		0x4E, 0x94, 0x7F, 0x17, 0x25, 0xB6, 0x43, 0x76, 0xAE, 0xF8,
		0xAE, 0x33, 0x34, 0xCF, 0xBD, 0xAD, 0x08, 0x0D, 0x65, 0x71,
		0x74, 0x29, 0x74, 0x41, 0xBA, 0x6D, 0xDB, 0xCA, 0x53, 0xE2,
		0xC3, 0x6F, 0x8D, 0x2B, 0x9B, 0xDD, 0xA7, 0x42, 0xB8, 0xB1,
		0x95, 0xC3, 0xAD, 0x13, 0x81, 0x27, 0x38, 0xE2, 0x76, 0xCB,
		0xB6, 0xFD, 0xC7, 0x92, 0x1D, 0xE9, 0x72, 0x26, 0x5A, 0xDE,
		0x68, 0xD8, 0x3A, 0x5B, 0x0B, 0x30, 0x5A, 0x6D, 0xB1, 0x34,
		0x79, 0xBF, 0x0B, 0x2C, 0x04, 0x92, 0x12, 0xE6, 0x67, 0x14,
		0x4B, 0x43, 0xA3, 0x8D, 0x3D, 0xCF, 0x9F, 0xE7, 0x99, 0x4D,
		0x11, 0x20, 0x01, 0xC4, 0xDE, 0x99, 0x56, 0xBF, 0xF9, 0xE9,
		0x3B, 0x84, 0x26, 0x1A, 0x59, 0xF3, 0xCD, 0x66, 0x7A, 0xB7,
		0x36, 0x83, 0x46, 0x5B, 0xA2, 0x46, 0x72, 0x90, 0x1A, 0x05,
		0x19, 0x96, 0xF2, 0xD8, 0x4E, 0xD2, 0xB2, 0xD8, 0x93, 0x7E,
		0x49, 0x49, 0x69, 0xAE, 0x70, 0xEE, 0x8A, 0x15, 0x1E, 0x58,
		0xD5, 0xB9, 0x7E, 0x8B, 0x25, 0x5F, 0xF3, 0xB4, 0x9C, 0x71,
		0x18, 0x9F, 0x08, 0x63, 0x98, 0xAC, 0x73, 0xB0, 0x2F, 0x77,
		0x4B, 0xA2, 0x20, 0xF8, 0x43, 0x5B, 0xE3, 0x3A, 0x10, 0x3F,
		0xB8, 0x8A, 0x2F, 0x23, 0x69, 0xB1, 0x42, 0x65, 0x88, 0x7D,
		0x4A, 0xFE, 0x7A, 0xB3, 0x77, 0x8C, 0xA4, 0x4D, 0xE8, 0x80,
		0x2E, 0x43, 0x9F, 0x4D, 0x7B, 0x3E, 0x6A, 0xC1, 0xF1, 0xAE,
		0xDD, 0x3E, 0xB3, 0x17, 0x25, 0x8F, 0x2C, 0xAD, 0xB0, 0x80,
		0x2D, 0xA1, 0xF3, 0xCF, 0xC5, 0xC7, 0xE0, 0x8A, 0xB9, 0x34,
		0x0A, 0xC0, 0x78, 0xC3, 0x57, 0xDE, 0x1A, 0x5B, 0xD0, 0x19,
		0x32, 0x70, 0x07, 0x7C, 0x4D, 0x10, 0xC6, 0x5B, 0x3E, 0x31,
		0xC1, 0xAF, 0x0D, 0x60, 0x44, 0x85, 0x21, 0xBA, 0xA9, 0x4B,
		0x58, 0x9D, 0x31, 0xF8, 0x2A, 0x2A, 0x4E, 0x79, 0x72, 0x63,
		0xC1, 0xED, 0x5C, 0xBE, 0xE1, 0x72, 0x26, 0x15, 0x89, 0x6F,
		0x12, 0x38, 0x28, 0xEB, 0x2D, 0x35, 0xEB, 0xC6, 0x3D, 0xCA,
		0x42, 0xDB, 0x49, 0x2D, 0xD8, 0x14, 0xE2, 0x2D, 0x91, 0xD6,
		0x7F, 0xD8, 0x93, 0x66, 0x1C, 0xEF, 0xE4, 0xF9, 0x27, 0x82,
		0x4F, 0x63, 0x95, 0xF3, 0x78, 0x30, 0xB1, 0x13, 0x26, 0xBC,
		0x01, 0xCE, 0x52, 0x05, 0xCA, 0xDD, 0xDD, 0x40, 0xB7, 0x09,
		0xF4, 0x76, 0x78, 0x1F, 0x62, 0x8B, 0x68, 0xD9, 0xFE, 0x54,
		0xF1, 0xEC, 0x95, 0x8F, 0xCF, 0x0C, 0xB9, 0x26, 0x6D, 0x90,
		0x1D, 0xD6, 0xA4, 0x0A, 0xF6, 0xE2, 0x64, 0xFB, 0x40, 0x1B,
		0x43, 0xD5, 0xE5, 0xBD, 0x11, 0x1F, 0x70, 0x4E, 0x1A, 0xE6,
		0xC0, 0xCE, 0xEA, 0xCA, 0xC7, 0xF8, 0xB0, 0xB7, 0x01, 0x0B,
		0xF8, 0x20, 0x16, 0x9D, 0xC6, 0x06, 0x95, 0xE9, 0x6E, 0x7C,
		0x72, 0x29, 0xA8, 0xE9, 0x1F, 0xB9, 0xE4, 0x45, 0x46, 0xD1,
		0x18, 0xB2, 0x47, 0x84, 0x2D, 0xE3, 0xDC, 0x3A, 0xF3, 0x64,
		0x5E, 0xD7, 0x25, 0xE8, 0x05, 0x62, 0x8F, 0xCA, 0x02, 0x58,
		0xF1, 0x88, 0x00, 0x61, 0x6B, 0xEE, 0xB2, 0xC7, 0x05, 0xBB,
		0x5C, 0x0E, 0x8E, 0x27, 0xF3, 0xFE, 0x83, 0xA5, 0x68, 0xC9,
		0x5E, 0x87, 0x71, 0x6F, 0xDD, 0xDE, 0x17, 0x5A, 0xB1, 0x24,
		0x74, 0x41, 0x54, 0xD5, 0xF9, 0x0B, 0xB0, 0xCA, 0x2A, 0x96,
		0xAC, 0x3B, 0xC1, 0x65, 0xD3, 0x3C, 0xB5, 0x3D, 0x43, 0xB6,
		0x87, 0x03, 0x0C, 0x56, 0x2E, 0xEA, 0x94, 0xB1, 0x2C, 0x95,
		0x84, 0x5F, 0x02, 0xC6, 0x20, 0x30, 0x4B, 0x44, 0x64, 0x37,
		0xB4, 0x0A, 0xA9, 0x74, 0x20, 0x8C, 0x71, 0x05, 0x20, 0x0A,
		0x74, 0x9C, 0xBA, 0xB4, 0xEA, 0x95, 0x72, 0x9F, 0x1A, 0x5D,
		0x42, 0x64, 0xD5, 0xD3, 0x0B, 0xCC, 0xC7, 0x33, 0x6B, 0xF4,
		0x46, 0x29, 0xD9, 0xFD, 0x6A, 0x71, 0x99, 0x7E, 0xF6, 0x48,
		0x48, 0xD7, 0x15, 0x01, 0x69, 0x8F, 0x2C, 0xCA, 0xF9, 0x8C,
		0x24, 0x91, 0x6E, 0x3D, 0x34, 0x1E, 0x7C, 0x43, 0xF3, 0x16,
		0xD4, 0xBA, 0xDD, 0x36, 0x0C, 0xDD, 0x2C, 0xE6, 0x7E, 0xE3,
		0x22, 0x96, 0x31, 0x88, 0x9B, 0x77, 0x83, 0x51, 0x16, 0xFE,
		0xE3, 0x9C, 0xED, 0x48, 0x2A, 0x52, 0x07, 0x05, 0xE3, 0xE9,
		0x55, 0xBB, 0x15, 0x27, 0xD0, 0xE3, 0x4E, 0x99, 0x53, 0x80,
		0x8E, 0x0F, 0x67, 0xE6, 0xD5, 0x4E, 0xE4, 0x76, 0x43, 0x18,
		0x08, 0xD1, 0xFE, 0x08, 0x3E, 0xA5, 0x9D, 0x2C, 0xCF, 0x5F,
		0x5E, 0xBA, 0xF5, 0xE6, 0xCA, 0xF7, 0xE5, 0x15, 0xFE, 0xDE,
		0x90, 0x8B, 0x23, 0x15, 0x06, 0x21, 0x41, 0x8A, 0x74, 0xF9,
		0xB2, 0x66, 0xC0, 0xD9, 0xCF, 0x20, 0xB2, 0xFA, 0x3D, 0xCE,
		0xC2, 0xDF, 0x24, 0x76, 0x96, 0x94, 0x30, 0xA8, 0x71, 0x49,
		0x48, 0x97, 0xDC, 0x03, 0xF2, 0xD7, 0xF1, 0xB1, 0x02, 0x7B,
		0x73, 0x5F, 0x0B, 0xF3, 0xD6, 0xF4, 0x4F, 0x26, 0x8F, 0x3B,
		0x40, 0x2D, 0xAC, 0xDA, 0xE2, 0x82, 0x63, 0x14, 0xC6, 0x09,
		0x9C, 0xB5, 0xEA, 0x80, 0xED, 0x88, 0x5B, 0x0F, 0x00, 0x5E,
		0x5C, 0xF3, 0x28, 0xEA, 0xA9, 0xA4, 0x62, 0x88, 0x8B, 0x5D,
		0x9E, 0xCD, 0x5E, 0x5C, 0xC9, 0x35, 0x1F, 0x37, 0x12, 0xF2,
		0x36, 0x4D, 0xA5, 0x2B, 0x65, 0x1D, 0x04, 0x6D, 0x7C, 0x61,
		0x52, 0xED, 0xF3, 0xED, 0x97, 0x0A, 0x63, 0xCC, 0x12, 0xF4,
		0x75, 0x9A, 0x62, 0x6B, 0x34, 0x02, 0xD3, 0xEF, 0xED, 0x69,
		0xEA, 0xAD, 0xFC, 0xF3, 0x4C, 0xF4, 0xFE, 0x8B, 0xB7, 0xBC,
		0x99, 0xD7, 0x22, 0xEE, 0xC8, 0xD3, 0xD0, 0x31, 0x4B, 0xB7,
		0x9F, 0x09, 0xE5, 0x0E, 0xE6, 0x83, 0x41, 0xA3, 0xB7, 0x1D,
		0xDA, 0x9E, 0x45, 0x9E, 0x4F, 0x1A, 0xC0, 0xC3, 0x1F, 0xE3,
		0xC2, 0x02, 0xD6, 0xF7, 0x03, 0x5B, 0x1D, 0xA4, 0x0F, 0xEC,
		0x1F, 0xAD, 0x8D, 0x33, 0x08, 0x39, 0xC8, 0xBC, 0xB7, 0x6B,
		0xD7, 0xC1, 0x8E, 0x8D, 0xB4, 0x9D, 0x33, 0x3F, 0x92, 0xD8,
		0xBF, 0xEF, 0xDF, 0xD8, 0xEB, 0xA7, 0x43, 0xAD, 0xCB, 0x9E,
		0xEE, 0x19, 0x64, 0x45, 0xAA, 0x8E, 0x3B, 0x79, 0xDE, 0xE2,
		0xF3, 0x7F, 0xB3, 0xFE, 0xBC, 0xAE, 0xA9, 0x19, 0xA5, 0xCF,
		0xFA, 0x0C, 0xBD, 0x74, 0x32, 0xCA, 0x3C, 0xC9, 0xAC, 0xE7,
		0x1B, 0xBB, 0x3C, 0x6B, 0x6A, 0x93, 0x6F, 0x38, 0xA7, 0x59,
		0x6C, 0x2D, 0x4B, 0x50, 0x4C, 0xF1, 0xBA, 0x1B, 0xA2, 0x98,
		0x8E, 0x5F, 0x69, 0x07, 0x09, 0x58, 0xAC, 0xD4, 0x7D, 0x52,
		0xE8, 0xA8, 0xA7, 0x2A, 0x32, 0x40, 0x71, 0x6E, 0x1A, 0x5E,
		0x46, 0x37, 0x79, 0x0E, 0xD7, 0x03, 0x64, 0x6A, 0x48, 0xB0,
		0x1D, 0x25, 0x72, 0x54, 0x11, 0x4D, 0xFB, 0xAD, 0xD9, 0x84,
		0xF2, 0xB6, 0x70, 0x20, 0x68, 0x44, 0x31, 0xB0, 0xBA, 0x95,
		0xA1, 0xD7, 0x40, 0x14, 0xEC, 0xA0, 0xAC, 0x65, 0xB5, 0xE3,
		0xDD, 0x9E, 0x8A, 0xEE, 0x73, 0x51, 0xEC, 0xD8, 0xDB, 0xEC,
		0xD5, 0x13
	];
	let key: [u8; 63] = [
		0x46, 0x5B, 0x73, 0x33, 0xD8, 0x26, 0x74, 0x8C, 0xE7,
		0x42, 0x09, 0xD3, 0x9E, 0x90, 0x76, 0x5F, 0x2C, 0xED,
		0x3A, 0xD1, 0xC0, 0x20, 0xDF, 0x66, 0xE9, 0x48, 0x46,
		0xDC, 0xD0, 0x18, 0xF2, 0xC3, 0x69, 0x40, 0x68, 0xB7,
		0x74, 0x19, 0x8D, 0xE7, 0x37, 0x76, 0x03, 0xBD, 0x05,
		0x73, 0x36, 0xBA, 0x2D, 0xBD, 0xF1, 0xCE, 0xAF, 0x3F,
		0x53, 0xAE, 0x12, 0xE7, 0x4C, 0x65, 0x0D, 0xAB, 0x02
	];
	let cust: [u8; 10] = [
		0xEC, 0x80, 0xE9, 0xFF, 0x52, 0x6C, 0xE8, 0x21, 0x10, 0x13
	];
	let exp: [u8; 16] = [
		0xD7, 0xB9, 0x00, 0x54, 0x59, 0x3C,
		0x63, 0xF3, 0xE1, 0x62, 0x88, 0x52,
		0x3D, 0x68, 0x47, 0xC4
	];

	lc_rust_kmac_one(lcr_kmac_type::lcr_kmac_256, &key, &cust, &msg, &exp)
}

#[test]
fn lc_rust_kmac_128() {
	let msg: [u8; 6764] = [
		0x4E, 0xE1, 0xC6, 0xCC, 0xA6, 0x57, 0x66, 0x32, 0xE1, 0x10,
		0x49, 0xE3, 0x59, 0xB2, 0xF7, 0x57, 0x09, 0xAC, 0x91, 0x13,
		0xFB, 0x20, 0x16, 0x55, 0x9B, 0x6B, 0xED, 0x16, 0xB2, 0xC9,
		0x48, 0x51, 0x7D, 0xB1, 0xFD, 0xBF, 0xED, 0x84, 0x15, 0xA2,
		0xC8, 0x61, 0x4B, 0xF0, 0xEC, 0xD1, 0xDD, 0xD6, 0x50, 0x36,
		0xD6, 0x86, 0x8F, 0xD5, 0x00, 0x4C, 0xA0, 0xB3, 0x50, 0xBB,
		0xF9, 0x2F, 0x94, 0x2C, 0x00, 0x36, 0xB5, 0x36, 0x89, 0x48,
		0x8D, 0xE0, 0x33, 0x20, 0xE8, 0xFC, 0x6D, 0xB7, 0x09, 0xEC,
		0x26, 0xE7, 0x46, 0x47, 0xF6, 0x49, 0x88, 0x32, 0xE8, 0x86,
		0x07, 0x7B, 0x64, 0x49, 0xC3, 0xB0, 0x5E, 0xB8, 0xA7, 0xF6,
		0xBD, 0x04, 0xBE, 0x96, 0x7D, 0xCC, 0xD0, 0x17, 0x07, 0xE9,
		0xD3, 0xA7, 0x2E, 0xB5, 0x05, 0xBD, 0xE0, 0xF2, 0xE4, 0xA8,
		0x2C, 0x73, 0x3B, 0x47, 0xDE, 0x0A, 0xAD, 0x04, 0x38, 0xF0,
		0xE5, 0x40, 0x5E, 0x38, 0xA1, 0x2C, 0xA5, 0x26, 0x3A, 0xDB,
		0xF1, 0xB4, 0xA5, 0x23, 0xCE, 0xE5, 0x85, 0x11, 0x4D, 0xFB,
		0xA1, 0x43, 0xF8, 0x06, 0x2E, 0x5E, 0x30, 0x64, 0xBC, 0xB7,
		0xA8, 0xC2, 0xCC, 0xAD, 0x0C, 0xA1, 0x8D, 0xB4, 0xC6, 0x52,
		0xB9, 0x81, 0x36, 0xDD, 0xAE, 0x5B, 0xD2, 0x6B, 0xAB, 0x9B,
		0xC1, 0x55, 0xC6, 0x04, 0x9E, 0xF2, 0xE2, 0xEC, 0xED, 0x67,
		0x7F, 0x5E, 0xA7, 0x70, 0xB8, 0xA0, 0x10, 0x78, 0x8F, 0x7A,
		0x80, 0x47, 0x67, 0x14, 0xFF, 0x3F, 0x35, 0xDA, 0xF2, 0x1B,
		0xC6, 0xAF, 0xC2, 0x64, 0x27, 0x60, 0x63, 0xDC, 0x25, 0x9C,
		0x1C, 0x0E, 0xFB, 0x2E, 0x67, 0xB9, 0xC0, 0x6B, 0xB7, 0xA1,
		0x77, 0x99, 0x1B, 0x6E, 0xFF, 0xE9, 0xB6, 0x2A, 0xE0, 0xC4,
		0x8E, 0xFE, 0xFA, 0xE9, 0x96, 0xE6, 0x18, 0x68, 0xDB, 0x70,
		0x41, 0x88, 0x11, 0xAA, 0xE8, 0x80, 0x30, 0x1E, 0x3E, 0xE0,
		0x03, 0xC3, 0xD8, 0xED, 0x04, 0x5C, 0x96, 0xDA, 0xCC, 0x03,
		0xD1, 0xA4, 0x64, 0x36, 0x46, 0x04, 0x8E, 0x2C, 0x4C, 0xA6,
		0x9A, 0xC4, 0x2E, 0x90, 0x8F, 0x7C, 0xF5, 0xB1, 0x49, 0xCB,
		0xEC, 0x88, 0xCE, 0x45, 0xFA, 0xD4, 0x03, 0x89, 0x15, 0xE2,
		0x8C, 0xB2, 0x2C, 0x0A, 0x01, 0xD0, 0x04, 0x07, 0x5B, 0x32,
		0x4C, 0x1D, 0xA6, 0x0D, 0x26, 0xFE, 0x98, 0xDB, 0x77, 0x1A,
		0x3F, 0x22, 0x8C, 0xF8, 0x0D, 0xD3, 0x75, 0x69, 0x3D, 0xA1,
		0x37, 0xDA, 0x49, 0xF2, 0xB3, 0xC0, 0xC2, 0xC7, 0x6C, 0x06,
		0x97, 0x35, 0xD3, 0xD6, 0xC4, 0xFA, 0x5B, 0x44, 0xAF, 0x38,
		0x76, 0x3A, 0x64, 0x5F, 0xD4, 0x40, 0x72, 0xEE, 0xD1, 0x6E,
		0xD7, 0x9C, 0x3C, 0x2F, 0x09, 0x10, 0x17, 0x01, 0x51, 0xD6,
		0x37, 0x0E, 0x16, 0xDC, 0x79, 0x14, 0x62, 0xFF, 0xE7, 0xDF,
		0x41, 0x4C, 0x05, 0x7A, 0xD1, 0x40, 0xEB, 0x61, 0xF4, 0x05,
		0x96, 0xF9, 0xD5, 0x82, 0xE2, 0x11, 0x49, 0x36, 0xD6, 0xAC,
		0xD9, 0x9C, 0x91, 0xB5, 0xB3, 0x47, 0x7A, 0x4F, 0xC0, 0xC8,
		0x45, 0x09, 0xE8, 0xA7, 0xF0, 0xB9, 0xAD, 0x85, 0x66, 0xDF,
		0xBC, 0xF1, 0x9E, 0xE3, 0x8B, 0xEA, 0x56, 0x90, 0x2B, 0x3A,
		0xDC, 0xA8, 0x09, 0xD0, 0x87, 0x32, 0x06, 0xB3, 0xFF, 0x37,
		0x1F, 0xA7, 0x52, 0x92, 0xE8, 0x22, 0xE4, 0x2C, 0x13, 0x0E,
		0x4B, 0x26, 0xDF, 0xDF, 0x7C, 0x60, 0xCE, 0x98, 0x69, 0xB2,
		0x7F, 0x9A, 0xF7, 0x12, 0x5B, 0x2E, 0x2C, 0xCD, 0xD4, 0x04,
		0xC8, 0xF9, 0xAB, 0x45, 0xBE, 0x38, 0x52, 0x6D, 0x7B, 0xF4,
		0x9D, 0xD6, 0x2A, 0x23, 0x4D, 0x38, 0xD6, 0x35, 0x9D, 0x5B,
		0x36, 0xCF, 0xAC, 0xAB, 0x7D, 0xDE, 0xE0, 0xA4, 0xFF, 0x63,
		0x17, 0x81, 0x02, 0x3E, 0x32, 0x98, 0x5E, 0xB6, 0x14, 0x1A,
		0xBC, 0x05, 0x67, 0x15, 0x5E, 0xFD, 0x78, 0xD8, 0xB7, 0x5A,
		0x92, 0x43, 0x4E, 0xD7, 0x42, 0xFA, 0x91, 0x0A, 0xFC, 0xD3,
		0x52, 0x0E, 0x72, 0xE2, 0x0A, 0xFC, 0xD3, 0xA3, 0xFB, 0x04,
		0x92, 0x69, 0x79, 0xEC, 0x32, 0x79, 0xC9, 0x3E, 0xB6, 0x79,
		0xB1, 0x33, 0xB2, 0x71, 0x46, 0x21, 0x7D, 0xD6, 0xE1, 0x2E,
		0x9D, 0xF0, 0x1F, 0x8E, 0x06, 0xEE, 0x7F, 0x31, 0x53, 0x15,
		0x77, 0x2A, 0x0C, 0x69, 0xCC, 0xD8, 0x65, 0xCC, 0xAD, 0x24,
		0x28, 0x2E, 0xFA, 0x1C, 0x61, 0x59, 0xFC, 0xF2, 0x13, 0x5D,
		0x11, 0x4A, 0xCD, 0x07, 0x6D, 0x5E, 0x42, 0xDF, 0xB8, 0xDF,
		0x9E, 0xF2, 0x6C, 0xB1, 0x96, 0x13, 0x8C, 0xAE, 0x63, 0x5A,
		0x7F, 0x8F, 0x20, 0xAF, 0x35, 0x0A, 0x4F, 0x61, 0x6D, 0xBE,
		0xCD, 0x1D, 0x23, 0xC0, 0xC6, 0x0D, 0x3E, 0x98, 0x18, 0x30,
		0x33, 0xE5, 0x5B, 0xCE, 0xD4, 0x5E, 0x39, 0x4A, 0x7E, 0xCF,
		0x18, 0xCA, 0x0F, 0x37, 0x1E, 0xD7, 0x57, 0x0C, 0xA0, 0x86,
		0x8E, 0x75, 0xC4, 0x14, 0xAD, 0x72, 0x51, 0xA6, 0xFB, 0x14,
		0xE3, 0xA4, 0x79, 0x73, 0x91, 0x9E, 0x66, 0x78, 0xBB, 0xD6,
		0x91, 0xDD, 0xD7, 0x32, 0xD3, 0x7B, 0x90, 0x74, 0x67, 0xC1,
		0x21, 0x21, 0x3D, 0x91, 0xA3, 0xC4, 0x83, 0xC2, 0x46, 0x8F,
		0x35, 0xCA, 0x27, 0x55, 0x01, 0xEB, 0x46, 0xA0, 0xAB, 0x85,
		0x98, 0x78, 0x2E, 0xDE, 0x18, 0x32, 0x3A, 0x48, 0xD6, 0xF4,
		0x02, 0x86, 0x71, 0x23, 0x69, 0xE4, 0x90, 0xBF, 0x13, 0x3D,
		0x23, 0xFF, 0xF1, 0xE9, 0xB5, 0xA9, 0x10, 0x11, 0x7A, 0xA9,
		0x70, 0x92, 0xFA, 0xD8, 0xE4, 0x8B, 0x06, 0xCD, 0x51, 0xB7,
		0x02, 0xA7, 0xC1, 0x38, 0x79, 0xEA, 0x74, 0x37, 0x67, 0x7B,
		0x94, 0xB6, 0x32, 0xA6, 0x5E, 0x7D, 0x03, 0x34, 0xAC, 0x69,
		0xEA, 0x3C, 0xEF, 0x98, 0x2C, 0x7B, 0x71, 0xE6, 0x5A, 0xD1,
		0x60, 0x24, 0x76, 0x46, 0xE4, 0x82, 0x7D, 0xF5, 0x1E, 0x61,
		0x13, 0xFA, 0x4E, 0x6A, 0x41, 0x9D, 0x0E, 0x6C, 0xBF, 0x3F,
		0x5C, 0x67, 0xF2, 0xC7, 0xFE, 0xE6, 0x29, 0x3E, 0x2D, 0x97,
		0x4C, 0x87, 0x31, 0xB5, 0xE5, 0xD6, 0x54, 0x41, 0x12, 0x1D,
		0xBC, 0x60, 0x9B, 0x9F, 0x31, 0xE8, 0x23, 0x19, 0xD7, 0x5A,
		0x13, 0xBD, 0x6E, 0x8C, 0x87, 0x46, 0xFB, 0xCA, 0x87, 0xF6,
		0x9F, 0x39, 0x55, 0xBA, 0x00, 0x45, 0xE2, 0x71, 0x83, 0x18,
		0xF8, 0x22, 0x29, 0x2D, 0xCC, 0xE3, 0xAD, 0xEE, 0x92, 0x85,
		0xDD, 0xC9, 0x78, 0x83, 0x52, 0x4D, 0xCD, 0x14, 0x27, 0x69,
		0xEF, 0x4C, 0x63, 0x3C, 0x4B, 0x89, 0x43, 0x49, 0x8D, 0xC4,
		0x6E, 0x00, 0xE3, 0x4F, 0xDD, 0x66, 0x0B, 0x3B, 0xA1, 0xAE,
		0xAF, 0xEB, 0x17, 0xEB, 0x33, 0xEB, 0xC3, 0x98, 0x03, 0x8E,
		0x09, 0x50, 0x72, 0x67, 0x68, 0xD5, 0x88, 0x34, 0x68, 0x6D,
		0x92, 0x5C, 0xD9, 0xA9, 0x71, 0x11, 0xB4, 0x3B, 0x1C, 0x55,
		0xDB, 0xD6, 0x64, 0x9A, 0xD0, 0x58, 0x22, 0xEB, 0x71, 0xB3,
		0xF6, 0xD5, 0xA6, 0x66, 0x6D, 0xDF, 0x11, 0x98, 0x8B, 0xF9,
		0xBB, 0xE6, 0x11, 0x05, 0x49, 0x80, 0x43, 0x39, 0xD7, 0xA4,
		0x9E, 0xB9, 0x6E, 0xA9, 0xF4, 0x76, 0x2F, 0x84, 0xFF, 0xF4,
		0xCF, 0x43, 0x1B, 0x7A, 0x73, 0xDB, 0xDD, 0x1D, 0xE2, 0x48,
		0xB5, 0x53, 0xF5, 0x8A, 0xFF, 0xD0, 0x4C, 0x94, 0x1E, 0xCF,
		0x4C, 0xF1, 0x2F, 0x8C, 0x2D, 0x6A, 0xB0, 0x83, 0x83, 0x50,
		0x27, 0xC7, 0xEB, 0x80, 0x84, 0xBA, 0xB6, 0x5E, 0x99, 0x67,
		0x4F, 0x90, 0x2B, 0x06, 0x31, 0xCC, 0x23, 0x1E, 0x2C, 0x78,
		0xC7, 0x56, 0x4B, 0xEB, 0x74, 0x53, 0xFE, 0xD1, 0xD8, 0x0D,
		0x42, 0x2C, 0xAD, 0x0D, 0x4E, 0xE7, 0x20, 0x19, 0xA9, 0x2B,
		0x48, 0x14, 0xEA, 0x6A, 0xBA, 0xC1, 0x45, 0x2E, 0x5E, 0x06,
		0x77, 0xF3, 0xB0, 0x31, 0x5E, 0x76, 0xAF, 0x0B, 0x99, 0x2D,
		0xE6, 0xFF, 0x72, 0x84, 0xD1, 0x0A, 0x51, 0x70, 0x35, 0x8F,
		0xD2, 0x9B, 0x11, 0xDC, 0xA9, 0x25, 0xCC, 0xFA, 0xF7, 0x0E,
		0x80, 0xE5, 0xD1, 0x4A, 0x92, 0x14, 0x37, 0xFE, 0x7E, 0x84,
		0x77, 0xF2, 0xB4, 0xBB, 0x08, 0xF2, 0x39, 0xB7, 0x65, 0xB1,
		0xCF, 0xC6, 0x13, 0x2F, 0x50, 0x09, 0x07, 0xE7, 0x2F, 0xDD,
		0xFA, 0x26, 0xFE, 0x4E, 0xE8, 0xEF, 0xF2, 0xAC, 0x3C, 0x37,
		0xBC, 0x21, 0x36, 0x48, 0xBF, 0x23, 0xA4, 0x19, 0xC4, 0xD9,
		0x02, 0xC7, 0x21, 0xF5, 0x5C, 0x68, 0xD4, 0xC3, 0xF6, 0xCE,
		0x4B, 0xF4, 0x1F, 0xB5, 0xCC, 0x37, 0x15, 0x58, 0xD4, 0xCD,
		0xCC, 0x3C, 0x7E, 0x86, 0xA7, 0x2C, 0x2F, 0xE7, 0xF3, 0x47,
		0xD7, 0x53, 0xDC, 0x4F, 0x92, 0xC7, 0x5D, 0x14, 0xA6, 0xBC,
		0xBC, 0xBE, 0xE8, 0xBA, 0xBD, 0x64, 0x4E, 0x66, 0x30, 0xB3,
		0x6B, 0x7C, 0x6C, 0xA1, 0x29, 0x95, 0x8E, 0x7E, 0x09, 0x03,
		0x25, 0x90, 0x81, 0xDF, 0x89, 0x09, 0x58, 0x6D, 0x1B, 0xFD,
		0xEA, 0x81, 0xD4, 0x41, 0x6E, 0xF5, 0xD1, 0x0C, 0x05, 0xB1,
		0x30, 0x54, 0x78, 0x75, 0x27, 0x61, 0x5C, 0x35, 0xBD, 0xAE,
		0xC4, 0x48, 0x95, 0x7B, 0xFE, 0x2A, 0xB3, 0xBA, 0xC7, 0xC3,
		0x15, 0x7E, 0xBA, 0xA8, 0x28, 0x92, 0xDD, 0xC9, 0xD6, 0xE6,
		0x43, 0x52, 0xC6, 0x9F, 0x7F, 0x23, 0xBB, 0x35, 0x9A, 0x50,
		0xCC, 0x6E, 0xEC, 0xD6, 0xA6, 0x13, 0x13, 0xB1, 0x0B, 0x44,
		0xDC, 0x15, 0x5E, 0xCC, 0x50, 0xAF, 0x78, 0x33, 0xAB, 0xC1,
		0xB1, 0x29, 0xA3, 0xA9, 0x84, 0x6A, 0x5B, 0xA9, 0x2D, 0x09,
		0x6F, 0x8F, 0xD0, 0x26, 0x8F, 0x65, 0x2E, 0x33, 0xDC, 0xD3,
		0x54, 0x8F, 0xF2, 0x79, 0x90, 0xC8, 0xBC, 0x22, 0x48, 0xAC,
		0x2D, 0x2A, 0xC5, 0x6A, 0x77, 0x5B, 0xD4, 0x1A, 0x0A, 0x82,
		0xE6, 0x68, 0x41, 0xFD, 0xBE, 0x6C, 0xDE, 0x1B, 0xEE, 0x99,
		0x5C, 0xA6, 0x5E, 0x70, 0x2C, 0x29, 0x84, 0x77, 0xC4, 0x8F,
		0xEC, 0x42, 0x11, 0x6E, 0x07, 0x2E, 0xFC, 0xAE, 0x41, 0x7D,
		0xFF, 0x8A, 0xDA, 0x3A, 0x44, 0x81, 0xB8, 0x77, 0xF1, 0x55,
		0x57, 0xA9, 0x64, 0x5A, 0x1B, 0x34, 0x58, 0xC1, 0x19, 0xB9,
		0xD9, 0xD3, 0xA4, 0xE4, 0xBD, 0x2B, 0xE9, 0xF9, 0x75, 0xB0,
		0xC1, 0x36, 0xC4, 0x52, 0xF3, 0x77, 0xDB, 0xB7, 0x32, 0x46,
		0xE2, 0x5E, 0x4B, 0x69, 0x73, 0xAA, 0x6D, 0xFA, 0x68, 0x7C,
		0xDE, 0xBE, 0xE5, 0x06, 0xD0, 0xF2, 0x01, 0xB5, 0xF2, 0x77,
		0xC7, 0x9A, 0xD5, 0x3C, 0xBE, 0xCB, 0x53, 0xD2, 0x55, 0x83,
		0xB1, 0x37, 0x80, 0xFE, 0xBF, 0xAA, 0xC5, 0xA6, 0x99, 0xAF,
		0xA2, 0xB1, 0x94, 0x4A, 0xEC, 0x7F, 0x01, 0x17, 0x24, 0x48,
		0xFA, 0x8C, 0xEE, 0xB2, 0x45, 0xA2, 0x08, 0x10, 0x0F, 0x0D,
		0x59, 0x10, 0xAC, 0xFC, 0xC7, 0xCB, 0x3D, 0xE6, 0x1E, 0xF5,
		0x3A, 0xFC, 0x41, 0xE0, 0x2A, 0x0D, 0x3C, 0xDE, 0x4D, 0xCD,
		0xD8, 0xAD, 0xE2, 0x01, 0x80, 0x78, 0x02, 0xBD, 0xE5, 0xDD,
		0x9F, 0x00, 0xA2, 0xDB, 0x90, 0xCC, 0xBE, 0xD1, 0x28, 0x3A,
		0x4D, 0xDA, 0xC4, 0x07, 0x49, 0x7B, 0x05, 0xF0, 0x70, 0x2B,
		0x7F, 0xA8, 0x11, 0xB1, 0xE8, 0xEA, 0xDD, 0x6A, 0xCE, 0xF0,
		0x05, 0x8A, 0xC1, 0xE3, 0xFC, 0x84, 0x77, 0x46, 0xD1, 0x9A,
		0x38, 0xBC, 0x86, 0x5E, 0x7E, 0xC0, 0x97, 0xBF, 0x6B, 0x4E,
		0x68, 0x07, 0x48, 0xA7, 0xC1, 0x9C, 0x84, 0x19, 0x21, 0x61,
		0x67, 0xF6, 0x8E, 0xAE, 0x98, 0xCB, 0x9A, 0x5C, 0xC6, 0xB5,
		0xC9, 0x6D, 0xF8, 0xB9, 0xD2, 0x41, 0x44, 0x95, 0xCB, 0x0A,
		0x4B, 0x44, 0x61, 0x5C, 0x9E, 0x9E, 0x56, 0x5C, 0x3F, 0xC3,
		0x39, 0x41, 0xB4, 0xC9, 0x54, 0x0A, 0xFE, 0xCB, 0xF3, 0x4A,
		0xCA, 0xE3, 0x04, 0xCC, 0xAB, 0x07, 0xA6, 0x9F, 0x03, 0x00,
		0x60, 0xBD, 0x5B, 0xD6, 0x2B, 0xBB, 0x01, 0x8C, 0x7A, 0x8C,
		0x16, 0x65, 0x3F, 0xCB, 0x82, 0x75, 0x75, 0x9E, 0x58, 0xD5,
		0x41, 0xA5, 0x52, 0x26, 0xE0, 0x77, 0x00, 0x63, 0xE9, 0x19,
		0x2B, 0x6C, 0xC1, 0x7E, 0x8F, 0xBB, 0xB6, 0x29, 0x67, 0xAC,
		0xE0, 0x40, 0x70, 0x46, 0x71, 0xAB, 0x5A, 0x04, 0x2F, 0x95,
		0x20, 0xD1, 0x0C, 0xC5, 0x7F, 0xEF, 0x78, 0x12, 0x00, 0x93,
		0x68, 0xE2, 0xCC, 0xEA, 0xD3, 0x87, 0x11, 0x99, 0xD0, 0xB2,
		0x0A, 0x96, 0x6F, 0x65, 0x9A, 0xC1, 0x95, 0xCB, 0x38, 0xE0,
		0x0A, 0x68, 0x45, 0xB6, 0x03, 0xFE, 0xF4, 0xD3, 0x18, 0x5F,
		0x6D, 0xC1, 0xBF, 0xD7, 0x11, 0x7B, 0x01, 0x06, 0x11, 0xAA,
		0x8C, 0x64, 0x6D, 0x75, 0x4B, 0x24, 0xCF, 0x00, 0x98, 0x97,
		0x4F, 0xE5, 0xD3, 0xC0, 0x6A, 0x79, 0x5D, 0xCD, 0xF0, 0xE5,
		0x31, 0x9D, 0xFE, 0x3D, 0x27, 0x56, 0xBA, 0x3F, 0x65, 0x33,
		0x58, 0xE4, 0xB1, 0x0C, 0xA1, 0x74, 0xE2, 0xA2, 0x84, 0xF4,
		0xE3, 0x26, 0x70, 0x16, 0x9C, 0x94, 0x17, 0x7A, 0x16, 0x77,
		0xAE, 0x5A, 0xEA, 0xC2, 0xB6, 0x2D, 0xD9, 0x4F, 0x18, 0xCF,
		0x19, 0xB3, 0x27, 0x09, 0xAE, 0xEC, 0x7A, 0xEB, 0x3B, 0x81,
		0xF1, 0x0C, 0x28, 0xDB, 0xA8, 0xE8, 0xCB, 0xD7, 0x8C, 0x3F,
		0x25, 0xD4, 0xF3, 0x8C, 0x3C, 0x07, 0xD0, 0x2B, 0xF7, 0xDC,
		0xF6, 0xD4, 0xAF, 0xAC, 0xD0, 0x8C, 0x0E, 0x3B, 0xEE, 0x29,
		0xD3, 0x3D, 0x4F, 0x09, 0x04, 0xEB, 0x61, 0xBB, 0xEE, 0xAE,
		0x09, 0xCF, 0x23, 0xC7, 0x7C, 0xEA, 0x3F, 0x35, 0x45, 0x9C,
		0x15, 0x4E, 0x28, 0x8A, 0x8B, 0xF6, 0xF7, 0x89, 0x97, 0x96,
		0x97, 0xD4, 0xA7, 0xB0, 0x07, 0xA2, 0xBE, 0x32, 0xC2, 0x81,
		0xA3, 0xE6, 0x79, 0x20, 0x66, 0x81, 0xCA, 0xA7, 0x36, 0x68,
		0x22, 0xA5, 0x10, 0x95, 0xDA, 0xDB, 0xDF, 0x0E, 0x3C, 0xC7,
		0x6D, 0x05, 0xA2, 0x67, 0xCF, 0x70, 0x51, 0xEE, 0x6F, 0xB2,
		0x52, 0x62, 0xDB, 0x55, 0x32, 0xF0, 0x29, 0x08, 0x98, 0xC8,
		0xBD, 0x64, 0x8E, 0x39, 0xFE, 0x66, 0x45, 0xF9, 0x9F, 0x76,
		0x54, 0x11, 0x95, 0x9E, 0xD7, 0xD5, 0xD6, 0xA2, 0x29, 0xFF,
		0x49, 0x87, 0x0C, 0xC5, 0xA1, 0x20, 0x0C, 0x56, 0x8E, 0x79,
		0x51, 0x34, 0x1F, 0xD1, 0x65, 0x44, 0xDB, 0x79, 0x0A, 0xF9,
		0x6F, 0x9B, 0x75, 0x7B, 0x77, 0xF4, 0x83, 0xE8, 0xAB, 0xC1,
		0x69, 0x7A, 0x3D, 0xB8, 0x99, 0x22, 0xA9, 0x6A, 0xB2, 0x09,
		0x60, 0xB5, 0x94, 0x14, 0xF1, 0xCC, 0x05, 0x18, 0x86, 0x83,
		0x97, 0x40, 0x1D, 0x67, 0xB7, 0xE3, 0xAA, 0xB5, 0x2F, 0x54,
		0x2A, 0x6C, 0x0B, 0x23, 0x8E, 0x42, 0x01, 0x24, 0xEA, 0xEB,
		0xC7, 0x72, 0xB9, 0x28, 0x61, 0x27, 0x7F, 0x20, 0x30, 0xA7,
		0x48, 0xAD, 0x55, 0xB7, 0xE4, 0xF8, 0xBD, 0x3F, 0xDC, 0x84,
		0x0D, 0x71, 0xDC, 0x6C, 0x31, 0xC1, 0xDC, 0x13, 0x35, 0x22,
		0x74, 0x4D, 0xC5, 0xDB, 0x3A, 0x7B, 0x39, 0xDB, 0xA7, 0x94,
		0x67, 0xB8, 0x27, 0x0C, 0xC0, 0x5B, 0x2E, 0xE8, 0x37, 0xC3,
		0x65, 0x75, 0x72, 0xAF, 0xC6, 0x02, 0xEE, 0xF2, 0x03, 0x08,
		0xBB, 0xE6, 0xD8, 0x58, 0xB3, 0x08, 0x08, 0x83, 0x9F, 0x1B,
		0xBC, 0x22, 0xB5, 0x4F, 0x33, 0xD4, 0x74, 0xCA, 0x5F, 0x35,
		0xCA, 0x98, 0xFA, 0x0D, 0x7A, 0x27, 0x56, 0xCE, 0x5E, 0xDA,
		0x97, 0xF5, 0xFE, 0x5D, 0xE3, 0x9B, 0x8C, 0x9B, 0xD6, 0x1F,
		0x90, 0xF9, 0x80, 0x31, 0xD1, 0xDF, 0x9D, 0xC9, 0x7E, 0x25,
		0xDD, 0x85, 0xE8, 0x01, 0x65, 0x9E, 0x82, 0x65, 0xF3, 0x8A,
		0x28, 0x99, 0x7C, 0x20, 0x4D, 0x8D, 0x02, 0x53, 0x18, 0xB2,
		0xF9, 0x51, 0x50, 0x47, 0x36, 0x30, 0x37, 0x7D, 0x72, 0x6D,
		0xB1, 0x7F, 0x68, 0xDB, 0xFA, 0xE5, 0x83, 0x01, 0xB0, 0xFB,
		0x4E, 0x86, 0xEC, 0x27, 0x67, 0xC7, 0x29, 0xB7, 0xAF, 0x44,
		0x4D, 0x4F, 0x48, 0xE6, 0x00, 0xD8, 0x6B, 0x6B, 0x02, 0x16,
		0x9A, 0xD1, 0xEB, 0xE0, 0xB1, 0x90, 0x61, 0xA6, 0xD8, 0xFD,
		0xD2, 0x50, 0xA2, 0x60, 0x99, 0xFA, 0x29, 0xBC, 0xB3, 0x39,
		0x46, 0xD1, 0x4E, 0x5E, 0x4E, 0x63, 0x0F, 0xC9, 0x81, 0xD2,
		0xE2, 0x55, 0x51, 0x52, 0x5C, 0xE7, 0x80, 0x16, 0xE3, 0x1A,
		0x01, 0x70, 0x89, 0xFE, 0xA2, 0x9C, 0xB2, 0x58, 0xB0, 0x82,
		0xAB, 0x4E, 0xC1, 0x54, 0x68, 0xF7, 0xB9, 0x6C, 0xF7, 0x95,
		0xAB, 0x62, 0x16, 0x61, 0xFB, 0x74, 0xEC, 0x54, 0xB8, 0x4A,
		0x7D, 0xD5, 0x0A, 0x74, 0x8E, 0xAF, 0xC9, 0x07, 0xCA, 0x45,
		0xF9, 0xBA, 0x2B, 0x98, 0x34, 0xDE, 0xCA, 0x3E, 0xA7, 0x15,
		0x32, 0xFE, 0xEB, 0xEE, 0x8F, 0x8F, 0x1F, 0x52, 0x06, 0x6C,
		0xE1, 0xAB, 0x12, 0xDC, 0x1B, 0xDF, 0x2C, 0x8B, 0x93, 0x10,
		0x0C, 0xEB, 0x6A, 0xE2, 0x84, 0x3A, 0xF1, 0x35, 0xB4, 0x01,
		0xE1, 0xCE, 0x00, 0xCB, 0x26, 0x98, 0xB4, 0x37, 0x00, 0x21,
		0xBF, 0xD6, 0xF3, 0x35, 0x28, 0x7D, 0xF5, 0x1D, 0xCE, 0xE9,
		0xDF, 0xAC, 0x0D, 0xA8, 0xE1, 0x58, 0x97, 0xA0, 0x26, 0x90,
		0xB1, 0xDE, 0x28, 0xF9, 0xA2, 0x09, 0x29, 0x08, 0xFA, 0xE9,
		0xC8, 0x0C, 0x37, 0xA0, 0xB8, 0xEE, 0x44, 0x02, 0x83, 0xAA,
		0xCB, 0xA3, 0x14, 0xDD, 0xFA, 0x83, 0xD7, 0x2A, 0xD4, 0x21,
		0x82, 0x7D, 0x44, 0x38, 0xC2, 0x77, 0xC4, 0x37, 0x37, 0xAB,
		0x27, 0xC0, 0xC0, 0xE2, 0xC6, 0xDA, 0x2F, 0xE4, 0xD2, 0xB6,
		0x79, 0x19, 0x1E, 0x6A, 0xC8, 0x73, 0x56, 0xED, 0x30, 0xD9,
		0xB5, 0x50, 0xC9, 0xBB, 0xCB, 0x8F, 0xE7, 0x67, 0x85, 0x8A,
		0xA4, 0xBE, 0x1C, 0xA2, 0x14, 0x93, 0x16, 0x6A, 0xCA, 0x2B,
		0x5D, 0x8D, 0x7A, 0x9E, 0xDC, 0x4E, 0x0D, 0x2B, 0x9F, 0x47,
		0x01, 0x68, 0xF5, 0x53, 0x4E, 0x3C, 0x54, 0xD7, 0x4C, 0x91,
		0x05, 0x8D, 0x72, 0x4D, 0x54, 0x56, 0x68, 0x2C, 0x95, 0x68,
		0xBA, 0x57, 0x56, 0x25, 0x2D, 0x03, 0x8D, 0x1E, 0x92, 0x18,
		0x21, 0xED, 0xF4, 0xF7, 0xD7, 0xF8, 0x14, 0x84, 0xCE, 0xAA,
		0xFC, 0xD1, 0xC2, 0x21, 0x50, 0x8B, 0x91, 0x08, 0xBC, 0x6F,
		0x66, 0x2E, 0x18, 0x94, 0x3F, 0xD9, 0x61, 0x09, 0xD8, 0x1C,
		0xE6, 0x51, 0xA8, 0x0E, 0xB2, 0x56, 0x18, 0x03, 0x55, 0x11,
		0xC5, 0xCE, 0x5A, 0xAE, 0xFD, 0xF9, 0x31, 0xB7, 0x2F, 0x06,
		0x1F, 0x74, 0x6F, 0x3A, 0x27, 0xD4, 0x43, 0x0D, 0xFB, 0x2D,
		0x0A, 0xDA, 0x53, 0x29, 0x58, 0xF9, 0xCA, 0xF3, 0xD8, 0x80,
		0x91, 0xB1, 0x7C, 0x2C, 0xA9, 0x8E, 0x03, 0x5C, 0xA2, 0x3A,
		0x06, 0x6B, 0xE4, 0x2B, 0x45, 0x3F, 0x4B, 0x73, 0x02, 0x8B,
		0xD7, 0xA9, 0xB4, 0xD2, 0x3D, 0x6F, 0x5D, 0x5D, 0x0A, 0x9F,
		0x34, 0x38, 0x23, 0x13, 0x31, 0x65, 0xB3, 0xBA, 0xAD, 0x29,
		0x5F, 0x1B, 0xB6, 0xAF, 0xFF, 0x48, 0x49, 0xB0, 0x87, 0x63,
		0x99, 0xFE, 0x2B, 0xFC, 0x94, 0xC2, 0x16, 0x30, 0xE7, 0x57,
		0xB8, 0x62, 0x77, 0x59, 0xEA, 0x05, 0xB3, 0x2D, 0xDC, 0xCC,
		0xA2, 0xE4, 0x72, 0x2E, 0x88, 0x6C, 0x6C, 0xCB, 0x02, 0x25,
		0xC7, 0x15, 0xC0, 0x81, 0xF8, 0x1E, 0x3E, 0xD7, 0x41, 0x0D,
		0x31, 0x56, 0x9F, 0x1F, 0x58, 0xE7, 0x87, 0x23, 0x7A, 0x01,
		0x6D, 0x10, 0xC5, 0x47, 0xE5, 0x91, 0x1D, 0x69, 0xD3, 0xD6,
		0xD0, 0x48, 0x86, 0x86, 0x77, 0x10, 0x4A, 0xFE, 0x7C, 0x65,
		0x07, 0x21, 0x3E, 0xA1, 0x14, 0x87, 0x76, 0x8F, 0x98, 0x5B,
		0x42, 0x3D, 0x13, 0x89, 0xAB, 0x3F, 0x25, 0x0B, 0x80, 0x75,
		0x83, 0x45, 0xF8, 0x7F, 0xAF, 0x46, 0xDE, 0x15, 0xAF, 0xF0,
		0x43, 0xC9, 0x60, 0x8E, 0x8B, 0x19, 0x4F, 0xEB, 0x53, 0x65,
		0x06, 0xA0, 0xAF, 0xCC, 0x56, 0x80, 0x8E, 0x65, 0x9B, 0x27,
		0x00, 0x8E, 0xC8, 0xDC, 0xF3, 0x92, 0x3A, 0x8E, 0x8C, 0x67,
		0xED, 0x7D, 0x30, 0x47, 0x28, 0x98, 0xBD, 0xA2, 0xEE, 0x47,
		0x66, 0x26, 0x99, 0x48, 0x51, 0x77, 0xD8, 0x33, 0x0E, 0x3C,
		0xAD, 0xCC, 0x92, 0xDF, 0x37, 0x6B, 0x62, 0x1D, 0x30, 0x79,
		0x61, 0xB5, 0x11, 0xEC, 0xAE, 0x58, 0x17, 0x87, 0xFF, 0x64,
		0xBC, 0x3C, 0x31, 0x2E, 0x06, 0x38, 0xFA, 0x2B, 0xFA, 0x24,
		0x0C, 0x9C, 0x44, 0x9D, 0x54, 0x9B, 0xD5, 0x47, 0xCF, 0x27,
		0xFD, 0x79, 0xDE, 0x20, 0x76, 0x6C, 0x5E, 0x93, 0x73, 0x4C,
		0x9B, 0xE4, 0xCF, 0x52, 0xD9, 0x01, 0x67, 0x68, 0x27, 0x07,
		0x93, 0x5F, 0x97, 0x0F, 0x9A, 0xD7, 0x22, 0x6F, 0x80, 0x22,
		0xA5, 0xB3, 0x6C, 0xCF, 0xB6, 0x96, 0x3E, 0x99, 0x96, 0x2D,
		0x40, 0x8D, 0x8A, 0x91, 0x34, 0xA7, 0x73, 0x66, 0x0D, 0xD4,
		0xBE, 0x7B, 0xFB, 0x3C, 0xF7, 0xB3, 0x0F, 0x1D, 0xB9, 0x0C,
		0x9A, 0x9E, 0x21, 0xFE, 0xDD, 0xD0, 0x8C, 0x43, 0xB8, 0xE7,
		0xD6, 0x26, 0xE3, 0x22, 0x43, 0x8A, 0x13, 0xA6, 0x2D, 0x54,
		0xE4, 0xBD, 0x0E, 0xA2, 0x21, 0x83, 0x83, 0xF8, 0x12, 0x6F,
		0x7E, 0x15, 0xBF, 0x79, 0x50, 0x83, 0x18, 0x9E, 0x38, 0x61,
		0xFB, 0x57, 0x23, 0x1C, 0x76, 0xDF, 0xFE, 0x7E, 0x34, 0x09,
		0x23, 0x8D, 0x32, 0x79, 0x86, 0xEB, 0x22, 0x3A, 0x9C, 0x16,
		0x31, 0x87, 0x9B, 0xDC, 0x04, 0x4D, 0x05, 0xE1, 0xA1, 0x78,
		0x94, 0x36, 0x34, 0x5F, 0xCD, 0x6A, 0xA8, 0x31, 0xDC, 0xBD,
		0x99, 0xB5, 0x80, 0xAD, 0x4C, 0x82, 0x62, 0xC4, 0x5B, 0xF2,
		0xFC, 0xAD, 0xD4, 0x6C, 0xC1, 0xC9, 0xC2, 0xCC, 0x4E, 0x53,
		0x9D, 0x97, 0x34, 0xB5, 0xB7, 0x6F, 0x10, 0x95, 0x3E, 0xE1,
		0xDD, 0x1E, 0xE7, 0x71, 0x82, 0xF3, 0x50, 0x5D, 0xA9, 0x52,
		0xA0, 0xA2, 0x78, 0xD5, 0x2D, 0x4C, 0xA6, 0x44, 0xF4, 0x0B,
		0x4C, 0x85, 0x50, 0xC4, 0x2F, 0x1B, 0xA8, 0xD0, 0x0B, 0xF5,
		0xD5, 0x03, 0x67, 0xD3, 0xC8, 0x29, 0xF9, 0x7A, 0xB0, 0xB6,
		0xD4, 0x0E, 0x52, 0x9E, 0x17, 0xCC, 0xB3, 0xDC, 0x86, 0xC3,
		0x65, 0xE5, 0x98, 0xA5, 0xB5, 0xA5, 0x48, 0xE1, 0xEC, 0x33,
		0xA9, 0x25, 0x6C, 0xFB, 0xF0, 0xB7, 0x6E, 0x2B, 0xB2, 0xB3,
		0x0F, 0x0B, 0x29, 0x8D, 0xB0, 0x01, 0x2B, 0x98, 0xA5, 0x1B,
		0x4F, 0x52, 0xF6, 0x78, 0xF1, 0xC5, 0xBB, 0xD5, 0xFF, 0x7A,
		0xDD, 0x09, 0x85, 0xDA, 0x43, 0x85, 0x10, 0x7D, 0x36, 0xBC,
		0xE7, 0x02, 0x2F, 0x7C, 0x3B, 0x56, 0xAB, 0x53, 0x7A, 0xB5,
		0x20, 0x26, 0xEB, 0x72, 0x0E, 0xED, 0xAB, 0x7D, 0x79, 0x27,
		0x74, 0x56, 0x2E, 0xE1, 0x30, 0x4B, 0x61, 0xD5, 0xE1, 0x2B,
		0xB7, 0x4B, 0xBE, 0x91, 0x8D, 0x5D, 0x3F, 0xA2, 0xF6, 0x6A,
		0xED, 0x2F, 0x2B, 0x2E, 0x5C, 0x0E, 0x55, 0xE7, 0xB0, 0x24,
		0xB1, 0xC7, 0xC9, 0xF5, 0x4D, 0x74, 0x60, 0x6E, 0x1E, 0x32,
		0xC1, 0x98, 0x83, 0xA6, 0xE3, 0x7D, 0x34, 0x41, 0x59, 0xBE,
		0x12, 0x5D, 0x45, 0x42, 0x1C, 0xC2, 0xCA, 0x8F, 0x50, 0x4F,
		0x7D, 0x23, 0x6D, 0x64, 0xD2, 0x9A, 0xC9, 0xF2, 0xD2, 0xE7,
		0x1C, 0x52, 0xFF, 0x3B, 0xAD, 0x0A, 0xF2, 0xF6, 0xBD, 0xAD,
		0x5D, 0xD3, 0x0B, 0x1D, 0x89, 0x83, 0x62, 0x92, 0xE8, 0x20,
		0x5C, 0x7C, 0xC1, 0x1E, 0x6D, 0x6E, 0xA2, 0x76, 0x9D, 0xA0,
		0x2B, 0x91, 0x14, 0x65, 0xDC, 0x9C, 0xE1, 0xA7, 0xD5, 0x04,
		0xE2, 0x26, 0x0B, 0x12, 0x6D, 0xEB, 0x92, 0x10, 0xDA, 0x6F,
		0x3B, 0xF4, 0x84, 0xE6, 0x4C, 0xF8, 0xEE, 0xE0, 0xE3, 0x83,
		0x0E, 0xD5, 0x1D, 0xC9, 0xC0, 0xCB, 0xBE, 0x9D, 0x2E, 0x64,
		0xF3, 0x9E, 0x87, 0x6C, 0x07, 0xA8, 0x75, 0xAF, 0xFE, 0x50,
		0xF2, 0xB1, 0x4B, 0xBB, 0x6A, 0xEB, 0x5C, 0xD3, 0x02, 0x0A,
		0x00, 0xD4, 0x45, 0x5D, 0x71, 0xE9, 0xCF, 0x81, 0xC1, 0x6C,
		0x3B, 0x84, 0xC2, 0x26, 0x2A, 0xA5, 0xDF, 0xE1, 0xBA, 0x4F,
		0x03, 0xEE, 0xB5, 0x29, 0xE0, 0x80, 0xBF, 0xED, 0x4A, 0x81,
		0x37, 0xFA, 0x21, 0xF0, 0x8A, 0x1E, 0x43, 0x67, 0x4D, 0xC1,
		0xF7, 0x78, 0x9E, 0xB3, 0x79, 0xD2, 0x31, 0x0A, 0xA8, 0xB4,
		0x6D, 0x94, 0xCE, 0x24, 0xFA, 0x8B, 0xF3, 0x89, 0xC4, 0xBF,
		0xAB, 0xC7, 0xFA, 0x60, 0x20, 0xB5, 0x56, 0xF3, 0x31, 0xD3,
		0x5A, 0x4D, 0x75, 0x88, 0x74, 0xFE, 0x7A, 0xA8, 0xB3, 0xA4,
		0x3F, 0x6A, 0xB4, 0x44, 0xE2, 0x57, 0xB0, 0x78, 0xE4, 0xBC,
		0xF1, 0x9E, 0xB4, 0xC3, 0x4A, 0x6C, 0x76, 0x18, 0x5B, 0x2C,
		0x77, 0x70, 0x75, 0x5C, 0xF9, 0x4F, 0xEC, 0x7E, 0x9D, 0xDE,
		0x47, 0xE4, 0x28, 0x78, 0x82, 0xF5, 0x33, 0x70, 0x79, 0x7C,
		0xE2, 0xCD, 0x64, 0xA6, 0x30, 0x53, 0x9C, 0x62, 0x07, 0x11,
		0xDD, 0xC1, 0x8C, 0xDB, 0x1C, 0xC1, 0x60, 0xE8, 0xB5, 0x5B,
		0xD3, 0x93, 0x46, 0x48, 0x62, 0x21, 0xF1, 0x50, 0x25, 0x52,
		0x7D, 0x53, 0xDA, 0xE8, 0x08, 0x93, 0x32, 0xD5, 0x45, 0xAE,
		0xAA, 0xC0, 0x12, 0xE4, 0x27, 0xAA, 0x06, 0x1A, 0xC8, 0xA8,
		0x47, 0xB1, 0xC4, 0x54, 0x65, 0xEA, 0x61, 0x97, 0x7C, 0x75,
		0x01, 0x47, 0x22, 0x1A, 0xF6, 0x04, 0x6C, 0x93, 0xF5, 0xE5,
		0x6A, 0xA7, 0x99, 0x7A, 0xAF, 0x39, 0x7E, 0xA8, 0xE0, 0xDE,
		0x97, 0xF3, 0x98, 0x48, 0x6C, 0x74, 0x58, 0x47, 0x7C, 0xFD,
		0x79, 0x9D, 0x7D, 0xDE, 0x28, 0x5A, 0xF9, 0x90, 0x8D, 0x07,
		0x89, 0x9D, 0x0D, 0x60, 0xF1, 0xB3, 0x25, 0x1D, 0x9B, 0xE1,
		0xE8, 0xDB, 0x34, 0xF9, 0x53, 0x2A, 0xA2, 0x1B, 0xAA, 0x84,
		0x0C, 0xF5, 0x31, 0x03, 0x46, 0x5B, 0x5B, 0xBF, 0x97, 0x4A,
		0x1A, 0x66, 0x83, 0x14, 0xD8, 0xEA, 0x1D, 0x3A, 0x64, 0x16,
		0xF8, 0x65, 0x3C, 0xAD, 0x00, 0xC0, 0xF8, 0xD8, 0x9D, 0xC0,
		0xA5, 0x13, 0xF7, 0x11, 0x1C, 0x61, 0x88, 0x2B, 0x9B, 0x55,
		0xCA, 0x1E, 0x82, 0xD7, 0x8F, 0xE9, 0x9B, 0x2D, 0xD1, 0xF1,
		0xEA, 0x47, 0x2F, 0xA1, 0x95, 0x29, 0x56, 0xB9, 0x2E, 0xE1,
		0xB0, 0x25, 0xDD, 0x19, 0x22, 0xA9, 0x50, 0x1F, 0x42, 0xAA,
		0xDC, 0x08, 0x81, 0x94, 0xB8, 0x8D, 0x8E, 0x93, 0x66, 0x2F,
		0xA1, 0xA9, 0xFE, 0x7B, 0xDA, 0xC7, 0xBD, 0x47, 0x30, 0x5F,
		0x0B, 0x65, 0x34, 0x1C, 0xE9, 0x82, 0xAC, 0x10, 0x6B, 0xF5,
		0x19, 0x3C, 0xD7, 0xD0, 0x6D, 0xC4, 0x0F, 0xED, 0xE9, 0xC5,
		0x88, 0x62, 0x1E, 0x62, 0xFE, 0x9A, 0xA7, 0x89, 0x2A, 0x8D,
		0x9F, 0x6A, 0x5B, 0x19, 0x7A, 0x4A, 0xCC, 0x90, 0x81, 0xA1,
		0x53, 0x16, 0xF5, 0x85, 0x92, 0x9F, 0x44, 0xA6, 0x05, 0x04,
		0x55, 0x5C, 0xB2, 0x8C, 0x78, 0x85, 0xAF, 0x01, 0xAD, 0x6A,
		0x24, 0xFF, 0x26, 0x61, 0x7B, 0xAC, 0xDC, 0x1B, 0x76, 0x73,
		0xC9, 0x65, 0x4A, 0x9B, 0x52, 0x00, 0xE8, 0xCD, 0x64, 0xAB,
		0x05, 0xCC, 0x89, 0x51, 0xEF, 0x1A, 0x16, 0x38, 0x35, 0xF0,
		0xBD, 0x96, 0xF6, 0xCB, 0x3D, 0xA5, 0x16, 0xDE, 0x53, 0xF6,
		0x4B, 0x49, 0x72, 0x8F, 0x55, 0x44, 0x77, 0xC8, 0xC5, 0xAD,
		0x76, 0x6C, 0x97, 0xA3, 0x36, 0x61, 0xA5, 0x3D, 0x3F, 0xCE,
		0xA3, 0x28, 0x65, 0xE8, 0x02, 0x1E, 0xAA, 0xC1, 0x64, 0x17,
		0xAE, 0x5E, 0x39, 0xFB, 0x4F, 0x3B, 0x2D, 0x2F, 0x8E, 0x2E,
		0xE9, 0xB9, 0x61, 0x9A, 0xD8, 0xBF, 0xB4, 0xA0, 0x8D, 0xA1,
		0xBD, 0xC5, 0x3C, 0x8C, 0xF2, 0x9B, 0x85, 0x9F, 0xBF, 0xE4,
		0x73, 0x60, 0x82, 0x26, 0x48, 0x4E, 0x2B, 0x4B, 0x43, 0xC3,
		0x03, 0x62, 0x02, 0x7B, 0x5B, 0x95, 0xC3, 0xEA, 0x87, 0xC1,
		0x02, 0x1D, 0x22, 0xE6, 0x3D, 0xF6, 0xAC, 0x65, 0x47, 0x93,
		0xFF, 0x30, 0x92, 0x91, 0xAA, 0x44, 0x4E, 0x24, 0x85, 0x60,
		0x81, 0xB7, 0xBE, 0xB0, 0x88, 0xFA, 0xBB, 0xE9, 0xF2, 0x39,
		0x2E, 0x85, 0xD0, 0x51, 0x75, 0x1D, 0xA5, 0xD1, 0x4B, 0xCD,
		0xAF, 0xAD, 0x03, 0x02, 0x73, 0x8B, 0x5D, 0xBE, 0x65, 0x4D,
		0xCE, 0xF2, 0x70, 0x58, 0x13, 0x77, 0x3B, 0xC8, 0xB7, 0x88,
		0xC3, 0x5B, 0x13, 0x4A, 0x95, 0x73, 0xBC, 0xEF, 0x4F, 0x0F,
		0x92, 0x1D, 0x17, 0xF5, 0xB8, 0x95, 0x2F, 0xF9, 0x78, 0x3F,
		0x83, 0x7F, 0x45, 0x05, 0x84, 0xC4, 0xF8, 0x8D, 0x41, 0xC4,
		0x2D, 0x37, 0xEE, 0x62, 0xC2, 0x68, 0x42, 0x33, 0xFB, 0xEB,
		0xCB, 0xE0, 0x3D, 0xAA, 0x53, 0x52, 0xB6, 0x68, 0xE5, 0x55,
		0x3C, 0x69, 0xFF, 0xA6, 0xEB, 0x86, 0xC3, 0x55, 0x7D, 0x91,
		0x5F, 0x88, 0xD0, 0x8D, 0xB9, 0xE6, 0x70, 0xCD, 0x5E, 0x9E,
		0xEA, 0x6C, 0x62, 0x77, 0xE6, 0x6F, 0x05, 0x09, 0x5B, 0x01,
		0x55, 0x58, 0x15, 0x12, 0x52, 0xE7, 0x1D, 0x5F, 0x30, 0x21,
		0x06, 0x2C, 0x9F, 0x5E, 0x63, 0xB1, 0x50, 0x43, 0xB3, 0xAF,
		0x54, 0x0B, 0xC1, 0xFA, 0x67, 0x8E, 0x50, 0x1D, 0x05, 0x1B,
		0xA7, 0xCA, 0x82, 0x1D, 0xEF, 0xC3, 0x39, 0x20, 0xD1, 0x6D,
		0x95, 0xC4, 0xD9, 0x1F, 0x99, 0x51, 0xFB, 0xB3, 0x95, 0xEB,
		0x8B, 0xAD, 0xD3, 0xDE, 0x1E, 0x45, 0x1B, 0x4E, 0x33, 0x23,
		0x55, 0xB3, 0x23, 0x09, 0x40, 0x9F, 0x7E, 0x9A, 0x11, 0xF8,
		0x9D, 0xDB, 0xF1, 0x87, 0x8E, 0x73, 0x51, 0xDC, 0x49, 0x9A,
		0xB2, 0xA5, 0xD1, 0x29, 0x25, 0xE8, 0x6A, 0x00, 0xB7, 0x01,
		0xBC, 0xD7, 0xFC, 0xC2, 0xF5, 0x23, 0x38, 0x2A, 0xFC, 0xE3,
		0xEB, 0xAD, 0x58, 0x66, 0x14, 0xB0, 0x95, 0x3E, 0xDE, 0xD6,
		0x47, 0xD7, 0x26, 0x2F, 0xE6, 0x4A, 0xD7, 0x5C, 0x34, 0x3C,
		0x83, 0x69, 0x47, 0x6A, 0x5F, 0xB2, 0x8C, 0x86, 0xC6, 0x6C,
		0x4A, 0x98, 0xB4, 0x6F, 0xCA, 0x98, 0xEF, 0xA8, 0x85, 0x59,
		0x24, 0xED, 0x6A, 0xFF, 0x6C, 0x1B, 0x89, 0x88, 0x49, 0xBB,
		0x90, 0x82, 0xC3, 0xB0, 0x53, 0xB5, 0x45, 0x48, 0x0F, 0x9A,
		0x0A, 0xBA, 0x2C, 0x49, 0xD6, 0xB3, 0xF3, 0xAF, 0xB7, 0x2F,
		0x05, 0x68, 0xDE, 0xF7, 0xEC, 0xD6, 0x1F, 0x8D, 0xB3, 0x6E,
		0x4C, 0x96, 0x41, 0xF2, 0xB6, 0x20, 0x4C, 0xA7, 0x1F, 0x81,
		0xC7, 0xF5, 0x10, 0x64, 0x51, 0x11, 0xC4, 0x8A, 0x8D, 0x1F,
		0x8A, 0xD6, 0x1E, 0x8A, 0xAD, 0x46, 0xA2, 0x0F, 0x57, 0x72,
		0x37, 0x0D, 0x3E, 0x20, 0xFF, 0x5B, 0xCC, 0xF0, 0xD3, 0x63,
		0xD3, 0x7D, 0x85, 0xD6, 0xB9, 0xB3, 0xAF, 0x81, 0x06, 0x44,
		0x49, 0x57, 0x9B, 0xF1, 0x8E, 0x4B, 0xF2, 0x0D, 0xE2, 0x8A,
		0x83, 0x20, 0x24, 0xA5, 0x58, 0x81, 0xA8, 0xC0, 0xE6, 0x57,
		0x56, 0x0D, 0xE2, 0x28, 0x11, 0x18, 0x22, 0x11, 0x97, 0x8F,
		0xD0, 0x26, 0x2F, 0x7B, 0x18, 0x90, 0x8F, 0xE6, 0xE0, 0xE5,
		0xB3, 0x22, 0xFD, 0xB6, 0x8C, 0xE5, 0x95, 0xA8, 0xE8, 0x51,
		0x42, 0x80, 0xA7, 0x36, 0xBF, 0x61, 0xE8, 0x2A, 0xD8, 0x6E,
		0x31, 0xA5, 0x2F, 0xDF, 0x40, 0x0F, 0x05, 0xB9, 0x4D, 0x41,
		0xD2, 0xEB, 0x40, 0x8D, 0x8E, 0x15, 0x6E, 0xCA, 0x53, 0xE9,
		0x87, 0x53, 0xB7, 0xC6, 0x6F, 0x11, 0x35, 0x35, 0x35, 0x66,
		0xD1, 0x62, 0xD2, 0x20, 0x73, 0x42, 0x38, 0x6C, 0x81, 0xDF,
		0xCF, 0x1D, 0xFB, 0x9F, 0x18, 0xB2, 0x91, 0xDD, 0x6C, 0xF5,
		0xC7, 0xBA, 0x8C, 0xAA, 0x8F, 0xA4, 0xE7, 0x67, 0x4B, 0xC1,
		0xEE, 0xD5, 0xD7, 0x49, 0xE0, 0xEA, 0x9D, 0xD1, 0x6A, 0xFA,
		0x0B, 0x36, 0x85, 0xE0, 0x94, 0x17, 0xF7, 0x9F, 0x2E, 0xBB,
		0xD2, 0x24, 0x0A, 0x0C, 0xAF, 0xB2, 0x4C, 0x70, 0xF2, 0xDD,
		0x3C, 0x43, 0xA2, 0x06, 0xBF, 0x63, 0x34, 0xF3, 0x3E, 0x46,
		0x6D, 0x63, 0x01, 0x77, 0x01, 0x77, 0x6C, 0xD9, 0x1A, 0x81,
		0x82, 0x24, 0xAA, 0xEE, 0xC7, 0xAE, 0x38, 0x0B, 0x3D, 0x1F,
		0x31, 0x0B, 0x35, 0x6C, 0x72, 0x0C, 0xAD, 0xF2, 0xBA, 0xAE,
		0x98, 0x79, 0xC3, 0xD8, 0xA9, 0x9A, 0xD3, 0x43, 0x17, 0xEC,
		0xD1, 0x7E, 0x7B, 0x27, 0x4A, 0xAE, 0x4F, 0x4F, 0xF9, 0x7A,
		0x12, 0xCD, 0x6D, 0x02, 0x1F, 0x17, 0xFD, 0xD4, 0xBA, 0x19,
		0x63, 0x3E, 0x58, 0x96, 0xA9, 0xCC, 0xD3, 0xFA, 0x3E, 0xC0,
		0x29, 0x4B, 0xD8, 0xAA, 0x21, 0x50, 0x23, 0x33, 0x41, 0x2D,
		0x25, 0x11, 0x30, 0x12, 0x45, 0x42, 0x46, 0x97, 0xD3, 0xE7,
		0x5B, 0xFC, 0x8E, 0xF0, 0x97, 0xF3, 0xCE, 0x9F, 0x5A, 0x8B,
		0xDB, 0x75, 0x60, 0x2A, 0x8C, 0x57, 0x0F, 0xB4, 0x99, 0xF2,
		0xFF, 0x58, 0xD1, 0x49, 0x09, 0xFB, 0x31, 0x26, 0x35, 0x04,
		0x58, 0xAF, 0x70, 0x9E, 0xEB, 0xB5, 0xA2, 0xE8, 0x5E, 0xBB,
		0x42, 0x22, 0x8C, 0x12, 0x76, 0x2F, 0x0C, 0x62, 0x38, 0x7A,
		0x47, 0x9E, 0x47, 0x43, 0x86, 0x46, 0xB2, 0x31, 0xD8, 0xF5,
		0xAE, 0x11, 0xC1, 0xE7, 0x9E, 0x52, 0xA5, 0x6D, 0x8C, 0xC5,
		0xD2, 0x57, 0x52, 0xE0, 0x36, 0x86, 0x9F, 0x83, 0x7B, 0xE2,
		0xF1, 0xE0, 0x37, 0xF7, 0xEB, 0xF6, 0x02, 0x15, 0x0C, 0x77,
		0x7B, 0x75, 0x3F, 0xB1, 0x9C, 0x79, 0xA9, 0x2B, 0x96, 0xF2,
		0x11, 0xFD, 0x43, 0x70, 0x71, 0x33, 0x95, 0x49, 0x59, 0x93,
		0xBE, 0x2D, 0xA7, 0x18, 0xA7, 0x4B, 0x8A, 0x6A, 0x12, 0x35,
		0x45, 0xB6, 0x6A, 0xCF, 0x35, 0x44, 0x8A, 0xE8, 0x2A, 0x90,
		0x19, 0x5D, 0xB1, 0x22, 0xFA, 0x98, 0x2B, 0xC2, 0x62, 0xFE,
		0x2C, 0x47, 0xCC, 0x77, 0x47, 0xB0, 0xB9, 0xCA, 0x9A, 0xC6,
		0x09, 0xC2, 0x88, 0x48, 0x15, 0x0C, 0x96, 0x6E, 0x32, 0xD4,
		0x31, 0xC1, 0x90, 0x26, 0x0A, 0x05, 0xDE, 0x60, 0xE3, 0x41,
		0x12, 0x94, 0x2D, 0x9A, 0x7D, 0x5A, 0xC2, 0x01, 0x3E, 0x67,
		0x18, 0xBC, 0xAA, 0xBB, 0xB6, 0x23, 0x62, 0x8F, 0x83, 0xDD,
		0xDA, 0x3D, 0x56, 0x8F, 0x98, 0xA9, 0xA2, 0x8F, 0x78, 0x85,
		0x18, 0xD2, 0x87, 0x45, 0x81, 0xBF, 0xE6, 0xEB, 0xD7, 0x99,
		0xEA, 0x70, 0xAC, 0x88, 0x63, 0xCA, 0x6B, 0x33, 0xB5, 0x83,
		0x8B, 0xF6, 0x32, 0x6A, 0x86, 0x93, 0x53, 0x13, 0x85, 0x54,
		0xF4, 0xAC, 0x72, 0xEA, 0xF0, 0x7C, 0x86, 0x16, 0xEE, 0x6B,
		0xA7, 0xEE, 0x86, 0xB6, 0x48, 0x61, 0xD1, 0xA9, 0xA3, 0x1E,
		0xFE, 0x96, 0x3B, 0xEA, 0x70, 0x09, 0x4E, 0x0D, 0x5B, 0x70,
		0x2B, 0x8D, 0xD9, 0x5D, 0x4F, 0x32, 0xD2, 0x7D, 0x32, 0x07,
		0xE8, 0x69, 0xCA, 0xF1, 0x6E, 0xFC, 0xFF, 0x81, 0x77, 0xE2,
		0x04, 0x7F, 0xA1, 0x5F, 0xB2, 0x43, 0xDF, 0x44, 0x68, 0x22,
		0x83, 0x6E, 0x80, 0xB8, 0x5C, 0x15, 0xEA, 0x86, 0x37, 0xC0,
		0x43, 0xD6, 0x29, 0x44, 0x7E, 0xB0, 0xDA, 0x1B, 0x2F, 0xD9,
		0x2B, 0xD2, 0x7C, 0xDB, 0x29, 0x88, 0xA2, 0xC4, 0xE3, 0x87,
		0xF2, 0x16, 0x54, 0x5B, 0x4C, 0x66, 0x1A, 0xA4, 0x0C, 0x9E,
		0xD5, 0x2B, 0x53, 0xF3, 0xBD, 0x2E, 0x07, 0x44, 0x24, 0x96,
		0xCB, 0xD1, 0x5B, 0xBC, 0x7E, 0xD0, 0xD5, 0x9A, 0x5D, 0x69,
		0xB3, 0xC0, 0xB1, 0x8A, 0xB4, 0x01, 0x04, 0x30, 0x89, 0xA2,
		0x47, 0xA5, 0xED, 0x8B, 0x95, 0xAA, 0xA2, 0x51, 0x76, 0x87,
		0xF1, 0x56, 0x84, 0x95, 0x12, 0xA6, 0x18, 0x58, 0x8B, 0xD5,
		0x08, 0x62, 0x4A, 0xF9, 0x29, 0x6A, 0xD5, 0xF5, 0x45, 0x34,
		0xB5, 0xF9, 0x9A, 0xA8, 0x2C, 0xDD, 0x78, 0x73, 0x15, 0x0A,
		0xA5, 0xDD, 0xFD, 0xA8, 0x63, 0xDD, 0x2B, 0x87, 0x80, 0xF9,
		0x35, 0xB9, 0x29, 0x23, 0x4A, 0x73, 0x83, 0x65, 0xFF, 0xAA,
		0xB4, 0x48, 0x2F, 0x73, 0x2F, 0x54, 0x33, 0xAF, 0x1D, 0x92,
		0x9F, 0x4E, 0x12, 0xB5, 0xF9, 0x53, 0xD5, 0xCE, 0x5F, 0x29,
		0x15, 0x8A, 0x3D, 0x74, 0xA7, 0xE5, 0x4D, 0xCD, 0x58, 0x49,
		0x70, 0x18, 0xC2, 0x27, 0xBF, 0x9F, 0x31, 0xD3, 0x76, 0xC2,
		0x06, 0x40, 0x12, 0xBE, 0x6D, 0xDD, 0x9E, 0x77, 0xF1, 0xA3,
		0xBD, 0xF3, 0x7E, 0xC1, 0xD1, 0x74, 0xE7, 0x60, 0x46, 0xA6,
		0x17, 0x39, 0xF0, 0x2F, 0x51, 0x0D, 0x2B, 0xB1, 0x2C, 0xEA,
		0xF6, 0x93, 0xC2, 0xDE, 0xEA, 0xE7, 0xFE, 0x25, 0xD5, 0xEE,
		0x8B, 0x65, 0x07, 0x48, 0x79, 0xC1, 0x26, 0xED, 0xD8, 0xEC,
		0x89, 0x27, 0x66, 0x38, 0xA0, 0xF1, 0x28, 0x11, 0xE3, 0x43,
		0xCB, 0x2A, 0xDD, 0xF8, 0xEB, 0x36, 0xB5, 0xA4, 0xAE, 0xAA,
		0x60, 0x71, 0x88, 0xAD, 0xDF, 0xB8, 0x2A, 0x18, 0x40, 0x35,
		0x50, 0xF6, 0xDB, 0x09, 0xE3, 0xE2, 0xCA, 0x77, 0x82, 0x99,
		0x92, 0x5F, 0x96, 0x5D, 0x9B, 0x6A, 0x1E, 0x32, 0x61, 0xBA,
		0xF6, 0x77, 0xE3, 0xB3, 0xF6, 0x6D, 0x09, 0x30, 0xB0, 0x23,
		0x8A, 0xBA, 0x70, 0x7A, 0xA1, 0x66, 0x94, 0xFB, 0x2B, 0x4C,
		0xF1, 0xAC, 0x1C, 0x92, 0x5E, 0x56, 0xDF, 0x92, 0xDD, 0x8F,
		0xDC, 0xB5, 0xD2, 0xB1, 0xAF, 0xF1, 0x5D, 0x70, 0x05, 0x0F,
		0x2E, 0xBB, 0x24, 0x9E, 0x2C, 0xF0, 0xB5, 0x75, 0xF0, 0x79,
		0xFA, 0xFA, 0xE7, 0x34, 0x6B, 0xA6, 0x69, 0x40, 0x24, 0xFB,
		0x7C, 0x38, 0xEB, 0xD4, 0xC2, 0xB6, 0xAA, 0x26, 0xF0, 0x91,
		0xDF, 0x8E, 0xF6, 0x13, 0x80, 0x5C, 0x53, 0x85, 0xD3, 0x82,
		0x52, 0x5B, 0xCB, 0xC9, 0x64, 0x9F, 0x6C, 0x7D, 0xD7, 0xA0,
		0x83, 0x3C, 0xFF, 0xFB, 0x22, 0xAF, 0x17, 0x77, 0x24, 0xFF,
		0x73, 0x08, 0xD1, 0xBB, 0x42, 0x5E, 0x9A, 0x54, 0xB1, 0x42,
		0x47, 0xEC, 0x7D, 0x23, 0xF7, 0x9F, 0xE8, 0x03, 0x4C, 0xE7,
		0xB6, 0x1D, 0x2E, 0xC2, 0x4D, 0x22, 0xED, 0x90, 0x67, 0x32,
		0x9F, 0x9A, 0x72, 0x51, 0xA5, 0x49, 0xFA, 0xB3, 0xCD, 0xB1,
		0x86, 0x23, 0x9B, 0x05, 0x62, 0xD0, 0x5E, 0xEC, 0x8C, 0x8A,
		0xE2, 0xB7, 0x12, 0x96, 0x50, 0x18, 0x51, 0xCC, 0x6F, 0x0F,
		0x0F, 0x20, 0x77, 0x4D, 0x48, 0xAA, 0xBD, 0xDC, 0x0F, 0x83,
		0x29, 0xCA, 0xBA, 0x5F, 0x7D, 0x07, 0x36, 0x7F, 0x8E, 0xBC,
		0x32, 0xFD, 0xB1, 0xDF, 0x80, 0xE7, 0x44, 0x2F, 0x2D, 0xB5,
		0xCD, 0x24, 0x7E, 0x88, 0x10, 0x03, 0x2B, 0xD2, 0x85, 0xF1,
		0x8C, 0x91, 0xE9, 0x3D, 0x6A, 0x62, 0x9E, 0x57, 0xC5, 0x7E,
		0xCD, 0x1E, 0xF0, 0xED, 0x97, 0x1F, 0x24, 0xA2, 0x20, 0xCD,
		0x9B, 0x4A, 0x6D, 0xF9, 0xC1, 0x93, 0x2D, 0x59, 0xB6, 0x43,
		0x86, 0x04, 0x9E, 0x7E, 0xAA, 0xCA, 0x1D, 0xD5, 0x82, 0xA0,
		0xB5, 0xE0, 0x79, 0x9B, 0xAE, 0x8E, 0xB7, 0xAB, 0xE1, 0xC7,
		0x47, 0xB1, 0xD7, 0x23, 0x6F, 0x37, 0x5B, 0x06, 0x13, 0xEC,
		0x03, 0x54, 0x37, 0x3F, 0x19, 0x7C, 0xAD, 0xA7, 0x99, 0x79,
		0xE7, 0x6B, 0x54, 0xF7, 0xF0, 0x0A, 0x81, 0xC8, 0xFF, 0xD4,
		0x80, 0x30, 0xAE, 0xEE, 0x83, 0xA0, 0x80, 0x4B, 0x26, 0x43,
		0x43, 0x04, 0x41, 0xD7, 0x97, 0xA6, 0xEB, 0xFA, 0x64, 0xAE,
		0x63, 0x37, 0xDC, 0x8F, 0xB5, 0xCA, 0x51, 0x58, 0x6E, 0x96,
		0x7A, 0x9F, 0x0A, 0x54, 0xC4, 0x9B, 0xFA, 0x22, 0x3A, 0x7E,
		0x2A, 0x10, 0xDF, 0x76, 0x09, 0x9C, 0x36, 0xA5, 0x4F, 0x5F,
		0xA1, 0xFB, 0x68, 0x0A, 0x29, 0x44, 0xCC, 0xA8, 0x11, 0xBD,
		0x1E, 0xAB, 0x5B, 0x06, 0xB9, 0x4D, 0xA0, 0x77, 0x7C, 0xE7,
		0x05, 0x8C, 0xF6, 0xCC, 0xB9, 0x8C, 0x7C, 0xA8, 0x38, 0xC6,
		0x88, 0x6F, 0x03, 0xC9, 0x9C, 0x34, 0xEC, 0xAB, 0xCC, 0x41,
		0x82, 0xFF, 0xAF, 0x31, 0x4A, 0xE4, 0x9D, 0x5C, 0x1F, 0x41,
		0x62, 0xBE, 0x15, 0x53, 0x28, 0x3E, 0x30, 0x55, 0xB6, 0xF5,
		0x85, 0x4E, 0xA7, 0xC0, 0xEF, 0x68, 0x90, 0xF8, 0x62, 0xD9,
		0x8F, 0x03, 0x35, 0x92, 0x3A, 0x2D, 0xCA, 0x27, 0x21, 0x6F,
		0x82, 0x01, 0xF7, 0xE0, 0x86, 0x77, 0xD8, 0x93, 0xC2, 0xFF,
		0x0D, 0x7C, 0xAF, 0x02, 0xD3, 0xBC, 0x2D, 0x7A, 0x58, 0x6B,
		0x53, 0x20, 0x5D, 0x16, 0x21, 0xE9, 0x35, 0xED, 0xE7, 0xEB,
		0x3E, 0x3E, 0xD1, 0x65, 0xF8, 0x58, 0x0A, 0xD5, 0x3A, 0x47,
		0xD4, 0xE3, 0x18, 0xC0, 0xEF, 0xB1, 0xED, 0x6B, 0x80, 0x21,
		0x19, 0x08, 0x65, 0xF2, 0xD5, 0x7A, 0x5A, 0x02, 0xB5, 0xC1,
		0xBF, 0xAC, 0x77, 0xDB, 0xC0, 0xFD, 0x2D, 0x9C, 0x6D, 0x53,
		0xDD, 0x47, 0x83, 0xBE, 0x5E, 0x28, 0x9A, 0xB8, 0x18, 0xD7,
		0xAA, 0xF4, 0x9A, 0x5E, 0xC7, 0xFE, 0x3B, 0xFF, 0x01, 0x65,
		0xAC, 0xE8, 0x1C, 0xBD, 0x06, 0x33, 0x0C, 0xB3, 0x79, 0x08,
		0x88, 0x97, 0x60, 0x1A, 0x7F, 0xE0, 0x07, 0xA1, 0xF1, 0xD6,
		0xE0, 0x07, 0xEA, 0xD2, 0x7E, 0xC5, 0xDF, 0x84, 0x8F, 0x39,
		0x06, 0xB6, 0x34, 0xCD, 0x6E, 0x5D, 0x27, 0x8A, 0x5F, 0x50,
		0xFF, 0x7B, 0xBF, 0x03, 0xA9, 0x29, 0xBA, 0x7A, 0xE3, 0x56,
		0x37, 0x41, 0xB5, 0x35, 0x33, 0x1B, 0x46, 0x44, 0x47, 0x65,
		0x83, 0x0C, 0x2F, 0x38, 0x0C, 0x32, 0xC1, 0x1A, 0x95, 0x63,
		0x37, 0x3E, 0xE2, 0xA7, 0x0C, 0x13, 0x22, 0x30, 0x16, 0xC5,
		0x2A, 0x63, 0x91, 0xBE, 0x71, 0x12, 0xE9, 0xF5, 0xE7, 0x03,
		0xC1, 0x5F, 0xC8, 0x42, 0xF8, 0x0C, 0x8F, 0xE2, 0x34, 0xC0,
		0xEE, 0xE4, 0xDE, 0x74, 0x15, 0xA9, 0x53, 0x9E, 0x2B, 0x1B,
		0x1E, 0x07, 0xA4, 0x10, 0xA7, 0x7B, 0x18, 0xBE, 0x1A, 0x40,
		0xF2, 0xAF, 0xFF, 0x73, 0xD8, 0x4D, 0xE7, 0x36, 0x46, 0x37,
		0x2F, 0x40, 0x38, 0xBA, 0xD6, 0xD2, 0xDF, 0xAA, 0x7D, 0x8C,
		0x52, 0xDB, 0xA5, 0xE7, 0x76, 0x99, 0xAC, 0xB0, 0x78, 0x3E,
		0x44, 0xBE, 0x22, 0x6C, 0x7A, 0xC5, 0xB7, 0xBD, 0x1E, 0x92,
		0x35, 0xA6, 0x7A, 0xD0, 0x02, 0x95, 0xA0, 0x51, 0xC0, 0x20,
		0xBA, 0x5D, 0x7D, 0x3D, 0xAD, 0x6D, 0x75, 0xB6, 0x30, 0xF7,
		0xE0, 0x1D, 0xDD, 0xC0, 0x08, 0x31, 0xD6, 0xD8, 0x0B, 0x97,
		0xF3, 0x99, 0x07, 0xFA, 0xCA, 0x00, 0x48, 0x1A, 0x9D, 0xA7,
		0x90, 0x8D, 0xCE, 0x56, 0xFF, 0x4C, 0xF0, 0xF9, 0x45, 0x86,
		0x5C, 0xAA, 0x57, 0x26, 0x8D, 0x00, 0xA6, 0xC3, 0xEA, 0xA0,
		0x3D, 0x30, 0xDE, 0x8F, 0x76, 0xB9, 0xA9, 0x6D, 0x37, 0xF3,
		0xF3, 0xD1, 0x61, 0x88, 0x1D, 0x6D, 0x1D, 0x84, 0x84, 0xF2,
		0xD1, 0x13, 0x09, 0xE9, 0x26, 0x69, 0x52, 0xB0, 0x68, 0x01,
		0xF3, 0x3C, 0xEB, 0xFF, 0x2F, 0xAD, 0xFA, 0xEC, 0xDA, 0x83,
		0x3E, 0xC2, 0x0D, 0x4F, 0x70, 0x82, 0x14, 0xA9, 0xD6, 0xF5,
		0x01, 0x9A, 0x00, 0xD6, 0x9F, 0x6D, 0xDE, 0xD8, 0x7B, 0x50,
		0x3C, 0xE8, 0x2F, 0x44, 0x42, 0x64, 0x53, 0x75, 0x31, 0xF3,
		0x62, 0x1F, 0x1E, 0x84, 0xC9, 0x71, 0x81, 0xE8, 0xC2, 0xA1,
		0xBA, 0x8E, 0x98, 0x97, 0x45, 0x36, 0xA0, 0x3F, 0x74, 0x61,
		0xA1, 0x5D, 0x0B, 0xF8, 0x07, 0x08, 0x33, 0x8E, 0x37, 0x33,
		0x23, 0xCF, 0x38, 0x6E, 0x8D, 0x47, 0xD0, 0xA0, 0x15, 0x14,
		0xCF, 0x7F, 0x36, 0x87, 0x49, 0x43, 0xF0, 0x63, 0xD2, 0x7A,
		0x97, 0x77, 0x45, 0x6A, 0x62, 0xF0, 0x67, 0xEE, 0x25, 0xDF,
		0x85, 0x50, 0x67, 0xF8, 0x28, 0x04, 0xD2, 0x88, 0xFB, 0x18,
		0xB0, 0x37, 0xB6, 0xB9, 0xCC, 0x03, 0x4D, 0x72, 0xE7, 0x2C,
		0xE7, 0xB8, 0xDE, 0x9F, 0x2C, 0x4C, 0xD3, 0x0B, 0xF7, 0x88,
		0x60, 0x81, 0xD8, 0xCC, 0xCE, 0x9B, 0x82, 0x83, 0xF9, 0x31,
		0x70, 0xD7, 0x1B, 0xB1, 0x1A, 0xC3, 0xC3, 0x18, 0x75, 0x59,
		0x78, 0xC7, 0x70, 0xFD, 0xAB, 0x53, 0xEB, 0xEA, 0x7C, 0x22,
		0x35, 0x42, 0x5F, 0x17, 0xCA, 0xCB, 0xFA, 0x7A, 0xD2, 0x41,
		0x5C, 0xC5, 0x3A, 0x57, 0x2A, 0x93, 0x0B, 0xCA, 0x8A, 0xA8,
		0x6E, 0x08, 0x4E, 0x57, 0x9F, 0x2A, 0x4C, 0xD3, 0x47, 0xA0,
		0xDC, 0xDF, 0x49, 0x33, 0x88, 0x9A, 0x6A, 0x27, 0x7C, 0xA4,
		0x6B, 0x31, 0x21, 0xC3, 0xB0, 0x18, 0xF1, 0xBC, 0xD1, 0xF3,
		0xF3, 0xC8, 0xCA, 0xC5, 0x56, 0xBE, 0xDD, 0xCA, 0x3E, 0x47,
		0x6F, 0x66, 0xAC, 0xF3, 0x05, 0x0D, 0xDD, 0x2B, 0x14, 0xFB,
		0x86, 0xFF, 0x93, 0x54, 0x6E, 0xC4, 0xC7, 0x2B, 0x21, 0xDE,
		0x71, 0xBC, 0x3A, 0xF3, 0x94, 0xE2, 0xED, 0x27, 0x5F, 0x9D,
		0x95, 0xF1, 0x26, 0x0F, 0xF9, 0xD9, 0xF6, 0xE8, 0xEF, 0xD2,
		0xF6, 0x63, 0x78, 0x0B, 0xE6, 0x88, 0xAA, 0xB8, 0xED, 0xB5,
		0x5B, 0x73, 0xBB, 0x65, 0x56, 0x69, 0x40, 0xA2, 0xDC, 0xF4,
		0xE0, 0xD4, 0x8F, 0x8D, 0x06, 0x6C, 0x87, 0x3D, 0x0C, 0x5B,
		0x7A, 0x22, 0x9F, 0xDD, 0x43, 0xC8, 0x25, 0x7A, 0x95, 0x70,
		0xD6, 0xCE, 0xFB, 0xEF, 0x2A, 0x9C, 0x5D, 0xAE, 0x4D, 0x06,
		0x35, 0x70, 0x0B, 0x8E, 0xB0, 0x3A, 0x2C, 0x9E, 0x85, 0x7B,
		0xF8, 0x81, 0x86, 0xE8, 0xA4, 0x86, 0x0F, 0x19, 0x13, 0xA4,
		0x6F, 0x75, 0x12, 0xE7, 0x82, 0x90, 0x0E, 0xBB, 0x2B, 0x8C,
		0xE8, 0x80, 0xBF, 0x92, 0x44, 0x1F, 0xA9, 0xAF, 0xF7, 0x57,
		0x6D, 0x95, 0xF0, 0xA8, 0x58, 0x59, 0x5E, 0x09, 0xAA, 0x56,
		0x28, 0x46, 0x2E, 0x0C, 0x99, 0x79, 0x8F, 0xB5, 0x3D, 0xA7,
		0x91, 0x99, 0x33, 0x76, 0x24, 0x51, 0xDA, 0xE5, 0xF2, 0x4F,
		0xDD, 0x38, 0x60, 0x42, 0xB1, 0x08, 0xEF, 0xA2, 0x9C, 0xC0,
		0x38, 0xA0, 0x55, 0x5F, 0x02, 0x0C, 0xD1, 0x99, 0x77, 0x36,
		0x6F, 0x86, 0xE7, 0x57, 0x9A, 0x1E, 0x9E, 0xE0, 0x4A, 0xF5,
		0x1A, 0x97, 0xA9, 0x00, 0xB5, 0x2B, 0x53, 0x00, 0xA6, 0x11,
		0xB0, 0xFE, 0x3B, 0x25, 0xFF, 0x2C, 0xB8, 0xD1, 0x56, 0x6E,
		0x46, 0xF4, 0x7C, 0x9C, 0x4F, 0x92, 0x64, 0xB4, 0x43, 0x58,
		0x1A, 0x9F, 0xA0, 0xD8, 0xE5, 0x38, 0xCC, 0x0D, 0x30, 0x59,
		0x7B, 0x18, 0x8E, 0xF4, 0xD2, 0xBE, 0x19, 0x4C, 0x0C, 0xDD,
		0x76, 0xB8, 0x80, 0xEB, 0xCE, 0xFF, 0xCF, 0x75, 0x0D, 0xE6,
		0x85, 0xCF, 0x18, 0x35, 0x05, 0x83, 0xBD, 0x55, 0x71, 0x0B,
		0x09, 0x2F, 0x8A, 0xD6, 0x5A, 0x14, 0xF4, 0x4C, 0x12, 0x5D,
		0x79, 0xE9, 0x83, 0x5B, 0x8D, 0x20, 0x22, 0x5E, 0x2D, 0x57,
		0x43, 0xD6, 0x3C, 0xBB, 0x30, 0xD5, 0x2D, 0x83, 0x40, 0xE4,
		0x1C, 0x92, 0x36, 0x8F, 0x39, 0x04, 0xD8, 0xA2, 0x49, 0xFB,
		0x56, 0x13, 0x4B, 0x4A, 0xA1, 0xB6, 0xA5, 0x45, 0xB8, 0x46,
		0x67, 0xD9, 0xE8, 0xA7, 0x9B, 0x43, 0x33, 0x62, 0x79, 0xA4,
		0x10, 0x4B, 0x5F, 0x5E, 0xA7, 0x1A, 0xA0, 0x29, 0xA1, 0x46,
		0x66, 0x8A, 0x38, 0x15, 0xFE, 0xDF, 0x19, 0x70, 0x1D, 0xAA,
		0x7A, 0x3D, 0xCF, 0xA0, 0x43, 0xC4, 0xAD, 0xD6, 0xB9, 0xDB,
		0x4B, 0x3F, 0x2D, 0x82, 0x11, 0x55, 0xF2, 0x65, 0xBB, 0xBF,
		0x60, 0x1F, 0x83, 0x87, 0xE7, 0xAB, 0x9E, 0x95, 0x7D, 0x54,
		0x1E, 0x7E, 0xA2, 0xA7, 0x26, 0x22, 0xA1, 0x52, 0xC3, 0x63,
		0xDA, 0x00, 0x48, 0x25, 0xCE, 0x4E, 0x35, 0xD6, 0xAB, 0x7C,
		0x33, 0x37, 0xEF, 0xBF, 0x52, 0xA0, 0x60, 0x70, 0xF3, 0x06,
		0x27, 0x87, 0xD7, 0x01, 0x00, 0x19, 0x8D, 0x63, 0x58, 0xBD,
		0xD1, 0x26, 0x9D, 0xAA, 0x7F, 0xB3, 0xC0, 0xE4, 0xF7, 0xDA,
		0xB1, 0xC5, 0x1D, 0x54, 0x56, 0x1F, 0x98, 0x2F, 0x0B, 0x7F,
		0xE0, 0x72, 0x79, 0x60, 0x0D, 0x59, 0xB4, 0x08, 0x09, 0xC6,
		0x28, 0xE7, 0x9B, 0xC8, 0xD0, 0x8A, 0x57, 0x87, 0xEF, 0x5D,
		0x1F, 0x40, 0xE2, 0x6C, 0x66, 0x35, 0xE0, 0x8B, 0x0A, 0x0A,
		0xDF, 0xB6, 0x4A, 0xF4, 0x86, 0xA2, 0xB0, 0xF4, 0x32, 0x6E,
		0x10, 0xDF, 0x24, 0xDA, 0x46, 0x18, 0xFA, 0xFE, 0xDF, 0x60,
		0x2A, 0x77, 0xE5, 0x56, 0x73, 0x66, 0x14, 0x29, 0x54, 0x3F,
		0xC7, 0xC4, 0xE8, 0x2E, 0xA0, 0x6F, 0xFF, 0x96, 0xC6, 0xB0,
		0xEA, 0x81, 0xFF, 0xD5, 0xB3, 0x93, 0xA7, 0xF7, 0x96, 0xBB,
		0x74, 0xA8, 0x4F, 0xF3, 0x39, 0x80, 0x25, 0x4A, 0x8A, 0x4D,
		0x53, 0x9B, 0x75, 0x31, 0x43, 0xC4, 0x73, 0xCB, 0xD2, 0xDD,
		0xD4, 0x6F, 0xA6, 0x2F, 0x53, 0x63, 0xF8, 0x0B, 0xA0, 0xFE,
		0xB3, 0x68, 0xC5, 0xCA, 0x71, 0x0B, 0x89, 0x5B, 0xFA, 0x72,
		0x4C, 0xD3, 0x05, 0x6D, 0xE0, 0x3E, 0xE4, 0xF7, 0xE0, 0x54,
		0x91, 0x5A, 0x5F, 0x5C, 0x86, 0xE6, 0xB4, 0xD9, 0x4E, 0x33,
		0xF3, 0x72, 0x1E, 0x8F, 0xBA, 0x33, 0xDA, 0x5A, 0x29, 0x24,
		0x19, 0x04, 0x26, 0x31, 0x39, 0x5A, 0x56, 0x5A, 0x09, 0xA2,
		0x96, 0xEB, 0x7D, 0xBF, 0x8D, 0x6E, 0x9B, 0x9F, 0x25, 0x9A,
		0x30, 0xA2, 0x60, 0x20, 0xA4, 0x85, 0x2E, 0x5B, 0xC8, 0xE9,
		0xE1, 0x7B, 0xE3, 0x5E, 0x14, 0x1F, 0xC9, 0xFB, 0xFB, 0xAB,
		0x81, 0xE4, 0x7B, 0xAF, 0x54, 0x98, 0xF8, 0x4C, 0xEB, 0x83,
		0x1D, 0xBB, 0xD5, 0x0C, 0x83, 0x2D, 0x23, 0xF0, 0x2D, 0x35,
		0x05, 0xD7, 0x3F, 0x85, 0x52, 0xEA, 0x83, 0x76, 0x0C, 0x79,
		0xF1, 0xB9, 0x5B, 0xFA, 0xA7, 0xB4, 0xCF, 0x84, 0x10, 0xD1,
		0xBB, 0x34, 0x33, 0x08, 0x6F, 0x33, 0x65, 0x66, 0xD4, 0x20,
		0x98, 0xBB, 0xB5, 0x57, 0xA6, 0x28, 0x99, 0x4B, 0x33, 0x65,
		0x69, 0xB3, 0x80, 0x1E, 0x8B, 0x8C, 0xEF, 0x3B, 0x72, 0x58,
		0x4D, 0xBC, 0xAB, 0xE2, 0x8C, 0xAC, 0x00, 0xD8, 0xD0, 0x85,
		0x4F, 0xED, 0x29, 0x8C, 0x97, 0x7D, 0x46, 0x4B, 0x35, 0x1A,
		0x0C, 0x1D, 0x40, 0xBC, 0xB4, 0x47, 0xE0, 0x33, 0xE1, 0x3F,
		0xB9, 0xB5, 0x54, 0xFE, 0xAD, 0x30, 0x39, 0xFB, 0xA3, 0xDC,
		0x42, 0xA5, 0x68, 0x74, 0x21, 0x1E, 0xFB, 0x61, 0x52, 0x6C,
		0xCC, 0xA8, 0xF0, 0xC6, 0x56, 0xFA, 0xED, 0xE4, 0xDE, 0x2F,
		0x88, 0x43, 0x7E, 0x39, 0xFA, 0x71, 0x0B, 0x24, 0xED, 0xEA,
		0xEA, 0x12, 0xF4, 0x25, 0xE0, 0x63, 0x17, 0x2D, 0xA1, 0x92,
		0x77, 0x68, 0x76, 0x5F, 0x79, 0xB3, 0x26, 0x5F, 0x23, 0x9B,
		0x1A, 0x23, 0x0E, 0x97, 0x01, 0x23, 0x09, 0x80, 0xA2, 0xC3,
		0xB0, 0x73, 0xC9, 0xC8, 0xCC, 0x45, 0x5C, 0x16, 0x26, 0x79,
		0x95, 0x07, 0x08, 0xEC, 0xC7, 0x8A, 0x66, 0x87, 0xB9, 0x9C,
		0x34, 0xBC, 0x97, 0xB7, 0xB2, 0xB6, 0x58, 0x50, 0x3C, 0x07,
		0xDD, 0xAD, 0x35, 0xA9, 0x7F, 0x3E, 0xF2, 0xDD, 0xE7, 0x32,
		0x4F, 0x5A, 0x1C, 0xBA, 0x52, 0xCD, 0xDA, 0x67, 0xE7, 0x3E,
		0x2B, 0xE6, 0x2D, 0xD8, 0x6C, 0x30, 0xB2, 0xBB, 0xA6, 0xB4,
		0x78, 0xAC, 0x70, 0x43, 0x34, 0xE7, 0xD5, 0xE2, 0x54, 0xAE,
		0xC8, 0x92, 0x6E, 0xE7, 0x74, 0xA0, 0xED, 0x4A, 0x72, 0x59,
		0x6A, 0xD0, 0xFC, 0xAD, 0x4B, 0x12, 0x94, 0x0B, 0x8C, 0xD3,
		0xED, 0xBA, 0x6F, 0x56, 0x19, 0xC4, 0xFB, 0x0A, 0x04, 0xC5,
		0x26, 0x45, 0x38, 0xE3, 0xEB, 0x76, 0xF9, 0xC2, 0xE3, 0xF8,
		0x03, 0x9F, 0x76, 0x16, 0x0A, 0x92, 0x70, 0x6B, 0x4D, 0x39,
		0xAC, 0xFC, 0x27, 0x8C, 0x92, 0xB3, 0x08, 0x68, 0xA9, 0x58,
		0x99, 0xF1, 0xE9, 0xE8, 0x6F, 0x75, 0x63, 0x6D, 0x18, 0x98,
		0x4C, 0x41, 0x07, 0xB3, 0x5F, 0x7F, 0x6E, 0x78, 0xBE, 0xC3,
		0x1C, 0x94, 0xFB, 0xEE, 0xAD, 0x96, 0xBA, 0x9B, 0x51, 0x04,
		0x0F, 0x45, 0x03, 0xEB, 0xAC, 0x63, 0x1F, 0xE4, 0x8F, 0xC9,
		0x55, 0x8D, 0x00, 0x48, 0x28, 0x5A, 0x92, 0xE2, 0x27, 0xE3,
		0xB7, 0x18, 0xAD, 0xAD
	];
	let key: [u8; 45] = [
		0x43, 0x45, 0x54, 0x08, 0xCD, 0xFE, 0x3E, 0xB8, 0xFA,
		0x19, 0xE1, 0x74, 0x9C, 0x99, 0x95, 0x87, 0x63, 0x26,
		0x91, 0x6C, 0x70, 0x60, 0xDA, 0x4F, 0xF7, 0x7B, 0x94,
		0xF9, 0x54, 0x4C, 0x1D, 0xF1, 0xBA, 0xA0, 0x84, 0x42,
		0xEA, 0x82, 0x0A, 0x20, 0x36, 0xC5, 0xED, 0xDB, 0x98
	];
	let cust: [u8; 97] = [
		0x77, 0x34, 0xCE, 0xD9, 0x69, 0x1E, 0xFC, 0xB5, 0xCF, 0x1A,
		0x60, 0x7C, 0x87, 0x50, 0xFE, 0xB2, 0x31, 0x01, 0x55, 0xEC,
		0x70, 0x81, 0x09, 0xC3, 0x7C, 0xF9, 0xBC, 0xFA, 0xB9, 0x58,
		0xA4, 0x07, 0x67, 0xAD, 0x51, 0x89, 0x04, 0x23, 0x05, 0x35,
		0xBE, 0xF7, 0x02, 0x1A, 0x2F, 0x52, 0x3D, 0x46, 0xB1, 0xCC,
		0xB7, 0x36, 0xB2, 0xAD, 0xE7, 0xC9, 0xBE, 0xD3, 0xFD, 0x6E,
		0x54, 0xBB, 0xAC, 0xD2, 0x34, 0x8F, 0xCF, 0xC4, 0x96, 0xAF,
		0x7F, 0xE8, 0x38, 0x07, 0x99, 0x57, 0x4B, 0xC7, 0x96, 0x7A,
		0x68, 0x11, 0xB5, 0x66, 0x2A, 0xD9, 0x97, 0x36, 0x72, 0xE8,
		0x39, 0x83, 0x0C, 0x3D, 0x3D, 0x57, 0x4E
	];
	let exp: [u8; 89] = [
		0xf4, 0x3d, 0x2b, 0x36, 0x5c, 0xc9, 0x9d, 0xa0, 0x97, 0x00,
		0x22, 0xfb, 0xf1, 0x62, 0xaa, 0x0e, 0x15, 0xf5, 0xae, 0x20,
		0x0d, 0x6d, 0xb5, 0x78, 0xb9, 0x5d, 0x31, 0xfe, 0xda, 0x95,
		0x2d, 0x8a, 0xbd, 0x6b, 0x6c, 0x73, 0xd5, 0x6c, 0xcc, 0x25,
		0x74, 0xe0, 0xc0, 0x37, 0x45, 0x3f, 0xb8, 0xcb, 0x1a, 0xb0,
		0xfd, 0xb8, 0xcc, 0xca, 0x23, 0x27, 0x1d, 0x74, 0x2d, 0x3c,
		0xcd, 0x1f, 0x4b, 0x2d, 0x55, 0x92, 0xd6, 0x7a, 0x08, 0xa8,
		0xd1, 0x8e, 0x54, 0x06, 0x92, 0x22, 0xc8, 0xce, 0xe8, 0x57,
		0x75, 0x9b, 0xe9, 0xea, 0xa4, 0xdf, 0x2c, 0x4d, 0xb3
	];

	lc_rust_kmac_one(lcr_kmac_type::lcr_kmac_128, &key, &cust, &msg, &exp)
}
