/*
 * Copyright (C) 2025, Stephan Mueller <smueller@chronox.de>
 *
 * License: see LICENSE file in root directory
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ALL OF
 * WHICH ARE HEREBY DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF NOT ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

use leancrypto_sys::lcr_aead::lcr_aead;
use leancrypto_sys::lcr_aead::lcr_aead_type;
use leancrypto_sys::error::AeadError;

fn lc_rust_aead_one(aead_type: lcr_aead_type, key: &[u8], iv: &[u8], pt: &[u8],
		    ct: &mut [u8], aad: &[u8], tag: &mut [u8], exp_ct: &[u8],
		    exp_tag: &[u8]) {
	let mut aead = lcr_aead::new(aead_type);

	// One-shot encrypt
	let result = aead.setkey(key, iv);
	assert_eq!(result, Ok(()));
	let result = aead.encrypt(pt, ct, aad, tag);
	assert_eq!(result, Ok(()));
	assert_eq!(ct, exp_ct);
	assert_eq!(tag, exp_tag);

	// One-shot decrypt
	let result = aead.setkey(key, iv);
	assert_eq!(result, Ok(()));
	let mut new_pt = vec!(0u8; pt.len());
	let result = aead.decrypt(ct, &mut new_pt, aad, tag);
	assert_eq!(result, Ok(()));
	assert_eq!(pt, new_pt);

	// Stream encrypt
	let result = aead.setkey(key, iv);
	assert_eq!(result, Ok(()));

	let result = aead.enc_init(aad);
	assert_eq!(result, Ok(()));

	let result = aead.enc_update(pt, ct);
	assert_eq!(result, Ok(()));
	assert_eq!(ct, exp_ct);

	let result = aead.enc_final(tag);
	assert_eq!(result, Ok(()));
	assert_eq!(tag, exp_tag);

	// Stream decrypt
	let result = aead.setkey(key, iv);
	assert_eq!(result, Ok(()));

	let mut new_pt = vec!(0u8; pt.len());
	let result = aead.dec_init(aad);
	assert_eq!(result, Ok(()));

	let result = aead.dec_update(ct, &mut new_pt);
	assert_eq!(result, Ok(()));
	assert_eq!(new_pt, pt);

	let result = aead.dec_final(tag);
	assert_eq!(result, Ok(()));

	// Test authentication failure detection
	new_pt[0] ^= 0x01;

	let result = aead.setkey(key, iv);
	assert_eq!(result, Ok(()));
	let result = aead.encrypt(&new_pt, ct, aad, tag);
	assert_eq!(result, Ok(()));

	let result = aead.setkey(key, iv);
	assert_eq!(result, Ok(()));
	let result = aead.decrypt(exp_ct, &mut new_pt, aad, tag);
	assert_eq!(result, Err(AeadError::AuthenticationError));
}

#[test]
fn lc_rust_ascon_keccack_256() {
	let key: [u8; 32] = [
		0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
		0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
		0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
		0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
	];
	let iv: [u8; 16] = [
		0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
		0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
	];

	let pt: [u8; 64] = [
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
		0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13,
		0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d,
		0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
		0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31,
		0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
		0x3c, 0x3d, 0x3e, 0x3f,
	];
	let exp_ct: [u8; 64] = [
		0xbf, 0xdf, 0xeb, 0x80, 0x84, 0x88, 0xbe, 0xd1, 0xda, 0xdb,
		0x85, 0xda, 0xe2, 0x39, 0x18, 0xfc, 0x14, 0x20, 0xf1, 0x0b,
		0xc4, 0xd2, 0xaf, 0xc3, 0x1c, 0xee, 0x97, 0x0f, 0xad, 0x52,
		0xa0, 0xfa, 0xa6, 0x1a, 0x58, 0x0b, 0x56, 0x3f, 0xf6, 0xe8,
		0x03, 0x49, 0x43, 0xf1, 0x12, 0x0d, 0x5e, 0xb0, 0x82, 0x69,
		0xe2, 0xfd, 0xde, 0x02, 0xc2, 0x12, 0xd6, 0x91, 0x3b, 0x31,
		0x3d, 0x20, 0x54, 0x63
	];
	let exp_tag: [u8; 16] = [
		0xc5, 0x72, 0x34, 0x77, 0xa0, 0x60, 0x46, 0x0d, 0xc1, 0x74,
		0x21, 0x17, 0x6a, 0x28, 0xbb, 0x70
	];
	let mut ct: [u8; 64] = [0; 64];
	let mut tag: [u8; 16] = [0; 16];

	lc_rust_aead_one(lcr_aead_type::lcr_ascon_keccak_256, &key, &iv, &pt,
			 &mut ct, &pt, &mut tag, &exp_ct, &exp_tag)
}

#[test]
fn lc_rust_ascon_keccack_512() {
	let key: [u8; 64] = [
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
		0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13,
		0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d,
		0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
		0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31,
		0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
		0x3c, 0x3d, 0x3e, 0x3f
	];
	let iv: [u8; 16] = [
		0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
		0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
	];

	let pt: [u8; 192] = [
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
		0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13,
		0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d,
		0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
		0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31,
		0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
		0x3c, 0x3d, 0x3e, 0x3f, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
		0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
		0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
		0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
		0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d,
		0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
		0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x00, 0x01,
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
		0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15,
		0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29,
		0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33,
		0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d,
		0x3e, 0x3f,
	];
	let exp_ct: [u8; 192] = [
		0xed, 0x50, 0x43, 0x63, 0x83, 0x2f, 0xaf, 0x87, 0xf0, 0x17,
		0x47, 0x2c, 0x70, 0x24, 0xf3, 0x04, 0xea, 0x6c, 0xff, 0x9d,
		0x96, 0xd2, 0x04, 0xe7, 0xe4, 0x73, 0x90, 0x77, 0x85, 0xf5,
		0x45, 0xcd, 0xad, 0xc4, 0x88, 0x49, 0x01, 0x09, 0x1a, 0xb4,
		0xf2, 0xaf, 0x34, 0xa7, 0xdc, 0x6c, 0x02, 0x02, 0xdb, 0x27,
		0x50, 0xc0, 0x07, 0xda, 0xa2, 0xb7, 0x8f, 0x6c, 0xd1, 0xa0,
		0xa4, 0xa5, 0x8e, 0x97, 0xbf, 0xb2, 0x29, 0x08, 0xdd, 0xbc,
		0x5c, 0x89, 0x3b, 0x79, 0x5f, 0x9f, 0xf5, 0x6d, 0x02, 0xfd,
		0x8d, 0x09, 0x72, 0x7b, 0x1f, 0xf7, 0xc5, 0x4a, 0x68, 0xd5,
		0x6e, 0x91, 0x4d, 0x72, 0x64, 0xaf, 0x57, 0x55, 0x0c, 0x54,
		0xdc, 0xa6, 0xd5, 0xbc, 0x1c, 0x03, 0x6a, 0xf6, 0x9f, 0x03,
		0x7f, 0x37, 0xed, 0xae, 0x75, 0xbb, 0x03, 0x09, 0x62, 0x47,
		0x33, 0x18, 0xab, 0xc0, 0x2b, 0x2c, 0x2d, 0xb0, 0x88, 0xef,
		0x7c, 0x4b, 0x75, 0x56, 0x7d, 0x58, 0xbd, 0x81, 0x72, 0x24,
		0x7e, 0x55, 0x1b, 0x5c, 0x1f, 0x35, 0x9d, 0x1d, 0xa1, 0xf3,
		0x29, 0x28, 0x11, 0xab, 0xb5, 0xea, 0x56, 0xd0, 0x94, 0x2f,
		0x15, 0x72, 0x0a, 0x58, 0xf8, 0x4b, 0x8e, 0x61, 0xb9, 0xcf,
		0xdd, 0xcc, 0x42, 0xed, 0xf0, 0xe1, 0x6e, 0x5d, 0x85, 0xe7,
		0x35, 0x74, 0x20, 0xbb, 0x3e, 0x79, 0x2f, 0x59, 0xe3, 0xf4,
		0x5f, 0x54
	];
	let exp_tag: [u8; 16] = [
		0xb5, 0x63, 0x33, 0x41, 0x9a, 0xda, 0x82, 0xc1, 0xba, 0xdc,
		0xcd, 0x70, 0xe7, 0x98, 0xfa, 0xe5
	];
	let mut ct: [u8; 192] = [0; 192];
	let mut tag: [u8; 16] = [0; 16];

	lc_rust_aead_one(lcr_aead_type::lcr_ascon_keccak_512, &key, &iv, &pt,
			 &mut ct, &pt, &mut tag, &exp_ct, &exp_tag)
}

#[test]
fn lc_rust_ascon_128() {
	let key: [u8; 16] = [
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
		0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
	];
	let iv: [u8; 16] = [
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
		0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
	];

	let pt: [u8; 32] = [
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
		0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13,
		0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D,
		0x1E, 0x1F
	];
	let aad: [u8; 32] = [
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
		0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13,
		0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D,
		0x1E, 0x1F
	];
	let exp_ct: [u8; 32] = [
		0x4C, 0x08, 0x6D, 0x27, 0xA3, 0xB5, 0x1A, 0x23, 0x33, 0xCF,
		0xC7, 0xF2, 0x21, 0x72, 0xA9, 0xBC, 0xAD, 0x88, 0xB8, 0xD4,
		0xD7, 0x7E, 0x50, 0x62, 0x2D, 0x78, 0x83, 0x45, 0xFA, 0x7B,
		0xEE, 0x44
	];
	let exp_tag: [u8; 16] = [
		0x68, 0x91, 0x5D, 0x3F, 0x94, 0x22, 0x28, 0x9F, 0x23, 0x49,
		0xD6, 0xA3, 0xB4, 0x16, 0x03, 0x97
	];
	let mut ct: [u8; 32] = [0; 32];
	let mut tag: [u8; 16] = [0; 16];

	lc_rust_aead_one(lcr_aead_type::lcr_ascon_128, &key, &iv, &pt,
			 &mut ct, &aad, &mut tag, &exp_ct, &exp_tag)
}

#[test]
fn lc_rust_aes_cbc_sha2_512() {
	let key: [u8; 64] = [
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
		0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13,
		0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d,
		0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
		0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31,
		0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
		0x3c, 0x3d, 0x3e, 0x3f,
	];
	let iv: [u8; 16] = [
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
	];

	let pt: [u8; 64] = [
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
		0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13,
		0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d,
		0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
		0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31,
		0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
		0x3c, 0x3d, 0x3e, 0x3f,
	];
	let exp_ct: [u8; 64] = [
		0xf8, 0xf6, 0xf0, 0x2d, 0x2f, 0xb6, 0xee, 0x57, 0x92, 0x49,
		0xb8, 0xa2, 0xe7, 0xc1, 0xe0, 0x48, 0x6a, 0x0e, 0x0a, 0x46,
		0x24, 0x11, 0xef, 0x3b, 0x6a, 0x0b, 0xc9, 0x2a, 0xb8, 0x94,
		0xd5, 0xac, 0x3f, 0x0a, 0x22, 0x21, 0x61, 0x23, 0x81, 0x40,
		0x22, 0x3d, 0x72, 0x94, 0xe6, 0x4a, 0x05, 0x6c, 0x55, 0x9a,
		0x0d, 0x7d, 0x6c, 0x6a, 0xb3, 0x58, 0x69, 0x8d, 0xaa, 0x6c,
		0x9b, 0x53, 0xa1, 0x67
	];
	let exp_tag: [u8; 64] = [
		0xa9, 0xd1, 0x8a, 0x72, 0xed, 0xc2, 0x30, 0x26, 0xef, 0x4c,
		0x69, 0x1e, 0xf9, 0x67, 0x1b, 0x7c, 0xaf, 0x40, 0x59, 0x59,
		0x90, 0x63, 0xd5, 0x64, 0x5f, 0x19, 0x4a, 0x98, 0xf6, 0x4d,
		0x72, 0x2e, 0xf5, 0xc7, 0xcb, 0x67, 0x1d, 0x1a, 0x34, 0xf8,
		0x79, 0xd8, 0xc3, 0x36, 0x59, 0xbf, 0x9a, 0xcb, 0xb3, 0x58,
		0x62, 0xac, 0xc4, 0x83, 0x91, 0x97, 0x31, 0x19, 0x56, 0x8d,
		0x32, 0xbe, 0xf1, 0x30
	];
	let mut ct: [u8; 64] = [0; 64];
	let mut tag: [u8; 64] = [0; 64];

	lc_rust_aead_one(lcr_aead_type::lcr_aes_cbc_sha2_512, &key, &iv, &pt,
			 &mut ct, &pt, &mut tag, &exp_ct, &exp_tag)
}

#[test]
fn lc_rust_aes_cbc_cshake256() {
	let key: [u8; 64] = [
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
		0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13,
		0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d,
		0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
		0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31,
		0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
		0x3c, 0x3d, 0x3e, 0x3f,
	];
	let iv: [u8; 16] = [
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
	];

	let pt: [u8; 64] = [
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
		0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13,
		0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d,
		0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
		0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31,
		0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
		0x3c, 0x3d, 0x3e, 0x3f,
	];
	let exp_ct: [u8; 64] = [
		0xe1, 0xe3, 0x2f, 0x24, 0xee, 0x4c, 0x9b, 0x47, 0xa1, 0x31,
		0xb5, 0xbd, 0xc6, 0x36, 0x0f, 0x2a, 0x72, 0x6e, 0xc0, 0x43,
		0x89, 0xf7, 0x91, 0xba, 0x34, 0x87, 0xce, 0x1d, 0xe1, 0x38,
		0x91, 0x61, 0x40, 0x2a, 0x2c, 0xb7, 0xe9, 0x76, 0x80, 0xc3,
		0xc4, 0x40, 0x45, 0x16, 0x2e, 0xbd, 0xd2, 0x69, 0x15, 0x59,
		0xba, 0x9c, 0xd4, 0xae, 0x00, 0x59, 0x49, 0x27, 0x2d, 0x50,
		0xd9, 0xd8, 0x04, 0xeb
	];
	let exp_tag: [u8; 64] = [
		0xfa, 0x3c, 0xc4, 0x08, 0x17, 0xa4, 0x61, 0xfa, 0xa3, 0x78,
		0x63, 0x58, 0xef, 0x1e, 0xe0, 0x92, 0xf8, 0xf4, 0xe3, 0xfc,
		0xb6, 0xf7, 0xa1, 0xa1, 0x90, 0xc6, 0x33, 0xf0, 0x49, 0x0a,
		0x64, 0x58, 0x56, 0x51, 0x72, 0x58, 0x94, 0xf6, 0xc5, 0xb3,
		0x0d, 0x08, 0x2d, 0xc5, 0x97, 0x99, 0xd5, 0x52, 0x8a, 0x2a,
		0x9d, 0xd4, 0x0d, 0x00, 0x06, 0xcd, 0x72, 0x39, 0x8c, 0x03,
		0xb2, 0xeb, 0x6a, 0xa4
	];
	let mut ct: [u8; 64] = [0; 64];
	let mut tag: [u8; 64] = [0; 64];

	lc_rust_aead_one(lcr_aead_type::lcr_aes_cbc_cshake256, &key, &iv, &pt,
			 &mut ct, &pt, &mut tag, &exp_ct, &exp_tag)
}

#[test]
fn lc_rust_chacha20_poly1305() {
	let aad: [u8; 12] = [
		0x50, 0x51, 0x52, 0x53, 0xc0, 0xc1, 0xc2, 0xc3,
		0xc4, 0xc5, 0xc6, 0xc7
	];
	let key: [u8; 32] = [
		0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
		0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
		0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
		0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f
	];
	let iv: [u8; 8] = [
		0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47
	];

	let pt: [u8; 114] = [
		0x4c, 0x61, 0x64, 0x69, 0x65, 0x73, 0x20, 0x61,
		0x6e, 0x64, 0x20, 0x47, 0x65, 0x6e, 0x74, 0x6c,
		0x65, 0x6d, 0x65, 0x6e, 0x20, 0x6f, 0x66, 0x20,
		0x74, 0x68, 0x65, 0x20, 0x63, 0x6c, 0x61, 0x73,
		0x73, 0x20, 0x6f, 0x66, 0x20, 0x27, 0x39, 0x39,
		0x3a, 0x20, 0x49, 0x66, 0x20, 0x49, 0x20, 0x63,
		0x6f, 0x75, 0x6c, 0x64, 0x20, 0x6f, 0x66, 0x66,
		0x65, 0x72, 0x20, 0x79, 0x6f, 0x75, 0x20, 0x6f,
		0x6e, 0x6c, 0x79, 0x20, 0x6f, 0x6e, 0x65, 0x20,
		0x74, 0x69, 0x70, 0x20, 0x66, 0x6f, 0x72, 0x20,
		0x74, 0x68, 0x65, 0x20, 0x66, 0x75, 0x74, 0x75,
		0x72, 0x65, 0x2c, 0x20, 0x73, 0x75, 0x6e, 0x73,
		0x63, 0x72, 0x65, 0x65, 0x6e, 0x20, 0x77, 0x6f,
		0x75, 0x6c, 0x64, 0x20, 0x62, 0x65, 0x20, 0x69,
		0x74, 0x2e
	];
	let exp_ct: [u8; 114] = [
		0xd3, 0x1a, 0x8d, 0x34, 0x64, 0x8e, 0x60, 0xdb,
		0x7b, 0x86, 0xaf, 0xbc, 0x53, 0xef, 0x7e, 0xc2,
		0xa4, 0xad, 0xed, 0x51, 0x29, 0x6e, 0x08, 0xfe,
		0xa9, 0xe2, 0xb5, 0xa7, 0x36, 0xee, 0x62, 0xd6,
		0x3d, 0xbe, 0xa4, 0x5e, 0x8c, 0xa9, 0x67, 0x12,
		0x82, 0xfa, 0xfb, 0x69, 0xda, 0x92, 0x72, 0x8b,
		0x1a, 0x71, 0xde, 0x0a, 0x9e, 0x06, 0x0b, 0x29,
		0x05, 0xd6, 0xa5, 0xb6, 0x7e, 0xcd, 0x3b, 0x36,
		0x92, 0xdd, 0xbd, 0x7f, 0x2d, 0x77, 0x8b, 0x8c,
		0x98, 0x03, 0xae, 0xe3, 0x28, 0x09, 0x1b, 0x58,
		0xfa, 0xb3, 0x24, 0xe4, 0xfa, 0xd6, 0x75, 0x94,
		0x55, 0x85, 0x80, 0x8b, 0x48, 0x31, 0xd7, 0xbc,
		0x3f, 0xf4, 0xde, 0xf0, 0x8e, 0x4b, 0x7a, 0x9d,
		0xe5, 0x76, 0xd2, 0x65, 0x86, 0xce, 0xc6, 0x4b,
		0x61, 0x16
	];
	let exp_tag: [u8; 16] = [
		0x1a, 0xe1, 0x0b, 0x59, 0x4f, 0x09, 0xe2, 0x6a,
		0x7e, 0x90, 0x2e, 0xcb, 0xd0, 0x60, 0x06, 0x91
	];
	let mut ct: [u8; 114] = [0; 114];
	let mut tag: [u8; 16] = [0; 16];

	lc_rust_aead_one(lcr_aead_type::lcr_chacha20_poly1305, &key, &iv, &pt,
			 &mut ct, &aad, &mut tag, &exp_ct, &exp_tag)
}
