```toml
[advisory]
id = "RUSTSEC-2025-0001"
package = "gix-worktree-state"
date = "2025-01-18"
url = "https://github.com/GitoxideLabs/gitoxide/security/advisories/GHSA-fqmf-w4xh-33rh"
references = [
   "https://github.com/advisories/GHSA-fqmf-w4xh-33rh",
   "https://nvd.nist.gov/vuln/detail/CVE-2025-22620",
]
cvss = "CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:N/I:H/A:N"
keywords = ["permissions"]
aliases = ["CVE-2025-22620", "GHSA-fqmf-w4xh-33rh"]
license = "CC0-1.0"

[affected]

[affected.functions]
"gix_worktree_state::checkout" = ["< 0.17.0"]

[versions]
patched = [">= 0.17.0"]
```

# gix-worktree-state nonexclusive checkout sets executable files world-writable

### Summary

`gix-worktree-state` specifies 0777 permissions when checking out executable files, intending that the umask will restrict them appropriately. But one of the strategies it uses to set permissions is not subject to the umask. This causes files in a repository to be world-writable in some situations.

### Details

Git repositories track executable bits for regular files. In tree objects and the index, regular file modes are stored as 0644 if not executable, or 0755 if executable. But this is independent of how the permissions are set in the filesystem (where supported).

[`gix_worktree_state::checkout`](https://github.com/GitoxideLabs/gitoxide/blob/8d84818240d44e1f5fe78a231b5d9bffd0283918/gix-worktree-state/src/checkout/function.rs#L8-L35) has two strategies for checking out a file and marking it executable on a Unix-like operating system, one of which is vulnerable:

- If the file is created by assuming it does not already exist, correct permissions are applied, because permissions specified when opening a file are subject to the umask.
- If the file is considered possibly already to exist—even in a clean checkout if the application does not specify the option to treat the destination directory as empty—then permissions conferring unrestricted access to any user account on the system are wrongly applied, because permissions specified when calling chmod on an existing file are not subject to the umask. 

Specifically, [`checkout::entry::checkout`](https://github.com/GitoxideLabs/gitoxide/blob/8d84818240d44e1f5fe78a231b5d9bffd0283918/gix-worktree-state/src/checkout/entry.rs#L56-L191) chooses the strategy for each file. The same strategy is usually chosen for each executable file, if no [process](https://github.com/git/git/blob/a60673e9252b08d4eca90543b3729f4798b9aafd/Documentation/RelNotes/2.11.0.txt#L149-L154) (i.e. [long running](https://github.com/GitoxideLabs/gitoxide/discussions/996)) smudge filter is in use. The strategy depends on the [`checkout::Options::destination_is_initially_empty`](https://github.com/GitoxideLabs/gitoxide/blob/8d84818240d44e1f5fe78a231b5d9bffd0283918/gix-worktree-state/src/checkout/mod.rs#L50-L53) value, which is passed along to [`checkout::entry::open_file`](https://github.com/GitoxideLabs/gitoxide/blob/8d84818240d44e1f5fe78a231b5d9bffd0283918/gix-worktree-state/src/checkout/entry.rs#L253-L277), whose return value includes a flag indicating whether permissions still need to be set:

- With `destination_is_initially_empty: true`, executable permissions are specified when opening the file, via [`OpenOptionsEx::mode`](https://doc.rust-lang.org/std/os/unix/fs/trait.OpenOptionsExt.html#tymethod.mode), by its effect on the behavior of [`OpenOptions::open`](https://doc.rust-lang.org/std/fs/struct.OpenOptions.html#method.open). A mode of 0777 is safe here, for the same reason the default mode of 0666 is safe. When creating a file, the applied mode is the specified mode with any bits unset from it that are set in the umask.

   <https://github.com/GitoxideLabs/gitoxide/blob/8d84818240d44e1f5fe78a231b5d9bffd0283918/gix-worktree-state/src/checkout/entry.rs#L265-L268>

  The `set_executable_after_creation` flag in the `open_file` return value is then `false`.

- With `destination_is_initially_empty: false`, executable permissions are set in a separate step, via [`PermissionsExt::set_mode`](https://doc.rust-lang.org/beta/std/os/unix/fs/trait.PermissionsExt.html#tymethod.set_mode) and [`set_permissions`](https://doc.rust-lang.org/beta/std/fs/fn.set_permissions.html). A mode of 0777 is not safe here, because the umask is not applied. The vulnerable code appears in [`checkout::entry::finalize_entry`](https://github.com/GitoxideLabs/gitoxide/blob/8d84818240d44e1f5fe78a231b5d9bffd0283918/gix-worktree-state/src/checkout/entry.rs#L279-L299), which receives the `set_executable_after_creation` flag originally from `open_file`:

  <https://github.com/GitoxideLabs/gitoxide/blob/8d84818240d44e1f5fe78a231b5d9bffd0283918/gix-worktree-state/src/checkout/entry.rs#L288-L293>

  The file has unrestricted permissions.

`finalize_entry` is [likewise called](https://github.com/GitoxideLabs/gitoxide/blob/8d84818240d44e1f5fe78a231b5d9bffd0283918/gix-worktree-state/src/checkout/chunk.rs#L229-L236) from [`checkout::chunk::process_delayed_filter_results`](https://github.com/GitoxideLabs/gitoxide/blob/8d84818240d44e1f5fe78a231b5d9bffd0283918/gix-worktree-state/src/checkout/chunk.rs#L157-L259).

### PoC

1. On a Unix-like system such as GNU/Linux or macOS, create a new project and define its dependencies. While the vulnerability is in `gix-worktree-state`, this example will use vulnerable code through the `gix` crate, which exposes it. Run:

   ```sh
   cargo new checkout-index
   cd checkout-index
   cargo add gix gix-object
   ```

2. In the `checkout-index` directory, edit `src/main.rs` so that its entire contents are:

   ```rust
   fn main() -> Result<(), Box<dyn std::error::Error>> {
       let repo = gix::discover("has-executable")?;
       let mut index = repo.open_index()?;
       gix::worktree::state::checkout(
           &mut index,
           repo.work_dir().ok_or("need non-bare repo")?,
           gix_object::find::Never, // Can also use: repo.objects.clone()
           &gix::progress::Discard,
           &gix::progress::Discard,
           &Default::default(),
           Default::default(),
       )?;
       Ok(())
   }
   ```

3. Create the test repository that the vulnerable program will operate on. Still in the `checkout-index` directory, run:

   ```sh
   git init has-executable
   touch has-executable/a has-executable/b
   chmod +x has-executable/b
   git -C has-executable add .
   ```

   It is not necessary to commit the changes, only to stage them, since the test program will check  out the index.

4. *Optionally*, run `rm has-executable/[ab]` to remove the staged files from disk.

5. Run the program by issuing `cargo run`. The program uses `gix-worktree-state` to check out the index. It should terminate successfully and not issue any errors.

6. Run `ls -l has-executable` to inspect the permissions of the checked out files. Observe that owner, group, and other all have read, write, and execute permissions on `b`.

   ```text
   -rw-r--r-- 1 ek ek 0 Jan  9 03:38 a
   -rwxrwxrwx 1 ek ek 0 Jan  9 03:38 b
   ```

   With affected versions of `gix-worktree-state`, the output shows `-rwxrwxrwx` for `b`, whether the files were removed in step 4 or not.

7. It was not necessary to set `destination_is_initially_empty` to `false` explicitly to trigger the bug, because that is its default value. If desired, modify the program to pass `true` and rerun the experiment to verify that `b` is no longer created with excessive permissions. The modified program would change the last `checkout` argument from `Default::default(),` to:

   ```rust
           gix::worktree::state::checkout::Options {
               destination_is_initially_empty: true,
               ..Default::default()
           },
   ```

### Impact

Setting unlimited file permissions is a problem on systems where a user account exists on the system that should not have the ability to access and modify the files. That applies to multi-user systems, or when an account is used to run software with reduced abilities. (Some programs may also treat broad write permissions to mean less validation is required.)

This bug affects Unix-like systems but not Windows. The `gix clone` command is not believed to be affected, due to [`checkout_exclusive`](https://github.com/GitoxideLabs/gitoxide/blob/af704f57bb9480c47cdd393465264d586f1d4562/gitoxide-core/src/index/checkout.rs#L14-L172)'s [use](https://github.com/GitoxideLabs/gitoxide/blob/af704f57bb9480c47cdd393465264d586f1d4562/gitoxide-core/src/index/checkout.rs#L61) of `destination_is_initially_empty: true`. Specialized uses in which repositories are known never to have any files marked executable are unaffected. Repositories that no untrusted users can access, due to not having the ability to traverse the directories to them or due to sufficiently restrictive ACLs, are likewise unaffected.

The default value of `destination_is_initially_empty` is `false`, so some applications may be affected even if they don't attempt checkouts in nonempty directories. The 0777 permissions are applied to files that are created earlier in the same checkout, as well as those that already existed, regardless of their prior permissions. On preexisting files, 0777 is set *even if [`overwrite_existing`](https://github.com/GitoxideLabs/gitoxide/blob/8d84818240d44e1f5fe78a231b5d9bffd0283918/gix-worktree-state/src/checkout/mod.rs#L54-L58) is `false`*, as that prevents the checkout from changing file contents but not permissions.

Files not tracked/staged as executable are not checked out with insecure permissions. Such a file that previously existed keeps its old permissions. However, this may include executable permissions that no longer match repository metadata, as well as undesired write permissions acquired from a previous vulnerable checkout. `set_mode(0o777)` clears other bits, so the bug is not exacerbated by the presence of setuid/setgid bits. In some applications, the vulnerable strategy may be used only for files rewritten by a [long running](https://git-scm.com/docs/gitattributes/2.40.0#_long_running_filter_process) smudge filter or only in the presence of [delays](https://git-scm.com/docs/gitattributes/2.40.0#_delay).
