package symtab

import (
	"bytes"
	"fmt"
	"reflect"
	"strconv"
	"strings"
	"unsafe"
)

// ProcMapPermissions contains permission settings read from `/proc/[pid]/maps`.
type ProcMapPermissions struct {
	// mapping has the [R]ead flag set
	Read bool
	// mapping has the [W]rite flag set
	Write bool
	// mapping has the [X]ecutable flag set
	Execute bool
	// mapping has the [S]hared flag set
	Shared bool
	// mapping is marked as [P]rivate (copy on write)
	Private bool
}

func (p *ProcMapPermissions) String() string {
	var res string
	if p.Read {
		res += "r"
	} else {
		res += "-"
	}
	if p.Write {
		res += "w"
	} else {
		res += "-"
	}
	if p.Execute {
		res += "x"
	} else {
		res += "-"
	}
	if p.Shared {
		res += "s"
	} else {
		res += "-"
	}
	if p.Private {
		res += "p"
	} else {
		res += "-"
	}
	return res
}

// ProcMap contains the process memory-mappings of the process
// read from `/proc/[pid]/maps`.
type ProcMap struct {
	// The start address of current mapping.
	StartAddr uint64
	// The end address of the current mapping
	EndAddr uint64
	// The permissions for this mapping
	Perms *ProcMapPermissions
	// The current offset into the file/fd (e.g., shared libs)
	Offset uint64
	// Device owner of this mapping (major:minor) in Mkdev format.
	Dev uint64
	// The inode of the device above
	Inode uint64
	// The file or psuedofile (or empty==anonymous)
	Pathname string
}

func (p *ProcMap) String() string {
	return fmt.Sprintf("%x-%x %s %x %x:%x %s",
		p.StartAddr, p.EndAddr, p.Perms.String(), p.Offset, p.Dev, p.Inode, p.Pathname)
}

type ProcMaps []*ProcMap

func (p ProcMaps) String() string {
	var sb strings.Builder
	for _, m := range p {
		sb.WriteString(m.String())
		sb.WriteString("\n")
	}
	return sb.String()
}

type file struct {
	dev   uint64
	inode uint64
	path  string
}

func (m *ProcMap) file() file {
	return file{
		dev:   m.Dev,
		inode: m.Inode,
		path:  m.Pathname,
	}
}

// parseDevice parses the device token of a line and converts it to a dev_t
// (mkdev) like structure.
func parseDevice(s []byte) (uint64, error) {
	i := bytes.IndexByte(s, ':')
	if i == -1 {
		return 0, fmt.Errorf("unexpected number of fields")
	}
	majorBytes := s[:i]
	minorBytes := s[i+1:]

	major, err := strconv.ParseUint(tokenToStringUnsafe(majorBytes), 16, 0)
	if err != nil {
		return 0, err
	}

	minor, err := strconv.ParseUint(tokenToStringUnsafe(minorBytes), 16, 0)
	if err != nil {
		return 0, err
	}

	return mkdev(uint32(major), uint32(minor)), nil
}

// mkdev returns a Linux device number generated from the given major and minor
// components.
// this is a copy-paste from unix.Mkdev
func mkdev(major, minor uint32) uint64 {
	dev := (uint64(major) & 0x00000fff) << 8
	dev |= (uint64(major) & 0xfffff000) << 32
	dev |= (uint64(minor) & 0x000000ff) << 0
	dev |= (uint64(minor) & 0xffffff00) << 12
	return dev
}

// parseAddress converts a hex-string to a uintptr.
func parseAddress(s []byte) (uint64, error) {
	a, err := strconv.ParseUint(tokenToStringUnsafe(s), 16, 0)
	if err != nil {
		return 0, err
	}

	return a, nil
}

// parseAddresses parses the start-end address.
func parseAddresses(s []byte) (uint64, uint64, error) {
	i := bytes.IndexByte(s, '-')
	if i == -1 {
		return 0, 0, fmt.Errorf("invalid address")
	}
	saddrBytes := s[:i]
	eaddrBytes := s[i+1:]

	saddr, err := parseAddress(saddrBytes)
	if err != nil {
		return 0, 0, err
	}

	eaddr, err := parseAddress(eaddrBytes)
	if err != nil {
		return 0, 0, err
	}

	return saddr, eaddr, nil
}

// parsePermissions parses a token and returns any that are set.
func parsePermissions(s []byte) (*ProcMapPermissions, error) {
	if len(s) < 4 {
		return nil, fmt.Errorf("invalid permissions token")
	}

	perms := ProcMapPermissions{}
	for _, ch := range s {
		switch ch {
		case 'r':
			perms.Read = true
		case 'w':
			perms.Write = true
		case 'x':
			perms.Execute = true
		case 'p':
			perms.Private = true
		case 's':
			perms.Shared = true
		}
	}

	return &perms, nil
}

// ParseProcMapLine will attempt to parse a single line within a proc/[pid]/maps
// buffer.
// 7f5822ebe000-7f5822ec0000 r--p 00000000 09:00 533429                     /usr/lib/x86_64-linux-gnu/ld-linux-x86-64.so.2
// returns nil if entry is not executable
func ParseProcMapLine(line []byte, executableOnly bool) (*ProcMap, error) {
	var i int
	if i = bytes.IndexByte(line, ' '); i == -1 {
		return nil, fmt.Errorf("invalid procmap entry: %s", line)
	}
	addresesBytes := line[:i]
	line = line[i+1:]

	if i = bytes.IndexByte(line, ' '); i == -1 {
		return nil, fmt.Errorf("invalid procmap entry: %s", line)
	}
	permsBytes := line[:i]
	line = line[i+1:]

	if i = bytes.IndexByte(line, ' '); i == -1 {
		return nil, fmt.Errorf("invalid procmap entry: %s", line)
	}
	offsetBytes := line[:i]
	line = line[i+1:]

	if i = bytes.IndexByte(line, ' '); i == -1 {
		return nil, fmt.Errorf("invalid procmap entry: %s", line)
	}
	deviceBytes := line[:i]
	line = line[i+1:]

	var inodeBytes []byte
	if i = bytes.IndexByte(line, ' '); i == -1 {
		inodeBytes = line
		line = nil
	} else {
		inodeBytes = line[:i]
		line = line[i+1:]
	}

	perms, err := parsePermissions(permsBytes)
	if err != nil {
		return nil, err
	}

	if executableOnly && !perms.Execute {
		return nil, nil
	}

	saddr, eaddr, err := parseAddresses(addresesBytes)
	if err != nil {
		return nil, err
	}

	offset, err := strconv.ParseUint(tokenToStringUnsafe(offsetBytes), 16, 0)
	if err != nil {
		return nil, err
	}

	device, err := parseDevice(deviceBytes)
	if err != nil {
		return nil, err
	}

	inode, err := strconv.ParseUint(tokenToStringUnsafe(inodeBytes), 10, 0)
	if err != nil {
		return nil, err
	}

	pathname := ""

	for len(line) > 0 && line[0] == ' ' {
		line = line[1:]
	}
	if len(line) > 0 {
		pathname = string(line)
	}

	return &ProcMap{
		StartAddr: saddr,
		EndAddr:   eaddr,
		Perms:     perms,
		Offset:    offset,
		Dev:       device,
		Inode:     inode,
		Pathname:  pathname,
	}, nil
}

func ParseProcMapsExecutableModules(procMaps []byte, executableOnly bool) ([]*ProcMap, error) {
	var modules []*ProcMap
	for len(procMaps) > 0 {
		nl := bytes.IndexByte(procMaps, '\n')
		var line []byte
		if nl == -1 {
			line = procMaps
			procMaps = nil
		} else {
			line = procMaps[:nl]
			procMaps = procMaps[nl+1:]
		}
		if len(line) == 0 {
			continue
		}
		m, err := ParseProcMapLine(line, executableOnly)
		if err != nil {
			return nil, err
		}
		if m == nil { // not executable
			continue
		}
		modules = append(modules, m)
	}
	return modules, nil
}

func tokenToStringUnsafe(tok []byte) string {
	res := ""
	sh := (*reflect.StringHeader)(unsafe.Pointer(&res))
	sh.Data = uintptr(unsafe.Pointer(&tok[0]))
	sh.Len = len(tok)
	return res
}

func FindLastRXMap(maps []*ProcMap) *ProcMap {
	for i := len(maps) - 1; i >= 0; i-- {
		m := maps[i]
		if m.Perms.Read && m.Perms.Execute {
			return m
		}
	}
	return nil
}

// FindReadableMap return a map entry that is readable and not writable or nil
func FindReadableMap(maps []*ProcMap) *ProcMap {
	var readable *ProcMap
	for _, m := range maps {
		if m.Perms.Read && !m.Perms.Write {
			readable = m
			break
		}
	}
	return readable
}
