import { init, layout, minBend, solve } from "./utils";
/**
 * Create a new {@link TopologicalOperator}, bundled as {@link coordTopological}.
 */
export function topological(...args) {
    if (args.length) {
        throw new Error(`got arguments to topological(${args}), but constructor takes no arguments.`);
    }
    function topologicalCall(layers, nodeSize) {
        for (const layer of layers) {
            const numNodes = layer.reduce((count, node) => count + +("node" in node.data), 0);
            if (numNodes !== 1) {
                throw new Error("topological() only works with a topological layering");
            }
        }
        const inds = new Map();
        let i = 0;
        for (const layer of layers) {
            for (const node of layer) {
                if ("link" in node.data) {
                    inds.set(node, i++);
                }
            }
        }
        // we assign all real nodes the last index, knowing that the optimization
        // always assigns them the same coord: 0.
        for (const layer of layers) {
            for (const node of layer) {
                if ("node" in node.data) {
                    inds.set(node, i);
                }
            }
        }
        const [Q, c, A, b] = init(layers, inds, nodeSize);
        for (const layer of layers) {
            for (const par of layer) {
                const pind = inds.get(par);
                for (const node of par.ichildren()) {
                    const nind = inds.get(node);
                    if ("link" in node.data) {
                        for (const child of node.ichildren()) {
                            const cind = inds.get(child);
                            minBend(Q, pind, nind, cind, 1);
                        }
                    }
                }
            }
        }
        const solution = solve(Q, c, A, b);
        const width = layout(layers, nodeSize, inds, solution);
        if (width <= 0) {
            throw new Error("must assign nonzero width to at least one node");
        }
        return width;
    }
    return topologicalCall;
}
