/**
 * A {@link LaneOperator} that assigns lanes greedily, but quickly.
 *
 * @packageDocumentation
 */
import { least, median } from "d3-array";
import { map, reverse } from "../../iters";
/**
 * an indexer that only assigns non-negatable indices
 *
 * @internal
 */
class OneSidedIndexer {
    constructor(uncompressed) {
        this.uncompressed = uncompressed;
        this.indices = [];
    }
    getIndex(after, target) {
        var _a;
        const free = [];
        for (const [i, val] of this.indices.entries()) {
            if (val <= after) {
                free.push(i);
            }
        }
        if (this.uncompressed) {
            free.push(this.indices.length);
        }
        const targ = target !== null && target !== void 0 ? target : 0;
        const ind = (_a = least(free, (v) => [Math.abs(targ - v), v])) !== null && _a !== void 0 ? _a : this.indices.length;
        this.setIndex(ind, after);
        return ind;
    }
    setIndex(index, upTo) {
        this.indices[index] = upTo;
    }
    offset() {
        return 0;
    }
}
/**
 * an indexer that will assign positive and negative indices
 *
 * @internal
 */
class TwoSidedIndexer {
    constructor(uncompressed) {
        this.uncompressed = uncompressed;
        // We use two arrays so that appends on either side are efficient
        this.posIndices = [0];
        this.negIndices = [];
    }
    /** the next negative index */
    nextNeg() {
        return -this.negIndices.length - 1;
    }
    /** the next positive index */
    nextPos() {
        return this.posIndices.length;
    }
    getIndex(after, target) {
        var _a;
        const free = [];
        for (const [i, val] of this.negIndices.entries()) {
            if (val <= after) {
                free.push(-i - 1);
            }
        }
        for (const [i, val] of this.posIndices.entries()) {
            if (val <= after) {
                free.push(i);
            }
        }
        if (this.uncompressed) {
            free.push(this.nextNeg());
            free.push(this.nextPos());
        }
        const targ = target !== null && target !== void 0 ? target : 0;
        const empty = this.negIndices.length < this.posIndices.length - 1
            ? this.nextNeg()
            : this.nextPos();
        // tie break on distance to target, then closest to zero, then positive
        const ind = (_a = least(free, (v) => [Math.abs(targ - v), Math.abs(v), -v])) !== null && _a !== void 0 ? _a : empty;
        this.setIndex(ind, after);
        return ind;
    }
    setIndex(index, upTo) {
        if (index < 0) {
            this.negIndices[-index - 1] = upTo;
        }
        else {
            this.posIndices[index] = upTo;
        }
    }
    offset() {
        return this.negIndices.length;
    }
}
/**
 * create a new indexer
 *
 * @internal
 */
function indexer(compressed, bidirectional) {
    if (bidirectional) {
        return new TwoSidedIndexer(!compressed);
    }
    else {
        return new OneSidedIndexer(!compressed);
    }
}
/**
 * assign lanes based on top down approach
 *
 * This method assigns lanes to each of nodes children as soon as it finds
 * them, giving the farthest away children lanes first.
 *
 * @internal
 */
function topDownOp(nodes, inds) {
    // set all xs to undefined so we know what we've seen before
    for (const node of nodes) {
        node.x = undefined;
    }
    // if node is new (no children) give it an arbitrary index
    for (const node of nodes) {
        if (node.x === undefined) {
            node.x = inds.getIndex(node.y);
        }
        // iterate over children from farthest away to closest, assign each a lane
        // in order, trying to be as close to their parent as possible
        for (const child of [...node.ichildren()].sort((a, b) => b.y - a.y)) {
            if (child.x === undefined) {
                child.x = inds.getIndex(node.y, node.x);
                inds.setIndex(child.x, child.y);
            }
        }
    }
    // update according to offset
    const offset = inds.offset();
    for (const node of nodes) {
        node.x += offset;
    }
}
/**
 * assign lanes based on bottom up approach
 *
 * This method assigns a nodes highest parent the same lane as it, and
 * otherwise defers from assigning lanes.
 *
 * @internal
 */
function bottomUpOp(nodes, inds) {
    var _a;
    // create map of a node to their highest parent, these we assign automatically
    const highestParent = new Map();
    for (const node of nodes) {
        node.x = undefined;
        for (const child of node.ichildren()) {
            const current = highestParent.get(child);
            if (current === undefined || node.y < current.y) {
                highestParent.set(child, node);
            }
        }
    }
    for (const node of reverse(nodes)) {
        // if node wasn't a highest parent, find it a lane
        if (node.x === undefined) {
            const target = median(map(node.ichildren(), (c) => c.x));
            // note we invert y because we're going bottom up
            node.x = inds.getIndex(nodes.length - node.y, target);
        }
        // if node has a highest parent, assign it to the same lane
        const par = highestParent.get(node);
        if (par !== undefined) {
            (_a = par.x) !== null && _a !== void 0 ? _a : (par.x = node.x);
            inds.setIndex(node.x, nodes.length - par.y);
        }
    }
    // adjust for offset
    const offset = inds.offset();
    for (const node of nodes) {
        node.x += offset;
    }
}
/** @internal */
function buildOperator(topDownVal, compressedVal, bidirectionalVal) {
    function greedyCall(ordered) {
        const inds = indexer(compressedVal, bidirectionalVal);
        if (topDownVal) {
            topDownOp(ordered, inds);
        }
        else {
            bottomUpOp(ordered, inds);
        }
    }
    function topDown(val) {
        if (val === undefined) {
            return topDownVal;
        }
        else {
            return buildOperator(val, compressedVal, bidirectionalVal);
        }
    }
    greedyCall.topDown = topDown;
    function compressed(val) {
        if (val === undefined) {
            return compressedVal;
        }
        else {
            return buildOperator(topDownVal, val, bidirectionalVal);
        }
    }
    greedyCall.compressed = compressed;
    function bidirectional(val) {
        if (val === undefined) {
            return bidirectionalVal;
        }
        else {
            return buildOperator(topDownVal, compressedVal, val);
        }
    }
    greedyCall.bidirectional = bidirectional;
    return greedyCall;
}
/**
 * Create a default {@link GreedyOperator}, bundled as {@link laneGreedy}.
 */
export function greedy(...args) {
    if (args.length) {
        throw new Error(`got arguments to greedy(${args}), but constructor takes no arguments.`);
    }
    return buildOperator(true, true, false);
}
