var __rest = (this && this.__rest) || function (s, e) {
    var t = {};
    for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p) && e.indexOf(p) < 0)
        t[p] = s[p];
    if (s != null && typeof Object.getOwnPropertySymbols === "function")
        for (var i = 0, p = Object.getOwnPropertySymbols(s); i < p.length; i++) {
            if (e.indexOf(p[i]) < 0 && Object.prototype.propertyIsEnumerable.call(s, p[i]))
                t[p[i]] = s[p[i]];
        }
    return t;
};
import { entries, every, filter, flatMap, map, reduce, some } from "../iters";
import { dfs, js, listMultimapPush, setMultimapAdd, setPop, } from "../utils";
import { getParents } from "./utils";
/**********************
 * Dag Implementation *
 **********************/
class LayoutChildLink {
    constructor(child, data, points = [], reversed = false) {
        this.child = child;
        this.data = data;
        this.points = points;
        this.reversed = reversed;
    }
}
/**
 * The concrete class backing the {@link Link} interface.
 */
class LayoutLink {
    constructor(source, target, data, points, reversed) {
        this.source = source;
        this.target = target;
        this.data = data;
        this.points = points;
        this.reversed = reversed;
    }
}
/**
 * The concrete implementation backing {@link Dag}.
 */
class LayoutDag {
    constructor(roots) {
        if (roots) {
            this.proots = roots;
        }
    }
    [Symbol.iterator]() {
        return this.idescendants()[Symbol.iterator]();
    }
    iroots() {
        return { [Symbol.iterator]: () => this.proots[Symbol.iterator]() };
    }
    roots() {
        return [...this.iroots()];
    }
    *idepth() {
        const ch = (node) => node.ichildren();
        yield* dfs(ch, ...this.iroots());
    }
    *ibreadth() {
        const seen = new Set();
        let next = this.roots();
        let current = [];
        do {
            current = next.reverse();
            next = [];
            let node;
            while ((node = current.pop())) {
                if (!seen.has(node)) {
                    seen.add(node);
                    yield node;
                    next.push(...node.ichildren());
                }
            }
        } while (next.length);
    }
    *ibefore() {
        const numBefore = new Map();
        for (const node of this) {
            for (const child of node.ichildren()) {
                numBefore.set(child, (numBefore.get(child) || 0) + 1);
            }
        }
        const queue = this.roots();
        let node;
        while ((node = queue.pop())) {
            yield node;
            for (const child of node.ichildren()) {
                const before = numBefore.get(child);
                if (before > 1) {
                    numBefore.set(child, before - 1);
                }
                else {
                    queue.push(child);
                }
            }
        }
    }
    *iafter() {
        const queue = this.roots();
        const seen = new Set();
        let node;
        while ((node = queue.pop())) {
            if (seen.has(node)) {
                // noop
            }
            else if (every(node.ichildren(), (c) => seen.has(c))) {
                seen.add(node);
                yield node;
            }
            else {
                // need to revisit after children
                queue.push(node, ...node.ichildren());
            }
        }
    }
    idescendants(style = "depth") {
        if (style === "depth") {
            return this.idepth();
        }
        else if (style === "breadth") {
            return this.ibreadth();
        }
        else if (style === "before") {
            return this.ibefore();
        }
        else if (style === "after") {
            return this.iafter();
        }
        else {
            throw new Error(`unknown iteration style: ${style}`);
        }
    }
    descendants(style = "depth") {
        return [...this.idescendants(style)];
    }
    ilinks() {
        return flatMap(this, (node) => node.ichildLinks());
    }
    links() {
        return [...this.ilinks()];
    }
    size() {
        return reduce(this, (s) => s + 1, 0);
    }
    sum(callback) {
        const descendantVals = new Map();
        for (const [index, node] of entries(this.idescendants("after"))) {
            const val = callback(node, index);
            const nodeVals = new Map();
            nodeVals.set(node, val);
            for (const child of node.ichildren()) {
                const childMap = descendantVals.get(child);
                for (const [child, v] of childMap.entries()) {
                    nodeVals.set(child, v);
                }
            }
            node.value = reduce(nodeVals.values(), (a, b) => a + b, 0);
            descendantVals.set(node, nodeVals);
        }
        return this;
    }
    count() {
        const leaves = new Map();
        for (const node of this.idescendants("after")) {
            if (node.ichildren()[Symbol.iterator]().next().done) {
                leaves.set(node, new Set([node]));
                node.value = 1;
            }
            else {
                const nodeLeaves = new Set();
                for (const child of node.ichildren()) {
                    const childLeaves = leaves.get(child);
                    for (const leaf of childLeaves) {
                        nodeLeaves.add(leaf);
                    }
                }
                leaves.set(node, nodeLeaves);
                node.value = nodeLeaves.size;
            }
        }
        return this;
    }
    height() {
        for (const node of this.idescendants("after")) {
            node.value = Math.max(0, ...map(node.ichildren(), (child) => child.value + 1));
        }
        return this;
    }
    depth() {
        var _a;
        const parents = getParents(this);
        for (const node of this.idescendants("before")) {
            node.value = Math.max(0, ...map((_a = parents.get(node)) !== null && _a !== void 0 ? _a : [], (par) => par.value + 1));
        }
        return this;
    }
    *isplit() {
        // create parents
        const parents = getParents(this);
        // "children" function that returns children and parents
        function* graph(node) {
            var _a;
            yield* node.ichildren();
            yield* (_a = parents.get(node)) !== null && _a !== void 0 ? _a : [];
        }
        // dfs over roots, tracing parents too
        const available = new Set(this.iroots());
        for (const root of this.iroots()) {
            if (!available.delete(root))
                continue;
            const connected = [root];
            for (const node of dfs(graph, root)) {
                if (available.delete(node)) {
                    connected.push(node);
                }
            }
            // yield all connected roots
            yield connected.length > 1 ? new LayoutDag(connected) : connected[0];
        }
    }
    split() {
        return [...this.isplit()];
    }
    connected() {
        const iter = this.isplit()[Symbol.iterator]();
        iter.next();
        const { done } = iter.next();
        return !!done;
    }
    multidag() {
        if (this.pmultidag === undefined) {
            return (this.pmultidag = some(this.iroots(), (n) => n.multidag()));
        }
        else {
            return this.pmultidag;
        }
    }
}
/**
 * Concrete implementation of {@link DagNode}.
 */
class LayoutDagNode extends LayoutDag {
    constructor(data) {
        super();
        this.data = data;
        this.dataChildren = [];
    }
    childrenCountsMap() {
        var _a;
        if (this.cachedChildrenCounts === undefined) {
            const cache = new Map();
            for (const { child } of this.dataChildren) {
                cache.set(child, ((_a = cache.get(child)) !== null && _a !== void 0 ? _a : 0) + 1);
            }
            return (this.cachedChildrenCounts =
                cache.size === this.dataChildren.length ? null : cache);
        }
        else {
            return this.cachedChildrenCounts;
        }
    }
    // NOTE everything extends from iroots, so by overriding this, we allow
    // dag nodes to work effectively
    iroots() {
        const singleton = [this];
        return { [Symbol.iterator]: () => singleton[Symbol.iterator]() };
    }
    nchildren() {
        var _a, _b;
        return (_b = (_a = this.childrenCountsMap()) === null || _a === void 0 ? void 0 : _a.size) !== null && _b !== void 0 ? _b : this.dataChildren.length;
    }
    nchildLinks() {
        return this.dataChildren.length;
    }
    nchildLinksTo(node) {
        var _a, _b;
        return (_b = (_a = this.childrenCountsMap()) === null || _a === void 0 ? void 0 : _a.get(node)) !== null && _b !== void 0 ? _b : 1;
    }
    *ichildren() {
        const cache = this.childrenCountsMap();
        if (cache === null) {
            for (const { child } of this.dataChildren) {
                yield child;
            }
        }
        else {
            yield* cache.keys();
        }
    }
    children() {
        return [...this.ichildren()];
    }
    *ichildrenCounts() {
        const cache = this.childrenCountsMap();
        if (cache === null) {
            for (const { child } of this.dataChildren) {
                yield [child, 1];
            }
        }
        else {
            yield* cache;
        }
    }
    childrenCounts() {
        return [...this.ichildrenCounts()];
    }
    *ichildLinks() {
        for (const { child, data, points, reversed } of this.dataChildren) {
            yield new LayoutLink(this, child, data, points, reversed);
        }
    }
    childLinks() {
        return [...this.ichildLinks()];
    }
    // NOTE these are simpler for a single node, so we override
    isplit() {
        return this.iroots();
    }
    split() {
        return this.roots();
    }
    connected() {
        return true;
    }
    // NOTE we need to override this, or we get an infinity loop
    multidag() {
        if (this.pmultidag === undefined) {
            return (this.pmultidag =
                this.childrenCountsMap() !== null ||
                    some(this.ichildren(), (child) => child.multidag()));
        }
        else {
            return this.pmultidag;
        }
    }
}
/**
 * Verify an ID is a valid ID.
 */
function verifyId(id) {
    if (typeof id !== "string") {
        throw new Error(`id is supposed to be string but got type ${typeof id}`);
    }
    return id;
}
/**
 * Verify a DAG is valid.
 *
 * @param checkForCycles - if true will check for all cycles, if false, will only
 * check for trivial self-loops
 */
function verifyDag(roots, checkForCycles) {
    if (!roots.length) {
        throw new Error("dag contained no roots; this often indicates a cycle");
    }
    // make sure there's no self loops
    for (const node of new LayoutDag(roots)) {
        for (const child of node.ichildren()) {
            if (child === node) {
                throw new Error(js `node '${node.data}' contained a self loop`);
            }
        }
    }
    // return early if no loops
    if (!checkForCycles)
        return;
    // test that dag is free of cycles
    // we attempt to take every unique path from each root and see if we ever see
    // a node again
    const seen = new Set(); // already processed
    const past = new Set(); // seen down this path
    let rec = null;
    /** visit nodes returning cycle if found */
    function visit(node) {
        if (seen.has(node)) {
            return [];
        }
        else if (past.has(node)) {
            rec = node;
            return [node];
        }
        else {
            past.add(node);
            let result = [];
            for (const child of node.ichildren()) {
                result = visit(child);
                if (result.length)
                    break;
            }
            past.delete(node);
            seen.add(node);
            if (result.length && rec !== null)
                result.push(node);
            if (rec === node)
                rec = null;
            return result;
        }
    }
    for (const root of roots) {
        const msg = visit(root);
        if (msg.length) {
            const cycle = msg
                .reverse()
                .map(({ data }) => js `'${data}'`)
                .join(" -> ");
            throw new Error(`dag contained a cycle: ${cycle}`);
        }
    }
}
function setPopUndef(elems) {
    return elems && setPop(elems);
}
/**
 * Remove cycles from a constrcuted dag by reversing some edges
 *
 * This uses the algorithm from [P Eades, X Lin, WF Smyth - Information
 * Processing Letters
 * [1993]](https://www.sciencedirect.com/science/article/pii/002001909390079O)
 */
function removeCycles(nodes) {
    var _a, _b, _c;
    // NOTE this doesn't use getParents, because we're still modifying the dag,
    // we want to make sure not to cache children set
    const parents = new Map();
    for (const node of nodes) {
        for (const { child } of node.dataChildren) {
            listMultimapPush(parents, child, node);
        }
    }
    const augmented = new Map(map(nodes, (node) => {
        var _a, _b;
        return [
            node,
            {
                indeg: (_b = (_a = parents.get(node)) === null || _a === void 0 ? void 0 : _a.length) !== null && _b !== void 0 ? _b : 0,
                outdeg: node.nchildLinks(),
                node,
            },
        ];
    }));
    const maxDelta = Math.max(...map(filter(augmented.values(), ({ indeg, outdeg }) => indeg > 0 && outdeg > 0), ({ indeg, outdeg }) => outdeg - indeg));
    if (maxDelta === -Infinity) {
        // all nodes were sources or sinks, so roots are just all sources, and there are no cycles
        const roots = [];
        for (const { indeg, node } of augmented.values()) {
            if (indeg === 0) {
                roots.push(node);
            }
        }
        return roots;
    }
    const sources = new Set();
    const sinks = new Set();
    const buckets = new Map();
    for (const aug of augmented.values()) {
        const { outdeg, indeg } = aug;
        if (indeg === 0) {
            sources.add(aug);
        }
        else if (outdeg === 0) {
            sinks.add(aug);
        }
        else {
            const delta = outdeg - indeg;
            setMultimapAdd(buckets, delta, aug);
        }
    }
    let minRank = 0;
    let maxRank = nodes.length;
    let delta = maxDelta;
    // nodes in topological order after removing cycles
    const ordered = new Array(nodes.length);
    while (minRank < maxRank) {
        const aug = (_b = (_a = setPop(sources)) !== null && _a !== void 0 ? _a : setPop(sinks)) !== null && _b !== void 0 ? _b : setPopUndef(buckets.get(delta));
        if (aug === undefined) {
            --delta;
        }
        else {
            const { node, indeg, outdeg } = aug;
            const rank = (aug.rank =
                indeg > 0 && outdeg === 0 ? --maxRank : minRank++);
            ordered[rank] = node;
            // process parents
            for (const par of (_c = parents.get(node)) !== null && _c !== void 0 ? _c : []) {
                const paug = augmented.get(par);
                if (paug.rank === undefined && paug.indeg > 0 && paug.outdeg > 0) {
                    // rank: not already ranked already ranked
                    // indeg: once a source always a source
                    // outdeg: unranked parents can't have 0 outdeg
                    const pdelta = paug.outdeg - paug.indeg;
                    buckets.get(pdelta).delete(paug);
                    --paug.outdeg;
                    if (paug.outdeg > 0) {
                        setMultimapAdd(buckets, pdelta - 1, paug);
                    }
                    else {
                        sinks.add(paug);
                    }
                }
            }
            // process children
            for (const { child } of node.dataChildren) {
                const caug = augmented.get(child);
                if (caug.rank !== undefined || caug.indeg === 0) {
                    // no op, already ranked or impossible for unranked children to have 0 in degree
                }
                else if (caug.outdeg === 0) {
                    // already a sink, removing could turn into a source
                    --caug.indeg;
                    if (caug.indeg === 0) {
                        // moved from sink to a source
                        sinks.delete(caug);
                        sources.add(caug);
                    }
                }
                else {
                    const cdelta = caug.outdeg - caug.indeg;
                    if (cdelta === delta) {
                        // added a new node at a higher delta, need to step back
                        ++delta;
                    }
                    buckets.get(cdelta).delete(caug);
                    --caug.indeg;
                    if (caug.indeg === 0) {
                        sources.add(caug);
                    }
                    else {
                        setMultimapAdd(buckets, cdelta + 1, caug);
                    }
                }
            }
        }
    }
    const roots = [];
    const hasParents = new Set();
    for (const [rank, node] of ordered.entries()) {
        // compute new children by reversing children with higher rank
        const newDataChildren = [];
        for (const link of node.dataChildren) {
            const caug = augmented.get(link.child);
            if (caug.rank < rank) {
                // reversing edge
                caug.node.dataChildren.push(new LayoutChildLink(node, link.data, link.points, true));
            }
            else {
                newDataChildren.push(link);
            }
        }
        // add node to roots if it hasn't been a child yet
        if (newDataChildren.length === node.nchildLinks() &&
            !hasParents.has(node)) {
            roots.push(node);
        }
        // update has parents for all true children
        for (const { child } of newDataChildren) {
            hasParents.add(child);
        }
        // actually update children
        node.dataChildren = newDataChildren;
    }
    return roots;
}
function getConnectRoots(nodes, hasParents) {
    const roots = [];
    for (const [id, node] of nodes.entries()) {
        if (!hasParents.has(id)) {
            roots.push(node);
        }
    }
    return roots;
}
function buildConnect(operators) {
    function connect(data) {
        if (!data.length) {
            throw new Error("can't connect empty data");
        }
        const nodes = new Map();
        const hasParents = new Set();
        for (const [i, datum] of data.entries()) {
            // create dag
            const source = verifyId(operators.sourceId(datum, i));
            let sourceNode = nodes.get(source);
            if (sourceNode === undefined) {
                sourceNode = new LayoutDagNode(operators.nodeDatum(source, i));
                nodes.set(source, sourceNode);
            }
            const target = verifyId(operators.targetId(datum, i));
            let targetNode = nodes.get(target);
            if (targetNode === undefined) {
                targetNode = new LayoutDagNode(operators.nodeDatum(target, i));
                nodes.set(target, targetNode);
            }
            if (source !== target || !operators.single) {
                sourceNode.dataChildren.push(new LayoutChildLink(targetNode, datum));
                hasParents.add(target);
            }
        }
        const roots = operators.decycle
            ? removeCycles([...nodes.values()])
            : getConnectRoots(nodes, hasParents);
        verifyDag(roots, !operators.decycle);
        return roots.length > 1 ? new LayoutDag(roots) : roots[0];
    }
    function sourceId(id) {
        if (id === undefined) {
            return operators.sourceId;
        }
        else {
            const { sourceId: _ } = operators, rest = __rest(operators, ["sourceId"]);
            return buildConnect(Object.assign(Object.assign({}, rest), { sourceId: id }));
        }
    }
    connect.sourceId = sourceId;
    function targetId(id) {
        if (id === undefined) {
            return operators.targetId;
        }
        else {
            const { targetId: _ } = operators, rest = __rest(operators, ["targetId"]);
            return buildConnect(Object.assign(Object.assign({}, rest), { targetId: id }));
        }
    }
    connect.targetId = targetId;
    function nodeDatum(id) {
        if (id === undefined) {
            return operators.nodeDatum;
        }
        else {
            const { nodeDatum: _ } = operators, rest = __rest(operators, ["nodeDatum"]);
            return buildConnect(Object.assign(Object.assign({}, rest), { nodeDatum: id }));
        }
    }
    connect.nodeDatum = nodeDatum;
    function single(val) {
        if (val === undefined) {
            return operators.single;
        }
        else {
            return buildConnect(Object.assign(Object.assign({}, operators), { single: val }));
        }
    }
    connect.single = single;
    function decycle(val) {
        if (val === undefined) {
            return operators.decycle;
        }
        else {
            return buildConnect(Object.assign(Object.assign({}, operators), { decycle: val }));
        }
    }
    connect.decycle = decycle;
    return connect;
}
function isZeroString(d) {
    try {
        return typeof d[0] === "string";
    }
    catch (_a) {
        return false;
    }
}
function defaultSourceId(d) {
    if (isZeroString(d)) {
        return d[0];
    }
    else {
        throw new Error(`default source id expected datum[0] to be a string but got datum: ${d}`);
    }
}
function isOneString(d) {
    try {
        return typeof d[1] === "string";
    }
    catch (_a) {
        return false;
    }
}
function defaultTargetId(d) {
    if (isOneString(d)) {
        return d[1];
    }
    else {
        throw new Error(`default target id expected datum[1] to be a string but got datum: ${d}`);
    }
}
function defaultNodeDatum(id) {
    return { id };
}
/**
 * Creates a new {@link ConnectOperator} with the default settings. This is
 * bundled as {@link dagConnect}
 */
export function connect(...args) {
    if (args.length) {
        throw new Error(`got arguments to connect(${args}), but constructor takes no arguments. ` +
            "These were probably meant as data which should be called as connect()(...)");
    }
    else {
        // NOTE I think because source and target are both IdOperators, typescript
        // tries to cache the inference, but in so doing, gets it wrong.
        return buildConnect({
            sourceId: defaultSourceId,
            targetId: defaultTargetId,
            nodeDatum: defaultNodeDatum,
            single: false,
            decycle: false,
        });
    }
}
function buildHierarchy(operators) {
    function hierarchy(...data) {
        var _a;
        if (!data.length) {
            throw new Error("must pass in at least one node");
        }
        const mapping = new Map();
        const queue = [];
        function nodify(datum) {
            let node = mapping.get(datum);
            if (node === undefined) {
                node = new LayoutDagNode(datum);
                mapping.set(datum, node);
                queue.push(node);
            }
            return node;
        }
        const roots = data.map(nodify);
        let node;
        let i = 0;
        while ((node = queue.pop())) {
            node.dataChildren = ((_a = operators.childrenData(node.data, i++)) !== null && _a !== void 0 ? _a : []).map(([childDatum, linkDatum]) => new LayoutChildLink(nodify(childDatum), linkDatum));
        }
        // verify roots are roots
        const rootSet = new Set(roots);
        for (const node of mapping.values()) {
            for (const { child } of node.dataChildren) {
                if (rootSet.delete(child) && operators.roots) {
                    throw new Error(js `node '${node.data}' pointed to a root`);
                }
            }
        }
        // NOTE if rootSet is empty then we have a cycle, but we defer to verifyDag
        // to get better printing
        const trueRoots = operators.decycle
            ? removeCycles([...mapping.values()])
            : rootSet.size
                ? [...rootSet]
                : roots;
        // create dag
        verifyDag(trueRoots, !operators.decycle);
        return trueRoots.length > 1 ? new LayoutDag(trueRoots) : trueRoots[0];
    }
    function children(childs) {
        if (childs === undefined) {
            return operators.children;
        }
        else {
            const { children: _, childrenData: __ } = operators, rest = __rest(operators, ["children", "childrenData"]);
            return buildHierarchy(Object.assign(Object.assign({}, rest), { children: childs, childrenData: wrapChildren(childs) }));
        }
    }
    hierarchy.children = children;
    function childrenData(data) {
        if (data === undefined) {
            return operators.childrenData;
        }
        else {
            const { children: _, childrenData: __ } = operators, rest = __rest(operators, ["children", "childrenData"]);
            return buildHierarchy(Object.assign(Object.assign({}, rest), { children: wrapChildrenData(data), childrenData: data }));
        }
    }
    hierarchy.childrenData = childrenData;
    function roots(val) {
        if (val === undefined) {
            return operators.roots;
        }
        else {
            return buildHierarchy(Object.assign(Object.assign({}, operators), { roots: val }));
        }
    }
    hierarchy.roots = roots;
    function decycle(val) {
        if (val === undefined) {
            return operators.decycle;
        }
        else {
            return buildHierarchy(Object.assign(Object.assign({}, operators), { decycle: val }));
        }
    }
    hierarchy.decycle = decycle;
    return hierarchy;
}
function wrapChildren(children) {
    function wrapped(d, i) {
        var _a;
        return ((_a = children(d, i)) !== null && _a !== void 0 ? _a : []).map((d) => [d, undefined]);
    }
    wrapped.wrapped = children;
    return wrapped;
}
function wrapChildrenData(childrenData) {
    function wrapped(d, i) {
        var _a;
        return ((_a = childrenData(d, i)) !== null && _a !== void 0 ? _a : []).map(([d]) => d);
    }
    wrapped.wrapped = childrenData;
    return wrapped;
}
function hasChildren(d) {
    try {
        const children = d.children;
        return children === undefined || children instanceof Array;
    }
    catch (_a) {
        return false;
    }
}
function defaultChildren(d) {
    if (hasChildren(d)) {
        return d.children;
    }
    else {
        throw new Error(js `default children function expected datum to have a children field but got: ${d}`);
    }
}
/**
 * Constructs a new {@link HierarchyOperator} with default settings. This is
 * bundled as {@link dagHierarchy}.
 */
export function hierarchy(...args) {
    if (args.length) {
        throw new Error(`got arguments to hierarchy(${args}), but constructor takes no arguments. ` +
            "These were probably meant as data which should be called as hierarchy()(...)");
    }
    else {
        return buildHierarchy({
            children: defaultChildren,
            childrenData: wrapChildren(defaultChildren),
            roots: true,
            decycle: false,
        });
    }
}
function buildStratify(operators) {
    function stratify(data) {
        var _a;
        if (!data.length)
            throw new Error("can't stratify empty data");
        const mapping = new Map();
        for (const [i, datum] of data.entries()) {
            const nid = verifyId(operators.id(datum, i));
            const pdata = (_a = operators.parentData(datum, i)) !== null && _a !== void 0 ? _a : [];
            const node = new LayoutDagNode(datum);
            if (mapping.has(nid)) {
                throw new Error(`found a duplicate id: ${id}`);
            }
            else {
                mapping.set(nid, [node, pdata]);
            }
        }
        const baseRoots = [];
        for (const [node, pdata] of mapping.values()) {
            for (const [pid, linkData] of pdata) {
                const info = mapping.get(pid);
                if (!info)
                    throw new Error(`missing id: ${pid}`);
                const [par] = info;
                par.dataChildren.push(new LayoutChildLink(node, linkData));
            }
            if (!pdata.length) {
                baseRoots.push(node);
            }
        }
        const roots = operators.decycle
            ? removeCycles([...map(mapping.values(), ([node]) => node)])
            : baseRoots;
        verifyDag(roots, !operators.decycle);
        return roots.length > 1 ? new LayoutDag(roots) : roots[0];
    }
    function id(op) {
        if (op === undefined) {
            return operators.id;
        }
        else {
            const { id: _ } = operators, rest = __rest(operators, ["id"]);
            return buildStratify(Object.assign(Object.assign({}, rest), { id: op }));
        }
    }
    stratify.id = id;
    function parentData(data) {
        if (data === undefined) {
            return operators.parentData;
        }
        else {
            const { parentIds: _, parentData: __ } = operators, rest = __rest(operators, ["parentIds", "parentData"]);
            return buildStratify(Object.assign(Object.assign({}, rest), { parentIds: wrapParentData(data), parentData: data }));
        }
    }
    stratify.parentData = parentData;
    function parentIds(ids) {
        if (ids === undefined) {
            return operators.parentIds;
        }
        else {
            const { parentIds: _, parentData: __ } = operators, rest = __rest(operators, ["parentIds", "parentData"]);
            return buildStratify(Object.assign(Object.assign({}, rest), { parentIds: ids, parentData: wrapParentIds(ids) }));
        }
    }
    stratify.parentIds = parentIds;
    function decycle(val) {
        if (val === undefined) {
            return operators.decycle;
        }
        else {
            return buildStratify(Object.assign(Object.assign({}, operators), { decycle: val }));
        }
    }
    stratify.decycle = decycle;
    return stratify;
}
function wrapParentIds(parentIds) {
    function wrapper(d, i) {
        var _a;
        return ((_a = parentIds(d, i)) !== null && _a !== void 0 ? _a : []).map((id) => [id, undefined]);
    }
    wrapper.wrapped = parentIds;
    return wrapper;
}
function wrapParentData(parentData) {
    function wrapper(d, i) {
        var _a;
        return ((_a = parentData(d, i)) !== null && _a !== void 0 ? _a : []).map(([id]) => id);
    }
    wrapper.wrapped = parentData;
    return wrapper;
}
function hasId(d) {
    try {
        return typeof d.id === "string";
    }
    catch (_a) {
        return false;
    }
}
function defaultId(data) {
    if (hasId(data)) {
        return data.id;
    }
    else {
        throw new Error(js `default id function expected datum to have an id field but got '${data}'`);
    }
}
function hasParentIds(d) {
    try {
        const parentIds = d.parentIds;
        return (parentIds === undefined ||
            (parentIds instanceof Array &&
                parentIds.every((id) => typeof id === "string")));
    }
    catch (_a) {
        return false;
    }
}
function defaultParentIds(d) {
    if (hasParentIds(d)) {
        return d.parentIds;
    }
    else {
        throw new Error(`default parentIds function expected datum to have a parentIds field but got: ${d}`);
    }
}
/**
 * Constructs a new {@link StratifyOperator} with the default settings. This is
 * bundled as {@link dagStratify}.
 */
export function stratify(...args) {
    if (args.length) {
        throw new Error(`got arguments to stratify(${args}), but constructor takes no arguments. ` +
            "These were probably meant as data which should be called as stratify()(...)");
    }
    else {
        return buildStratify({
            id: defaultId,
            parentIds: defaultParentIds,
            parentData: wrapParentIds(defaultParentIds),
            decycle: false,
        });
    }
}
