defmodule TrentoWeb.ConnCase do
  @moduledoc """
  This module defines the test case to be used by
  tests that require setting up a connection.

  Such tests rely on `Phoenix.ConnTest` and also
  import other functionality to make it easier
  to build common data structures and query the data layer.

  Finally, if the test case interacts with the database,
  we enable the SQL sandbox, so changes done to the database
  are reverted at the end of every test. If you are using
  PostgreSQL, you can even run database tests asynchronously
  by setting `use TrentoWeb.ConnCase, async: true`, although
  this option is not recommended for other databases.
  """

  use ExUnit.CaseTemplate

  use Trento.TaskCase

  alias Ecto.Adapters.SQL.Sandbox

  using do
    quote do
      # Import conveniences for testing with connections

      @endpoint TrentoWeb.Endpoint
      use TrentoWeb, :verified_routes
      import Plug.Conn
      import Phoenix.ConnTest
      import TrentoWeb.ConnCase

      # The default endpoint for testing
      @endpoint TrentoWeb.Endpoint

      use TrentoWeb, :verified_routes
    end
  end

  setup tags do
    pid = Sandbox.start_owner!(Trento.Repo, shared: not tags[:async])

    on_exit(fn ->
      wait_for_tasks_completion()

      Sandbox.stop_owner(pid)
    end)

    {:ok, conn: Phoenix.ConnTest.build_conn()}
  end
end
