defmodule Faker.Aws.En do
  import Faker, only: [sampler: 2]

  @moduledoc """
  Functions for generating AWS information in English
  """

  @doc """
  Returns a random region name available on AWS

  ## Examples

      iex> Faker.Aws.En.region_name()
      "Asia Pacific (Tokyo)"
      iex> Faker.Aws.En.region_name()
      "US East (Ohio)"
      iex> Faker.Aws.En.region_name()
      "Europe (Milan)"
      iex> Faker.Aws.En.region_name()
      "Africa (Cape Town)"
  """
  @spec region_name() :: String.t()
  sampler(:region_name, [
    "US East (N. Virginia)",
    "US East (Ohio)",
    "US West (N. California)",
    "US West (Oregon)",
    "Africa (Cape Town)",
    "Asia Pacific (Hong Kong)",
    "Asia Pacific (Mumbai)",
    "Asia Pacific (Seoul)",
    "Asia Pacific (Singapore)",
    "Asia Pacific (Sydney)",
    "Asia Pacific (Tokyo)",
    "Canada (Central)",
    "Europe (Frankfurt)",
    "Europe (Ireland)",
    "Europe (London)",
    "Europe (Milan)",
    "Europe (Paris)",
    "Europe (Stockholm)",
    "Middle East (Bahrain)",
    "South America (São Paulo)"
  ])

  @doc """
  Returns a random region code available on AWS

  ## Examples

      iex> Faker.Aws.En.region_code()
      "ap-northeast-1"
      iex> Faker.Aws.En.region_code()
      "us-east-2"
      iex> Faker.Aws.En.region_code()
      "eu-south-1"
      iex> Faker.Aws.En.region_code()
      "af-south-1"
  """
  @spec region_code() :: String.t()
  sampler(:region_code, [
    "us-east-1",
    "us-east-2",
    "us-west-1",
    "us-west-2",
    "af-south-1",
    "ap-east-1",
    "ap-south-1",
    "ap-northeast-2",
    "ap-southeast-1",
    "ap-southeast-2",
    "ap-northeast-1",
    "ca-central-1",
    "eu-central-1",
    "eu-west-1",
    "eu-west-2",
    "eu-south-1",
    "eu-west-3",
    "eu-north-1",
    "me-south-1",
    "sa-east-1"
  ])

  @doc """
  Returns a random AWS service

  ## Examples

    iex> Faker.Aws.En.service()
    "AWS Compute Optimizer"
    iex> Faker.Aws.En.service()
    "Ground Station"
    iex> Faker.Aws.En.service()
    "Neptune"
    iex> Faker.Aws.En.service()
    "DataSync"
  """
  @spec service() :: String.t()
  sampler(:service, [
    "EC2",
    "Lightsail",
    "Lambda",
    "Batch",
    "Elastic Beanstalk",
    "Serverless Application Repository",
    "AWS Outposts",
    "EC2 Image Builder",
    "S3",
    "EFS",
    "FSx",
    "S3 Glacier",
    "Storage Gateway",
    "AWS Backup",
    "RDS",
    "DynamoDB",
    "ElastiCache",
    "Neptune",
    "Amazon Redshift",
    "Amazon QLDB",
    "Amazon DocumentDB",
    "Amazon Keyspaces",
    "AWS Migration Hub",
    "Application Discovery Service",
    "Database Migration Service",
    "Server Migration Service",
    "AWS Transfer Family",
    "AWS Snow Family",
    "DataSync",
    "VPC",
    "CloudFront",
    "Route 53",
    "API Gateway",
    "Direct Connect",
    "AWS App Mesh",
    "AWS Cloud Map",
    "Global Accelerator",
    "CodeStar",
    "CodeCommit",
    "CodeArtifact",
    "CodeBuild",
    "CodeDeploy",
    "CodePipeline",
    "Cloud9",
    "X-Ray",
    "AWS RoboMaker",
    "AWS IQ",
    "Support",
    "Managed Services",
    "Amazon Managed Blockchain",
    "Ground Station",
    "Amazon Braket",
    "AWS Organizations",
    "CloudWatch",
    "AWS Auto Scaling",
    "CloudFormation",
    "CloudTrail",
    "Config",
    "OpsWorks",
    "Service Catalog",
    "Systems Manager",
    "AWS AppConfig",
    "Trusted Advisor",
    "Control Tower",
    "AWS License Manager",
    "AWS Well-Architected Tool",
    "Personal Health Dashboard",
    "AWS Chatbot",
    "Launch Wizard",
    "AWS Compute Optimizer",
    "Kinesis Video Streams",
    "MediaConnect",
    "MediaConvert",
    "MediaLive",
    "MediaPackage",
    "MediaStore",
    "MediaTailor",
    "Elemental Appliances & Software",
    "Amazon Interactive Video Service",
    "Elastic Transcoder",
    "Amazon SageMaker",
    "Amazon Augmented AI",
    "Amazon CodeGuru",
    "Amazon Comprehend",
    "Amazon Forecast",
    "Amazon Fraud Detector",
    "Amazon Kendra",
    "Amazon Lex",
    "Amazon Personalize",
    "Amazon Polly",
    "Amazon Rekognition",
    "Amazon Textract",
    "Amazon Transcribe",
    "Amazon Translate",
    "AWS DeepComposer",
    "AWS DeepLens",
    "AWS DeepRacer",
    "Athena",
    "EMR",
    "CloudSearch",
    "Elasticsearch Service",
    "Kinesis",
    "QuickSight",
    "Data Pipeline",
    "AWS Data Exchange",
    "AWS Glue",
    "AWS Lake Formation",
    "MSK",
    "IAM",
    "Resource Access Manager",
    "Cognito",
    "Secrets Manager",
    "GuardDuty",
    "Inspector",
    "Amazon Macie",
    "AWS Single Sign-On",
    "Certificate Manager",
    "Key Management Service",
    "CloudHSM",
    "Directory Service",
    "WAF & Shield",
    "AWS Firewall Manager",
    "Artifact",
    "Security Hub",
    "Detective",
    "AWS Amplify",
    "Mobile Hub",
    "AWS AppSync",
    "Device Farm",
    "Amazon Sumerian",
    "Step Functions",
    "Amazon AppFlow",
    "Amazon EventBridge",
    "Amazon MQ",
    "Simple Notification Service",
    "Simple Queue Service",
    "SWF",
    "AWS Cost Explorer",
    "AWS Budgets",
    "AWS Marketplace Subscriptions",
    "Amazon Connect",
    "Pinpoint",
    "Simple Email Service"
  ])

  @doc """
  Returns an AWS S3 Action

  ## Example

    iex> Faker.Aws.En.s3_action()
    "DeleteBucketTagging"
    iex> Faker.Aws.En.s3_action()
    "DeleteObjects"
    iex> Faker.Aws.En.s3_action()
    "PutPublicAccessBlock"
    iex> Faker.Aws.En.s3_action()
    "PutBucketReplication"
  """
  @spec s3_action() :: String.t()
  sampler(:s3_action, [
    "AbortMultipartUpload",
    "CompleteMultipartUpload",
    "CopyObject",
    "CreateBucket",
    "CreateMultipartUpload",
    "DeleteBucket",
    "DeleteBucketAnalyticsConfiguration",
    "DeleteBucketCors",
    "DeleteBucketEncryption",
    "DeleteBucketInventoryConfiguration",
    "DeleteBucketLifecycle",
    "DeleteBucketMetricsConfiguration",
    "DeleteBucketPolicy",
    "DeleteBucketReplication",
    "DeleteBucketTagging",
    "DeleteBucketWebsite",
    "DeleteObject",
    "DeleteObjects",
    "DeleteObjectTagging",
    "DeletePublicAccessBlock",
    "GetBucketAccelerateConfiguration",
    "GetBucketAcl",
    "GetBucketAnalyticsConfiguration",
    "GetBucketCors",
    "GetBucketEncryption",
    "GetBucketInventoryConfiguration",
    "GetBucketLifecycle",
    "GetBucketLifecycleConfiguration",
    "GetBucketLocation",
    "GetBucketLogging",
    "GetBucketMetricsConfiguration",
    "GetBucketNotification",
    "GetBucketNotificationConfiguration",
    "GetBucketPolicy",
    "GetBucketPolicyStatus",
    "GetBucketReplication",
    "GetBucketRequestPayment",
    "GetBucketTagging",
    "GetBucketVersioning",
    "GetBucketWebsite",
    "GetObject",
    "GetObjectAcl",
    "GetObjectLegalHold",
    "GetObjectLockConfiguration",
    "GetObjectRetention",
    "GetObjectTagging",
    "GetObjectTorrent",
    "GetPublicAccessBlock",
    "HeadBucket",
    "HeadObject",
    "ListBucketAnalyticsConfigurations",
    "ListBucketInventoryConfigurations",
    "ListBucketMetricsConfigurations",
    "ListBuckets",
    "ListMultipartUploads",
    "ListObjects",
    "ListObjectsV2",
    "ListObjectVersions",
    "ListParts",
    "PutBucketAccelerateConfiguration",
    "PutBucketAcl",
    "PutBucketAnalyticsConfiguration",
    "PutBucketCors",
    "PutBucketEncryption",
    "PutBucketInventoryConfiguration",
    "PutBucketLifecycle",
    "PutBucketLifecycleConfiguration",
    "PutBucketLogging",
    "PutBucketMetricsConfiguration",
    "PutBucketNotification",
    "PutBucketNotificationConfiguration",
    "PutBucketPolicy",
    "PutBucketReplication",
    "PutBucketRequestPayment",
    "PutBucketTagging",
    "PutBucketVersioning",
    "PutBucketWebsite",
    "PutObject",
    "PutObjectAcl",
    "PutObjectLegalHold",
    "PutObjectLockConfiguration",
    "PutObjectRetention",
    "PutObjectTagging",
    "PutPublicAccessBlock",
    "RestoreObject",
    "SelectObjectContent",
    "UploadPart",
    "UploadPartCopy"
  ])

  @doc """
  Returns an AWS RDS Action

  ## Example

    iex> Faker.Aws.En.rds_action()
    "DeleteDBClusterEndpoint"
    iex> Faker.Aws.En.rds_action()
    "CopyDBSnapshot"
    iex> Faker.Aws.En.rds_action()
    "ModifyDBParameterGroup"
    iex> Faker.Aws.En.rds_action()
    "DescribeDBClusterSnapshots"
  """
  @spec rds_action() :: String.t()
  sampler(:rds_action, [
    "AddRoleToDBCluster",
    "AddRoleToDBInstance",
    "AddSourceIdentifierToSubscription",
    "AddTagsToResource",
    "ApplyPendingMaintenanceAction",
    "AuthorizeDBSecurityGroupIngress",
    "BacktrackDBCluster",
    "CancelExportTask",
    "CopyDBClusterParameterGroup",
    "CopyDBClusterSnapshot",
    "CopyDBParameterGroup",
    "CopyDBSnapshot",
    "CopyOptionGroup",
    "CreateCustomAvailabilityZone",
    "CreateDBCluster",
    "CreateDBClusterEndpoint",
    "CreateDBClusterParameterGroup",
    "CreateDBClusterSnapshot",
    "CreateDBInstance",
    "CreateDBInstanceReadReplica",
    "CreateDBParameterGroup",
    "CreateDBProxy",
    "CreateDBSecurityGroup",
    "CreateDBSnapshot",
    "CreateDBSubnetGroup",
    "CreateEventSubscription",
    "CreateGlobalCluster",
    "CreateOptionGroup",
    "DeleteCustomAvailabilityZone",
    "DeleteDBCluster",
    "DeleteDBClusterEndpoint",
    "DeleteDBClusterParameterGroup",
    "DeleteDBClusterSnapshot",
    "DeleteDBInstance",
    "DeleteDBInstanceAutomatedBackup",
    "DeleteDBParameterGroup",
    "DeleteDBProxy",
    "DeleteDBSecurityGroup",
    "DeleteDBSnapshot",
    "DeleteDBSubnetGroup",
    "DeleteEventSubscription",
    "DeleteGlobalCluster",
    "DeleteInstallationMedia",
    "DeleteOptionGroup",
    "DeregisterDBProxyTargets",
    "DescribeAccountAttributes",
    "DescribeCertificates",
    "DescribeCustomAvailabilityZones",
    "DescribeDBClusterBacktracks",
    "DescribeDBClusterEndpoints",
    "DescribeDBClusterParameterGroups",
    "DescribeDBClusterParameters",
    "DescribeDBClusters",
    "DescribeDBClusterSnapshotAttributes",
    "DescribeDBClusterSnapshots",
    "DescribeDBEngineVersions",
    "DescribeDBInstanceAutomatedBackups",
    "DescribeDBInstances",
    "DescribeDBLogFiles",
    "DescribeDBParameterGroups",
    "DescribeDBParameters",
    "DescribeDBProxies",
    "DescribeDBProxyTargetGroups",
    "DescribeDBProxyTargets",
    "DescribeDBSecurityGroups",
    "DescribeDBSnapshotAttributes",
    "DescribeDBSnapshots",
    "DescribeDBSubnetGroups",
    "DescribeEngineDefaultClusterParameters",
    "DescribeEngineDefaultParameters",
    "DescribeEventCategories",
    "DescribeEvents",
    "DescribeEventSubscriptions",
    "DescribeExportTasks",
    "DescribeGlobalClusters",
    "DescribeInstallationMedia",
    "DescribeOptionGroupOptions",
    "DescribeOptionGroups",
    "DescribeOrderableDBInstanceOptions",
    "DescribePendingMaintenanceActions",
    "DescribeReservedDBInstances",
    "DescribeReservedDBInstancesOfferings",
    "DescribeSourceRegions",
    "DescribeValidDBInstanceModifications",
    "DownloadDBLogFilePortion",
    "FailoverDBCluster",
    "ImportInstallationMedia",
    "ListTagsForResource",
    "ModifyCertificates",
    "ModifyCurrentDBClusterCapacity",
    "ModifyDBCluster",
    "ModifyDBClusterEndpoint",
    "ModifyDBClusterParameterGroup",
    "ModifyDBClusterSnapshotAttribute",
    "ModifyDBInstance",
    "ModifyDBParameterGroup",
    "ModifyDBProxy",
    "ModifyDBProxyTargetGroup",
    "ModifyDBSnapshot",
    "ModifyDBSnapshotAttribute",
    "ModifyDBSubnetGroup",
    "ModifyEventSubscription",
    "ModifyGlobalCluster",
    "ModifyOptionGroup",
    "PromoteReadReplica",
    "PromoteReadReplicaDBCluster",
    "PurchaseReservedDBInstancesOffering",
    "RebootDBInstance",
    "RegisterDBProxyTargets",
    "RemoveFromGlobalCluster",
    "RemoveRoleFromDBCluster",
    "RemoveRoleFromDBInstance",
    "RemoveSourceIdentifierFromSubscription",
    "RemoveTagsFromResource",
    "ResetDBClusterParameterGroup",
    "ResetDBParameterGroup",
    "RestoreDBClusterFromS3",
    "RestoreDBClusterFromSnapshot",
    "RestoreDBClusterToPointInTime",
    "RestoreDBInstanceFromDBSnapshot",
    "RestoreDBInstanceFromS3",
    "RestoreDBInstanceToPointInTime",
    "RevokeDBSecurityGroupIngress",
    "StartActivityStream",
    "StartDBCluster",
    "StartDBInstance",
    "StartExportTask",
    "StopActivityStream",
    "StopDBCluster",
    "StopDBInstance"
  ])

  @doc """
  Returns an AWS EC2 Action

  ## Example

    iex> Faker.Aws.En.ec2_action()
    "CreateVpcEndpoint"
    iex> Faker.Aws.En.ec2_action()
    "RevokeSecurityGroupEgress"
    iex> Faker.Aws.En.ec2_action()
    "GetTransitGatewayRouteTableAssociations"
    iex> Faker.Aws.En.ec2_action()
    "RunScheduledInstances"
  """
  @spec ec2_action() :: String.t()
  sampler(:ec2_action, [
    "AcceptReservedInstancesExchangeQuote",
    "AcceptTransitGatewayPeeringAttachment",
    "AcceptTransitGatewayVpcAttachment",
    "AcceptVpcEndpointConnections",
    "AcceptVpcPeeringConnection",
    "AdvertiseByoipCidr",
    "AllocateAddress",
    "AllocateHosts",
    "ApplySecurityGroupsToClientVpnTargetNetwork",
    "AssignIpv6Addresses",
    "AssignPrivateIpAddresses",
    "AssociateAddress",
    "AssociateClientVpnTargetNetwork",
    "AssociateDhcpOptions",
    "AssociateIamInstanceProfile",
    "AssociateRouteTable",
    "AssociateSubnetCidrBlock",
    "AssociateTransitGatewayMulticastDomain",
    "AssociateTransitGatewayRouteTable",
    "AssociateVpcCidrBlock",
    "AttachClassicLinkVpc",
    "AttachInternetGateway",
    "AttachNetworkInterface",
    "AttachVolume",
    "AttachVpnGateway",
    "AuthorizeClientVpnIngress",
    "AuthorizeSecurityGroupEgress",
    "AuthorizeSecurityGroupIngress",
    "BundleInstance",
    "CancelBundleTask",
    "CancelCapacityReservation",
    "CancelConversionTask",
    "CancelExportTask",
    "CancelImportTask",
    "CancelReservedInstancesListing",
    "CancelSpotFleetRequests",
    "CancelSpotInstanceRequests",
    "ConfirmProductInstance",
    "CopyFpgaImage",
    "CopyImage",
    "CopySnapshot",
    "CreateCapacityReservation",
    "CreateCarrierGateway",
    "CreateClientVpnEndpoint",
    "CreateClientVpnRoute",
    "CreateCustomerGateway",
    "CreateDefaultSubnet",
    "CreateDefaultVpc",
    "CreateDhcpOptions",
    "CreateEgressOnlyInternetGateway",
    "CreateFleet",
    "CreateFlowLogs",
    "CreateFpgaImage",
    "CreateImage",
    "CreateInstanceExportTask",
    "CreateInternetGateway",
    "CreateKeyPair",
    "CreateLaunchTemplate",
    "CreateLaunchTemplateVersion",
    "CreateLocalGatewayRoute",
    "CreateLocalGatewayRouteTableVpcAssociation",
    "CreateManagedPrefixList",
    "CreateNatGateway",
    "CreateNetworkAcl",
    "CreateNetworkAclEntry",
    "CreateNetworkInterface",
    "CreateNetworkInterfacePermission",
    "CreatePlacementGroup",
    "CreateReservedInstancesListing",
    "CreateRoute",
    "CreateRouteTable",
    "CreateSecurityGroup",
    "CreateSnapshot",
    "CreateSnapshots",
    "CreateSpotDatafeedSubscription",
    "CreateSubnet",
    "CreateTags",
    "CreateTrafficMirrorFilter",
    "CreateTrafficMirrorFilterRule",
    "CreateTrafficMirrorSession",
    "CreateTrafficMirrorTarget",
    "CreateTransitGateway",
    "CreateTransitGatewayMulticastDomain",
    "CreateTransitGatewayPeeringAttachment",
    "CreateTransitGatewayPrefixListReference",
    "CreateTransitGatewayRoute",
    "CreateTransitGatewayRouteTable",
    "CreateTransitGatewayVpcAttachment",
    "CreateVolume",
    "CreateVpc",
    "CreateVpcEndpoint",
    "CreateVpcEndpointConnectionNotification",
    "CreateVpcEndpointServiceConfiguration",
    "CreateVpcPeeringConnection",
    "CreateVpnConnection",
    "CreateVpnConnectionRoute",
    "CreateVpnGateway",
    "DeleteCarrierGateway",
    "DeleteClientVpnEndpoint",
    "DeleteClientVpnRoute",
    "DeleteCustomerGateway",
    "DeleteDhcpOptions",
    "DeleteEgressOnlyInternetGateway",
    "DeleteFleets",
    "DeleteFlowLogs",
    "DeleteFpgaImage",
    "DeleteInternetGateway",
    "DeleteKeyPair",
    "DeleteLaunchTemplate",
    "DeleteLaunchTemplateVersions",
    "DeleteLocalGatewayRoute",
    "DeleteLocalGatewayRouteTableVpcAssociation",
    "DeleteManagedPrefixList",
    "DeleteNatGateway",
    "DeleteNetworkAcl",
    "DeleteNetworkAclEntry",
    "DeleteNetworkInterface",
    "DeleteNetworkInterfacePermission",
    "DeletePlacementGroup",
    "DeleteQueuedReservedInstances",
    "DeleteRoute",
    "DeleteRouteTable",
    "DeleteSecurityGroup",
    "DeleteSnapshot",
    "DeleteSpotDatafeedSubscription",
    "DeleteSubnet",
    "DeleteTags",
    "DeleteTrafficMirrorFilter",
    "DeleteTrafficMirrorFilterRule",
    "DeleteTrafficMirrorSession",
    "DeleteTrafficMirrorTarget",
    "DeleteTransitGateway",
    "DeleteTransitGatewayMulticastDomain",
    "DeleteTransitGatewayPeeringAttachment",
    "DeleteTransitGatewayPrefixListReference",
    "DeleteTransitGatewayRoute",
    "DeleteTransitGatewayRouteTable",
    "DeleteTransitGatewayVpcAttachment",
    "DeleteVolume",
    "DeleteVpc",
    "DeleteVpcEndpointConnectionNotifications",
    "DeleteVpcEndpoints",
    "DeleteVpcEndpointServiceConfigurations",
    "DeleteVpcPeeringConnection",
    "DeleteVpnConnection",
    "DeleteVpnConnectionRoute",
    "DeleteVpnGateway",
    "DeprovisionByoipCidr",
    "DeregisterImage",
    "DeregisterInstanceEventNotificationAttributes",
    "DeregisterTransitGatewayMulticastGroupMembers",
    "DeregisterTransitGatewayMulticastGroupSources",
    "DescribeAccountAttributes",
    "DescribeAddresses",
    "DescribeAggregateIdFormat",
    "DescribeAvailabilityZones",
    "DescribeBundleTasks",
    "DescribeByoipCidrs",
    "DescribeCapacityReservations",
    "DescribeCarrierGateways",
    "DescribeClassicLinkInstances",
    "DescribeClientVpnAuthorizationRules",
    "DescribeClientVpnConnections",
    "DescribeClientVpnEndpoints",
    "DescribeClientVpnRoutes",
    "DescribeClientVpnTargetNetworks",
    "DescribeCoipPools",
    "DescribeConversionTasks",
    "DescribeCustomerGateways",
    "DescribeDhcpOptions",
    "DescribeEgressOnlyInternetGateways",
    "DescribeElasticGpus",
    "DescribeExportImageTasks",
    "DescribeExportTasks",
    "DescribeFastSnapshotRestores",
    "DescribeFleetHistory",
    "DescribeFleetInstances",
    "DescribeFleets",
    "DescribeFlowLogs",
    "DescribeFpgaImageAttribute",
    "DescribeFpgaImages",
    "DescribeHostReservationOfferings",
    "DescribeHostReservations",
    "DescribeHosts",
    "DescribeIamInstanceProfileAssociations",
    "DescribeIdentityIdFormat",
    "DescribeIdFormat",
    "DescribeImageAttribute",
    "DescribeImages",
    "DescribeImportImageTasks",
    "DescribeImportSnapshotTasks",
    "DescribeInstanceAttribute",
    "DescribeInstanceCreditSpecifications",
    "DescribeInstanceEventNotificationAttributes",
    "DescribeInstances",
    "DescribeInstanceStatus",
    "DescribeInstanceTypeOfferings",
    "DescribeInstanceTypes",
    "DescribeInternetGateways",
    "DescribeIpv6Pools",
    "DescribeKeyPairs",
    "DescribeLaunchTemplates",
    "DescribeLaunchTemplateVersions",
    "DescribeLocalGatewayRouteTables",
    "DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociations",
    "DescribeLocalGatewayRouteTableVpcAssociations",
    "DescribeLocalGateways",
    "DescribeLocalGatewayVirtualInterfaceGroups",
    "DescribeLocalGatewayVirtualInterfaces",
    "DescribeManagedPrefixLists",
    "DescribeMovingAddresses",
    "DescribeNatGateways",
    "DescribeNetworkAcls",
    "DescribeNetworkInterfaceAttribute",
    "DescribeNetworkInterfacePermissions",
    "DescribeNetworkInterfaces",
    "DescribePlacementGroups",
    "DescribePrefixLists",
    "DescribePrincipalIdFormat",
    "DescribePublicIpv4Pools",
    "DescribeRegions",
    "DescribeReservedInstances",
    "DescribeReservedInstancesListings",
    "DescribeReservedInstancesModifications",
    "DescribeReservedInstancesOfferings",
    "DescribeRouteTables",
    "DescribeScheduledInstanceAvailability",
    "DescribeScheduledInstances",
    "DescribeSecurityGroupReferences",
    "DescribeSecurityGroups",
    "DescribeSnapshotAttribute",
    "DescribeSnapshots",
    "DescribeSpotDatafeedSubscription",
    "DescribeSpotFleetInstances",
    "DescribeSpotFleetRequestHistory",
    "DescribeSpotFleetRequests",
    "DescribeSpotInstanceRequests",
    "DescribeSpotPriceHistory",
    "DescribeStaleSecurityGroups",
    "DescribeSubnets",
    "DescribeTags",
    "DescribeTrafficMirrorFilters",
    "DescribeTrafficMirrorSessions",
    "DescribeTrafficMirrorTargets",
    "DescribeTransitGatewayAttachments",
    "DescribeTransitGatewayMulticastDomains",
    "DescribeTransitGatewayPeeringAttachments",
    "DescribeTransitGatewayRouteTables",
    "DescribeTransitGateways",
    "DescribeTransitGatewayVpcAttachments",
    "DescribeVolumeAttribute",
    "DescribeVolumes",
    "DescribeVolumesModifications",
    "DescribeVolumeStatus",
    "DescribeVpcAttribute",
    "DescribeVpcClassicLink",
    "DescribeVpcClassicLinkDnsSupport",
    "DescribeVpcEndpointConnectionNotifications",
    "DescribeVpcEndpointConnections",
    "DescribeVpcEndpoints",
    "DescribeVpcEndpointServiceConfigurations",
    "DescribeVpcEndpointServicePermissions",
    "DescribeVpcEndpointServices",
    "DescribeVpcPeeringConnections",
    "DescribeVpcs",
    "DescribeVpnConnections",
    "DescribeVpnGateways",
    "DetachClassicLinkVpc",
    "DetachInternetGateway",
    "DetachNetworkInterface",
    "DetachVolume",
    "DetachVpnGateway",
    "DisableEbsEncryptionByDefault",
    "DisableFastSnapshotRestores",
    "DisableTransitGatewayRouteTablePropagation",
    "DisableVgwRoutePropagation",
    "DisableVpcClassicLink",
    "DisableVpcClassicLinkDnsSupport",
    "DisassociateAddress",
    "DisassociateClientVpnTargetNetwork",
    "DisassociateIamInstanceProfile",
    "DisassociateRouteTable",
    "DisassociateSubnetCidrBlock",
    "DisassociateTransitGatewayMulticastDomain",
    "DisassociateTransitGatewayRouteTable",
    "DisassociateVpcCidrBlock",
    "EnableEbsEncryptionByDefault",
    "EnableFastSnapshotRestores",
    "EnableTransitGatewayRouteTablePropagation",
    "EnableVgwRoutePropagation",
    "EnableVolumeIO",
    "EnableVpcClassicLink",
    "EnableVpcClassicLinkDnsSupport",
    "ExportClientVpnClientCertificateRevocationList",
    "ExportClientVpnClientConfiguration",
    "ExportImage",
    "ExportTransitGatewayRoutes",
    "GetAssociatedIpv6PoolCidrs",
    "GetCapacityReservationUsage",
    "GetCoipPoolUsage",
    "GetConsoleOutput",
    "GetConsoleScreenshot",
    "GetDefaultCreditSpecification",
    "GetEbsDefaultKmsKeyId",
    "GetEbsEncryptionByDefault",
    "GetGroupsForCapacityReservation",
    "GetHostReservationPurchasePreview",
    "GetLaunchTemplateData",
    "GetManagedPrefixListAssociations",
    "GetManagedPrefixListEntries",
    "GetPasswordData",
    "GetReservedInstancesExchangeQuote",
    "GetTransitGatewayAttachmentPropagations",
    "GetTransitGatewayMulticastDomainAssociations",
    "GetTransitGatewayPrefixListReferences",
    "GetTransitGatewayRouteTableAssociations",
    "GetTransitGatewayRouteTablePropagations",
    "ImportClientVpnClientCertificateRevocationList",
    "ImportImage",
    "ImportInstance",
    "ImportKeyPair",
    "ImportSnapshot",
    "ImportVolume",
    "ModifyAvailabilityZoneGroup",
    "ModifyCapacityReservation",
    "ModifyClientVpnEndpoint",
    "ModifyDefaultCreditSpecification",
    "ModifyEbsDefaultKmsKeyId",
    "ModifyFleet",
    "ModifyFpgaImageAttribute",
    "ModifyHosts",
    "ModifyIdentityIdFormat",
    "ModifyIdFormat",
    "ModifyImageAttribute",
    "ModifyInstanceAttribute",
    "ModifyInstanceCapacityReservationAttributes",
    "ModifyInstanceCreditSpecification",
    "ModifyInstanceEventStartTime",
    "ModifyInstanceMetadataOptions",
    "ModifyInstancePlacement",
    "ModifyLaunchTemplate",
    "ModifyManagedPrefixList",
    "ModifyNetworkInterfaceAttribute",
    "ModifyReservedInstances",
    "ModifySnapshotAttribute",
    "ModifySpotFleetRequest",
    "ModifySubnetAttribute",
    "ModifyTrafficMirrorFilterNetworkServices",
    "ModifyTrafficMirrorFilterRule",
    "ModifyTrafficMirrorSession",
    "ModifyTransitGateway",
    "ModifyTransitGatewayPrefixListReference",
    "ModifyTransitGatewayVpcAttachment",
    "ModifyVolume",
    "ModifyVolumeAttribute",
    "ModifyVpcAttribute",
    "ModifyVpcEndpoint",
    "ModifyVpcEndpointConnectionNotification",
    "ModifyVpcEndpointServiceConfiguration",
    "ModifyVpcEndpointServicePermissions",
    "ModifyVpcPeeringConnectionOptions",
    "ModifyVpcTenancy",
    "ModifyVpnConnection",
    "ModifyVpnConnectionOptions",
    "ModifyVpnTunnelCertificate",
    "ModifyVpnTunnelOptions",
    "MonitorInstances",
    "MoveAddressToVpc",
    "ProvisionByoipCidr",
    "PurchaseHostReservation",
    "PurchaseReservedInstancesOffering",
    "PurchaseScheduledInstances",
    "RebootInstances",
    "RegisterImage",
    "RegisterInstanceEventNotificationAttributes",
    "RegisterTransitGatewayMulticastGroupMembers",
    "RegisterTransitGatewayMulticastGroupSources",
    "RejectTransitGatewayPeeringAttachment",
    "RejectTransitGatewayVpcAttachment",
    "RejectVpcEndpointConnections",
    "RejectVpcPeeringConnection",
    "ReleaseAddress",
    "ReleaseHosts",
    "ReplaceIamInstanceProfileAssociation",
    "ReplaceNetworkAclAssociation",
    "ReplaceNetworkAclEntry",
    "ReplaceRoute",
    "ReplaceRouteTableAssociation",
    "ReplaceTransitGatewayRoute",
    "ReportInstanceStatus",
    "RequestSpotFleet",
    "RequestSpotInstances",
    "ResetEbsDefaultKmsKeyId",
    "ResetFpgaImageAttribute",
    "ResetImageAttribute",
    "ResetInstanceAttribute",
    "ResetNetworkInterfaceAttribute",
    "ResetSnapshotAttribute",
    "RestoreAddressToClassic",
    "RestoreManagedPrefixListVersion",
    "RevokeClientVpnIngress",
    "RevokeSecurityGroupEgress",
    "RevokeSecurityGroupIngress",
    "RunInstances",
    "RunScheduledInstances",
    "SearchLocalGatewayRoutes",
    "SearchTransitGatewayMulticastGroups",
    "SearchTransitGatewayRoutes",
    "SendDiagnosticInterrupt",
    "StartInstances",
    "StartVpcEndpointServicePrivateDnsVerification",
    "StopInstances",
    "TerminateClientVpnConnections",
    "TerminateInstances",
    "UnassignIpv6Addresses",
    "UnassignPrivateIpAddresses",
    "UnmonitorInstances",
    "UpdateSecurityGroupRuleDescriptionsEgress",
    "UpdateSecurityGroupRuleDescriptionsIngress",
    "WithdrawByoipCidr"
  ])
end
