/*
 * Copyright (c) 2005 Kungliga Tekniska Högskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "hdb_locl.h"
#include <getarg.h>

static int help_flag;
static int version_flag;

struct getargs args[] = {
    { "help",		'h',	arg_flag,   &help_flag,    NULL, NULL },
    { "version",	0,	arg_flag,   &version_flag, NULL, NULL }
};

static int num_args = sizeof(args) / sizeof(args[0]);

/*
 * Prove that HDB_EntryOrAlias being a CHOICE of hdb_entry or hdb_entry_alias
 * adds nothing to the encoding of those types.
 */
static
void
check_HDB_EntryOrAlias(krb5_context context)
{
    HDB_EntryOrAlias eoa;
    hdb_entry entry;
    hdb_entry_alias alias;
    krb5_data v;
    size_t len;
    int ret;

    memset(&entry, 0, sizeof(entry));
    memset(&alias, 0, sizeof(alias));
    memset(&eoa, 0, sizeof(eoa));
    krb5_data_zero(&v);

    ret = krb5_make_principal(context, &alias.principal, "KTH.SE", "foo",
                              NULL);
    if (ret)
        krb5_err(context, 1, ret, "krb5_make_principal");
    ASN1_MALLOC_ENCODE(HDB_entry_alias, v.data, v.length, &alias, &len, ret);
    if (ret)
        krb5_err(context, 1, ret, "encode_HDB_EntryOrAlias");
    if (v.length != len)
        abort();
    ret = decode_HDB_EntryOrAlias(v.data, v.length, &eoa, &len);
    if (ret)
        krb5_err(context, 1, ret, "decode_HDB_EntryOrAlias");
    if (v.length != len)
        abort();
    free_HDB_EntryOrAlias(&eoa);
    free_HDB_entry_alias(&alias);
    krb5_data_free(&v);

    ret = krb5_make_principal(context, &entry.principal, "KTH.SE", "foo",
                              NULL);
    if (ret)
        krb5_err(context, 1, ret, "krb5_make_principal");
    entry.kvno = 5;
    entry.flags.initial = 1;
    ASN1_MALLOC_ENCODE(HDB_entry, v.data, v.length, &entry, &len, ret);
    if (ret)
        krb5_err(context, 1, ret, "encode_HDB_EntryOrAlias");
    if (v.length != len)
        abort();
    ret = decode_HDB_EntryOrAlias(v.data, v.length, &eoa, &len);
    if (ret)
        krb5_err(context, 1, ret, "decode_HDB_EntryOrAlias");
    if (v.length != len)
        abort();
    free_HDB_EntryOrAlias(&eoa);
    free_HDB_entry(&entry);
    krb5_data_free(&v);
}

int
main(int argc, char **argv)
{
    struct hdb_dbinfo *info, *d;
    krb5_context context;
    int ret, o = 0;

    setprogname(argv[0]);

    if(getarg(args, num_args, argc, argv, &o))
	krb5_std_usage(1, args, num_args);

    if(help_flag)
	krb5_std_usage(0, args, num_args);

    if(version_flag){
	print_version(NULL);
	exit(0);
    }

    ret = krb5_init_context(&context);
    if (ret)
	errx (1, "krb5_init_context failed: %d", ret);

    check_HDB_EntryOrAlias(context);

    ret = hdb_get_dbinfo(context, &info);
    if (ret)
	krb5_err(context, 1, ret, "hdb_get_dbinfo");

    d = NULL;
    while ((d = hdb_dbinfo_get_next(info, d)) != NULL) {
	const char *s;
	s = hdb_dbinfo_get_label(context, d);
	printf("label: %s\n", s ? s : "no label");
	s = hdb_dbinfo_get_realm(context, d);
	printf("\trealm: %s\n", s ? s : "no realm");
	s = hdb_dbinfo_get_dbname(context, d);
	printf("\tdbname: %s\n", s ? s : "no dbname");
	s = hdb_dbinfo_get_mkey_file(context, d);
	printf("\tmkey_file: %s\n", s ? s : "no mkey file");
	s = hdb_dbinfo_get_acl_file(context, d);
	printf("\tacl_file: %s\n", s ? s : "no acl file");
    }

    hdb_free_dbinfo(context, &info);

    krb5_free_context(context);

    return 0;
}
