package org.codehaus.modello;

/*
 * Copyright (c) 2005, Codehaus.org
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of
 * this software and associated documentation files (the "Software"), to deal in
 * the Software without restriction, including without limitation the rights to
 * use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is furnished to do
 * so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import java.io.File;
import java.io.IOException;
import java.io.Reader;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import org.codehaus.plexus.ContainerConfiguration;
import org.codehaus.plexus.PlexusConstants;
import org.codehaus.plexus.PlexusTestCase;
import org.codehaus.plexus.util.FileUtils;
import org.codehaus.plexus.util.xml.XmlStreamReader;
import org.xml.sax.SAXException;

/**
 * Abstract class for Modello plugins unit-tests that check output generated by the plugin.
 *
 * @author <a href="mailto:trygvis@inamo.no">Trygve Laugst&oslash;l</a>
 */
public abstract class AbstractModelloGeneratorTest extends PlexusTestCase {
    private String name;

    protected AbstractModelloGeneratorTest(String name) {
        this.name = name;
    }

    protected void setUp() throws Exception {
        super.setUp();

        FileUtils.deleteDirectory(getOutputDirectory());

        assertTrue(getOutputDirectory().mkdirs());
    }

    protected File getOutputDirectory() {
        return getTestFile("target/generator-results/" + getName());
    }

    public String getName() {
        return name;
    }

    protected Map<String, Object> getModelloParameters() {
        Map<String, Object> parameters = new HashMap<>();

        parameters.put("modello.output.directory", getOutputDirectory().getAbsolutePath());

        return parameters;
    }

    protected Map<String, Object> getModelloParameters(String version) {
        return getModelloParameters(version, null);
    }

    protected Map<String, Object> getModelloParameters(String version, Integer javaSource) {
        Map<String, Object> parameters = getModelloParameters();

        parameters.put("modello.package.with.version", Boolean.toString(false));
        parameters.put("modello.version", version);
        Optional.ofNullable(javaSource)
                .ifPresent(a -> parameters.put("modello.output.java.source", Integer.toString(a)));

        return parameters;
    }

    protected Reader getXmlResourceReader(String name) throws IOException {
        return new XmlStreamReader(getClass().getResourceAsStream(name));
    }

    protected SAXParser createSaxParserWithSchema(String generatedXsdName)
            throws ParserConfigurationException, SAXException {
        SAXParserFactory factory = SAXParserFactory.newInstance();
        factory.setValidating(true);
        factory.setNamespaceAware(true);
        SAXParser saxParser = factory.newSAXParser();
        saxParser.setProperty(
                "http://java.sun.com/xml/jaxp/properties/schemaLanguage", "http://www.w3.org/2001/XMLSchema");
        saxParser.setProperty(
                "http://java.sun.com/xml/jaxp/properties/schemaSource",
                new File(getOutputDirectory(), generatedXsdName));
        return saxParser;
    }

    @Override
    protected void customizeContainerConfiguration(ContainerConfiguration containerConfiguration) {
        containerConfiguration.setClassPathScanning("cache");
        containerConfiguration.setAutoWiring(true);
        containerConfiguration.setClassPathScanning(PlexusConstants.SCANNING_INDEX);
    }
}
