var __rest = (this && this.__rest) || function (s, e) {
    var t = {};
    for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p) && e.indexOf(p) < 0)
        t[p] = s[p];
    if (s != null && typeof Object.getOwnPropertySymbols === "function")
        for (var i = 0, p = Object.getOwnPropertySymbols(s); i < p.length; i++) {
            if (e.indexOf(p[i]) < 0 && Object.prototype.propertyIsEnumerable.call(s, p[i]))
                t[p[i]] = s[p[i]];
        }
    return t;
};
import { flatMap, map, some } from "../../iters";
import { bigrams, dfs } from "../../utils";
import { componentMap, indices, init, layout, minBend, minDist, solve, splitComponentLayers, } from "./utils";
/**
 * a function for creating a {@link ConstAccessor}
 */
export function createConstAccessor(value) {
    if (value < 0) {
        throw new Error("const accessors should return non-negative values");
    }
    const accessor = () => value;
    accessor.value = value;
    return accessor;
}
/**
 * If an accessor is a const accessor
 */
function isConstAccessor(acc) {
    return "value" in acc && typeof acc.value === "number";
}
/**
 * cache the vert weak function to avoid duplicate calls
 *
 * Due to the nature of how it's called, this is easier to implement as a
 * function of source and target, rather then links, but to build up the index,
 * we then have to iterate over the links in advance.
 */
function cacheVertWeak(vertWeak, layers) {
    if (isConstAccessor(vertWeak)) {
        // verify that it's actually const since we'll never actually call it normally
        const val = vertWeak.value;
        for (const node of flatMap(layers, (l) => l)) {
            if ("node" in node.data) {
                const source = node.data.node;
                for (const link of source.ichildLinks()) {
                    if (vertWeak(link) !== val) {
                        throw new Error("passed in a vertWeak accessor with a `value` property that wasn't a const accessor");
                    }
                }
            }
        }
        return () => val;
    }
    else {
        const vertWeakMap = new Map();
        for (const node of flatMap(layers, (l) => l)) {
            if ("node" in node.data) {
                // regular node
                const source = node.data.node;
                const targetLinks = new Map(map(source.ichildLinks(), (link) => [link.target, vertWeak(link)]));
                vertWeakMap.set(source, targetLinks);
            }
        }
        return (src, targ) => vertWeakMap.get(src).get(targ);
    }
}
/**
 * cache an arbitrary link weight accessor
 */
function cacheLinkWeightAccessor(accessor) {
    if (isConstAccessor(accessor)) {
        // don't need to cache constant accessors
        return accessor;
    }
    else {
        const cache = new Map();
        return (link) => {
            const { source, target } = link;
            let targets = cache.get(source);
            if (targets === undefined) {
                targets = new Map();
                cache.set(source, targets);
            }
            const cached = targets.get(target);
            if (cached === undefined) {
                const val = accessor(link);
                if (val < 0) {
                    throw new Error("link weights must be non-negative");
                }
                targets.set(target, val);
                return val;
            }
            else {
                return cached;
            }
        };
    }
}
/**
 * cache an arbitrary node weight accessor
 */
function cacheNodeWeightAccessor(accessor) {
    if (isConstAccessor(accessor)) {
        return accessor;
    }
    else {
        const cache = new Map();
        return (node) => {
            const cached = cache.get(node);
            if (cached === undefined) {
                const val = accessor(node);
                if (val < 0) {
                    throw new Error("node weights must be non-negative");
                }
                cache.set(node, val);
                return val;
            }
            else {
                return cached;
            }
        };
    }
}
function buildOperator(opts) {
    function quadComponent(layers, nodeSize, compMap) {
        var _a;
        const { comp } = opts;
        const inds = indices(layers);
        const [Q, c, A, b] = init(layers, inds, nodeSize);
        const cachedVertWeak = cacheVertWeak(opts.vertWeak, layers);
        const cachedVertStrong = cacheLinkWeightAccessor(opts.vertStrong);
        const cachedLinkCurve = cacheLinkWeightAccessor(opts.linkCurve);
        const cachedNodeCurve = cacheNodeWeightAccessor(opts.nodeCurve);
        // add loss for nearby nodes and for curve of nodes
        for (const par of flatMap(layers, (l) => l)) {
            const pind = inds.get(par);
            const pdata = par.data;
            const source = "node" in pdata ? pdata.node : pdata.link.source;
            for (const node of par.ichildren()) {
                const nind = inds.get(node);
                const ndata = node.data;
                const target = "node" in ndata ? ndata.node : ndata.link.target;
                const wpdist = "node" in pdata
                    ? cachedVertWeak(source, target)
                    : cachedVertStrong(pdata.link);
                const wndist = "node" in ndata
                    ? cachedVertWeak(source, target)
                    : cachedVertStrong(ndata.link);
                const wcurve = "node" in ndata
                    ? cachedNodeCurve(ndata.node)
                    : cachedLinkCurve(ndata.link);
                minDist(Q, pind, nind, wpdist + wndist);
                for (const child of node.ichildren()) {
                    const cind = inds.get(child);
                    minBend(Q, pind, nind, cind, wcurve);
                }
            }
        }
        // for disconnected dags, add loss for being too far apart
        // However, we only need to do this if a component is strictly to one side
        // of the other component. We can compute this by first making a graph
        // between components representing "to the left of" and then look for
        // cycles, which don't need to be constrained
        // create left map
        const leftOf = new Map();
        for (const layer of layers) {
            for (const [first, second] of bigrams(layer)) {
                const firstComp = compMap.get(first);
                const secondComp = compMap.get(second);
                if (firstComp !== secondComp) {
                    const rights = leftOf.get(firstComp);
                    if (rights === undefined) {
                        leftOf.set(firstComp, new Set([secondComp]));
                    }
                    else {
                        rights.add(secondComp);
                    }
                }
            }
        }
        // preserve links in left map if they're not part of a cycle
        // Enumerating all cycles would be prohibitive, but there are a few ways we
        // could speed up the current implementation
        // 1. Check if left appears in any rights which we could do by making
        //    another ancillary set. This doesn't improve worst case complexity,
        //    but will probably speed up a lot of common slow paths.
        // 2. Changing dfs to also return any cycles found along the way. This will
        //    require more space and also won't change worst case time, but will
        //    allow us to prune a lot of the available cycles and will at least
        //    remove some redundant computation in terms of the length of a cycle.
        // 3. Gate this behind a flag to prevent the long execution in large
        //    disconnected dags.
        const cons = new Map();
        for (const [left, rights] of leftOf.entries()) {
            const newRights = new Set();
            for (const right of rights) {
                const reachable = dfs((c) => { var _a; return (_a = leftOf.get(c)) !== null && _a !== void 0 ? _a : []; }, right);
                const inCycle = some(reachable, (c) => c === left);
                if (!inCycle) {
                    newRights.add(right);
                }
            }
            cons.set(left, newRights);
        }
        // add constraints if they're still there
        for (const layer of layers) {
            for (const [first, second] of bigrams(layer)) {
                const firstComp = compMap.get(first);
                const secondComp = compMap.get(second);
                if (firstComp !== secondComp && ((_a = cons.get(firstComp)) === null || _a === void 0 ? void 0 : _a.has(secondComp))) {
                    minDist(Q, inds.get(first), inds.get(second), comp);
                }
            }
        }
        // get actual solution
        try {
            const solution = solve(Q, c, A, b);
            return layout(layers, nodeSize, inds, solution);
        }
        catch (ex) {
            /* istanbul ignore else */
            if (ex instanceof Error &&
                ex.message ===
                    "quadratic program failed: matrix D in quadratic function is not positive definite!") {
                throw new Error("quad objective wasn't well defined, this happens when too many of the weights were set to zero (or really small). Try changing the weight accessors to return nonzero values in more instances.");
            }
            else {
                throw ex;
            }
        }
    }
    function quadCall(layers, nodeSize) {
        // split components
        const compMap = componentMap(layers);
        const components = splitComponentLayers(layers, compMap);
        // layout each component and get width
        const widths = components.map((compon) => quadComponent(compon, nodeSize, compMap));
        // center components
        const maxWidth = Math.max(...widths);
        if (maxWidth <= 0) {
            throw new Error("must assign nonzero width to at least one node");
        }
        for (const [i, compon] of components.entries()) {
            const offset = (maxWidth - widths[i]) / 2;
            for (const layer of compon) {
                for (const node of layer) {
                    node.x += offset;
                }
            }
        }
        return maxWidth;
    }
    function vertical(val) {
        if (val === undefined) {
            const { vertWeak, vertStrong } = opts;
            if (isConstAccessor(vertWeak) && isConstAccessor(vertStrong)) {
                return [vertWeak.value, vertStrong.value];
            }
            else {
                return null;
            }
        }
        else {
            const [vertNode, vertDummy] = val;
            if (vertNode < 0 || vertDummy < 0) {
                throw new Error(`weights must be non-negative, but were ${vertNode} and ${vertDummy}`);
            }
            else {
                const { vertWeak: _, vertStrong: __ } = opts, rest = __rest(opts, ["vertWeak", "vertStrong"]);
                return buildOperator(Object.assign(Object.assign({}, rest), { vertWeak: createConstAccessor(vertNode), vertStrong: createConstAccessor(vertDummy) }));
            }
        }
    }
    quadCall.vertical = vertical;
    function vertWeak(val) {
        if (val === undefined) {
            return opts.vertWeak;
        }
        else {
            const { vertWeak: _ } = opts, rest = __rest(opts, ["vertWeak"]);
            return buildOperator(Object.assign(Object.assign({}, rest), { vertWeak: val }));
        }
    }
    quadCall.vertWeak = vertWeak;
    function vertStrong(val) {
        if (val === undefined) {
            return opts.vertStrong;
        }
        else {
            const { vertStrong: _ } = opts, rest = __rest(opts, ["vertStrong"]);
            return buildOperator(Object.assign(Object.assign({}, rest), { vertStrong: val }));
        }
    }
    quadCall.vertStrong = vertStrong;
    function curve(val) {
        if (val === undefined) {
            const { linkCurve, nodeCurve } = opts;
            if (isConstAccessor(linkCurve) && isConstAccessor(nodeCurve)) {
                return [nodeCurve.value, linkCurve.value];
            }
            else {
                return null;
            }
        }
        else {
            const [curveNode, curveDummy] = val;
            if (curveNode < 0 || curveDummy < 0) {
                throw new Error(`weights must be non-negative, but were ${curveNode} and ${curveDummy}`);
            }
            else {
                const { linkCurve: _, nodeCurve: __ } = opts, rest = __rest(opts, ["linkCurve", "nodeCurve"]);
                return buildOperator(Object.assign(Object.assign({}, rest), { linkCurve: createConstAccessor(curveDummy), nodeCurve: createConstAccessor(curveNode) }));
            }
        }
    }
    quadCall.curve = curve;
    function linkCurve(val) {
        if (val === undefined) {
            return opts.linkCurve;
        }
        else {
            const { linkCurve: _ } = opts, rest = __rest(opts, ["linkCurve"]);
            return buildOperator(Object.assign(Object.assign({}, rest), { linkCurve: val }));
        }
    }
    quadCall.linkCurve = linkCurve;
    function nodeCurve(val) {
        if (val === undefined) {
            return opts.nodeCurve;
        }
        else {
            const { nodeCurve: _ } = opts, rest = __rest(opts, ["nodeCurve"]);
            return buildOperator(Object.assign(Object.assign({}, rest), { nodeCurve: val }));
        }
    }
    quadCall.nodeCurve = nodeCurve;
    function component(val) {
        if (val === undefined) {
            return opts.comp;
        }
        else if (val <= 0) {
            throw new Error(`weight must be positive, but was ${val}`);
        }
        else {
            return buildOperator(Object.assign(Object.assign({}, opts), { comp: val }));
        }
    }
    quadCall.component = component;
    return quadCall;
}
/**
 * Create a default {@link QuadOperator}, bundled as {@link coordQuad}.
 */
export function quad(...args) {
    if (args.length) {
        throw new Error(`got arguments to quad(${args}), but constructor takes no arguments.`);
    }
    return buildOperator({
        vertWeak: createConstAccessor(1),
        vertStrong: createConstAccessor(0),
        linkCurve: createConstAccessor(1),
        nodeCurve: createConstAccessor(0),
        comp: 1,
    });
}
