var __rest = (this && this.__rest) || function (s, e) {
    var t = {};
    for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p) && e.indexOf(p) < 0)
        t[p] = s[p];
    if (s != null && typeof Object.getOwnPropertySymbols === "function")
        for (var i = 0, p = Object.getOwnPropertySymbols(s); i < p.length; i++) {
            if (e.indexOf(p[i]) < 0 && Object.prototype.propertyIsEnumerable.call(s, p[i]))
                t[p[i]] = s[p[i]];
        }
    return t;
};
import { before } from "../dag/utils";
import { map } from "../iters";
import { js, setEqual } from "../utils";
import { greedy } from "./lane/greedy";
/**
 * Verify that nodes were assigned valid lanes
 *
 * @internal
 */
function verifyLanes(ordered) {
    for (const node of ordered) {
        if (node.x === undefined) {
            throw new Error(js `coord didn't assign an x to node '${node}'`);
        }
        else if (node.x < 0) {
            throw new Error(`coord assigned an x (${node.x}) less than 0`);
        }
    }
    const uniqueExes = new Set(ordered.map((n) => n.x));
    if (!setEqual(uniqueExes, new Set(map(uniqueExes, (_, i) => i)))) {
        const exStr = [...uniqueExes].join(", ");
        throw new Error(`didn't assign increasing positive integers for x coordinates: ${exStr}`);
    }
    const parentIndex = new Map();
    for (const [ind, node] of ordered.entries()) {
        // test that no nodes overlap with edges
        const topIndex = parentIndex.get(node);
        if (topIndex !== undefined) {
            for (const above of ordered.slice(topIndex + 1, ind)) {
                if (above.x === node.x) {
                    throw new Error(js `node ${above} was assigned an overlapping lane with ${node}`);
                }
            }
        }
        // update parent index
        for (const child of node.ichildren()) {
            if (!parentIndex.has(child)) {
                parentIndex.set(child, ind);
            }
        }
    }
    return uniqueExes.size;
}
/** @internal */
function buildOperator(options) {
    function gridCall(dag) {
        if (dag.multidag()) {
            throw new Error("grid layout doesn't make sense to apply to multidags, consider pruning the edges");
        }
        const { nodeWidth, nodeHeight, size, lane } = options;
        // topological sort
        const ordered = [...before(dag, options.rank)];
        // assign ys
        for (const [i, node] of ordered.entries()) {
            node.y = i;
        }
        // get lanes
        lane(ordered);
        const numLanes = verifyLanes(ordered);
        // adjust x and y by nodeSize
        for (const node of ordered) {
            node.x = (node.x + 0.5) * nodeWidth;
            node.y = (node.y + 0.5) * nodeHeight;
        }
        // scale for size
        let width = numLanes * nodeWidth;
        let height = ordered.length * nodeHeight;
        if (size !== null) {
            const [newWidth, newHeight] = size;
            for (const node of ordered) {
                node.x *= newWidth / width;
                node.y *= newHeight / height;
            }
            width = newWidth;
            height = newHeight;
        }
        // assign link points
        for (const { source, target, points } of dag.ilinks()) {
            points.length = 0;
            if (source.x !== target.x) {
                points.push({ x: source.x, y: source.y });
            }
            points.push({ x: target.x, y: source.y });
            points.push({ x: target.x, y: target.y });
        }
        return { width, height };
    }
    function lane(val) {
        if (val === undefined) {
            return options.lane;
        }
        else {
            const { lane: _ } = options, rest = __rest(options, ["lane"]);
            return buildOperator(Object.assign(Object.assign({}, rest), { lane: val }));
        }
    }
    gridCall.lane = lane;
    function rank(val) {
        if (val === undefined) {
            return options.rank;
        }
        else {
            const { rank: _ } = options, rest = __rest(options, ["rank"]);
            return buildOperator(Object.assign(Object.assign({}, rest), { rank: val }));
        }
    }
    gridCall.rank = rank;
    function nodeSize(val) {
        if (val === undefined) {
            const { nodeWidth, nodeHeight } = options;
            return [nodeWidth, nodeHeight];
        }
        else {
            const [nodeWidth, nodeHeight] = val;
            return buildOperator(Object.assign(Object.assign({}, options), { nodeWidth, nodeHeight }));
        }
    }
    gridCall.nodeSize = nodeSize;
    function size(val) {
        if (val !== undefined) {
            return buildOperator(Object.assign(Object.assign({}, options), { size: val }));
        }
        else if (options.size === null) {
            return null;
        }
        else {
            const [width, height] = options.size;
            return [width, height];
        }
    }
    gridCall.size = size;
    return gridCall;
}
/**
 * Create a new {@link GridOperator} with default settings.
 */
export function grid(...args) {
    if (args.length) {
        throw new Error(`got arguments to grid(${args}), but constructor takes no arguments.`);
    }
    return buildOperator({
        lane: greedy(),
        rank: () => undefined,
        nodeWidth: 1,
        nodeHeight: 1,
        size: null,
    });
}
