/**
 * An {@link AggOperator} that orders nodes based on the aggregation of their
 * parents' or children's indices.
 *
 * @packageDocumentation
 */
import { median } from "d3-array";
import { map } from "../../iters";
import { listMultimapPush } from "../../utils";
class Mean {
    constructor() {
        this.mean = 0.0;
        this.count = 0;
    }
    add(val) {
        this.mean += (val - this.mean) / ++this.count;
    }
    val() {
        return this.count ? this.mean : undefined;
    }
}
/**
 * A {@link AggFactory | factory} that creates mean {@link Aggregator}s,
 * bundled as {@link aggMeanFactory}.
 */
export const meanFactory = () => new Mean();
class Median {
    constructor() {
        this.vals = [];
    }
    add(val) {
        this.vals.push(val);
    }
    val() {
        return median(this.vals);
    }
}
/**
 * A {@link AggFactory | factory} that creates median {@link Aggregator}s,
 * bundled as {@link aggMedianFactory}.
 */
export const medianFactory = () => new Median();
class WeightedMedian {
    constructor() {
        this.vals = [];
    }
    add(val) {
        this.vals.push(val);
    }
    val() {
        // NOTE this could be linear time, but we already do other sorts, so
        // probably not terrible
        this.vals.sort((a, b) => a - b);
        if (this.vals.length === 0) {
            return undefined;
        }
        else if (this.vals.length === 2) {
            return (this.vals[0] + this.vals[1]) / 2;
        }
        else if (this.vals.length % 2 === 0) {
            const ind = this.vals.length / 2;
            const first = this.vals[0];
            const left = this.vals[ind - 1];
            const right = this.vals[ind];
            const last = this.vals[this.vals.length - 1];
            // all elements are guaranteed to be different, so we don't need to worry
            // about leftDiff or rightDiff being 0
            const leftDiff = left - first;
            const rightDiff = last - right;
            return (left * rightDiff + right * leftDiff) / (leftDiff + rightDiff);
        }
        else {
            return this.vals[(this.vals.length - 1) / 2];
        }
    }
}
/**
 * A {@link AggFactory | factory} that creates weighted median {@link Aggregator}s,
 * bundled as {@link aggWeightedMedianFactory}.
 *
 * @remarks
 * This is slightly slower than the {@link medianFactory}.
 */
export const weightedMedianFactory = () => new WeightedMedian();
/** perform aggregation over an iterable */
function aggregate(aggregator, iter) {
    const agg = aggregator();
    for (const val of iter) {
        agg.add(val);
    }
    return agg.val();
}
/**
 * Order a layer with respect to numeric values
 *
 * Nodes without a value will be placed in the final order with as close as
 * possible to their initial position. This is defined as the position with the
 * minimum rank inversion relative to the initial ordering.
 *
 * @remarks
 *
 * See this {@link
 * https://cs.stackexchange.com/questions/140295/complexity-to-insert-subset-of-array-to-minimize-order-inversions
 * | Stack Exchange} post for algorithmic details.
 */
function order(layer, poses) {
    // first group by number and preserve order, this makes ties resolve to the
    // same order as layer
    const orderMap = new Map();
    for (const node of layer) {
        const val = poses.get(node);
        if (val === undefined) {
            continue;
        }
        listMultimapPush(orderMap, val, node);
    }
    const ordered = [...orderMap.entries()]
        .sort(([a], [b]) => a - b)
        .flatMap(([, nodes]) => nodes);
    // initialize gaps for unassigned nodes
    const inds = new Map(layer.map((n, i) => [n, i]));
    const unassigned = layer.filter((n) => poses.get(n) === undefined);
    const placements = new Array(unassigned.length).fill(null);
    // recursively split optimal placement
    function recurse(ustart, uend, ostart, oend) {
        if (uend <= ustart)
            return;
        const umid = Math.floor((ustart + uend) / 2);
        const node = unassigned[umid];
        const nind = inds.get(node);
        let last = 0;
        const inversions = [last];
        for (let i = ostart; i < oend; ++i) {
            last += inds.get(ordered[i]) < nind ? -1 : 1;
            inversions.push(last);
        }
        const placement = ostart + inversions.indexOf(Math.min(...inversions));
        placements[umid] = placement;
        recurse(ustart, umid, ostart, placement);
        recurse(umid + 1, uend, placement, oend);
    }
    recurse(0, unassigned.length, 0, ordered.length);
    // place nodes
    placements.push(ordered.length + 1); // sentinel
    let insert = 0;
    let uind = 0;
    for (const [i, node] of ordered.entries()) {
        while (placements[uind] == i) {
            layer[insert++] = unassigned[uind++];
        }
        layer[insert++] = node;
    }
    while (placements[uind] == ordered.length) {
        layer[insert++] = unassigned[uind++];
    }
}
function buildOperator({ factory, }) {
    function aggCall(topLayer, bottomLayer, topDown) {
        if (topDown) {
            const incr = new Map(bottomLayer.map((node) => [node, factory()]));
            for (const [i, node] of topLayer.entries()) {
                for (const child of node.ichildren()) {
                    incr.get(child).add(i);
                }
            }
            const aggs = new Map([...incr.entries()].map(([node, agg]) => [node, agg.val()]));
            order(bottomLayer, aggs);
        }
        else {
            const inds = new Map(bottomLayer.map((node, i) => [node, i]));
            const aggs = new Map(topLayer.map((node) => {
                const agg = aggregate(factory, map(node.ichildren(), (child) => inds.get(child)));
                return [node, agg];
            }));
            order(topLayer, aggs);
        }
    }
    function aggregator(val) {
        if (val === undefined) {
            return factory;
        }
        else {
            return buildOperator({ factory: val });
        }
    }
    aggCall.aggregator = aggregator;
    return aggCall;
}
/**
 * Create a default {@link AggOperator}, bundled as {@link twolayerAgg}.
 */
export function agg(...args) {
    if (args.length) {
        throw new Error(`got arguments to agg(${args}), but constructor takes no arguments.`);
    }
    return buildOperator({ factory: weightedMedianFactory });
}
