/**
 * A {@link SimplexOperator} that assigns layers to minimize the number of
 * dummy nodes.
 *
 * @packageDocumentation
 */
import { GroupAccessor, LayeringOperator, RankAccessor } from ".";
import { Up } from "../../utils";
/** simplex operator operators */
export interface Operators<N = never, L = never> {
    /** rank operator */
    rank: RankAccessor<N, L>;
    /** group operator */
    group: GroupAccessor<N, L>;
}
/** the node datum of a set of operators */
export declare type OpsNodeDatum<Ops extends Operators> = Ops extends Operators<infer N, never> ? N : never;
/** the link datum of a set of operators */
export declare type OpsLinkDatum<Ops extends Operators> = Ops extends Operators<never, infer L> ? L : never;
/**
 * A layering operator that assigns layers to minimize the number of dummy
 * nodes (long edges) added to the layout.
 *
 * Computing this layering requires solving an integer linear program, which
 * may take a long time, although in practice is often quite fast. This is
 * often known as the network simplex layering from
 * {@link https://www.graphviz.org/Documentation/TSE93.pdf | Gansner et al.
 * [1993]}.
 *
 * Because this is solving a linear program, it is relatively easy to add new
 * constraints. The current implementation allows specifying {@link rank}
 * constriants that indicate which nodes should be above other nodes, or
 * {@link group} constraints that say which nodes should be on the same layer.
 * Note that adding these constraints can cause the optimization to become
 * ill-defined.
 *
 * Create with {@link simplex}.
 *
 * <img alt="simplex example" src="media://sugi-simplex-opt-quad.png" width="400">
 */
export interface SimplexOperator<Ops extends Operators = Operators> extends LayeringOperator<OpsNodeDatum<Ops>, OpsLinkDatum<Ops>> {
    /**
     * Set the {@link RankAccessor}. Any node with a rank assigned will have a second
     * ordering enforcing ordering of the ranks. Note, this can cause the simplex
     * optimization to be ill-defined, and may result in an error during layout.
     */
    rank<NewRank extends RankAccessor>(newRank: NewRank): SimplexOperator<Up<Ops, {
        /** new rank */
        rank: NewRank;
    }>>;
    /**
     * Get the current {@link RankAccessor}.
     */
    rank(): Ops["rank"];
    /**
     * Set the {@link GroupAccessor}. Any node with a group assigned will have a second
     * ordering enforcing all nodes with the same group have the same layer.
     * Note, this can cause the simplex optimization to be ill-defined, and may
     * result in an error during layout.
     */
    group<NewGroup extends GroupAccessor>(newGroup: NewGroup): SimplexOperator<Up<Ops, {
        /** new group */
        group: NewGroup;
    }>>;
    /**
     * Get the current {@link GroupAccessor}.
     */
    group(): Ops["group"];
}
/** default simplex operator */
export declare type DefaultSimplexOperator = SimplexOperator<{
    /** unconstrained rank */
    rank: RankAccessor<unknown, unknown>;
    /** unconstrained group */
    group: GroupAccessor<unknown, unknown>;
}>;
/**
 * Create a default {@link SimplexOperator}, bundled as {@link layeringSimplex}.
 */
export declare function simplex(...args: never[]): DefaultSimplexOperator;
