/**
 * A {@link TwoLayerOperator} heuristic for reducing the number of crossings in
 * large dags efficiently.
 *
 * @packageDocumentation
 */
import { DecrossOperator } from ".";
import { Up } from "../../utils";
import { TwolayerOperator as OrderOperator } from "../twolayer";
import { AggOperator } from "../twolayer/agg";
import { GreedyOperator } from "../twolayer/greedy";
import { DfsOperator } from "./dfs";
/** initializers */
export declare type Inits<N = never, L = never> = readonly [
    DecrossOperator<N, L>,
    ...DecrossOperator<N, L>[]
];
/** two layer operators */
export interface Operators<N = never, L = never> {
    /** the order operator */
    order: OrderOperator<N, L>;
    /** the initializers */
    inits: Inits<N, L>;
}
/** the node datum of a set of operators */
export declare type OpNodeDatum<O extends Operators> = O extends Operators<infer N, never> ? N : never;
/** the link datum of a set of operators */
export declare type OpLinkDatum<O extends Operators> = O extends Operators<never, infer L> ? L : never;
/**
 * A decrossing operator that minimizes the number of crossings by looking at every pair of layers.
 *
 * This method can be very fast and is a general heuristic for efficient
 * minimization. Customize with a two layer operator with {@link order} or use
 * one of the built-in {@link TwolayerOperator}s.
 *
 * This method can also make multiple {@link passes} in an attempt to produce a
 * better layout.
 *
 * <img alt="two layer example" src="media://sugi-simplex-twolayer-quad.png" width="400">
 */
export interface TwoLayerOperator<Ops extends Operators = Operators> extends DecrossOperator<OpNodeDatum<Ops>, OpLinkDatum<Ops>> {
    /**
     * Sets the order accessor to the specified {@link TwolayerOperator} and returns
     * a new operator. (default: {@link AggOperator}).
     */
    order<NewOrder extends OrderOperator>(val: NewOrder): TwoLayerOperator<Up<Ops, {
        /** new order */
        order: NewOrder;
    }>>;
    /**
     * Get the current {@link TwolayerOperator} for ordering.
     */
    order(): Ops["order"];
    /**
     * Sets the initialization passes to the specified {@link DecrossOperator}s and returns
     * a new operator.
     *
     * For every initialization operator, this will run the two layer heuristic,
     * ultimately choosing the ordering that minimized overall crossings. For
     * this reason, only quick decrossing operators should be used, not expensive
     * ones.  (default: [noop])
     */
    inits<NewInits extends Inits>(val: NewInits): TwoLayerOperator<Up<Ops, {
        /** new inits */
        inits: NewInits;
    }>>;
    /**
     * Get the current initialization passes
     */
    inits(): Ops["inits"];
    /**
     * Sets the number of passes to make, more takes longer, but might result in
     * a better output. (default: 24)
     */
    passes(val: number): TwoLayerOperator<Ops>;
    /**
     * Get the current number of passes
     */
    passes(): number;
}
/** default two layer operator */
export declare type DefaultTwoLayerOperator = TwoLayerOperator<{
    /** default order */
    order: GreedyOperator<AggOperator>;
    /** default inits, both dfs based */
    inits: [DfsOperator, DfsOperator];
}>;
/**
 * Create a default {@link TwoLayerOperator}, bundled as
 * {@link decrossTwoLayer}.
 */
export declare function twoLayer(...args: never[]): DefaultTwoLayerOperator;
