"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebVitalsWithAttribution = void 0;
var attribution_1 = require("web-vitals/attribution");
var faro_core_1 = require("@grafana/faro-core");
var utils_1 = require("../../utils");
var instrumentationConstants_1 = require("../instrumentationConstants");
// duplicate keys saved in variables to save bundle size
// refs: https://github.com/grafana/faro-web-sdk/pull/595#discussion_r1615833968
var loadStateKey = 'load_state';
var timeToFirstByteKey = 'time_to_first_byte';
var WebVitalsWithAttribution = /** @class */ (function () {
    function WebVitalsWithAttribution(corePushMeasurement, webVitalConfig) {
        this.corePushMeasurement = corePushMeasurement;
        this.webVitalConfig = webVitalConfig;
    }
    WebVitalsWithAttribution.prototype.initialize = function () {
        this.measureCLS();
        this.measureFCP();
        this.measureFID();
        this.measureINP();
        this.measureLCP();
        this.measureTTFB();
    };
    WebVitalsWithAttribution.prototype.measureCLS = function () {
        var _this = this;
        var _a;
        (0, attribution_1.onCLS)(function (metric) {
            var _a = metric.attribution, loadState = _a.loadState, largestShiftValue = _a.largestShiftValue, largestShiftTime = _a.largestShiftTime, largestShiftTarget = _a.largestShiftTarget;
            var values = _this.buildInitialValues(metric);
            _this.addIfPresent(values, 'largest_shift_value', largestShiftValue);
            _this.addIfPresent(values, 'largest_shift_time', largestShiftTime);
            var context = _this.buildInitialContext(metric);
            _this.addIfPresent(context, loadStateKey, loadState);
            _this.addIfPresent(context, 'largest_shift_target', largestShiftTarget);
            _this.pushMeasurement(values, context);
        }, { reportAllChanges: (_a = this.webVitalConfig) === null || _a === void 0 ? void 0 : _a.reportAllChanges });
    };
    WebVitalsWithAttribution.prototype.measureFCP = function () {
        var _this = this;
        var _a;
        (0, attribution_1.onFCP)(function (metric) {
            var _a = metric.attribution, firstByteToFCP = _a.firstByteToFCP, timeToFirstByte = _a.timeToFirstByte, loadState = _a.loadState;
            var values = _this.buildInitialValues(metric);
            _this.addIfPresent(values, 'first_byte_to_fcp', firstByteToFCP);
            _this.addIfPresent(values, timeToFirstByteKey, timeToFirstByte);
            var context = _this.buildInitialContext(metric);
            _this.addIfPresent(context, loadStateKey, loadState);
            _this.pushMeasurement(values, context);
        }, { reportAllChanges: (_a = this.webVitalConfig) === null || _a === void 0 ? void 0 : _a.reportAllChanges });
    };
    WebVitalsWithAttribution.prototype.measureFID = function () {
        var _this = this;
        var _a;
        (0, attribution_1.onFID)(function (metric) {
            var _a = metric.attribution, eventTime = _a.eventTime, eventTarget = _a.eventTarget, eventType = _a.eventType, loadState = _a.loadState;
            var values = _this.buildInitialValues(metric);
            _this.addIfPresent(values, 'event_time', eventTime);
            var context = _this.buildInitialContext(metric);
            _this.addIfPresent(context, 'event_target', eventTarget);
            _this.addIfPresent(context, 'event_type', eventType);
            _this.addIfPresent(context, loadStateKey, loadState);
            _this.pushMeasurement(values, context);
        }, { reportAllChanges: (_a = this.webVitalConfig) === null || _a === void 0 ? void 0 : _a.reportAllChanges });
    };
    WebVitalsWithAttribution.prototype.measureINP = function () {
        var _this = this;
        var _a;
        (0, attribution_1.onINP)(function (metric) {
            var _a = metric.attribution, interactionTime = _a.interactionTime, presentationDelay = _a.presentationDelay, inputDelay = _a.inputDelay, processingDuration = _a.processingDuration, nextPaintTime = _a.nextPaintTime, loadState = _a.loadState, interactionTarget = _a.interactionTarget, interactionType = _a.interactionType;
            var values = _this.buildInitialValues(metric);
            _this.addIfPresent(values, 'interaction_time', interactionTime);
            _this.addIfPresent(values, 'presentation_delay', presentationDelay);
            _this.addIfPresent(values, 'input_delay', inputDelay);
            _this.addIfPresent(values, 'processing_duration', processingDuration);
            _this.addIfPresent(values, 'next_paint_time', nextPaintTime);
            var context = _this.buildInitialContext(metric);
            _this.addIfPresent(context, loadStateKey, loadState);
            _this.addIfPresent(context, 'interaction_target', interactionTarget);
            _this.addIfPresent(context, 'interaction_type', interactionType);
            _this.pushMeasurement(values, context);
        }, { reportAllChanges: (_a = this.webVitalConfig) === null || _a === void 0 ? void 0 : _a.reportAllChanges });
    };
    WebVitalsWithAttribution.prototype.measureLCP = function () {
        var _this = this;
        var _a;
        (0, attribution_1.onLCP)(function (metric) {
            var _a = metric.attribution, elementRenderDelay = _a.elementRenderDelay, resourceLoadDelay = _a.resourceLoadDelay, resourceLoadDuration = _a.resourceLoadDuration, timeToFirstByte = _a.timeToFirstByte, element = _a.element;
            var values = _this.buildInitialValues(metric);
            _this.addIfPresent(values, 'element_render_delay', elementRenderDelay);
            _this.addIfPresent(values, 'resource_load_delay', resourceLoadDelay);
            _this.addIfPresent(values, 'resource_load_duration', resourceLoadDuration);
            _this.addIfPresent(values, timeToFirstByteKey, timeToFirstByte);
            var context = _this.buildInitialContext(metric);
            _this.addIfPresent(context, 'element', element);
            _this.pushMeasurement(values, context);
        }, { reportAllChanges: (_a = this.webVitalConfig) === null || _a === void 0 ? void 0 : _a.reportAllChanges });
    };
    WebVitalsWithAttribution.prototype.measureTTFB = function () {
        var _this = this;
        var _a;
        (0, attribution_1.onTTFB)(function (metric) {
            var _a = metric.attribution, dnsDuration = _a.dnsDuration, connectionDuration = _a.connectionDuration, requestDuration = _a.requestDuration, waitingDuration = _a.waitingDuration, cacheDuration = _a.cacheDuration;
            var values = _this.buildInitialValues(metric);
            _this.addIfPresent(values, 'dns_duration', dnsDuration);
            _this.addIfPresent(values, 'connection_duration', connectionDuration);
            _this.addIfPresent(values, 'request_duration', requestDuration);
            _this.addIfPresent(values, 'waiting_duration', waitingDuration);
            _this.addIfPresent(values, 'cache_duration', cacheDuration);
            var context = _this.buildInitialContext(metric);
            _this.pushMeasurement(values, context);
        }, { reportAllChanges: (_a = this.webVitalConfig) === null || _a === void 0 ? void 0 : _a.reportAllChanges });
    };
    WebVitalsWithAttribution.prototype.buildInitialValues = function (metric) {
        var _a;
        var indicator = metric.name.toLowerCase();
        return _a = {},
            _a[indicator] = metric.value,
            _a.delta = metric.delta,
            _a;
    };
    WebVitalsWithAttribution.prototype.buildInitialContext = function (metric) {
        var _a;
        var navigationEntryId = (_a = (0, utils_1.getItem)(instrumentationConstants_1.NAVIGATION_ID_STORAGE_KEY, utils_1.webStorageType.session)) !== null && _a !== void 0 ? _a : faro_core_1.unknownString;
        return {
            id: metric.id,
            rating: metric.rating,
            navigation_type: metric.navigationType,
            navigation_entry_id: navigationEntryId,
        };
    };
    WebVitalsWithAttribution.prototype.pushMeasurement = function (values, context) {
        var type = 'web-vitals';
        this.corePushMeasurement({ type: type, values: values }, { context: context });
    };
    WebVitalsWithAttribution.prototype.addIfPresent = function (source, key, metric) {
        if (metric) {
            source[key] = metric;
        }
    };
    return WebVitalsWithAttribution;
}());
exports.WebVitalsWithAttribution = WebVitalsWithAttribution;
//# sourceMappingURL=webVitalsWithAttribution.js.map