import { FieldType } from '../types/dataFrame.js';
import { guessFieldTypeFromNameAndValue } from './processDataFrame.js';

const ENTITY_MAP = {
  Inf: Infinity,
  NegInf: -Infinity,
  Undef: void 0,
  NaN: NaN
};
function decodeFieldValueEntities(lookup, values) {
  let key;
  for (key in lookup) {
    const repl = ENTITY_MAP[key];
    for (const idx of lookup[key]) {
      if (idx < values.length) {
        values[idx] = repl;
      }
    }
  }
}
function decodeFieldValueEnums(lookup, values) {
  for (let i = 0; i < values.length; i++) {
    values[i] = lookup[Number(values[i])];
  }
}
function guessFieldType(name, values) {
  for (const v of values) {
    if (v != null) {
      return guessFieldTypeFromNameAndValue(name, v);
    }
  }
  return FieldType.other;
}
function dataFrameFromJSON(dto) {
  const { schema, data } = dto;
  if (!schema || !schema.fields) {
    throw new Error("JSON needs a fields definition");
  }
  const length = data ? data.values.reduce((max, vals) => Math.max(max, vals.length), 0) : 0;
  const fields = schema.fields.map((f, index) => {
    var _a, _b, _c, _d;
    let buffer = data ? data.values[index] : [];
    let origLen = buffer.length;
    let type = f.type;
    if (origLen !== length) {
      buffer.length = length;
      buffer.fill(void 0, origLen);
    }
    let entities = (_a = data == null ? void 0 : data.entities) == null ? void 0 : _a[index];
    if (entities) {
      decodeFieldValueEntities(entities, buffer);
    }
    let enums = (_b = data == null ? void 0 : data.enums) == null ? void 0 : _b[index];
    if (enums) {
      decodeFieldValueEnums(enums, buffer);
      type = FieldType.string;
    }
    const nanos = (_c = data == null ? void 0 : data.nanos) == null ? void 0 : _c[index];
    const dataFrameField = {
      ...f,
      type: type != null ? type : guessFieldType(f.name, buffer),
      config: (_d = f.config) != null ? _d : {},
      values: buffer,
      // the presence of this prop is an optimization signal & lookup for consumers
      entities: entities != null ? entities : {}
    };
    if (nanos != null) {
      dataFrameField.nanos = nanos;
    }
    return dataFrameField;
  });
  return {
    ...schema,
    fields,
    length
  };
}
function dataFrameToJSON(frame) {
  const data = {
    values: []
  };
  const allNanos = [];
  let hasNanos = false;
  const schema = {
    refId: frame.refId,
    meta: frame.meta,
    name: frame.name,
    fields: frame.fields.map((f) => {
      const { values, nanos, state, display, ...sfield } = f;
      if ("entities" in sfield) {
        delete sfield.entities;
      }
      data.values.push(values);
      if (nanos != null) {
        allNanos.push(nanos);
        hasNanos = true;
      } else {
        allNanos.push(null);
      }
      return sfield;
    })
  };
  if (hasNanos) {
    data.nanos = allNanos;
  }
  return {
    schema,
    data
  };
}

export { dataFrameFromJSON, dataFrameToJSON, decodeFieldValueEntities, decodeFieldValueEnums };
//# sourceMappingURL=DataFrameJSON.js.map
