/*
 * livepatch_bsc1254196
 *
 * Fix for CVE-2025-40212, bsc#1254196
 *
 *  Copyright (c) 2026 SUSE
 *  Author: Lidong Zhong <lidong.zhong@suse.com>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */



/* klp-ccp: from fs/nfsd/nfsfh.c */
#include <linux/exportfs.h>
#include <linux/sunrpc/svcauth_gss.h>

/* klp-ccp: from fs/nfsd/nfsd.h */
#include <linux/types.h>
#include <linux/mount.h>

#include <linux/nfs.h>

/* klp-ccp: from include/uapi/linux/nfs3.h */
#define NFS3_FHSIZE		64

/* klp-ccp: from fs/nfsd/nfsd.h */
#include <linux/nfs4.h>
#include <linux/sunrpc/svc.h>
#include <linux/sunrpc/svc_xprt.h>
#include <linux/sunrpc/msg_prot.h>

#include <uapi/linux/nfsd/debug.h>

/* klp-ccp: from fs/nfsd/export.h */
#include <linux/sunrpc/cache.h>
#include <linux/percpu_counter.h>
#include <uapi/linux/nfsd/export.h>
#include <linux/nfs4.h>

#include "livepatch_bsc1254196.h"

struct nfsd4_fs_locations {
	uint32_t locations_count;
	struct nfsd4_fs_location *locations;
/* If we're not actually serving this data ourselves (only providing a
 * list of replicas that do serve it) then we set "migrated": */
	int migrated;
};

#define MAX_SECINFO_LIST	8

struct exp_flavor_info {
	u32	pseudoflavor;
	u32	flags;
};

enum {
	EXP_STATS_FH_STALE,
	EXP_STATS_IO_READ,
	EXP_STATS_IO_WRITE,
	EXP_STATS_COUNTERS_NUM
};

struct export_stats {
	time64_t		start_time;
	struct percpu_counter	counter[EXP_STATS_COUNTERS_NUM];
};

struct svc_export {
	struct cache_head	h;
	struct auth_domain *	ex_client;
	int			ex_flags;
	int			ex_fsid;
	struct path		ex_path;
	kuid_t			ex_anon_uid;
	kgid_t			ex_anon_gid;
	unsigned char *		ex_uuid; /* 16 byte fsid */
	struct nfsd4_fs_locations ex_fslocs;
	uint32_t		ex_nflavors;
	struct exp_flavor_info	ex_flavors[MAX_SECINFO_LIST];
	u32			ex_layout_types;
	struct nfsd4_deviceid_map *ex_devid_map;
	struct cache_detail	*cd;
	struct rcu_head		ex_rcu;
	unsigned long		ex_xprtsec_modes;
	struct export_stats	*ex_stats;
};

#define EX_WGATHER(exp)		((exp)->ex_flags & NFSEXP_GATHERED_WRITES)

__be32 check_nfsd_access(struct svc_export *exp, struct svc_rqst *rqstp,
			 bool may_bypass_gss);

static inline void exp_put(struct svc_export *exp)
{
	cache_put(&exp->h, exp->cd);
}

struct svc_export *rqst_exp_find(struct cache_req *reqp, struct net *net,
				 struct auth_domain *cl, struct auth_domain *gsscl,
				 int fsid_type, u32 *fsidv);

/* klp-ccp: from fs/nfsd/nfsd.h */
#define	NFSD_MAXVERS			4
#define NFSD_SUPPORTED_MINOR_VERSION	2

/* klp-ccp: from fs/nfsd/netns.h */
#include <net/net_namespace.h>
#include <net/netns/generic.h>
#include <linux/filelock.h>
#include <linux/nfs4.h>
#include <linux/percpu_counter.h>
#include <linux/percpu-refcount.h>
#include <linux/siphash.h>
#include <linux/sunrpc/stats.h>

enum {
	/* cache misses due only to checksum comparison failures */
	NFSD_STATS_PAYLOAD_MISSES,
	/* amount of memory (in bytes) currently consumed by the DRC */
	NFSD_STATS_DRC_MEM_USAGE,
	NFSD_STATS_RC_HITS,		/* repcache hits */
	NFSD_STATS_RC_MISSES,		/* repcache misses */
	NFSD_STATS_RC_NOCACHE,		/* uncached reqs */
	NFSD_STATS_FH_STALE,		/* FH stale error */
	NFSD_STATS_IO_READ,		/* bytes returned to read requests */
	NFSD_STATS_IO_WRITE,		/* bytes passed in write requests */
#ifdef CONFIG_NFSD_V4
	NFSD_STATS_FIRST_NFS4_OP,	/* count of individual nfsv4 operations */
	NFSD_STATS_LAST_NFS4_OP = NFSD_STATS_FIRST_NFS4_OP + LAST_NFS4_OP,
	NFSD_STATS_WDELEG_GETATTR,	/* count of getattr conflict with wdeleg */
#else
#error "klp-ccp: a preceeding branch should have been taken"
#endif
	NFSD_STATS_COUNTERS_NUM
};

struct nfsd_net {
	struct cld_net *cld_net;

	struct cache_detail *svc_expkey_cache;
	struct cache_detail *svc_export_cache;

	struct cache_detail *idtoname_cache;
	struct cache_detail *nametoid_cache;

	struct lock_manager nfsd4_manager;
	bool grace_ended;
	time64_t boot_time;

	struct dentry *nfsd_client_dir;

	/*
	 * reclaim_str_hashtbl[] holds known client info from previous reset/reboot
	 * used in reboot/reset lease grace period processing
	 *
	 * conf_id_hashtbl[], and conf_name_tree hold confirmed
	 * setclientid_confirmed info.
	 *
	 * unconf_str_hastbl[] and unconf_name_tree hold unconfirmed
	 * setclientid info.
	 */
	struct list_head *reclaim_str_hashtbl;
	int reclaim_str_hashtbl_size;
	struct list_head *conf_id_hashtbl;
	struct rb_root conf_name_tree;
	struct list_head *unconf_id_hashtbl;
	struct rb_root unconf_name_tree;
	struct list_head *sessionid_hashtbl;
	/*
	 * client_lru holds client queue ordered by nfs4_client.cl_time
	 * for lease renewal.
	 *
	 * close_lru holds (open) stateowner queue ordered by nfs4_stateowner.so_time
	 * for last close replay.
	 *
	 * All of the above fields are protected by the client_mutex.
	 */
	struct list_head client_lru;
	struct list_head close_lru;
	struct list_head del_recall_lru;

	/* protected by blocked_locks_lock */
	struct list_head blocked_locks_lru;

	struct delayed_work laundromat_work;

	/* client_lock protects the client lru list and session hash table */
	spinlock_t client_lock;

	/* protects blocked_locks_lru */
	spinlock_t blocked_locks_lock;

	struct file *rec_file;
	bool in_grace;
	const struct nfsd4_client_tracking_ops *client_tracking_ops;

	time64_t nfsd4_lease;
	time64_t nfsd4_grace;
	bool somebody_reclaimed;

	bool track_reclaim_completes;
	atomic_t nr_reclaim_complete;

	bool nfsd_net_up;
	bool lockd_up;

	seqlock_t writeverf_lock;
	unsigned char writeverf[8];

	/*
	 * Max number of connections this nfsd container will allow. Defaults
	 * to '0' which is means that it bases this on the number of threads.
	 */
	unsigned int max_connections;

	u32 clientid_base;
	u32 clientid_counter;
	u32 clverifier_counter;

	struct svc_info nfsd_info;
	struct percpu_ref nfsd_serv_ref;
	struct completion nfsd_serv_confirm_done;
	struct completion nfsd_serv_free_done;

	/*
	 * clientid and stateid data for construction of net unique COPY
	 * stateids.
	 */
	u32		s2s_cp_cl_id;
	struct idr	s2s_cp_stateids;
	spinlock_t	s2s_cp_lock;
	atomic_t	pending_async_copies;

	/*
	 * Version information
	 */
	bool nfsd_versions[NFSD_MAXVERS + 1];
	bool nfsd4_minorversions[NFSD_SUPPORTED_MINOR_VERSION + 1];

	/*
	 * Duplicate reply cache
	 */
	struct nfsd_drc_bucket   *drc_hashtbl;

	/* max number of entries allowed in the cache */
	unsigned int             max_drc_entries;

	/* number of significant bits in the hash value */
	unsigned int             maskbits;
	unsigned int             drc_hashsize;

	/*
	 * Stats and other tracking of on the duplicate reply cache.
	 * The longest_chain* fields are modified with only the per-bucket
	 * cache lock, which isn't really safe and should be fixed if we want
	 * these statistics to be completely accurate.
	 */

	/* total number of entries */
	atomic_t                 num_drc_entries;

	/* Per-netns stats counters */
	struct percpu_counter    counter[NFSD_STATS_COUNTERS_NUM];

	/* sunrpc svc stats */
	struct svc_stat          nfsd_svcstats;

	/* longest hash chain seen */
	unsigned int             longest_chain;

	/* size of cache when we saw the longest hash chain */
	unsigned int             longest_chain_cachesize;

	struct shrinker		*nfsd_reply_cache_shrinker;

	/* tracking server-to-server copy mounts */
	spinlock_t              nfsd_ssc_lock;
	struct list_head        nfsd_ssc_mount_list;
	wait_queue_head_t       nfsd_ssc_waitq;

	/* utsname taken from the process that starts the server */
	char			nfsd_name[UNX_MAXNODENAME+1];

	struct nfsd_fcache_disposal *fcache_disposal;

	siphash_key_t		siphash_key;

	atomic_t		nfs4_client_count;
	int			nfs4_max_clients;

	atomic_t		nfsd_courtesy_clients;
	struct shrinker		*nfsd_client_shrinker;
	struct work_struct	nfsd_shrinker_work;

	/* last time an admin-revoke happened for NFSv4.0 */
	time64_t		nfs40_last_revoke;

#if IS_ENABLED(CONFIG_NFS_LOCALIO)
#error "klp-ccp: non-taken branch"
#endif
};

extern unsigned int nfsd_net_id;

/* klp-ccp: from fs/nfsd/stats.h */
#include <uapi/linux/nfsd/stats.h>
#include <linux/percpu_counter.h>

static inline void nfsd_stats_fh_stale_inc(struct nfsd_net *nn,
					   struct svc_export *exp)
{
	percpu_counter_inc(&nn->counter[NFSD_STATS_FH_STALE]);
	if (exp && exp->ex_stats)
		percpu_counter_inc(&exp->ex_stats->counter[EXP_STATS_FH_STALE]);
}

/* klp-ccp: from fs/nfsd/nfsd.h */
#define	nfs_ok			cpu_to_be32(NFS_OK)

#define	nfserr_notdir		cpu_to_be32(NFSERR_NOTDIR)
#define	nfserr_isdir		cpu_to_be32(NFSERR_ISDIR)

#define	nfserr_stale		cpu_to_be32(NFSERR_STALE)

#define	nfserr_badhandle	cpu_to_be32(NFSERR_BADHANDLE)

#define	nfserr_nofilehandle	cpu_to_be32(NFSERR_NOFILEHANDLE)

#define	nfserr_symlink		cpu_to_be32(NFSERR_SYMLINK)

#define nfserr_wrong_type		cpu_to_be32(NFS4ERR_WRONG_TYPE)

enum {
	NFSERR_DROPIT = NFS4ERR_FIRST_FREE,
/* if a request fails due to kmalloc failure, it gets dropped.
 *  Client should resend eventually
 */

/* end-of-file indicator in readdir */
	NFSERR_EOF,

/* replay detected */
	NFSERR_REPLAY_ME,

/* nfs41 replay detected */
	NFSERR_REPLAY_CACHE,

/* symlink found where dir expected - handled differently to
 * other symlink found errors by NFSv3.
 */
	NFSERR_SYMLINK_NOT_DIR,
#define	nfserr_symlink_not_dir	cpu_to_be32(NFSERR_SYMLINK_NOT_DIR)
};

/* klp-ccp: from fs/nfsd/vfs.h */
#include <linux/fs.h>

/* klp-ccp: from fs/nfsd/nfsfh.h */
#include <linux/crc32.h>
#include <linux/sunrpc/svc.h>

#include <linux/exportfs.h>
#include <linux/nfs4.h>

struct knfsd_fh {
	unsigned int	fh_size;	/*
					 * Points to the current size while
					 * building a new file handle.
					 */
	union {
		char			fh_raw[NFS4_FHSIZE];
		struct {
			u8		fh_version;	/* == 1 */
			u8		fh_auth_type;	/* deprecated */
			u8		fh_fsid_type;
			u8		fh_fileid_type;
			u32		fh_fsid[]; /* flexible-array member */
		};
	};
};

struct svc_fh {
	struct knfsd_fh		fh_handle;	/* FH data */
	int			fh_maxsize;	/* max size for fh_handle */
	struct dentry *		fh_dentry;	/* validated dentry */
	struct svc_export *	fh_export;	/* export pointer */

	bool			fh_want_write;	/* remount protection taken */
	bool			fh_no_wcc;	/* no wcc data needed */
	bool			fh_no_atomic_attr;
						/*
						 * wcc data is not atomic with
						 * operation
						 */
	bool			fh_use_wgather;	/* NFSv2 wgather option */
	bool			fh_64bit_cookies;/* readdir cookie size */
	int			fh_flags;	/* FH flags */
	bool			fh_post_saved;	/* post-op attrs saved */
	bool			fh_pre_saved;	/* pre-op attrs saved */

	/* Pre-op attributes saved when inode is locked */
	__u64			fh_pre_size;	/* size before operation */
	struct timespec64	fh_pre_mtime;	/* mtime before oper */
	struct timespec64	fh_pre_ctime;	/* ctime before oper */
	/*
	 * pre-op nfsv4 change attr: note must check IS_I_VERSION(inode)
	 *  to find out if it is valid.
	 */
	u64			fh_pre_change;

	/* Post-op attributes saved in fh_fill_post_attrs() */
	struct kstat		fh_post_attr;	/* full attrs after operation */
	u64			fh_post_change; /* nfsv4 change; see above */
};

enum nfsd_fsid {
	FSID_DEV = 0,
	FSID_NUM,
	FSID_MAJOR_MINOR,
	FSID_ENCODE_DEV,
	FSID_UUID4_INUM,
	FSID_UUID8,
	FSID_UUID16,
	FSID_UUID16_INUM,
};

static inline int key_len(int type)
{
	switch(type) {
	case FSID_DEV:		return 8;
	case FSID_NUM: 		return 4;
	case FSID_MAJOR_MINOR:	return 12;
	case FSID_ENCODE_DEV:	return 8;
	case FSID_UUID4_INUM:	return 8;
	case FSID_UUID8:	return 8;
	case FSID_UUID16:	return 16;
	case FSID_UUID16_INUM:	return 24;
	default: return 0;
	}
}

/* klp-ccp: from fs/nfsd/vfs.h */
#define NFSD_MAY_LOCK			0x020

#define NFSD_MAY_BYPASS_GSS_ON_ROOT	0x100

#define NFSD_MAY_BYPASS_GSS		0x400

__be32		nfserrno (int errno);

__be32		nfsd_permission(struct svc_cred *cred, struct svc_export *exp,
				struct dentry *dentry, int acc);

/* klp-ccp: from fs/nfsd/trace.h */
#if !defined(_NFSD_TRACE_H) || defined(TRACE_HEADER_MULTI_READ)

#include <linux/tracepoint.h>
#include <linux/sunrpc/clnt.h>
#include <linux/sunrpc/xprt.h>
#include <trace/misc/nfs.h>
#include <trace/misc/sunrpc.h>

/* klp-ccp: from fs/nfsd/state.h */
#include <linux/idr.h>
#include <linux/refcount.h>
#include <linux/sunrpc/svc_xprt.h>

/* klp-ccp: from fs/nfsd/trace.h */


/* klp-ccp: not from file */
#undef inline

/* klp-ccp: from fs/nfsd/trace.h */
#include "klp_trace.h"
KLPR_TRACE_EVENT_CONDITION(nfsd, nfsd_fh_verify,
	TP_PROTO(
		const struct svc_rqst *rqstp,
		const struct svc_fh *fhp,
		umode_t type,
		int access
	),
	TP_ARGS(rqstp, fhp, type, access),
	TP_CONDITION(rqstp != NULL)
);
KLPR_TRACE_EVENT_CONDITION(nfsd, nfsd_fh_verify_err,
	TP_PROTO(
		const struct svc_rqst *rqstp,
		const struct svc_fh *fhp,
		umode_t type,
		int access,
		__be32 error
	),
	TP_ARGS(rqstp, fhp, type, access, error),
	TP_CONDITION(rqstp != NULL && error)
);

#define DEFINE_NFSD_FH_ERR_EVENT(name)			\
KLPR_TRACE_EVENT_CONDITION(nfsd, nfsd_##name,	\
	TP_PROTO(struct svc_rqst *rqstp,		\
		 struct svc_fh	*fhp,			\
		 int		status),		\
	TP_ARGS(rqstp, fhp, status),			\
	TP_CONDITION(rqstp != NULL))

DEFINE_NFSD_FH_ERR_EVENT(set_fh_dentry_badexport);
DEFINE_NFSD_FH_ERR_EVENT(set_fh_dentry_badhandle);



/* klp-ccp: from fs/nfsd/cache.h */
#include <linux/sunrpc/svc.h>

#else
#error "klp-ccp: a preceeding branch should have been taken"
/* klp-ccp: from fs/nfsd/trace.h */
#endif /* _NFSD_TRACE_H */

#include <trace/define_trace.h>

/* klp-ccp: from fs/nfsd/nfsfh.c */
extern int nfsd_acceptable(void *expv, struct dentry *dentry);

/* klp-ccp: from include/linux/compiler_types.h */
#define inline inline __gnu_inline __inline_maybe_unused notrace

/* klp-ccp: from fs/nfsd/nfsfh.c */
static inline __be32
nfsd_mode_check(struct dentry *dentry, umode_t requested)
{
	umode_t mode = d_inode(dentry)->i_mode & S_IFMT;

	if (requested == 0) /* the caller doesn't care */
		return nfs_ok;
	if (mode == requested) {
		if (mode == S_IFDIR && !d_can_lookup(dentry)) {
			WARN_ON_ONCE(1);
			return nfserr_notdir;
		}
		return nfs_ok;
	}
	if (mode == S_IFLNK) {
		if (requested == S_IFDIR)
			return nfserr_symlink_not_dir;
		return nfserr_symlink;
	}
	if (requested == S_IFDIR)
		return nfserr_notdir;
	if (mode == S_IFDIR)
		return nfserr_isdir;
	return nfserr_wrong_type;
}

extern __be32 nfsd_setuser_and_check_port(struct svc_rqst *rqstp,
					  struct svc_cred *cred,
					  struct svc_export *exp);

static inline __be32 check_pseudo_root(struct dentry *dentry,
				       struct svc_export *exp)
{
	if (!(exp->ex_flags & NFSEXP_V4ROOT))
		return nfs_ok;
	/*
	 * We're exposing only the directories and symlinks that have to be
	 * traversed on the way to real exports:
	 */
	if (unlikely(!d_is_dir(dentry) &&
		     !d_is_symlink(dentry)))
		return nfserr_stale;
	/*
	 * A pseudoroot export gives permission to access only one
	 * single directory; the kernel has to make another upcall
	 * before granting access to anything else under it:
	 */
	if (unlikely(dentry != exp->ex_path.dentry))
		return nfserr_stale;
	return nfs_ok;
}

static __be32 klpp_nfsd_set_fh_dentry(struct svc_rqst *rqstp, struct net *net,
				 struct svc_cred *cred,
				 struct auth_domain *client,
				 struct auth_domain *gssclient,
				 struct svc_fh *fhp)
{
	struct knfsd_fh	*fh = &fhp->fh_handle;
	struct fid *fid = NULL;
	struct svc_export *exp;
	struct dentry *dentry;
	int fileid_type;
	int data_left = fh->fh_size/4;
	int len;
	__be32 error;

	error = nfserr_badhandle;
	if (fh->fh_size == 0)
		return nfserr_nofilehandle;

	if (fh->fh_version != 1)
		return error;

	if (--data_left < 0)
		return error;
	if (fh->fh_auth_type != 0)
		return error;
	len = key_len(fh->fh_fsid_type) / 4;
	if (len == 0)
		return error;
	if (fh->fh_fsid_type == FSID_MAJOR_MINOR) {
		/* deprecated, convert to type 3 */
		len = key_len(FSID_ENCODE_DEV)/4;
		fh->fh_fsid_type = FSID_ENCODE_DEV;
		/*
		 * struct knfsd_fh uses host-endian fields, which are
		 * sometimes used to hold net-endian values. This
		 * confuses sparse, so we must use __force here to
		 * keep it from complaining.
		 */
		fh->fh_fsid[0] = new_encode_dev(MKDEV(ntohl((__force __be32)fh->fh_fsid[0]),
						      ntohl((__force __be32)fh->fh_fsid[1])));
		fh->fh_fsid[1] = fh->fh_fsid[2];
	}
	data_left -= len;
	if (data_left < 0)
		return error;
	exp = rqst_exp_find(rqstp ? &rqstp->rq_chandle : NULL,
			    net, client, gssclient,
			    fh->fh_fsid_type, fh->fh_fsid);
	fid = (struct fid *)(fh->fh_fsid + len);

	error = nfserr_stale;
	if (IS_ERR(exp)) {
		klpr_trace_nfsd_set_fh_dentry_badexport(rqstp, fhp, PTR_ERR(exp));

		if (PTR_ERR(exp) == -ENOENT)
			return error;

		return nfserrno(PTR_ERR(exp));
	}

	if (exp->ex_flags & NFSEXP_NOSUBTREECHECK) {
		/* Elevate privileges so that the lack of 'r' or 'x'
		 * permission on some parent directory will
		 * not stop exportfs_decode_fh from being able
		 * to reconnect a directory into the dentry cache.
		 * The same problem can affect "SUBTREECHECK" exports,
		 * but as nfsd_acceptable depends on correct
		 * access control settings being in effect, we cannot
		 * fix that case easily.
		 */
		struct cred *new = prepare_creds();
		if (!new) {
			error =  nfserrno(-ENOMEM);
			goto out;
		}
		new->cap_effective =
			cap_raise_nfsd_set(new->cap_effective,
					   new->cap_permitted);
		put_cred(override_creds(new));
		put_cred(new);
	} else {
		error = nfsd_setuser_and_check_port(rqstp, cred, exp);
		if (error)
			goto out;
	}

	/*
	 * Look up the dentry using the NFS file handle.
	 */
	error = nfserr_badhandle;

	fileid_type = fh->fh_fileid_type;

	if (fileid_type == FILEID_ROOT)
		dentry = dget(exp->ex_path.dentry);
	else {
		dentry = exportfs_decode_fh_raw(exp->ex_path.mnt, fid,
						data_left, fileid_type, 0,
						nfsd_acceptable, exp);
		if (IS_ERR_OR_NULL(dentry)) {
			klpr_trace_nfsd_set_fh_dentry_badhandle(rqstp, fhp,
					dentry ?  PTR_ERR(dentry) : -ESTALE);
			switch (PTR_ERR(dentry)) {
			case -ENOMEM:
			case -ETIMEDOUT:
				break;
			default:
				dentry = ERR_PTR(-ESTALE);
			}
		}
	}
	if (dentry == NULL)
		goto out;
	if (IS_ERR(dentry)) {
		if (PTR_ERR(dentry) != -EINVAL)
			error = nfserrno(PTR_ERR(dentry));
		goto out;
	}

	if (d_is_dir(dentry) &&
			(dentry->d_flags & DCACHE_DISCONNECTED)) {
		printk("nfsd: find_fh_dentry returned a DISCONNECTED directory: %pd2\n",
				dentry);
	}

	switch (fhp->fh_maxsize) {
	case NFS4_FHSIZE:
		if (dentry->d_sb->s_export_op->flags & EXPORT_OP_NOATOMIC_ATTR)
			fhp->fh_no_atomic_attr = true;
		fhp->fh_64bit_cookies = true;
		break;
	case NFS3_FHSIZE:
		if (dentry->d_sb->s_export_op->flags & EXPORT_OP_NOWCC)
			fhp->fh_no_wcc = true;
		fhp->fh_64bit_cookies = true;
		if (exp->ex_flags & NFSEXP_V4ROOT)
			goto out;
		break;
	case NFS_FHSIZE:
		fhp->fh_no_wcc = true;
		if (EX_WGATHER(exp))
			fhp->fh_use_wgather = true;
		if (exp->ex_flags & NFSEXP_V4ROOT)
			goto out;
	}

	fhp->fh_dentry = dentry;
	fhp->fh_export = exp;

	return 0;
out:
	exp_put(exp);
	return error;
}

__be32
klpp___fh_verify(struct svc_rqst *rqstp,
	    struct net *net, struct svc_cred *cred,
	    struct auth_domain *client,
	    struct auth_domain *gssclient,
	    struct svc_fh *fhp, umode_t type, int access)
{
	struct nfsd_net *nn = net_generic(net, nfsd_net_id);
	struct svc_export *exp = NULL;
	bool may_bypass_gss = false;
	struct dentry	*dentry;
	__be32		error;

	if (!fhp->fh_dentry) {
		error = klpp_nfsd_set_fh_dentry(rqstp, net, cred, client,
					   gssclient, fhp);
		if (error)
			goto out;
	}
	dentry = fhp->fh_dentry;
	exp = fhp->fh_export;

	klpr_trace_nfsd_fh_verify(rqstp, fhp, type, access);

	/*
	 * We still have to do all these permission checks, even when
	 * fh_dentry is already set:
	 * 	- fh_verify may be called multiple times with different
	 * 	  "access" arguments (e.g. nfsd_proc_create calls
	 * 	  fh_verify(...,NFSD_MAY_EXEC) first, then later (in
	 * 	  nfsd_create) calls fh_verify(...,NFSD_MAY_CREATE).
	 *	- in the NFSv4 case, the filehandle may have been filled
	 *	  in by fh_compose, and given a dentry, but further
	 *	  compound operations performed with that filehandle
	 *	  still need permissions checks.  In the worst case, a
	 *	  mountpoint crossing may have changed the export
	 *	  options, and we may now need to use a different uid
	 *	  (for example, if different id-squashing options are in
	 *	  effect on the new filesystem).
	 */
	error = check_pseudo_root(dentry, exp);
	if (error)
		goto out;

	error = nfsd_setuser_and_check_port(rqstp, cred, exp);
	if (error)
		goto out;

	error = nfsd_mode_check(dentry, type);
	if (error)
		goto out;

	/*
	 * pseudoflavor restrictions are not enforced on NLM,
	 * which clients virtually always use auth_sys for,
	 * even while using RPCSEC_GSS for NFS.
	 */
	if (access & NFSD_MAY_LOCK)
		goto skip_pseudoflavor_check;
	if (access & NFSD_MAY_BYPASS_GSS)
		may_bypass_gss = true;
	/*
	 * Clients may expect to be able to use auth_sys during mount,
	 * even if they use gss for everything else; see section 2.3.2
	 * of rfc 2623.
	 */
	if (access & NFSD_MAY_BYPASS_GSS_ON_ROOT
			&& exp->ex_path.dentry == dentry)
		may_bypass_gss = true;

	error = check_nfsd_access(exp, rqstp, may_bypass_gss);
	if (error)
		goto out;

skip_pseudoflavor_check:
	/* Finally, check access permissions. */
	error = nfsd_permission(cred, exp, dentry, access);
out:
	klpr_trace_nfsd_fh_verify_err(rqstp, fhp, type, access, error);
	if (error == nfserr_stale)
		nfsd_stats_fh_stale_inc(nn, exp);
	return error;
}



#include <linux/livepatch.h>

extern typeof(check_nfsd_access) check_nfsd_access
	 KLP_RELOC_SYMBOL(nfsd, nfsd, check_nfsd_access);
extern typeof(nfsd_acceptable) nfsd_acceptable
	 KLP_RELOC_SYMBOL(nfsd, nfsd, nfsd_acceptable);
extern typeof(nfsd_net_id) nfsd_net_id KLP_RELOC_SYMBOL(nfsd, nfsd, nfsd_net_id);
extern typeof(nfsd_permission) nfsd_permission
	 KLP_RELOC_SYMBOL(nfsd, nfsd, nfsd_permission);
extern typeof(nfsd_setuser_and_check_port) nfsd_setuser_and_check_port
	 KLP_RELOC_SYMBOL(nfsd, nfsd, nfsd_setuser_and_check_port);
extern typeof(nfserrno) nfserrno KLP_RELOC_SYMBOL(nfsd, nfsd, nfserrno);
extern typeof(rqst_exp_find) rqst_exp_find
	 KLP_RELOC_SYMBOL(nfsd, nfsd, rqst_exp_find);
