/*
 * livepatch_bsc1251956
 *
 * Fix for bsc#1251956
 *
 *  Copyright (c) 2025 SUSE
 *  Author: Vincenzo Mezzela <vincenzo.mezzela@suse.com>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#if IS_ENABLED(CONFIG_PPC_FTRACE_OUT_OF_LINE)

#include <linux/module.h>
#include <linux/elf.h>
#include <linux/moduleloader.h>
#include <linux/err.h>
#include <linux/vmalloc.h>
#include <linux/ftrace.h>
#include <linux/bug.h>
#include <linux/uaccess.h>
#include <linux/kernel.h>
#include <asm/module.h>
#include <asm/firmware.h>
#include <asm/text-patching.h>
#include <linux/sort.h>
#include <asm/setup.h>
#include <asm/sections.h>
#include <asm/inst.h>

extern void ftrace_caller(void);
extern void ftrace_regs_caller(void);
extern int patch_uint(void *addr, unsigned int val);
extern unsigned long stub_for_addr(const Elf64_Shdr *sechdrs, unsigned long
				   addr, struct module *me, const char *name);

struct ppc64_stub_entry {
	/*
	 * 28 byte jump instruction sequence (7 instructions) that can
	 * hold ppc64_stub_insns or stub_insns. Must be 8-byte aligned
	 * with PCREL kernels that use prefix instructions in the stub.
	 */
	u32 jump[7];
	/* Used by ftrace to identify stubs */
	u32 magic;
	/* Data for the above code */
	func_desc_t funcdata;
} __aligned(8);

static unsigned long stub_func_addr(func_desc_t func)
{
	return func.addr;
}



static int klpp_setup_ftrace_ool_stubs(const Elf64_Shdr *sechdrs, unsigned long
				       addr, struct module *me)
{
#ifdef CONFIG_PPC_FTRACE_OUT_OF_LINE
	unsigned int i, total_stubs, num_stubs;
	struct ppc64_stub_entry *stub;

	total_stubs = sechdrs[me->arch.stubs_section].sh_size / sizeof(*stub);
	num_stubs = roundup(me->arch.ool_stub_count * sizeof(struct ftrace_ool_stub),
			    sizeof(struct ppc64_stub_entry)) / sizeof(struct ppc64_stub_entry);

	/* Find the next available entry */
	stub = (void *)sechdrs[me->arch.stubs_section].sh_addr;
	for (i = 0; stub_func_addr(stub[i].funcdata); i++)
		if (WARN_ON(i >= total_stubs))
			return -1;

	if (WARN_ON(i + num_stubs > total_stubs))
		return -1;

	stub += i;
	me->arch.ool_stubs = (struct ftrace_ool_stub *)stub;

	/* reserve stubs */
	for (i = 0; i < num_stubs; i++)
		if (patch_u32((void *)&stub[i].funcdata, PPC_RAW_NOP()))
			return -1;
#endif

	return 0;
}

int klpp_module_finalize_ftrace(struct module *mod, const Elf_Shdr *sechdrs)
{
	mod->arch.tramp = stub_for_addr(sechdrs,
					(unsigned long)ftrace_caller,
					mod,
					"ftrace_caller");
#ifdef CONFIG_DYNAMIC_FTRACE_WITH_REGS
	mod->arch.tramp_regs = stub_for_addr(sechdrs,
					(unsigned long)ftrace_regs_caller,
					mod,
					"ftrace_regs_caller");
	if (!mod->arch.tramp_regs)
		return -ENOENT;
#endif

	if (!mod->arch.tramp)
		return -ENOENT;

	if (klpp_setup_ftrace_ool_stubs(sechdrs, mod->arch.tramp, mod))
		return -ENOENT;

	return 0;
}

#include <linux/livepatch.h>

extern typeof(stub_for_addr) stub_for_addr KLP_RELOC_SYMBOL(vmlinux, vmlinux,
							    stub_for_addr);
extern typeof(ftrace_caller) ftrace_caller KLP_RELOC_SYMBOL(vmlinux, vmlinux,
							    ftrace_caller);
extern typeof(ftrace_regs_caller) ftrace_regs_caller KLP_RELOC_SYMBOL(vmlinux,
								      vmlinux,
								      ftrace_regs_caller);
extern typeof(patch_uint) patch_uint KLP_RELOC_SYMBOL(vmlinux, vmlinux,
						      patch_uint);

#endif
