# Copyright (c) 2018 SUSE Linux LLC.  All rights reserved.
#
# This file is part of suse-migration-services.
#
# suse-migration-services is free software: you can redistribute it and/or
# modify it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# suse-migration-services is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with suse-migration-services. If not, see <http://www.gnu.org/licenses/>
#
import logging
import glob
import os
import shutil

# project
from suse_migration_services.command import Command
from suse_migration_services.fstab import Fstab
from suse_migration_services.defaults import Defaults
from suse_migration_services.logger import Logger
from suse_migration_services.migration_config import MigrationConfig

from suse_migration_services.exceptions import (
    DistMigrationHostNetworkException
)


def main():
    """
    DistMigration activate host network setup

    Setup and activate the network as it is setup on the host
    to become migrated. This includes the import of the network
    configuration from the migration host
    """
    Logger.setup()
    log = logging.getLogger(Defaults.get_migration_log_name())
    root_path = Defaults.get_system_root_path()

    system_mount = Fstab()
    system_mount.read(
        Defaults.get_system_mount_info_file()
    )

    sysconfig_network_providers = os.sep.join(
        [root_path, 'etc', 'sysconfig', 'network', 'providers']
    )
    sysconfig_network_setup = os.sep.join(
        [root_path, 'etc', 'sysconfig', 'network', '*']
    )
    try:
        log.info('Running setup host network service')
        Command.run(
            [
                'mount', '--bind', sysconfig_network_providers,
                '/etc/sysconfig/network/providers'
            ]
        )
        system_mount.add_entry(
            sysconfig_network_providers, '/etc/sysconfig/network/providers'
        )
        for network_setup in glob.glob(sysconfig_network_setup):
            if os.path.isfile(network_setup):
                shutil.copy(
                    network_setup, '/etc/sysconfig/network'
                )
        setup_interfaces(root_path)
        Command.run(
            ['systemctl', 'reload', 'network']
        )
        wicked2nm_migrate(root_path)
        system_mount.export(
            Defaults.get_system_mount_info_file()
        )
    except Exception as issue:
        log.error(
            'Preparation of migration host network failed with {0}'.format(
                issue
            )
        )
        raise DistMigrationHostNetworkException(
            'Preparation of migration host network failed with {0}'.format(
                issue
            )
        ) from issue


def log_network_details():
    """
    Provide detailed information about the current network setup

    The method must be called in an active and online network state to provide
    most useful information about the network interfaces and its setup.
    """
    log = logging.getLogger(Defaults.get_migration_log_name())
    log.info(
        'All Network Interfaces {0}{1}'.format(
            os.linesep, Command.run(
                ['ip', 'a'], raise_on_error=False
            ).output
        )
    )
    log.info(
        'Routing Tables {0}{1}'.format(
            os.linesep, Command.run(
                ['ip', 'r'], raise_on_error=False
            ).output
        )
    )
    log.info(
        'DNS Resolver {0}{1}'.format(
            os.linesep, Command.run(
                ['cat', '/etc/resolv.conf'], raise_on_error=False
            ).output
        )
    )
    bonding_paths = '/proc/net/bonding/bond*'
    if os.path.exists(os.path.dirname(bonding_paths)):
        log.info(
            'Network Bonding {0}{1}'.format(
                os.linesep, Command.run(
                    ['cat', bonding_paths], raise_on_error=False
                ).output
            )
        )


def wicked2nm_migrate(root_path):
    """
    Migrate from wicked to NetworkManager

    Requires the `wicked show-config` xml along with netconfig files to be
    present at `/var/cache/wicked_config`. These files are supposed to be
    generated by rpm scriptlets. If the files aren't present this part is
    skipped.
    """
    log = logging.getLogger(Defaults.get_migration_log_name())
    wicked_config_path = root_path + '/var/cache/wicked_config'
    migration_config = MigrationConfig()
    net_info = migration_config.get_network_info()

    if not os.path.exists(os.sep.join([wicked_config_path, 'config.xml'])):
        log.info('No wicked config present, skipping wicked2nm host network setup.')
        return
    if Command.run(['rpm', '--query', '--quiet', 'wicked2nm'], raise_on_error=False).returncode != 0:
        log.info('No wicked2nm present, skipping wicked2nm host network setup.')
        return
    if Command.run(['rpm', '--query', '--quiet', 'NetworkManager-config-server'], raise_on_error=False).returncode != 0:
        log.info('No NetworkManager-config-server present, skipping wicked2nm host network setup.')
        return

    log.info('Running wicked2nm')
    wicked2nm_cmd = [
        'wicked2nm', 'migrate', '--activate-connections',
        '--netconfig-path', os.sep.join([wicked_config_path, 'config']),
        '--netconfig-dhcp-path', os.sep.join([wicked_config_path, 'dhcp']),
        os.sep.join([wicked_config_path, 'config.xml'])
    ]
    if net_info and 'wicked2nm-continue-migration' in net_info and net_info['wicked2nm-continue-migration']:
        log.info('Ignoring wicked2nm warnings')
        wicked2nm_cmd = wicked2nm_cmd + ['--continue-migration']
    try:
        Command.run(wicked2nm_cmd)
    except Exception as issue:
        wicked2nm_cmd = [
            'wicked2nm', 'show',
            '--netconfig-path', os.sep.join([wicked_config_path, 'config']),
            '--netconfig-dhcp-path', os.sep.join([wicked_config_path, 'dhcp']),
            os.sep.join([wicked_config_path, 'config.xml'])
        ]
        log.info(
            'wicked2nm config {0}{1}'.format(
                os.linesep, Command.run(
                    wicked2nm_cmd, raise_on_error=False
                ).output
            )
        )
        log_network_details()

        raise DistMigrationHostNetworkException('Migration from wicked to NetworkManager failed with {0}'.format(issue))

    # Wait for NetworkManager online to fix dhcp race condition
    Command.run(
        ['nm-online', '-q']
    )


def setup_interfaces(root_path):
    """
    Sets up the hosts interfaces

    Due to a switch to predictable names this function applies some special udev
    rules that apply the persistent naming scheme to existing interfaces. The
    special udev rule is supposed to be generated by rpm scriptlets. If the file
    isn't present this part is skipped.
    """
    log = logging.getLogger(Defaults.get_migration_log_name())
    migration_udev_rules = root_path + '/var/cache/udev_rules/70-migration-persistent-net.rules'

    if not os.path.exists(migration_udev_rules):
        log.info('No migration udev rules present, skipping copy and application of migration rules')
        return

    target_file = '/etc/udev/rules.d/70-migration-persistent-net.rules'
    log.info('Copying migration udev rules')
    if not os.path.exists(os.path.dirname(target_file)):
        Command.run(
            ['mkdir', '-p', os.path.dirname(target_file)]
        )
    shutil.copy(migration_udev_rules, target_file)

    log.info('Running udevadm to apply network udev rules')
    Command.run(['udevadm', 'control', '--reload'])
    Command.run(['udevadm', 'trigger', '--subsystem-match=net', '--action=add'])
    Command.run(['udevadm', 'settle'])
