/*
 * Copyright (C) 2022 - 2025, Stephan Mueller <smueller@chronox.de>
 *
 * License: see LICENSE file in root directory
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ALL OF
 * WHICH ARE HEREBY DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF NOT ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

#include "compare.h"
#include "ext_headers.h"
#include "lc_ascon_lightweight.h"
#include "math_helper.h"
#include "visibility.h"

#include "sha3_c.h"
#include "sha3_arm_asm.h"
#include "sha3_arm_ce.h"
#include "sha3_arm_neon.h"
#include "sha3_avx2.h"
#include "sha3_avx512.h"
#include "sha3_riscv_asm.h"

static void ak_tester_enc(struct lc_aead_ctx *ak, const uint8_t *pt,
			  uint8_t *ct, size_t ptlen)
{
	size_t todo;

	while (ptlen) {
		todo = min_size(ptlen, 3);

		lc_aead_enc_update(ak, pt, ct, todo);
		pt += todo;
		ct += todo;
		ptlen -= todo;
	}
}

static void ak_tester_dec(struct lc_aead_ctx *ak, const uint8_t *ct,
			  uint8_t *pt, size_t ptlen)
{
	size_t todo;

	while (ptlen) {
		todo = min_size(ptlen, 3);

		lc_aead_dec_update(ak, ct, pt, todo);
		pt += todo;
		ct += todo;
		ptlen -= todo;
	}
}
static int ascon_tester_one(const uint8_t *pt, size_t ptlen,
			    const uint8_t *nonce, size_t noncelen,
			    const uint8_t *aad, size_t aadlen,
			    const uint8_t *key, size_t keylen,
			    const uint8_t *exp_ct, const uint8_t *exp_tag,
			    size_t exp_tag_len)
{
	struct lc_aead_ctx *al_heap = NULL;
	ssize_t ret;
	int ret_checked = 0;
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wvla"
	uint8_t out_enc[ptlen];
	uint8_t out_dec[ptlen];
	uint8_t tag[exp_tag_len];
#pragma GCC diagnostic pop
	LC_AL_CTX_ON_STACK(al);

	/* One shot encryption with pt ptr != ct ptr */
	if (lc_aead_setkey(al, key, keylen, nonce, noncelen))
		return -EFAULT;
	lc_aead_enc_init(al, aad, aadlen);
	ak_tester_enc(al, pt, out_enc, ptlen);
	lc_aead_enc_final(al, tag, exp_tag_len);
	lc_aead_zero(al);

	ret_checked +=
		lc_compare(out_enc, exp_ct, ptlen,
			   "Ascon lightweight crypt: Encryption, ciphertext");
	ret_checked += lc_compare(tag, exp_tag, exp_tag_len,
				  "Ascon lightweight crypt: Encryption, tag");

	//bin2print(out_enc, ptlen, stderr, "out_enc");
	//bin2print(tag, exp_tag_len, stderr, "tag");

	/* One shot encryption with pt ptr == ct ptr */
	if (lc_al_alloc(&al_heap))
		return 1;

	lc_aead_setkey(al_heap, key, keylen, nonce, noncelen);

	memcpy(out_enc, pt, ptlen);
	lc_aead_enc_init(al_heap, aad, aadlen);
	ak_tester_enc(al_heap, out_enc, out_enc, ptlen);
	lc_aead_enc_final(al_heap, tag, exp_tag_len);
	lc_aead_zero_free(al_heap);

	ret_checked += lc_compare(out_enc, exp_ct, ptlen,
				  "Ascon lightweight: Encryption, ciphertext");
	ret_checked += lc_compare(tag, exp_tag, exp_tag_len,
				  "Ascon lightweight: Encryption, tag");

	/* One shot decryption with pt ptr != ct ptr */
	if (lc_aead_setkey(al, key, keylen, nonce, noncelen))
		return -EFAULT;
	lc_aead_dec_init(al, aad, aadlen);
	ak_tester_dec(al, out_enc, out_dec, ptlen);
	ret = lc_aead_dec_final(al, tag, exp_tag_len);
	//bin2print(out_dec, ptlen, stderr, "out_enc");
	lc_aead_zero(al);
	if (ret < 0)
		return 1;

	ret_checked +=
		lc_compare(out_dec, pt, ptlen,
			   "Ascon lightweight crypt: Decryption, plaintext");

	/* Check authentication error */
	if (lc_aead_setkey(al, key, keylen, nonce, noncelen))
		return -EFAULT;

	out_enc[0] = (uint8_t)((out_enc[0] + 1) & 0xff);
	lc_aead_dec_init(al, aad, aadlen);
	ak_tester_dec(al, out_enc, out_dec, ptlen);
	ret = lc_aead_dec_final(al, tag, exp_tag_len);
	lc_aead_zero(al);
	if (ret != -EBADMSG)
		return 1;

	return ret_checked;
}

static int aascon_tester_128(void)
{
	/*
	 * Vector 1089 from genkat_crypto_aead_asconaead128_ref generated by
	 * code https://github.com/ascon/ascon-c
	 */
	static const uint8_t pt[] = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
				      0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D,
				      0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13, 0x14,
				      0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B,
				      0x1C, 0x1D, 0x1E, 0x1F };
	static const uint8_t aad[] = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
				       0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D,
				       0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13, 0x14,
				       0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B,
				       0x1C, 0x1D, 0x1E, 0x1F };
	static const uint8_t key[] = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
				       0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B,
				       0x0C, 0x0D, 0x0E, 0x0F };
	static const uint8_t nonce[] = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
					 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B,
					 0x0C, 0x0D, 0x0E, 0x0F };
	static const uint8_t exp_ct[] = { 0x4C, 0x08, 0x6D, 0x27, 0xA3, 0xB5,
					  0x1A, 0x23, 0x33, 0xCF, 0xC7, 0xF2,
					  0x21, 0x72, 0xA9, 0xBC, 0xAD, 0x88,
					  0xB8, 0xD4, 0xD7, 0x7E, 0x50, 0x62,
					  0x2D, 0x78, 0x83, 0x45, 0xFA, 0x7B,
					  0xEE, 0x44 };
	static const uint8_t exp_tag[] = { 0x68, 0x91, 0x5D, 0x3F, 0x94, 0x22,
					   0x28, 0x9F, 0x23, 0x49, 0xD6, 0xA3,
					   0xB4, 0x16, 0x03, 0x97 };

	printf("Ascon lightweight 128/128 crypt ctx len %u, state len %u\n",
	       (unsigned int)LC_AL_CTX_SIZE, (unsigned int)LC_AL_STATE_SIZE);
	return ascon_tester_one(pt, sizeof(pt), nonce, sizeof(nonce), aad,
				sizeof(aad), key, sizeof(key), exp_ct, exp_tag,
				sizeof(exp_tag));
}

LC_TEST_FUNC(int, main, int argc, char *argv[])
{
	int ret = 0;
	(void)argc;
	(void)argv;

	ret += aascon_tester_128();

	return ret;
}
