// Copyright (c) 2020 hxdmp developers
//
// Licensed under the Apache License, Version 2.0
// <LICENSE-APACHE or http://www.apache.org/licenses/LICENSE-2.0> or the MIT
// license <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. All files in the project carrying such notice may not be copied,
// modified, or distributed except according to those terms.

//! `hxdmp`
//!
//! A small utility to create hexdump output from byte slices on the given
//! writer.
//!
//! # Example
//!
//! ```
//! # use hxdmp::hexdump;
//! # use std::io::Result;
//! #
//! # fn main() -> Result<()> {
//!     let some_bytes = b"Hello, World! I'm hexy";
//!     let mut buffer = Vec::new();
//!     assert!(hexdump(some_bytes, &mut buffer).is_ok());
//!     assert_eq!(
//!         r#"0000: 48 65 6C 6C 6F 2C 20 57 6F 72 6C 64 21 20 49 27  Hello,.World!.I'
//! 0016: 6D 20 68 65 78 79                                m.hexy"#,
//!         String::from_utf8_lossy(&buffer)
//!     );
//! #     Ok(())
//! # }
// rustc lints
#![deny(
    absolute_paths_not_starting_with_crate,
    anonymous_parameters,
    array_into_iter,
    asm_sub_register,
    bare_trait_objects,
    bindings_with_variant_name,
    box_pointers,
    break_with_label_and_loop,
    cenum_impl_drop_cast,
    clashing_extern_declarations,
    coherence_leak_check,
    confusable_idents,
    const_evaluatable_unchecked,
    const_item_mutation,
    dead_code,
    deprecated,
    deprecated_in_future,
    drop_bounds,
    elided_lifetimes_in_paths,
    ellipsis_inclusive_range_patterns,
    explicit_outlives_requirements,
    exported_private_dependencies,
    forbidden_lint_groups,
    function_item_references,
    illegal_floating_point_literal_pattern,
    improper_ctypes,
    improper_ctypes_definitions,
    incomplete_features,
    indirect_structural_match,
    inline_no_sanitize,
    invalid_doc_attributes,
    invalid_value,
    irrefutable_let_patterns,
    keyword_idents,
    late_bound_lifetime_arguments,
    legacy_derive_helpers,
    macro_use_extern_crate,
    meta_variable_misuse,
    missing_abi,
    missing_copy_implementations,
    missing_debug_implementations,
    missing_docs,
    mixed_script_confusables,
    mutable_borrow_reservation_conflict,
    no_mangle_generic_items,
    non_ascii_idents,
    non_camel_case_types,
    non_shorthand_field_patterns,
    non_snake_case,
    non_upper_case_globals,
    nontrivial_structural_match,
    noop_method_call,
    overlapping_range_endpoints,
    path_statements,
    pointer_structural_match,
    private_in_public,
    proc_macro_back_compat,
    proc_macro_derive_resolution_fallback,
    redundant_semicolons,
    renamed_and_removed_lints,
    semicolon_in_expressions_from_macros,
    single_use_lifetimes,
    stable_features,
    temporary_cstring_as_ptr,
    trivial_bounds,
    trivial_casts,
    trivial_numeric_casts,
    type_alias_bounds,
    tyvar_behind_raw_pointer,
    unaligned_references,
    uncommon_codepoints,
    unconditional_recursion,
    uninhabited_static,
    unknown_lints,
    unnameable_test_items,
    unreachable_code,
    unreachable_patterns,
    unreachable_pub,
    unsafe_code,
    unsafe_op_in_unsafe_fn,
    unstable_features,
    unstable_name_collisions,
    unsupported_naked_functions,
    unused_allocation,
    unused_assignments,
    unused_attributes,
    unused_braces,
    unused_comparisons,
    unused_crate_dependencies,
    unused_doc_comments,
    unused_extern_crates,
    unused_features,
    unused_import_braces,
    unused_imports,
    unused_labels,
    unused_lifetimes,
    unused_macros,
    unused_must_use,
    unused_mut,
    unused_parens,
    unused_qualifications,
    unused_results,
    unused_unsafe,
    unused_variables,
    variant_size_differences,
    where_clauses_object_safety,
    while_true
)]

use std::io::{Result, Write};

/// Create a hexdump on the given writer
///
/// # Example
///
/// ```
/// # use hxdmp::hexdump;
/// # use std::io::Result;
/// #
/// # fn main() -> Result<()> {
///     let some_bytes = b"Hello, World! I'm hexy";
///     let mut buffer = Vec::new();
///     assert!(hexdump(some_bytes, &mut buffer).is_ok());
///     assert_eq!(
///         r#"0000: 48 65 6C 6C 6F 2C 20 57 6F 72 6C 64 21 20 49 27  Hello,.World!.I'
/// 0016: 6D 20 68 65 78 79                                m.hexy"#,
///         String::from_utf8_lossy(&buffer)
///     );
/// #     Ok(())
/// # }
pub fn hexdump<W>(buffer: &[u8], writer: &mut W) -> Result<()>
where
    W: Write,
{
    hexdumpm(buffer, None, writer)
}

/// Create a hexdump on the given writer, restricted to an optional maximum number
/// of lines
///
/// # Example
///
/// ```
/// # use hxdmp::hexdumpm;
/// # use std::io::Result;
/// #
/// # fn main() -> Result<()> {
///     let some_bytes = b"Hello, World! I'm hexy";
///     let mut buffer = Vec::new();
///     assert!(hexdumpm(some_bytes, Some(1), &mut buffer).is_ok());
///     assert_eq!(
///         r#"0000: 48 65 6C 6C 6F 2C 20 57 6F 72 6C 64 21 20 49 27  Hello,.World!.I'"#,
///         String::from_utf8_lossy(&buffer)
///     );
/// #     Ok(())
/// # }
pub fn hexdumpm<W>(buffer: &[u8], max_lines_opt: Option<usize>, writer: &mut W) -> Result<()>
where
    W: Write,
{
    let sixteen_iter = buffer.chunks(16).enumerate();

    if let Some(max) = max_lines_opt {
        for (line, parts) in sixteen_iter {
            if line < max {
                hex(line, parts, writer)?;
            } else {
                break;
            }
        }
    } else {
        for (line, parts) in sixteen_iter {
            hex(line, parts, writer)?;
        }
    }
    Ok(())
}

fn hex<W>(line: usize, parts: &[u8], writer: &mut W) -> Result<()>
where
    W: Write,
{
    if line > 0 {
        writeln!(writer)?;
    }
    write!(writer, "{:04}: ", line * 16)?;
    for b in parts {
        write!(writer, "{:02X} ", b)?;
    }

    for _ in parts.len()..16 {
        write!(writer, "   ")?;
    }

    write!(writer, " ")?;

    for b in parts {
        let ch = *b as char;
        if ch.is_ascii_graphic() {
            write!(writer, "{}", ch)?;
        } else {
            write!(writer, ".")?;
        }
    }
    Ok(())
}

#[cfg(test)]
mod tests {
    use super::hexdump;
    use lazy_static::lazy_static;

    lazy_static! {
        static ref TUPLE1: (Vec<u8>, &'static str) = {
            (
                vec![0, 1, 3, 4, 65],
                r#"0000: 00 01 03 04 41                                   ....A"#,
            )
        };
        static ref TUPLE2: (Vec<u8>, &'static str) = {
            (
                vec![
                    0x00, 0x00, 0x00, 0x13, 0x53, 0x53, 0x48, 0x2D, 0x32, 0x2E, 0x30, 0x2D, 0x74,
                    0x75, 0x73, 0x73, 0x68, 0x5F, 0x30, 0x2E, 0x31, 0x2E, 0x30,
                ],
                r#"0000: 00 00 00 13 53 53 48 2D 32 2E 30 2D 74 75 73 73  ....SSH-2.0-tuss
0016: 68 5F 30 2E 31 2E 30                             h_0.1.0"#,
            )
        };
        static ref TUPLE3: (Vec<u8>, &'static str) = {
            (
                vec![
                    0x00, 0x00, 0x00, 0x13, 0x53, 0x53, 0x48, 0x2D, 0x32, 0x2E, 0x30, 0x2D, 0x74,
                    0x75, 0x73, 0x73, 0x68, 0x5F, 0x30, 0x2E, 0x31, 0x2E, 0x30, 0x00, 0x00, 0x00,
                    0x13, 0x53, 0x53, 0x48, 0x2D, 0x32, 0x2E, 0x30, 0x2D, 0x4F, 0x70, 0x65, 0x6E,
                    0x53, 0x53, 0x48, 0x5F, 0x38, 0x2E, 0x32, 0x00, 0x00, 0x01, 0x0D, 0x14, 0xD4,
                    0x14, 0x01, 0x81, 0x25, 0x28, 0xC2, 0x47, 0xCE, 0xF9, 0x88, 0x6E, 0xE1, 0x46,
                    0xE7, 0x3E, 0x00, 0x00, 0x00, 0x2E, 0x63, 0x75, 0x72, 0x76, 0x65, 0x32, 0x35,
                    0x35, 0x31, 0x39, 0x2D, 0x73, 0x68, 0x61, 0x32, 0x35, 0x36, 0x2C, 0x63, 0x75,
                    0x72, 0x76, 0x65, 0x32, 0x35, 0x35, 0x31, 0x39, 0x2D, 0x73, 0x68, 0x61, 0x32,
                    0x35, 0x36, 0x40, 0x6C, 0x69, 0x62, 0x73, 0x73, 0x68, 0x2E, 0x6F, 0x72, 0x67,
                    0x00, 0x00, 0x00, 0x25, 0x73, 0x73, 0x68, 0x2D, 0x65, 0x64, 0x32, 0x35, 0x35,
                    0x31, 0x39, 0x2C, 0x72, 0x73, 0x61, 0x2D, 0x73, 0x68, 0x61, 0x32, 0x2D, 0x35,
                    0x31, 0x32, 0x2C, 0x72, 0x73, 0x61, 0x2D, 0x73, 0x68, 0x61, 0x32, 0x2D, 0x32,
                    0x35, 0x36, 0x00, 0x00, 0x00, 0x1D, 0x63, 0x68, 0x61, 0x63, 0x68, 0x61, 0x32,
                    0x30, 0x2D, 0x70, 0x6F, 0x6C, 0x79, 0x31, 0x33, 0x30, 0x35, 0x40, 0x6F, 0x70,
                    0x65, 0x6E, 0x73, 0x73, 0x68, 0x2E, 0x63, 0x6F, 0x6D, 0x00, 0x00, 0x00, 0x1D,
                    0x63, 0x68, 0x61, 0x63, 0x68, 0x61, 0x32, 0x30, 0x2D, 0x70, 0x6F, 0x6C, 0x79,
                    0x31, 0x33, 0x30, 0x35, 0x40, 0x6F, 0x70, 0x65, 0x6E, 0x73, 0x73, 0x68, 0x2E,
                    0x63, 0x6F, 0x6D, 0x00, 0x00, 0x00, 0x1D, 0x68, 0x6D, 0x61, 0x63, 0x2D, 0x73,
                    0x68, 0x61, 0x32, 0x2D, 0x35, 0x31, 0x32, 0x2D, 0x65, 0x74, 0x6D, 0x40, 0x6F,
                    0x70, 0x65, 0x6E, 0x73, 0x73, 0x68, 0x2E, 0x63, 0x6F, 0x6D, 0x00, 0x00, 0x00,
                    0x1D, 0x68, 0x6D, 0x61, 0x63, 0x2D, 0x73, 0x68, 0x61, 0x32, 0x2D, 0x35, 0x31,
                    0x32, 0x2D, 0x65, 0x74, 0x6D, 0x40, 0x6F, 0x70, 0x65, 0x6E, 0x73, 0x73, 0x68,
                    0x2E, 0x63, 0x6F, 0x6D, 0x00, 0x00, 0x00, 0x04, 0x6E, 0x6F, 0x6E, 0x65, 0x00,
                    0x00, 0x00, 0x04, 0x6E, 0x6F, 0x6E, 0x65, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0xDB, 0x14, 0x66,
                    0x1C, 0xC5, 0x92, 0xC1, 0x54, 0x57, 0x7D, 0xED, 0x0A, 0xAA, 0x20, 0xB9, 0x17,
                    0x62, 0x3D, 0x00, 0x00, 0x00, 0xE6, 0x63, 0x75, 0x72, 0x76, 0x65, 0x32, 0x35,
                    0x35, 0x31, 0x39, 0x2D, 0x73, 0x68, 0x61, 0x32, 0x35, 0x36, 0x2C, 0x63, 0x75,
                    0x72, 0x76, 0x65, 0x32, 0x35, 0x35, 0x31, 0x39, 0x2D, 0x73, 0x68, 0x61, 0x32,
                    0x35, 0x36, 0x40, 0x6C, 0x69, 0x62, 0x73, 0x73, 0x68, 0x2E, 0x6F, 0x72, 0x67,
                    0x2C, 0x65, 0x63, 0x64, 0x68, 0x2D, 0x73, 0x68, 0x61, 0x32, 0x2D, 0x6E, 0x69,
                    0x73, 0x74, 0x70, 0x32, 0x35, 0x36, 0x2C, 0x65, 0x63, 0x64, 0x68, 0x2D, 0x73,
                    0x68, 0x61, 0x32, 0x2D, 0x6E, 0x69, 0x73, 0x74, 0x70, 0x33, 0x38, 0x34, 0x2C,
                    0x65, 0x63, 0x64, 0x68, 0x2D, 0x73, 0x68, 0x61, 0x32, 0x2D, 0x6E, 0x69, 0x73,
                    0x74, 0x70, 0x35, 0x32, 0x31, 0x2C, 0x64, 0x69, 0x66, 0x66, 0x69, 0x65, 0x2D,
                    0x68, 0x65, 0x6C, 0x6C, 0x6D, 0x61, 0x6E, 0x2D, 0x67, 0x72, 0x6F, 0x75, 0x70,
                    0x2D, 0x65, 0x78, 0x63, 0x68, 0x61, 0x6E, 0x67, 0x65, 0x2D, 0x73, 0x68, 0x61,
                    0x32, 0x35, 0x36, 0x2C, 0x64, 0x69, 0x66, 0x66, 0x69, 0x65, 0x2D, 0x68, 0x65,
                    0x6C, 0x6C, 0x6D, 0x61, 0x6E, 0x2D, 0x67, 0x72, 0x6F, 0x75, 0x70, 0x31, 0x36,
                    0x2D, 0x73, 0x68, 0x61, 0x35, 0x31, 0x32, 0x2C, 0x64, 0x69, 0x66, 0x66, 0x69,
                    0x65, 0x2D, 0x68, 0x65, 0x6C, 0x6C, 0x6D, 0x61, 0x6E, 0x2D, 0x67, 0x72, 0x6F,
                    0x75, 0x70, 0x31, 0x38, 0x2D, 0x73, 0x68, 0x61, 0x35, 0x31, 0x32, 0x2C, 0x64,
                    0x69, 0x66, 0x66, 0x69, 0x65, 0x2D, 0x68, 0x65, 0x6C, 0x6C, 0x6D, 0x61, 0x6E,
                    0x2D, 0x67, 0x72, 0x6F, 0x75, 0x70, 0x31, 0x34, 0x2D, 0x73, 0x68, 0x61, 0x32,
                    0x35, 0x36, 0x00, 0x00, 0x00, 0x0B, 0x73, 0x73, 0x68, 0x2D, 0x65, 0x64, 0x32,
                    0x35, 0x35, 0x31, 0x39, 0x00, 0x00, 0x00, 0x6C, 0x63, 0x68, 0x61, 0x63, 0x68,
                    0x61, 0x32, 0x30, 0x2D, 0x70, 0x6F, 0x6C, 0x79, 0x31, 0x33, 0x30, 0x35, 0x40,
                    0x6F, 0x70, 0x65, 0x6E, 0x73, 0x73, 0x68, 0x2E, 0x63, 0x6F, 0x6D, 0x2C, 0x61,
                    0x65, 0x73, 0x31, 0x32, 0x38, 0x2D, 0x63, 0x74, 0x72, 0x2C, 0x61, 0x65, 0x73,
                    0x31, 0x39, 0x32, 0x2D, 0x63, 0x74, 0x72, 0x2C, 0x61, 0x65, 0x73, 0x32, 0x35,
                    0x36, 0x2D, 0x63, 0x74, 0x72, 0x2C, 0x61, 0x65, 0x73, 0x31, 0x32, 0x38, 0x2D,
                    0x67, 0x63, 0x6D, 0x40, 0x6F, 0x70, 0x65, 0x6E, 0x73, 0x73, 0x68, 0x2E, 0x63,
                    0x6F, 0x6D, 0x2C, 0x61, 0x65, 0x73, 0x32, 0x35, 0x36, 0x2D, 0x67, 0x63, 0x6D,
                    0x40, 0x6F, 0x70, 0x65, 0x6E, 0x73, 0x73, 0x68, 0x2E, 0x63, 0x6F, 0x6D, 0x00,
                    0x00, 0x00, 0x6C, 0x63, 0x68, 0x61, 0x63, 0x68, 0x61, 0x32, 0x30, 0x2D, 0x70,
                    0x6F, 0x6C, 0x79, 0x31, 0x33, 0x30, 0x35, 0x40, 0x6F, 0x70, 0x65, 0x6E, 0x73,
                    0x73, 0x68, 0x2E, 0x63, 0x6F, 0x6D, 0x2C, 0x61, 0x65, 0x73, 0x31, 0x32, 0x38,
                    0x2D, 0x63, 0x74, 0x72, 0x2C, 0x61, 0x65, 0x73, 0x31, 0x39, 0x32, 0x2D, 0x63,
                    0x74, 0x72, 0x2C, 0x61, 0x65, 0x73, 0x32, 0x35, 0x36, 0x2D, 0x63, 0x74, 0x72,
                    0x2C, 0x61, 0x65, 0x73, 0x31, 0x32, 0x38, 0x2D, 0x67, 0x63, 0x6D, 0x40, 0x6F,
                    0x70, 0x65, 0x6E, 0x73, 0x73, 0x68, 0x2E, 0x63, 0x6F, 0x6D, 0x2C, 0x61, 0x65,
                    0x73, 0x32, 0x35, 0x36, 0x2D, 0x67, 0x63, 0x6D, 0x40, 0x6F, 0x70, 0x65, 0x6E,
                    0x73, 0x73, 0x68, 0x2E, 0x63, 0x6F, 0x6D, 0x00, 0x00, 0x00, 0xD5, 0x75, 0x6D,
                    0x61, 0x63, 0x2D, 0x36, 0x34, 0x2D, 0x65, 0x74, 0x6D, 0x40, 0x6F, 0x70, 0x65,
                    0x6E, 0x73, 0x73, 0x68, 0x2E, 0x63, 0x6F, 0x6D, 0x2C, 0x75, 0x6D, 0x61, 0x63,
                    0x2D, 0x31, 0x32, 0x38, 0x2D, 0x65, 0x74, 0x6D, 0x40, 0x6F, 0x70, 0x65, 0x6E,
                    0x73, 0x73, 0x68, 0x2E, 0x63, 0x6F, 0x6D, 0x2C, 0x68, 0x6D, 0x61, 0x63, 0x2D,
                    0x73, 0x68, 0x61, 0x32, 0x2D, 0x32, 0x35, 0x36, 0x2D, 0x65, 0x74, 0x6D, 0x40,
                    0x6F, 0x70, 0x65, 0x6E, 0x73, 0x73, 0x68, 0x2E, 0x63, 0x6F, 0x6D, 0x2C, 0x68,
                    0x6D, 0x61, 0x63, 0x2D, 0x73, 0x68, 0x61, 0x32, 0x2D, 0x35, 0x31, 0x32, 0x2D,
                    0x65, 0x74, 0x6D, 0x40, 0x6F, 0x70, 0x65, 0x6E, 0x73, 0x73, 0x68, 0x2E, 0x63,
                    0x6F, 0x6D, 0x2C, 0x68, 0x6D, 0x61, 0x63, 0x2D, 0x73, 0x68, 0x61, 0x31, 0x2D,
                    0x65, 0x74, 0x6D, 0x40, 0x6F, 0x70, 0x65, 0x6E, 0x73, 0x73, 0x68, 0x2E, 0x63,
                    0x6F, 0x6D, 0x2C, 0x75, 0x6D, 0x61, 0x63, 0x2D, 0x36, 0x34, 0x40, 0x6F, 0x70,
                    0x65, 0x6E, 0x73, 0x73, 0x68, 0x2E, 0x63, 0x6F, 0x6D, 0x2C, 0x75, 0x6D, 0x61,
                    0x63, 0x2D, 0x31, 0x32, 0x38, 0x40, 0x6F, 0x70, 0x65, 0x6E, 0x73, 0x73, 0x68,
                    0x2E, 0x63, 0x6F, 0x6D, 0x2C, 0x68, 0x6D, 0x61, 0x63, 0x2D, 0x73, 0x68, 0x61,
                    0x32, 0x2D, 0x32, 0x35, 0x36, 0x2C, 0x68, 0x6D, 0x61, 0x63, 0x2D, 0x73, 0x68,
                    0x61, 0x32, 0x2D, 0x35, 0x31, 0x32, 0x2C, 0x68, 0x6D, 0x61, 0x63, 0x2D, 0x73,
                    0x68, 0x61, 0x31, 0x00, 0x00, 0x00, 0xD5, 0x75, 0x6D, 0x61, 0x63, 0x2D, 0x36,
                    0x34, 0x2D, 0x65, 0x74, 0x6D, 0x40, 0x6F, 0x70, 0x65, 0x6E, 0x73, 0x73, 0x68,
                    0x2E, 0x63, 0x6F, 0x6D, 0x2C, 0x75, 0x6D, 0x61, 0x63, 0x2D, 0x31, 0x32, 0x38,
                    0x2D, 0x65, 0x74, 0x6D, 0x40, 0x6F, 0x70, 0x65, 0x6E, 0x73, 0x73, 0x68, 0x2E,
                    0x63, 0x6F, 0x6D, 0x2C, 0x68, 0x6D, 0x61, 0x63, 0x2D, 0x73, 0x68, 0x61, 0x32,
                    0x2D, 0x32, 0x35, 0x36, 0x2D, 0x65, 0x74, 0x6D, 0x40, 0x6F, 0x70, 0x65, 0x6E,
                    0x73, 0x73, 0x68, 0x2E, 0x63, 0x6F, 0x6D, 0x2C, 0x68, 0x6D, 0x61, 0x63, 0x2D,
                    0x73, 0x68, 0x61, 0x32, 0x2D, 0x35, 0x31, 0x32, 0x2D, 0x65, 0x74, 0x6D, 0x40,
                    0x6F, 0x70, 0x65, 0x6E, 0x73, 0x73, 0x68, 0x2E, 0x63, 0x6F, 0x6D, 0x2C, 0x68,
                    0x6D, 0x61, 0x63, 0x2D, 0x73, 0x68, 0x61, 0x31, 0x2D, 0x65, 0x74, 0x6D, 0x40,
                    0x6F, 0x70, 0x65, 0x6E, 0x73, 0x73, 0x68, 0x2E, 0x63, 0x6F, 0x6D, 0x2C, 0x75,
                    0x6D, 0x61, 0x63, 0x2D, 0x36, 0x34, 0x40, 0x6F, 0x70, 0x65, 0x6E, 0x73, 0x73,
                    0x68, 0x2E, 0x63, 0x6F, 0x6D, 0x2C, 0x75, 0x6D, 0x61, 0x63, 0x2D, 0x31, 0x32,
                    0x38, 0x40, 0x6F, 0x70, 0x65, 0x6E, 0x73, 0x73, 0x68, 0x2E, 0x63, 0x6F, 0x6D,
                    0x2C, 0x68, 0x6D, 0x61, 0x63, 0x2D, 0x73, 0x68, 0x61, 0x32, 0x2D, 0x32, 0x35,
                    0x36, 0x2C, 0x68, 0x6D, 0x61, 0x63, 0x2D, 0x73, 0x68, 0x61, 0x32, 0x2D, 0x35,
                    0x31, 0x32, 0x2C, 0x68, 0x6D, 0x61, 0x63, 0x2D, 0x73, 0x68, 0x61, 0x31, 0x00,
                    0x00, 0x00, 0x15, 0x6E, 0x6F, 0x6E, 0x65, 0x2C, 0x7A, 0x6C, 0x69, 0x62, 0x40,
                    0x6F, 0x70, 0x65, 0x6E, 0x73, 0x73, 0x68, 0x2E, 0x63, 0x6F, 0x6D, 0x00, 0x00,
                    0x00, 0x15, 0x6E, 0x6F, 0x6E, 0x65, 0x2C, 0x7A, 0x6C, 0x69, 0x62, 0x40, 0x6F,
                    0x70, 0x65, 0x6E, 0x73, 0x73, 0x68, 0x2E, 0x63, 0x6F, 0x6D, 0x00, 0x00, 0x00,
                    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                    0x33, 0x00, 0x00, 0x00, 0x0B, 0x73, 0x73, 0x68, 0x2D, 0x65, 0x64, 0x32, 0x35,
                    0x35, 0x31, 0x39, 0x00, 0x00, 0x00, 0x20, 0xCA, 0x09, 0x52, 0x99, 0x4E, 0x66,
                    0xBE, 0xDD, 0x26, 0xCA, 0x3F, 0xE2, 0xC1, 0x3F, 0xB0, 0xC9, 0xF5, 0x3F, 0x7C,
                    0xBC, 0xB2, 0xF2, 0x1A, 0x4D, 0x6F, 0xB8, 0xCE, 0xEC, 0xC6, 0x96, 0x29, 0xF0,
                    0x00, 0x00, 0x00, 0x20, 0x40, 0x26, 0x0F, 0x76, 0x0F, 0x79, 0xB1, 0x90, 0x13,
                    0xFB, 0x03, 0x5B, 0x8A, 0x41, 0x63, 0xED, 0xBE, 0xBD, 0xB6, 0x1A, 0xD9, 0xC9,
                    0x23, 0x5E, 0xE9, 0x49, 0x9D, 0xD7, 0xD8, 0x95, 0x4C, 0x66, 0x00, 0x00, 0x00,
                    0x20, 0x02, 0x14, 0x1D, 0xB2, 0x15, 0xA7, 0x54, 0x67, 0xD2, 0x65, 0x24, 0x50,
                    0x03, 0xF5, 0x0D, 0x95, 0xAD, 0x1D, 0x97, 0x25, 0x62, 0xD9, 0xBC, 0xFF, 0xED,
                    0xB3, 0x5E, 0x11, 0x7C, 0x57, 0xC5, 0x08, 0x00, 0x00, 0x00, 0x21, 0x00, 0x86,
                    0xF8, 0x27, 0x69, 0xCC, 0x94, 0x9B, 0x70, 0x66, 0xB7, 0x50, 0x9F, 0x2A, 0x98,
                    0x0E, 0x52, 0x6D, 0x3E, 0x34, 0x65, 0x99, 0x43, 0xEA, 0x48, 0xCC, 0xCE, 0xF1,
                    0xC5, 0x8F, 0xF3, 0x77, 0x7C,
                ],
                r#"0000: 00 00 00 13 53 53 48 2D 32 2E 30 2D 74 75 73 73  ....SSH-2.0-tuss
0016: 68 5F 30 2E 31 2E 30 00 00 00 13 53 53 48 2D 32  h_0.1.0....SSH-2
0032: 2E 30 2D 4F 70 65 6E 53 53 48 5F 38 2E 32 00 00  .0-OpenSSH_8.2..
0048: 01 0D 14 D4 14 01 81 25 28 C2 47 CE F9 88 6E E1  .......%(.G...n.
0064: 46 E7 3E 00 00 00 2E 63 75 72 76 65 32 35 35 31  F.>....curve2551
0080: 39 2D 73 68 61 32 35 36 2C 63 75 72 76 65 32 35  9-sha256,curve25
0096: 35 31 39 2D 73 68 61 32 35 36 40 6C 69 62 73 73  519-sha256@libss
0112: 68 2E 6F 72 67 00 00 00 25 73 73 68 2D 65 64 32  h.org...%ssh-ed2
0128: 35 35 31 39 2C 72 73 61 2D 73 68 61 32 2D 35 31  5519,rsa-sha2-51
0144: 32 2C 72 73 61 2D 73 68 61 32 2D 32 35 36 00 00  2,rsa-sha2-256..
0160: 00 1D 63 68 61 63 68 61 32 30 2D 70 6F 6C 79 31  ..chacha20-poly1
0176: 33 30 35 40 6F 70 65 6E 73 73 68 2E 63 6F 6D 00  305@openssh.com.
0192: 00 00 1D 63 68 61 63 68 61 32 30 2D 70 6F 6C 79  ...chacha20-poly
0208: 31 33 30 35 40 6F 70 65 6E 73 73 68 2E 63 6F 6D  1305@openssh.com
0224: 00 00 00 1D 68 6D 61 63 2D 73 68 61 32 2D 35 31  ....hmac-sha2-51
0240: 32 2D 65 74 6D 40 6F 70 65 6E 73 73 68 2E 63 6F  2-etm@openssh.co
0256: 6D 00 00 00 1D 68 6D 61 63 2D 73 68 61 32 2D 35  m....hmac-sha2-5
0272: 31 32 2D 65 74 6D 40 6F 70 65 6E 73 73 68 2E 63  12-etm@openssh.c
0288: 6F 6D 00 00 00 04 6E 6F 6E 65 00 00 00 04 6E 6F  om....none....no
0304: 6E 65 00 00 00 00 00 00 00 00 00 00 00 00 00 00  ne..............
0320: 00 03 DB 14 66 1C C5 92 C1 54 57 7D ED 0A AA 20  ....f....TW}....
0336: B9 17 62 3D 00 00 00 E6 63 75 72 76 65 32 35 35  ..b=....curve255
0352: 31 39 2D 73 68 61 32 35 36 2C 63 75 72 76 65 32  19-sha256,curve2
0368: 35 35 31 39 2D 73 68 61 32 35 36 40 6C 69 62 73  5519-sha256@libs
0384: 73 68 2E 6F 72 67 2C 65 63 64 68 2D 73 68 61 32  sh.org,ecdh-sha2
0400: 2D 6E 69 73 74 70 32 35 36 2C 65 63 64 68 2D 73  -nistp256,ecdh-s
0416: 68 61 32 2D 6E 69 73 74 70 33 38 34 2C 65 63 64  ha2-nistp384,ecd
0432: 68 2D 73 68 61 32 2D 6E 69 73 74 70 35 32 31 2C  h-sha2-nistp521,
0448: 64 69 66 66 69 65 2D 68 65 6C 6C 6D 61 6E 2D 67  diffie-hellman-g
0464: 72 6F 75 70 2D 65 78 63 68 61 6E 67 65 2D 73 68  roup-exchange-sh
0480: 61 32 35 36 2C 64 69 66 66 69 65 2D 68 65 6C 6C  a256,diffie-hell
0496: 6D 61 6E 2D 67 72 6F 75 70 31 36 2D 73 68 61 35  man-group16-sha5
0512: 31 32 2C 64 69 66 66 69 65 2D 68 65 6C 6C 6D 61  12,diffie-hellma
0528: 6E 2D 67 72 6F 75 70 31 38 2D 73 68 61 35 31 32  n-group18-sha512
0544: 2C 64 69 66 66 69 65 2D 68 65 6C 6C 6D 61 6E 2D  ,diffie-hellman-
0560: 67 72 6F 75 70 31 34 2D 73 68 61 32 35 36 00 00  group14-sha256..
0576: 00 0B 73 73 68 2D 65 64 32 35 35 31 39 00 00 00  ..ssh-ed25519...
0592: 6C 63 68 61 63 68 61 32 30 2D 70 6F 6C 79 31 33  lchacha20-poly13
0608: 30 35 40 6F 70 65 6E 73 73 68 2E 63 6F 6D 2C 61  05@openssh.com,a
0624: 65 73 31 32 38 2D 63 74 72 2C 61 65 73 31 39 32  es128-ctr,aes192
0640: 2D 63 74 72 2C 61 65 73 32 35 36 2D 63 74 72 2C  -ctr,aes256-ctr,
0656: 61 65 73 31 32 38 2D 67 63 6D 40 6F 70 65 6E 73  aes128-gcm@opens
0672: 73 68 2E 63 6F 6D 2C 61 65 73 32 35 36 2D 67 63  sh.com,aes256-gc
0688: 6D 40 6F 70 65 6E 73 73 68 2E 63 6F 6D 00 00 00  m@openssh.com...
0704: 6C 63 68 61 63 68 61 32 30 2D 70 6F 6C 79 31 33  lchacha20-poly13
0720: 30 35 40 6F 70 65 6E 73 73 68 2E 63 6F 6D 2C 61  05@openssh.com,a
0736: 65 73 31 32 38 2D 63 74 72 2C 61 65 73 31 39 32  es128-ctr,aes192
0752: 2D 63 74 72 2C 61 65 73 32 35 36 2D 63 74 72 2C  -ctr,aes256-ctr,
0768: 61 65 73 31 32 38 2D 67 63 6D 40 6F 70 65 6E 73  aes128-gcm@opens
0784: 73 68 2E 63 6F 6D 2C 61 65 73 32 35 36 2D 67 63  sh.com,aes256-gc
0800: 6D 40 6F 70 65 6E 73 73 68 2E 63 6F 6D 00 00 00  m@openssh.com...
0816: D5 75 6D 61 63 2D 36 34 2D 65 74 6D 40 6F 70 65  .umac-64-etm@ope
0832: 6E 73 73 68 2E 63 6F 6D 2C 75 6D 61 63 2D 31 32  nssh.com,umac-12
0848: 38 2D 65 74 6D 40 6F 70 65 6E 73 73 68 2E 63 6F  8-etm@openssh.co
0864: 6D 2C 68 6D 61 63 2D 73 68 61 32 2D 32 35 36 2D  m,hmac-sha2-256-
0880: 65 74 6D 40 6F 70 65 6E 73 73 68 2E 63 6F 6D 2C  etm@openssh.com,
0896: 68 6D 61 63 2D 73 68 61 32 2D 35 31 32 2D 65 74  hmac-sha2-512-et
0912: 6D 40 6F 70 65 6E 73 73 68 2E 63 6F 6D 2C 68 6D  m@openssh.com,hm
0928: 61 63 2D 73 68 61 31 2D 65 74 6D 40 6F 70 65 6E  ac-sha1-etm@open
0944: 73 73 68 2E 63 6F 6D 2C 75 6D 61 63 2D 36 34 40  ssh.com,umac-64@
0960: 6F 70 65 6E 73 73 68 2E 63 6F 6D 2C 75 6D 61 63  openssh.com,umac
0976: 2D 31 32 38 40 6F 70 65 6E 73 73 68 2E 63 6F 6D  -128@openssh.com
0992: 2C 68 6D 61 63 2D 73 68 61 32 2D 32 35 36 2C 68  ,hmac-sha2-256,h
1008: 6D 61 63 2D 73 68 61 32 2D 35 31 32 2C 68 6D 61  mac-sha2-512,hma
1024: 63 2D 73 68 61 31 00 00 00 D5 75 6D 61 63 2D 36  c-sha1....umac-6
1040: 34 2D 65 74 6D 40 6F 70 65 6E 73 73 68 2E 63 6F  4-etm@openssh.co
1056: 6D 2C 75 6D 61 63 2D 31 32 38 2D 65 74 6D 40 6F  m,umac-128-etm@o
1072: 70 65 6E 73 73 68 2E 63 6F 6D 2C 68 6D 61 63 2D  penssh.com,hmac-
1088: 73 68 61 32 2D 32 35 36 2D 65 74 6D 40 6F 70 65  sha2-256-etm@ope
1104: 6E 73 73 68 2E 63 6F 6D 2C 68 6D 61 63 2D 73 68  nssh.com,hmac-sh
1120: 61 32 2D 35 31 32 2D 65 74 6D 40 6F 70 65 6E 73  a2-512-etm@opens
1136: 73 68 2E 63 6F 6D 2C 68 6D 61 63 2D 73 68 61 31  sh.com,hmac-sha1
1152: 2D 65 74 6D 40 6F 70 65 6E 73 73 68 2E 63 6F 6D  -etm@openssh.com
1168: 2C 75 6D 61 63 2D 36 34 40 6F 70 65 6E 73 73 68  ,umac-64@openssh
1184: 2E 63 6F 6D 2C 75 6D 61 63 2D 31 32 38 40 6F 70  .com,umac-128@op
1200: 65 6E 73 73 68 2E 63 6F 6D 2C 68 6D 61 63 2D 73  enssh.com,hmac-s
1216: 68 61 32 2D 32 35 36 2C 68 6D 61 63 2D 73 68 61  ha2-256,hmac-sha
1232: 32 2D 35 31 32 2C 68 6D 61 63 2D 73 68 61 31 00  2-512,hmac-sha1.
1248: 00 00 15 6E 6F 6E 65 2C 7A 6C 69 62 40 6F 70 65  ...none,zlib@ope
1264: 6E 73 73 68 2E 63 6F 6D 00 00 00 15 6E 6F 6E 65  nssh.com....none
1280: 2C 7A 6C 69 62 40 6F 70 65 6E 73 73 68 2E 63 6F  ,zlib@openssh.co
1296: 6D 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  m...............
1312: 00 33 00 00 00 0B 73 73 68 2D 65 64 32 35 35 31  .3....ssh-ed2551
1328: 39 00 00 00 20 CA 09 52 99 4E 66 BE DD 26 CA 3F  9......R.Nf..&.?
1344: E2 C1 3F B0 C9 F5 3F 7C BC B2 F2 1A 4D 6F B8 CE  ..?...?|....Mo..
1360: EC C6 96 29 F0 00 00 00 20 40 26 0F 76 0F 79 B1  ...).....@&.v.y.
1376: 90 13 FB 03 5B 8A 41 63 ED BE BD B6 1A D9 C9 23  ....[.Ac.......#
1392: 5E E9 49 9D D7 D8 95 4C 66 00 00 00 20 02 14 1D  ^.I....Lf.......
1408: B2 15 A7 54 67 D2 65 24 50 03 F5 0D 95 AD 1D 97  ...Tg.e$P.......
1424: 25 62 D9 BC FF ED B3 5E 11 7C 57 C5 08 00 00 00  %b.....^.|W.....
1440: 21 00 86 F8 27 69 CC 94 9B 70 66 B7 50 9F 2A 98  !...'i...pf.P.*.
1456: 0E 52 6D 3E 34 65 99 43 EA 48 CC CE F1 C5 8F F3  .Rm>4e.C.H......
1472: 77 7C                                            w|"#,
            )
        };
        static ref TEST_CASES: Vec<(Vec<u8>, &'static str)> = {
            let mut test_cases = Vec::new();
            test_cases.push(TUPLE1.clone());
            test_cases.push(TUPLE2.clone());
            test_cases.push(TUPLE3.clone());
            test_cases
        };
    }

    #[test]
    fn dump_the_hex() {
        let mut buffer = Vec::new();

        for (actual, expected) in &*TEST_CASES {
            assert!(hexdump(&actual, &mut buffer).is_ok());
            assert_eq!(*expected, String::from_utf8_lossy(&buffer));
            buffer.clear();
        }
    }
}
