/**
 * A topological layout using {@link ZherebkoOperator}.
 *
 * @packageDocumentation
 */
import { Dag } from "../dag";
/**
 * The return from calling {@link ZherebkoOperator}
 *
 * This is the final width and height of the laid out dag.
 */
export interface ZherebkoInfo {
    /** total width after layout */
    width: number;
    /** total height after layout */
    height: number;
}
/**
 * A simple topological layout operator.
 *
 * This layout algorithm constructs a topological representation of the dag
 * meant for visualization. The algorithm is based off a PR by D. Zherebko. The
 * nodes are topologically ordered, and edges are then positioned into "lanes"
 * to the left and right of the nodes.
 *
 * Create with {@link zherebko}.
 *
 * @example
 * <img alt="zherebko example" src="media://zherebko.png" width="1000">
 *
 * @example
 * ```typescript
 * const data = [["parent", "child"], ...];
 * const create = connect();
 * const dag = create(data);
 * const layout = zherebko();
 * const { width, height } = layout(dag);
 * for (const node of dag) {
 *   console.log(node.x, node.y);
 * }
 * ```
 */
export interface ZherebkoOperator {
    /** Layout the input dag */
    (dag: Dag): ZherebkoInfo;
    /**
     * Set the zherebko layout's node size
     *
     * Set the size to the specified three-element array of numbers [
     * *nodeWidth*, *nodeHeight*, *edgeGap* ] and returns a new operator. Nodes
     * are spaced apart vertically by nodeHeight, and each node is given space of
     * nodeWidth. Edges that wrap around the nodes are given space edgeGap.
     *
     * (default: [1, 1, 1])
     */
    nodeSize(val: readonly [number, number, number]): ZherebkoOperator;
    /** Get the current node size. */
    nodeSize(): [number, number, number];
    /**
     * Set the zherebko layouts full size
     *
     * Set the size to the specified two-element array of numbers [ *width*,
     * *height* ] and returns a new operator. The dag is resized to fit within
     * width and height if they are specified. (default: null)
     */
    size(val: null | readonly [number, number]): ZherebkoOperator;
    /** Get the current size */
    size(): null | [number, number];
}
/**
 * Create a new {@link ZherebkoOperator} with default settings.
 */
export declare function zherebko(...args: never[]): ZherebkoOperator;
