/**
 * General utilities for use throughout the package
 *
 * @packageDocumentation
 */
/** assert something */
export function assert(statement) {
    if (!statement) {
        throw new Error("internal error: failed assert");
    }
}
/** determines if two sets are equal */
export function setEqual(first, second) {
    if (second.size !== first.size) {
        return false;
    }
    else {
        for (const element of first) {
            if (!second.has(element)) {
                return false;
            }
        }
        return true;
    }
}
/** determines if two sets intersect */
export function setIntersect(first, second) {
    if (second.size < first.size) {
        [second, first] = [first, second];
    }
    for (const element of first) {
        if (second.has(element)) {
            return true;
        }
    }
    return false;
}
/**
 * removes a single arbitrary element from the Set, or undefined is missing
 *
 * @remarks
 * if the set contains undefined, then this doesn't distinguish in output,
 * but will properly remove it.
 */
export function setPop(elems) {
    const { done, value } = elems.values().next();
    if (done) {
        return undefined;
    }
    else {
        elems.delete(value);
        return value;
    }
}
/**
 * push val onto key list for multimap
 */
export function listMultimapPush(multimap, key, val) {
    const value = multimap.get(key);
    if (value === undefined) {
        multimap.set(key, [val]);
    }
    else {
        value.push(val);
    }
}
/**
 * add val to key set for multimap
 */
export function setMultimapAdd(multimap, key, val) {
    const value = multimap.get(key);
    if (value === undefined) {
        multimap.set(key, new Set([val]));
    }
    else {
        value.add(val);
    }
}
/** replacer for serializing possibly circular json */
export function getCircularReplacer() {
    const seen = new WeakSet();
    return (key, value) => {
        if (typeof value === "object" && value !== null) {
            if (seen.has(value)) {
                return "[circular]";
            }
            seen.add(value);
        }
        return value;
    };
}
/** format tag for converting inputs to json */
export function js(strings, ...values) {
    const [base, ...rest] = strings;
    return [base]
        .concat(...rest.map((str, i) => [
        JSON.stringify(values[i], getCircularReplacer()),
        str,
    ]))
        .join("");
}
/** iterate over bigrams of an array */
export function* bigrams(array) {
    let [first, ...rest] = array;
    for (const second of rest) {
        yield [first, second];
        first = second;
    }
}
/** depth first search for arbitrary types */
export function* dfs(children, ...queue) {
    const seen = new Set();
    let node;
    while ((node = queue.pop()) !== undefined) {
        if (seen.has(node))
            continue;
        yield node;
        seen.add(node);
        queue.push(...children(node));
    }
}
