/**
 * A {@link CoffmanGrahamOperator} that prevents the width of the dag from
 * being too large.
 *
 * @packageDocumentation
 */
import FastPriorityQueue from "fastpriorityqueue";
import { getParentCounts } from "../../dag/utils";
import { map } from "../../iters";
/** @internal */
function buildOperator(options) {
    function coffmanGrahamCall(dag) {
        var _a;
        const maxWidth = options.width || Math.floor(Math.sqrt(dag.size() + 0.5));
        // create queue
        function comp([leftBefore], [rightBefore]) {
            for (const [i, leftb] of leftBefore.entries()) {
                const rightb = rightBefore[i];
                if (rightb === undefined) {
                    return false;
                }
                else if (leftb < rightb) {
                    return true;
                }
                else if (rightb < leftb) {
                    return false;
                }
            }
            return true;
        }
        const queue = new FastPriorityQueue(comp);
        // initialize node data
        const beforeInds = new Map(map(dag, (node) => [node, []]));
        const parents = getParentCounts(dag);
        // start with root nodes
        for (const root of dag.iroots()) {
            queue.add([[], root]);
        }
        let i = 0; // node index
        let layer = 0; // layer assigning
        let width = 0; // current width
        let next;
        while ((next = queue.poll())) {
            const [, node] = next;
            // NOTE for clarity we compute this early, but we don't need to if width
            // >= maxWidth which is a little inefficient
            const limit = (_a = parents
                .get(node)) === null || _a === void 0 ? void 0 : _a.reduce((l, [p, c]) => Math.max(l, p.value + +(c > 1)), 0);
            if (width < maxWidth && (limit === undefined || limit < layer)) {
                node.value = layer;
                width++;
            }
            else {
                console.log;
                node.value = layer = Math.max(limit !== null && limit !== void 0 ? limit : 0, layer) + 1;
                width = 1;
            }
            for (const child of node.ichildren()) {
                const before = beforeInds.get(child);
                before.push(i);
                if (before.length === parents.get(child).length) {
                    queue.add([before.reverse(), child]);
                }
            }
            i++;
        }
    }
    function width(maxWidth) {
        if (maxWidth === undefined) {
            return options.width;
        }
        else if (maxWidth < 0) {
            throw new Error(`width must be non-negative: ${maxWidth}`);
        }
        else {
            return buildOperator(Object.assign(Object.assign({}, options), { width: maxWidth }));
        }
    }
    coffmanGrahamCall.width = width;
    return coffmanGrahamCall;
}
/**
 * Create a default {@link CoffmanGrahamOperator}, bundled as
 * {@link layeringCoffmanGraham}.
 */
export function coffmanGraham(...args) {
    if (args.length) {
        throw new Error(`got arguments to coffmanGraham(${args}), but constructor takes no arguments.`);
    }
    return buildOperator({ width: 0 });
}
