var __rest = (this && this.__rest) || function (s, e) {
    var t = {};
    for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p) && e.indexOf(p) < 0)
        t[p] = s[p];
    if (s != null && typeof Object.getOwnPropertySymbols === "function")
        for (var i = 0, p = Object.getOwnPropertySymbols(s); i < p.length; i++) {
            if (e.indexOf(p[i]) < 0 && Object.prototype.propertyIsEnumerable.call(s, p[i]))
                t[p[i]] = s[p[i]];
        }
    return t;
};
import { js } from "../utils";
import { simplex as coordSimplex, } from "./coord/simplex";
import { twoLayer, } from "./decross/two-layer";
import { simplex as layerSimplex, } from "./layering/simplex";
import { scaleLayers, sugify, unsugify, verifyCoordAssignment, } from "./utils";
/**
 * wrap a {@link NodeSizeAccessor} turning it into an {@link SugiNodeSizeAccessor}
 *
 * Mostly useful for running the steps of {@link sugiyama} independently.
 */
export function wrapNodeSizeAccessor(acc) {
    const empty = acc();
    function sugiNodeSizeAccessor(node) {
        return "node" in node.data ? acc(node.data.node) : empty;
    }
    sugiNodeSizeAccessor.wrapped = acc;
    return sugiNodeSizeAccessor;
}
/**
 * Verify, cache, and split the results of an {@link SugiNodeSizeAccessor} into
 * an x and y {@link CoordNodeSizeAccessor}.
 *
 * This allows you to split an {@link SugiNodeSizeAccessor} into independent x
 * and y accessors, while also caching the result to prevent potentially
 * expensive computation from being duplicated.
 *
 * The only real reason to use this would be to run the steps of {@link
 * sugiyama} independently.
 */
export function cachedNodeSize(nodeSize, check = true) {
    const cache = new Map();
    function cached(node) {
        let val = cache.get(node);
        if (val === undefined) {
            val = nodeSize(node);
            const [width, height] = val;
            if (check && (width < 0 || height < 0)) {
                throw new Error(js `all node sizes must be non-negative, but got width ${width} and height ${height} for node '${node}'`);
            }
            cache.set(node, val);
        }
        return val;
    }
    const cachedX = (node) => cached(node)[0];
    const cachedY = (node) => cached(node)[1];
    return [cachedX, cachedY];
}
/**
 * Given layers and node heights, assign y coordinates.
 *
 * This is only exported so that each step of {@link sugiyama} can be executed
 * independently or controlled. In the future it may make sense to make
 * vertical coordinates part of the sugiyama operators.
 */
export function coordVertical(layers, size) {
    let height = 0;
    for (const layer of layers) {
        const layerHeight = Math.max(...layer.map(size));
        for (const node of layer) {
            node.y = height + layerHeight / 2;
        }
        height += layerHeight;
    }
    return height;
}
/** @internal */
function buildOperator(options) {
    function sugiyama(dag) {
        // compute layers
        options.layering(dag);
        // create layers
        const layers = sugify(dag);
        // cache and split node sizes
        const [xSize, ySize] = cachedNodeSize(options.sugiNodeSize);
        // assign y
        let height = coordVertical(layers, ySize);
        if (height <= 0) {
            throw new Error("at least one node must have positive height, but total height was zero");
        }
        // minimize edge crossings
        options.decross(layers);
        // assign coordinates
        let width = options.coord(layers, xSize);
        // verify
        verifyCoordAssignment(layers, width);
        // scale x
        if (options.size !== null) {
            const [newWidth, newHeight] = options.size;
            scaleLayers(layers, newWidth / width, newHeight / height);
            width = newWidth;
            height = newHeight;
        }
        // Update original dag with values
        unsugify(layers);
        // layout info
        return { width, height };
    }
    function layering(layer) {
        if (layer === undefined) {
            return options.layering;
        }
        else {
            const { layering: _ } = options, rest = __rest(options, ["layering"]);
            return buildOperator(Object.assign(Object.assign({}, rest), { layering: layer }));
        }
    }
    sugiyama.layering = layering;
    function decross(dec) {
        if (dec === undefined) {
            return options.decross;
        }
        else {
            const { decross: _ } = options, rest = __rest(options, ["decross"]);
            return buildOperator(Object.assign(Object.assign({}, rest), { decross: dec }));
        }
    }
    sugiyama.decross = decross;
    function coord(crd) {
        if (crd === undefined) {
            return options.coord;
        }
        else {
            const { coord: _ } = options, rest = __rest(options, ["coord"]);
            return buildOperator(Object.assign(Object.assign({}, rest), { coord: crd }));
        }
    }
    sugiyama.coord = coord;
    function size(sz) {
        if (sz !== undefined) {
            return buildOperator(Object.assign(Object.assign({}, options), { size: sz }));
        }
        else {
            return options.size;
        }
    }
    sugiyama.size = size;
    function nodeSize(sz) {
        if (sz !== undefined) {
            const { nodeSize: _, sugiNodeSize: __ } = options, rest = __rest(options, ["nodeSize", "sugiNodeSize"]);
            return buildOperator(Object.assign(Object.assign({}, rest), { nodeSize: sz, sugiNodeSize: wrapNodeSizeAccessor(sz) }));
        }
        else {
            return options.nodeSize;
        }
    }
    sugiyama.nodeSize = nodeSize;
    function sugiNodeSize(sz) {
        if (sz !== undefined) {
            const { sugiNodeSize: _, nodeSize: __ } = options, rest = __rest(options, ["sugiNodeSize", "nodeSize"]);
            return buildOperator(Object.assign(Object.assign({}, rest), { sugiNodeSize: sz, nodeSize: null }));
        }
        else {
            return options.sugiNodeSize;
        }
    }
    sugiyama.sugiNodeSize = sugiNodeSize;
    return sugiyama;
}
/** @internal */
function defaultNodeSize(node) {
    return [+(node !== undefined), 1];
}
/**
 * Construct a new {@link SugiyamaOperator} with the default settings.
 *
 * @example
 * ```typescript
 * const dag = hierarchy()(...);
 * const layout = sugiyama().nodeSize(d => d === undefined ? [0, 0] : [d.width, d.height]);
 * layout(dag);
 * for (const node of dag) {
 *   console.log(node.x, node.y);
 * }
 * ```
 */
export function sugiyama(...args) {
    if (args.length) {
        throw new Error(`got arguments to sugiyama(${args}), but constructor takes no arguments.`);
    }
    else {
        return buildOperator({
            layering: layerSimplex(),
            decross: twoLayer(),
            coord: coordSimplex(),
            size: null,
            nodeSize: defaultNodeSize,
            sugiNodeSize: wrapNodeSizeAccessor(defaultNodeSize),
        });
    }
}
